package net.dongliu.commons.sequence;

import net.dongliu.commons.collection.*;
import org.checkerframework.checker.nullness.qual.NonNull;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.*;
import java.util.function.*;
import java.util.stream.Stream;

import static java.util.Comparator.naturalOrder;
import static java.util.Objects.requireNonNull;

/**
 * A sequence of elements supporting support map and reduce operations.
 * Sequence like {@link java.util.stream.Stream}, but comes with more convenient methods.
 * Sequence not support parallel operations, so the implementation can be simpler.
 */
public interface Sequence<T> extends Iterator<T> {

    @Override
    boolean hasNext();

    @Override
    T next();

    @Deprecated
    @Override
    default void remove() {
        throw new UnsupportedOperationException("remove");
    }

    /**
     * Returns a sequence of values built from the elements of this sequence and the other sequence with the same index.
     * The resulting sequence ends as soon as the shortest input sequence ends.
     */
    static <S, T> Sequence<Pair<S, T>> zip(Sequence<@NonNull S> s1, Sequence<@NonNull T> s2) {
        return new ZippedSequence<>(s1, s2);
    }

    /**
     * Create sequence from iterator
     */
    static <T> Sequence<T> of(Iterator<T> iterator) {
        requireNonNull(iterator);
        if (!iterator.hasNext()) {
            return of();
        }
        return new IteratorSequence<>((iterator));
    }


    /**
     * Create sequence from iterable
     */
    static <T> Sequence<T> of(Iterable<T> iterable) {
        requireNonNull(iterable);
        return of(iterable.iterator());
    }

    /**
     * Create sequence from collection.
     * This method do not do defensive copy.
     */
    static <T> Sequence<T> of(Collection<T> collection) {
        requireNonNull(collection);
        if (collection.isEmpty()) {
            return of();
        }
        if (collection instanceof List && collection instanceof RandomAccess) {
            return new RandomAccessSequence<>((List<T>) collection);
        }
        return of(collection.iterator());
    }

    /**
     * Create a new Sequence from stream. The stream is then terminated.
     */
    static <T> Sequence<T> of(Stream<T> stream) {
        return Sequence.of(stream.iterator());
    }

    /**
     * Return a empty sequence
     *
     * @param <T> the element type
     */
    @SuppressWarnings("unchecked")
    static <T> Sequence<T> of() {
        return (Sequence<T>) EmptySequence.instance;
    }

    /**
     * Return a empty sequence
     *
     * @param <T> the element type
     */
    static <T> Sequence<T> of(T value) {
        return new SingletonSequence<>(value);
    }

    /**
     * Create sequence from values, or array.
     * This method do not do defensive copy.
     */
    @SafeVarargs
    static <T> Sequence<T> of(T... values) {
        return new ArraySequence<>(values);
    }

    /**
     * Return a empty sequence
     *
     * @param <T> the element type
     */
    static <T> Sequence<T> of(Optional<T> optional) {
        if (optional.isPresent()) {
            return new SingletonSequence<>(optional.get());
        }
        return of();
    }

    /**
     * Generate a sequence, from value supplier
     *
     * @param <T> the value type
     * @return a sequence
     */
    static <T> Sequence<T> generate(Supplier<T> supplier) {
        requireNonNull(supplier);
        return new SupplierGeneratedSequence<>(supplier);
    }

    /**
     * Generate a sequence, according to current index
     *
     * @param <T> the value type
     * @return a sequence
     */
    static <T> Sequence<T> generate(LongFunction<T> generator) {
        requireNonNull(generator);
        return new IndexGeneratedSequence<>(generator);
    }

    /**
     * Generate a sequence, from initial value
     *
     * @param initial   the initial value supplier
     * @param generator generate value from previous value
     * @param <T>       the value type
     * @return a sequence
     */
    static <T> Sequence<T> generate(Supplier<T> initial, UnaryOperator<T> generator) {
        requireNonNull(initial);
        requireNonNull(generator);
        return new ProgressiveGeneratedSequence<>(initial, generator);
    }

    /**
     * Return a Sequence contains elements in both Sequence.
     */
    default Sequence<T> concat(Sequence<T> sequence) {
        requireNonNull(sequence);
        if (!sequence.hasNext()) {
            return this;
        }
        if (!this.hasNext()) {
            return sequence;
        }
        return new AggregatedSequence<>(Sequence.of(this, sequence));
    }

    /**
     * Return a Sequence contains elements in all Sequences.
     */
    default Sequence<T> concat(Collection<? extends Sequence<T>> sequences) {
        requireNonNull(sequences);
        if (sequences.isEmpty()) {
            return this;
        }
        if (!this.hasNext()) {
            return new AggregatedSequence<>(sequences.iterator());
        }
        List<Sequence<T>> list = new ArrayList<>(sequences.size() + 1);
        list.add(this);
        list.addAll(sequences);
        return new AggregatedSequence<>(list.iterator());
    }

    /**
     * map operator
     */
    default <R> Sequence<R> map(Function<? super T, ? extends R> mapper) {
        if (!hasNext()) {
            return of();
        }
        return new MappedSequence<>(this, requireNonNull(mapper));
    }

    /**
     * filter operator
     */
    default Sequence<T> filter(Predicate<? super T> filter) {
        if (!hasNext()) {
            return of();
        }
        return new FilterSequence<>(this, requireNonNull(filter));
    }

    /**
     * filter operator. If element is not accepted, call consumer for it.
     */
    default Sequence<T> filterOrConsume(Predicate<? super T> filter, Consumer<T> consumer) {
        requireNonNull(filter);
        requireNonNull(consumer);
        if (!hasNext()) {
            return of();
        }
        return new FilterSequence<>(this, e -> {
            boolean flag = filter.test(e);
            if (!flag) {
                consumer.accept(e);
            }
            return flag;
        });
    }

    /**
     * flat map operator
     */
    default <R> Sequence<R> flatMap(Function<? super T, ? extends Sequence<R>> mapper) {
        if (!hasNext()) {
            return of();
        }
        Sequence<? extends Sequence<R>> sequences = map(mapper);
        return new AggregatedSequence<>(sequences);
    }

    /**
     * Return a sequence do not contains duplicated elements.
     *
     * @param keyMapper function to get a element key, to judge if elements are duplicated.
     */
    default <E> Sequence<T> distinctBy(Function<? super T, E> keyMapper) {
        requireNonNull(keyMapper);
        if (!hasNext()) {
            return of();
        }
        Set<E> set = new HashSet<>();
        return new FilterSequence<>(this, v -> set.add(keyMapper.apply(v)));
    }

    /**
     * Return a sequence do not contains duplicated elements.
     */
    default Sequence<T> distinct() {
        return distinctBy(Function.identity());
    }


    /**
     * Return a Sequence only contains non-null values.
     */
    default Sequence<T> filterNonNull() {
        return filter(Objects::nonNull);
    }

    /**
     * Drop first N elements
     *
     * @param size the elements number to drop
     */
    default Sequence<T> drop(long size) {
        Utils.checkCount(size);
        if (size == 0) {
            return this;
        }
        return new SliceSequence<>(this, size, Long.MAX_VALUE);
    }

    /**
     * Drop elements while all meet elements pass the predicate
     */
    default Sequence<T> dropWhile(Predicate<? super T> predicate) {
        requireNonNull(predicate);
        return new DropWhileSequence<>(this, predicate);
    }

    /**
     * Only take first n elements
     */
    default Sequence<T> take(long size) {
        Utils.checkCount(size);
        if (size == 0) {
            return of();
        }
        return new SliceSequence<>(this, 0, size);
    }

    /**
     * Take elements while all meet elements pass the predicate
     */
    default Sequence<T> takeWhile(Predicate<? super T> predicate) {
        requireNonNull(predicate);
        return new TakeWhileSequence<>(this, predicate);
    }

    /**
     * Return the element at index, return empty Optional if this Sequence do not have enough elements.
     * The elements in Seq should not be null.
     *
     * @param seq the element index, start from zero
     */
    default Optional<T> at(long seq) {
        long index = 0;
        while (hasNext()) {
            T value = next();
            if (index++ == seq) {
                return Optional.of(value);
            }
        }
        return Optional.empty();
    }

    /**
     * Buffer this sequence into a sequence of lists each not exceeding the given size.
     *
     * @param size the max size for buffer
     */
    default Sequence<List<T>> chunked(int size) {
        Utils.checkSize(size);
        if (!hasNext()) {
            return of();
        }
        return new ChunkedSequence<>(this, size);
    }

    /**
     * Return a sorted Sequence. A comparator is needed for sort.
     * If T is sub type of Comparable and want to use it's compare impl for sorting, {@link Comparator#naturalOrder()} can be passed in.
     *
     * @param comparator the comparator to sort sequence
     */
    default Sequence<T> sortedBy(Comparator<? super T> comparator) {
        requireNonNull(comparator);
        return new SortedSequence<>(this, comparator);
    }

    /**
     * Return a sorted Sequence, Sequence element type T should implement {@link Comparable} interface.
     * The elements should not be null, or NPE maybe thrown.
     * {@link ClassCastException}  may be thrown if T not sub type of Comparable
     */
    @SuppressWarnings("unchecked")
    default Sequence<T> sorted() {
        return sortedBy((Comparator<T>) naturalOrder());
    }

    /**
     * Consumer the remained element in this Sequence.
     *
     * @param consumer the consumer
     */
    default void forEach(Consumer<? super T> consumer) {
        requireNonNull(consumer);
        while (hasNext()) {
            consumer.accept(next());
        }
    }

    /**
     * Return a Sequence, with has a side effect when a element is take out, it is consume by specified consumer.
     */
    default Sequence<T> peek(Consumer<? super T> consumer) {
        return new PeekSequence<>(this, requireNonNull(consumer));
    }

    /**
     * reduce operator
     *
     * @param initialValue the initial value
     * @param reducer      the reducer
     * @param <R>          the result value type
     * @return a value calculate by reducer
     */
    default <R> R reduce(R initialValue, BiFunction<? super R, ? super T, ? extends R> reducer) {
        requireNonNull(reducer);
        R value = initialValue;
        while (hasNext()) {
            value = reducer.apply(value, next());
        }
        return value;
    }


    /**
     * Collect elements to collection.
     *
     * @param collectConsumer to create a Collection instance
     * @param <R>             the returned Collection type
     */
    default <R> R collect(CollectConsumer<? super T, ? extends R> collectConsumer) {
        requireNonNull(collectConsumer);
        while (hasNext()) {
            collectConsumer.accept(next());
        }
        return collectConsumer.finish();
    }


    /**
     * Collect elements to collection.
     *
     * @param collectionSupplier to create a Collection instance
     * @param <R>                the returned Collection type
     */
    default <R extends Collection<T>> R toCollection(Supplier<R> collectionSupplier) {
        requireNonNull(collectionSupplier);
        R list = collectionSupplier.get();
        while (hasNext()) {
            list.add(next());
        }
        return list;
    }

    /**
     * reduce to array list.
     */
    default ArrayList<T> toArrayList() {
        return toCollection(ArrayList::new);
    }

    /**
     * reduce to immutable List.
     */
    default List<T> toImmutableList() {
        if (!hasNext()) {
            return Lists.of();
        }
        return Collections.unmodifiableList(toArrayList());
    }

    /**
     * reduce to hash set
     */
    default HashSet<T> toHashSet() {
        return toCollection(HashSet::new);
    }

    /**
     * reduce to immutable Set.
     */
    default Set<T> toImmutableSet() {
        if (!hasNext()) {
            return Sets.of();
        }
        return Collections.unmodifiableSet(toHashSet());
    }

    /**
     * reduce to hash map
     *
     * @param keyMapper   the mapper to calculate map key
     * @param valueMapper the mapper to calculate map value
     * @param <K>         map key type
     * @param <V>         map value type
     * @return a new map
     */
    default <K, V> HashMap<K, V> toHashMap(Function<? super T, ? extends K> keyMapper,
                                           Function<? super T, ? extends V> valueMapper) {
        return toMap(HashMap::new, keyMapper, valueMapper);
    }

    /**
     * collect elements to map
     *
     * @param mapSupplier to get a map instance
     * @param keyMapper   the mapper to calculate map key
     * @param valueMapper the mapper to calculate map value
     * @param <K>         map key type
     * @param <V>         map value type
     * @return a new map
     */
    default <K, V, R extends Map<K, V>> R toMap(Supplier<R> mapSupplier,
                                                Function<? super T, ? extends K> keyMapper,
                                                Function<? super T, ? extends V> valueMapper) {
        requireNonNull(mapSupplier);
        requireNonNull(keyMapper);
        requireNonNull(valueMapper);

        R map = mapSupplier.get();
        while (hasNext()) {
            T value = next();
            map.put(keyMapper.apply(value), valueMapper.apply(value));
        }
        return map;
    }

    /**
     * collect elements to immutable Map.
     */
    default <K, V> Map<K, V> toImmutableMap(Function<? super T, ? extends K> keyMapper,
                                            Function<? super T, ? extends V> valueMapper) {
        requireNonNull(keyMapper);
        requireNonNull(valueMapper);
        if (!hasNext()) {
            return Maps.of();
        }
        return Collections.unmodifiableMap(toHashMap(keyMapper, valueMapper));
    }

    /**
     * Group the element by key mapper; for per single key, reduce elements with this key to the result value.
     *
     * @param keyMapper get key from element
     * @param initial   initial value supplier for reducing
     * @param reducer   reduce function
     * @param <K>       the group key type
     * @param <R>       the reduce result type
     * @return a map contains grouped result. The is no guaranty for the map's immutability.
     */
    default <K, R> Map<K, R> groupAndReduce(Function<? super T, ? extends K> keyMapper, Supplier<R> initial,
                                            BiFunction<? super R, ? super T, ? extends R> reducer) {
        requireNonNull(keyMapper);
        requireNonNull(reducer);
        HashMap<K, R> map = new HashMap<K, R>();
        while (hasNext()) {
            T value = next();
            K key = keyMapper.apply(value);
            if (!map.containsKey(key)) {
                map.put(key, reducer.apply(initial.get(), value));
            } else {
                map.put(key, reducer.apply(map.get(key), value));
            }
        }
        return map;
    }

    /**
     * Group the element by key mapper; for per single key, collect elements with this key to the result value.
     *
     * @param keyMapper get key from element
     * @param collector the collector supplier
     * @param <K>       the group key type
     * @param <R>       the reduce result type
     * @return a map contains grouped result. The is no guaranty for the map's immutability.
     */
    default <K, R> Map<K, R> groupAndCollect(Function<? super T, ? extends K> keyMapper,
                                             Supplier<? extends CollectConsumer<? super T, ? extends R>> collector) {
        requireNonNull(keyMapper);
        requireNonNull(collector);
        HashMap<K, CollectConsumer<? super T, ? extends R>> map = new HashMap<>();
        while (hasNext()) {
            T value = next();
            K key = keyMapper.apply(value);
            map.computeIfAbsent(key, k -> collector.get()).accept(value);
        }
        return Maps.convert(map, k -> k, CollectConsumer::finish);
    }

    /**
     * Group the element by key mapper; for per single key, reduce elements with this key to the result value.
     *
     * @param keyMapper          get key from element
     * @param collectionSupplier supplier to make a Collection instance
     * @param <K>                the group key type
     * @param <R>                the reduce result type
     * @return a map contains grouped result. The is no guaranty for the map's immutability.
     */
    default <K, R extends Collection<T>> Map<K, R> groupToCollection(Function<? super T, ? extends K> keyMapper,
                                                                     Supplier<R> collectionSupplier) {
        requireNonNull(keyMapper);
        requireNonNull(collectionSupplier);
        HashMap<K, R> map = new HashMap<K, R>();
        while (hasNext()) {
            T value = next();
            K key = keyMapper.apply(value);
            map.computeIfAbsent(key, k -> collectionSupplier.get()).add(value);
        }
        return map;
    }

    /**
     * Group the element by key mapper; for per single key, a immutable list contains all elements with this key is constructed.
     *
     * @param keyMapper get key from element
     * @param <K>       the group key type
     * @return a map contains grouped result as immutable list. The is no guaranty for the returned map's, or the list's immutability.
     */
    default <K> Map<K, List<T>> groupToList(Function<? super T, ? extends K> keyMapper) {
        requireNonNull(keyMapper);
        return groupToCollection(keyMapper, ArrayList::new);
    }

    /**
     * Partition the elements by predicate, and then do reducer for both Seq of elements.
     *
     * @param predicate predicate for partition
     * @param initial   initial value supplier for reducing
     * @param reducer   reduce function
     * @param <R>       the reduce result type
     */
    default <R> PartitionResult<R> partitionAndReduce(Predicate<? super T> predicate, Supplier<R> initial,
                                                      BiFunction<? super R, ? super T, ? extends R> reducer) {
        requireNonNull(predicate);
        requireNonNull(reducer);
        R matched = initial.get();
        R missed = initial.get();
        while (hasNext()) {
            T value = next();
            if (predicate.test(value)) {
                matched = reducer.apply(matched, value);
            } else {
                missed = reducer.apply(missed, value);
            }
        }
        return new PartitionResult<>(matched, missed);
    }

    /**
     * Partition the elements by predicate, and then collect elements.
     *
     * @param predicate predicate for partition
     * @param supplier  collection supplier
     * @param <R>       the collection type
     * @return a PartitionResult contains grouped result
     */
    default <R extends Collection<T>> PartitionResult<R> partitionAndCollect(Predicate<? super T> predicate,
                                                                             Supplier<R> supplier) {
        requireNonNull(predicate);
        requireNonNull(supplier);
        R matched = supplier.get();
        R missed = supplier.get();
        while (hasNext()) {
            T value = next();
            if (predicate.test(value)) {
                matched.add(value);
            } else {
                missed.add(value);
            }
        }
        return new PartitionResult<>(matched, missed);
    }

    /**
     * Partition the elements by predicate, and then collect elements to immutable list.
     *
     * @param predicate predicate for partition
     * @return a PartitionResult contains grouped result as list. The is no guaranty for the returned lists's immutability.
     */
    default PartitionResult<List<T>> partitionToList(Predicate<? super T> predicate) {
        PartitionResult<ArrayList<T>> result = partitionAndCollect(predicate, ArrayList::new);
        return new PartitionResult<>(result.matched(), result.missed());
    }

    /**
     * Join the elements to String out
     *
     * @param appendable the appendable to append str
     * @param sep        the separator string between elements
     * @param prefix     the prefix to prepend before all elements
     * @param suffix     the suffix to append after elements
     */
    default void joinTo(Appendable appendable, CharSequence sep, CharSequence prefix, CharSequence suffix) {
        requireNonNull(appendable);
        requireNonNull(sep);
        requireNonNull(prefix);
        requireNonNull(suffix);
        try {
            appendable.append(prefix);
            if (hasNext()) {
                T value = next();
                while (hasNext()) {
                    appendable.append(String.valueOf(value)).append(sep);
                    value = next();
                }
                appendable.append(String.valueOf(value));
            }
            appendable.append(suffix);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    /**
     * Join the elements to String
     *
     * @param sep    the separator string between elements
     * @param prefix the prefix to prepend before all elements
     * @param suffix the suffix to append after elements
     */
    default String joinToString(CharSequence sep, CharSequence prefix, CharSequence suffix) {
        requireNonNull(sep);
        requireNonNull(prefix);
        requireNonNull(suffix);

        StringBuilder sb = new StringBuilder();
        joinTo(sb, sep, prefix, suffix);
        return sb.toString();
    }

    /**
     * Join the elements to String
     *
     * @param sep the separator string between elements
     */
    default String joinToString(CharSequence sep) {
        return joinToString(sep, "", "");
    }

    /**
     * return the count of elements
     */
    default long count() {
        long count = 0;
        while (hasNext()) {
            count++;
            next();
        }
        return count;
    }

    /**
     * Get the max value by comparator. If Sequence has no element, return empty Optional.
     * If T is sub type of Comparable and want to use it's compare impl for comparing, {@link Comparator#naturalOrder()} can be passed in.
     *
     * @param comparator the comparator
     */
    default Optional<T> maxBy(Comparator<? super T> comparator) {
        if (!hasNext()) {
            return Optional.empty();
        }
        T max = next();
        while (hasNext()) {
            T value = next();
            if (comparator.compare(max, value) < 0) {
                max = value;
            }
        }
        return Optional.of(max);
    }

    /**
     * Get the max value by natural order comparator. If Sequence has no element, return empty Optional.
     * Elements in this Sequence should not be null.
     * {@link ClassCastException}  may be thrown if T not sub type of Comparable
     */
    @SuppressWarnings("unchecked")
    default Optional<T> max() {
        return maxBy((Comparator<T>) naturalOrder());
    }

    /**
     * Get the min value by comparator. If Sequence has no element, return empty Optional.
     * If T is sub type of Comparable and want to use it's compare impl for comparing, {@link Comparator#naturalOrder()} can be passed in.
     *
     * @param comparator the comparator
     */
    default Optional<T> minBy(Comparator<? super T> comparator) {
        if (!hasNext()) {
            return Optional.empty();
        }
        T min = next();
        while (hasNext()) {
            T value = next();
            if (comparator.compare(min, value) > 0) {
                min = value;
            }
        }
        return Optional.of(min);
    }

    /**
     * Get the min value by natural order comparator. If Sequence has no element, return empty Optional.
     * Elements in this Sequence should not be null.
     * {@link ClassCastException}  may be thrown if T not sub type of Comparable
     */
    @SuppressWarnings("unchecked")
    default Optional<T> min() {
        return minBy((Comparator<T>) naturalOrder());
    }

    /**
     * Sum the int values calculated by the elements.
     *
     * @param function the function convert elements to int values
     */
    default int sumInt(ToIntFunction<T> function) {
        int total = 0;
        while (hasNext()) {
            total += function.applyAsInt(next());
        }
        return total;
    }

    /**
     * Sum the long values calculated by the elements.
     *
     * @param function the function convert elements to long values
     */
    default long sumLong(ToLongFunction<T> function) {
        long total = 0;
        while (hasNext()) {
            total += function.applyAsLong(next());
        }
        return total;
    }

    /**
     * Sum the double values calculated by the elements.
     *
     * @param function the function convert elements to double values
     */
    default double sumDouble(ToDoubleFunction<T> function) {
        double total = 0;
        while (hasNext()) {
            total += function.applyAsDouble(next());
        }
        return total;
    }

    /**
     * Calculate the average value of values calculated by the elements.
     *
     * @param function the function convert elements to long values
     */
    default double averageLong(ToLongFunction<T> function) {
        long total = 0;
        long count = 0;
        while (hasNext()) {
            total += function.applyAsLong(next());
            count++;
        }
        return (double) total / count;
    }

    /**
     * Calculate the average value of values calculated by the elements.
     *
     * @param function the function convert elements to double float values
     */
    default double averageDouble(ToDoubleFunction<T> function) {
        double total = 0;
        long count = 0;
        while (hasNext()) {
            total += function.applyAsDouble(next());
            count++;
        }
        return total / count;
    }

    /**
     * Return the first element of Sequence. If sequence is empty, return empty Optional.
     * The element in sequence should not be null.
     */
    default Optional<T> first() {
        if (!hasNext()) {
            return Optional.empty();
        }
        return Optional.of(next());
    }

    /**
     * Return the last element of Sequence. If sequence is empty, return empty Optional.
     * The element in sequence should not be null.
     */
    default Optional<T> last() {
        if (!hasNext()) {
            return Optional.empty();
        }

        T value = next();
        while (hasNext()) {
            value = next();
        }
        return Optional.of(value);
    }

    /**
     * Find and return the first element meet the predicate in Sequence. If none is found, return empty Optional.
     * The element in sequence should not be null.
     */
    default Optional<T> find(Predicate<? super T> predicate) {
        requireNonNull(predicate);
        while (hasNext()) {
            T value = next();
            if (predicate.test(value)) {
                return Optional.of(value);
            }
        }
        return Optional.empty();
    }

    /**
     * Find and return the lat element meet the predicate in Sequence. If none is found, return empty Optional.
     * The element in sequence should not be null.
     */
    default Optional<T> findLast(Predicate<? super T> predicate) {
        requireNonNull(predicate);
        T value = null;
        while (hasNext()) {
            T current = next();
            if (predicate.test(current)) {
                value = requireNonNull(current);
            }
        }
        return Optional.ofNullable(value);
    }

    /**
     * If any element in this sequence meet the predicate.
     * This method would always return false if Sequence is empty.
     */
    default boolean anyMatch(Predicate<? super T> predicate) {
        requireNonNull(predicate);
        while (hasNext()) {
            if (predicate.test(next())) {
                return true;
            }
        }
        return false;
    }

    /**
     * If all elements in this sequence meet the predicate.
     * This method would always return true if Sequence is empty.
     */
    default boolean allMatch(Predicate<? super T> predicate) {
        requireNonNull(predicate);
        while (hasNext()) {
            if (!predicate.test(next())) {
                return false;
            }
        }
        return true;
    }

    /**
     * If no elements in this sequence meet the predicate.
     * This method would always return true if Sequence is empty.
     */
    default boolean noneMatch(Predicate<? super T> predicate) {
        requireNonNull(predicate);
        while (hasNext()) {
            if (predicate.test(next())) {
                return false;
            }
        }
        return true;
    }

    /**
     * Return a iterable that wrap this sequence, and can iterate only once.
     */
    default Iterable<T> asIterable() {
        return () -> this;
    }

    /**
     * Return a Stream that wrap this sequence.
     */
    default Stream<T> asStream() {
        if (!hasNext()) {
            return Stream.empty();
        }
        return Iterators.stream(this);
    }

}
