/*
	Apache License
	Version 2.0, January 2004
	http://www.apache.org/licenses/

	TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION

	1. Definitions.

	"License" shall mean the terms and conditions for use, reproduction, and
	distribution as defined by Sections 1 through 9 of this document.

	"Licensor" shall mean the copyright owner or entity authorized by the copyright
	owner that is granting the License.

	"Legal Entity" shall mean the union of the acting entity and all other entities
	that control, are controlled by, or are under common control with that entity.
	For the purposes of this definition, "control" means (i) the power, direct or
	indirect, to cause the direction or management of such entity, whether by
	contract or otherwise, or (ii) ownership of fifty percent (50%) or more of the
	outstanding shares, or (iii) beneficial ownership of such entity.

	"You" (or "Your") shall mean an individual or Legal Entity exercising
	permissions granted by this License.

	"Source" form shall mean the preferred form for making modifications, including
	but not limited to software source code, documentation source, and configuration
	files.

	"Object" form shall mean any form resulting from mechanical transformation or
	translation of a Source form, including but not limited to compiled object code,
	generated documentation, and conversions to other media types.

	"Work" shall mean the work of authorship, whether in Source or Object form, made
	available under the License, as indicated by a copyright notice that is included
	in or attached to the work (an example is provided in the Appendix below).

	"Derivative Works" shall mean any work, whether in Source or Object form, that
	is based on (or derived from) the Work and for which the editorial revisions,
	annotations, elaborations, or other modifications represent, as a whole, an
	original work of authorship. For the purposes of this License, Derivative Works
	shall not include works that remain separable from, or merely link (or bind by
	name) to the interfaces of, the Work and Derivative Works thereof.

	"Contribution" shall mean any work of authorship, including the original version
	of the Work and any modifications or additions to that Work or Derivative Works
	thereof, that is intentionally submitted to Licensor for inclusion in the Work
	by the copyright owner or by an individual or Legal Entity authorized to submit
	on behalf of the copyright owner. For the purposes of this definition,
	"submitted" means any form of electronic, verbal, or written communication sent
	to the Licensor or its representatives, including but not limited to
	communication on electronic mailing lists, source code control systems, and
	issue tracking systems that are managed by, or on behalf of, the Licensor for
	the purpose of discussing and improving the Work, but excluding communication
	that is conspicuously marked or otherwise designated in writing by the copyright
	owner as "Not a Contribution."

	"Contributor" shall mean Licensor and any individual or Legal Entity on behalf
	of whom a Contribution has been received by Licensor and subsequently
	incorporated within the Work.

	2. Grant of Copyright License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable copyright license to reproduce, prepare Derivative Works of,
	publicly display, publicly perform, sublicense, and distribute the Work and such
	Derivative Works in Source or Object form.

	3. Grant of Patent License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable (except as stated in this section) patent license to make, have
	made, use, offer to sell, sell, import, and otherwise transfer the Work, where
	such license applies only to those patent claims licensable by such Contributor
	that are necessarily infringed by their Contribution(s) alone or by combination
	of their Contribution(s) with the Work to which such Contribution(s) was
	submitted. If You institute patent litigation against any entity (including a
	cross-claim or counterclaim in a lawsuit) alleging that the Work or a
	Contribution incorporated within the Work constitutes direct or contributory
	patent infringement, then any patent licenses granted to You under this License
	for that Work shall terminate as of the date such litigation is filed.

	4. Redistribution.

	You may reproduce and distribute copies of the Work or Derivative Works thereof
	in any medium, with or without modifications, and in Source or Object form,
	provided that You meet the following conditions:

	You must give any other recipients of the Work or Derivative Works a copy of
	this License; and
	You must cause any modified files to carry prominent notices stating that You
	changed the files; and
	You must retain, in the Source form of any Derivative Works that You distribute,
	all copyright, patent, trademark, and attribution notices from the Source form
	of the Work, excluding those notices that do not pertain to any part of the
	Derivative Works; and
	If the Work includes a "NOTICE" text file as part of its distribution, then any
	Derivative Works that You distribute must include a readable copy of the
	attribution notices contained within such NOTICE file, excluding those notices
	that do not pertain to any part of the Derivative Works, in at least one of the
	following places: within a NOTICE text file distributed as part of the
	Derivative Works; within the Source form or documentation, if provided along
	with the Derivative Works; or, within a display generated by the Derivative
	Works, if and wherever such third-party notices normally appear. The contents of
	the NOTICE file are for informational purposes only and do not modify the
	License. You may add Your own attribution notices within Derivative Works that
	You distribute, alongside or as an addendum to the NOTICE text from the Work,
	provided that such additional attribution notices cannot be construed as
	modifying the License.
	You may add Your own copyright statement to Your modifications and may provide
	additional or different license terms and conditions for use, reproduction, or
	distribution of Your modifications, or for any such Derivative Works as a whole,
	provided Your use, reproduction, and distribution of the Work otherwise complies
	with the conditions stated in this License.

	5. Submission of Contributions.

	Unless You explicitly state otherwise, any Contribution intentionally submitted
	for inclusion in the Work by You to the Licensor shall be under the terms and
	conditions of this License, without any additional terms or conditions.
	Notwithstanding the above, nothing herein shall supersede or modify the terms of
	any separate license agreement you may have executed with Licensor regarding
	such Contributions.

	6. Trademarks.

	This License does not grant permission to use the trade names, trademarks,
	service marks, or product names of the Licensor, except as required for
	reasonable and customary use in describing the origin of the Work and
	reproducing the content of the NOTICE file.

	7. Disclaimer of Warranty.

	Unless required by applicable law or agreed to in writing, Licensor provides the
	Work (and each Contributor provides its Contributions) on an "AS IS" BASIS,
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied,
	including, without limitation, any warranties or conditions of TITLE,
	NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A PARTICULAR PURPOSE. You are
	solely responsible for determining the appropriateness of using or
	redistributing the Work and assume any risks associated with Your exercise of
	permissions under this License.

	8. Limitation of Liability.

	In no event and under no legal theory, whether in tort (including negligence),
	contract, or otherwise, unless required by applicable law (such as deliberate
	and grossly negligent acts) or agreed to in writing, shall any Contributor be
	liable to You for damages, including any direct, indirect, special, incidental,
	or consequential damages of any character arising as a result of this License or
	out of the use or inability to use the Work (including but not limited to
	damages for loss of goodwill, work stoppage, computer failure or malfunction, or
	any and all other commercial damages or losses), even if such Contributor has
	been advised of the possibility of such damages.

	9. Accepting Warranty or Additional Liability.

	While redistributing the Work or Derivative Works thereof, You may choose to
	offer, and charge a fee for, acceptance of support, warranty, indemnity, or
	other liability obligations and/or rights consistent with this License. However,
	in accepting such obligations, You may act only on Your own behalf and on Your
	sole responsibility, not on behalf of any other Contributor, and only if You
	agree to indemnify, defend, and hold each Contributor harmless for any liability
	incurred by, or claims asserted against, such Contributor by reason of your
	accepting any such warranty or additional liability.

	END OF TERMS AND CONDITIONS

	APPENDIX: How to apply the Apache License to your work

	To apply the Apache License to your work, attach the following boilerplate
	notice, with the fields enclosed by brackets "{}" replaced with your own
	identifying information. (Don't include the brackets!) The text should be
	enclosed in the appropriate comment syntax for the file format. We also
	recommend that a file or class name and description of purpose be included on
	the same "printed page" as the copyright notice for easier identification within
	third-party archives.

	   Copyright 2018 JFinal

	   Licensed under the Apache License, Version 2.0 (the "License");
	   you may not use this file except in compliance with the License.
	   You may obtain a copy of the License at

	     http://www.apache.org/licenses/LICENSE-2.0

	   Unless required by applicable law or agreed to in writing, software
	   distributed under the License is distributed on an "AS IS" BASIS,
	   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	   See the License for the specific language governing permissions and
	   limitations under the License.
*/

package net.dreamlu.iot.mqtt.core.server.http.core;

import net.dreamlu.iot.mqtt.core.server.MqttConst;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tio.core.ChannelContext;
import org.tio.core.Tio;
import org.tio.core.TioConfig;
import org.tio.core.exception.TioDecodeException;
import org.tio.core.intf.Packet;
import org.tio.http.common.*;
import org.tio.http.common.handler.HttpRequestHandler;
import org.tio.server.intf.TioServerHandler;
import org.tio.utils.hutool.StrUtil;
import org.tio.websocket.common.*;
import org.tio.websocket.common.util.BASE64Util;
import org.tio.websocket.common.util.SHA1Util;
import org.tio.websocket.server.handler.IWsMsgHandler;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.util.*;

/**
 * websocket 和 http 共存
 *
 * @author tanyaowu
 * @author L.cm
 */
public class MqttWebServerAioHandler implements TioServerHandler {
	private static final Logger log = LoggerFactory.getLogger(MqttWebServerAioHandler.class);
	/**
	 * value: List<WsRequest>
	 */
	private static final String NOT_FINAL_WEBSOCKET_PACKET_PARTS = "TIO_N_F_W_P_P";
	/**
	 * SEC_WEBSOCKET_KEY 后缀
	 */
	private static final String SEC_WEBSOCKET_KEY_SUFFIX = "258EAFA5-E914-47DA-95CA-C5AB0DC85B11";
	private static final byte[] SEC_WEBSOCKET_KEY_SUFFIX_BYTES = SEC_WEBSOCKET_KEY_SUFFIX.getBytes();
	/**
	 * websocket子协议
	 */
	private static final String SEC_WEBSOCKET_PROTOCOL = "sec-websocket-protocol";
	private static final HeaderName HEADER_NAME_SEC_WEBSOCKET_PROTOCOL = HeaderName.from(SEC_WEBSOCKET_PROTOCOL);
	private final HttpConfig httpConfig;
	private final HttpRequestHandler requestHandler;
	private final IWsMsgHandler wsMsgHandler;

	public MqttWebServerAioHandler(HttpConfig httpConfig,
								   HttpRequestHandler requestHandler,
								   IWsMsgHandler wsMsgHandler) {
		this.httpConfig = httpConfig;
		this.requestHandler = requestHandler;
		this.wsMsgHandler = wsMsgHandler;
	}

	@SuppressWarnings("unchecked")
	@Override
	public Packet decode(ByteBuffer buffer, int limit, int position, int readableLength, ChannelContext context) throws TioDecodeException {
		WsSessionContext wsSessionContext = (WsSessionContext) context.get();
		// 尚未握手
		if (wsSessionContext == null) {
			HttpRequest request = HttpRequestDecoder.decode(buffer, limit, position, readableLength, context, httpConfig);
			if (request == null) {
				return null;
			}
			HttpResponse httpResponse = updateWebSocketProtocol(request);
			// 普通 http 非 websocket
			if (httpResponse == null) {
				context.set(MqttConst.IS_HTTP, (byte) 1);
				return request;
			}
			wsSessionContext = new WsSessionContext();
			context.set(wsSessionContext);
			wsSessionContext.setHandshakeRequest(request);
			wsSessionContext.setHandshakeResponse(httpResponse);
			WsRequest wsRequestPacket = new WsRequest();
			wsRequestPacket.setHandShake(true);
			return wsRequestPacket;
		}
		WsRequest websocketPacket = WsServerDecoder.decode(buffer, context);
		if (websocketPacket != null) {
			// 数据包尚未完成
			if (!websocketPacket.isWsEof()) {
				List<WsRequest> parts = (List<WsRequest>) context.get(NOT_FINAL_WEBSOCKET_PACKET_PARTS);
				if (parts == null) {
					parts = new ArrayList<>();
					context.set(NOT_FINAL_WEBSOCKET_PACKET_PARTS, parts);
				}
				parts.add(websocketPacket);
			} else {
				List<WsRequest> parts = (List<WsRequest>) context.get(NOT_FINAL_WEBSOCKET_PACKET_PARTS);
				if (parts != null) {
					context.set(NOT_FINAL_WEBSOCKET_PACKET_PARTS, null);

					parts.add(websocketPacket);
					WsRequest first = parts.get(0);
					websocketPacket.setWsOpcode(first.getWsOpcode());

					int allBodyLength = 0;
					for (WsRequest wsRequest : parts) {
						allBodyLength += wsRequest.getBody().length;
					}

					byte[] allBody = new byte[allBodyLength];
					int index = 0;
					for (WsRequest wsRequest : parts) {
						System.arraycopy(wsRequest.getBody(), 0, allBody, index, wsRequest.getBody().length);
						index += wsRequest.getBody().length;
					}
					websocketPacket.setBody(allBody);
				}
				HttpRequest handshakeRequest = wsSessionContext.getHandshakeRequest();
				if (websocketPacket.getWsOpcode() != Opcode.BINARY) {
					byte[] bodyBs = websocketPacket.getBody();
					if (bodyBs != null) {
						try {
							String text = new String(bodyBs, handshakeRequest.getCharset());
							websocketPacket.setWsBodyText(text);
						} catch (UnsupportedEncodingException e) {
							log.error(e.toString(), e);
						}
					}
				}
			}
		}
		return websocketPacket;
	}

	@Override
	public ByteBuffer encode(Packet packet, TioConfig tioConfig, ChannelContext channelContext) {
		if (packet == null) {
			return null;
		} else if (packet instanceof HttpResponse) {
			try {
				return HttpResponseEncoder.encode((HttpResponse) packet, tioConfig, channelContext);
			} catch (UnsupportedEncodingException e) {
				log.error(e.toString(), e);
				return null;
			}
		}
		// 处理 websocket 子协议
		WsResponse wsResponse;
		if (packet instanceof WsResponse) {
			wsResponse = (WsResponse) packet;
		} else {
			wsResponse = wsMsgHandler.encodeSubProtocol(packet, tioConfig, channelContext);
			Objects.requireNonNull(wsResponse, "IWsMsgHandler encodeSubProtocol WsResponse is null.");
		}
		// 握手包
		if (wsResponse.isHandShake()) {
			WsSessionContext imSessionContext = (WsSessionContext) channelContext.get();
			HttpResponse handshakeResponse = imSessionContext.getHandshakeResponse();
			try {
				return HttpResponseEncoder.encode(handshakeResponse, tioConfig, channelContext);
			} catch (UnsupportedEncodingException e) {
				log.error(e.toString(), e);
				return null;
			}
		}
		return WsServerEncoder.encode(wsResponse, tioConfig, channelContext);
	}

	private WsResponse handlerWs(WsRequest websocketPacket, byte[] bytes, Opcode opcode, ChannelContext channelContext) throws Exception {
		if (opcode == Opcode.TEXT) {
			if (bytes == null || bytes.length == 0) {
				Tio.remove(channelContext, "错误的websocket包，body为空");
				return null;
			}
			String text = new String(bytes, httpConfig.getCharset());
			Object retObj = wsMsgHandler.onText(websocketPacket, text, channelContext);
			String methodName = "onText";
			return processRetObj(retObj, methodName, channelContext);
		} else if (opcode == Opcode.BINARY) {
			if (bytes == null || bytes.length == 0) {
				Tio.remove(channelContext, "错误的websocket包，body为空");
				return null;
			}
			Object retObj = wsMsgHandler.onBytes(websocketPacket, bytes, channelContext);
			String methodName = "onBytes";
			return processRetObj(retObj, methodName, channelContext);
		} else if (opcode == Opcode.PING || opcode == Opcode.PONG) {
			log.debug("收到{}", opcode);
			return null;
		} else if (opcode == Opcode.CLOSE) {
			Object retObj = wsMsgHandler.onClose(websocketPacket, bytes, channelContext);
			String methodName = "onClose";
			return processRetObj(retObj, methodName, channelContext);
		} else {
			Tio.remove(channelContext, "错误的websocket包，错误的Opcode");
			return null;
		}
	}

	@Override
	public void handler(Packet packet, ChannelContext channelContext) throws Exception {
		if (packet instanceof HttpRequest) {
			HttpRequest request = (HttpRequest) packet;
			String ip = request.getClientIp();
			if (channelContext.tioConfig.ipBlacklist.isInBlacklist(ip)) {
				HttpResponse httpResponse = request.httpConfig.getRespForBlackIp();
				if (httpResponse != null) {
					Tio.send(channelContext, httpResponse);
				} else {
					Tio.remove(channelContext, ip + "在黑名单中");
				}
				return;
			}
			HttpResponse httpResponse = requestHandler.handler(request);
			if (httpResponse != null) {
				Tio.send(channelContext, httpResponse);
			} else {
				if (log.isInfoEnabled()) {
					log.info("{}, {}, handler return null, request line: {}", channelContext.tioConfig.getName(), channelContext, request.getRequestLine());
				}
				request.close("handler return null");
			}
			return;
		}
		WsRequest wsRequest = (WsRequest) packet;
		// 判断握手包
		if (wsRequest.isHandShake()) {
			WsSessionContext wsSessionContext = (WsSessionContext) channelContext.get();
			HttpRequest request = wsSessionContext.getHandshakeRequest();
			HttpResponse httpResponse = wsSessionContext.getHandshakeResponse();
			HttpResponse response = wsMsgHandler.handshake(request, httpResponse, channelContext);
			if (response == null) {
				Tio.remove(channelContext, "业务层不同意握手");
				return;
			}
			wsSessionContext.setHandshakeResponse(response);
			WsResponse wsResponse = new WsResponse();
			wsResponse.setHandShake(true);
			Tio.send(channelContext, wsResponse);
			wsSessionContext.setHandshaked(true);
			wsMsgHandler.onAfterHandshaked(request, httpResponse, channelContext);
			return;
		}
		if (!wsRequest.isWsEof()) {
			return;
		}
		WsResponse wsResponse = handlerWs(wsRequest, wsRequest.getBody(), wsRequest.getWsOpcode(), channelContext);
		if (wsResponse != null) {
			Tio.send(channelContext, wsResponse);
		}
	}

	private WsResponse processRetObj(Object obj, String methodName, ChannelContext channelContext) {
		if (obj == null) {
			return null;
		} else {
			if (obj instanceof String) {
				return WsResponse.fromText((String) obj, httpConfig.getCharset());
			} else if (obj instanceof byte[]) {
				return WsResponse.fromBytes((byte[]) obj);
			} else if (obj instanceof WsResponse) {
				return (WsResponse) obj;
			} else if (obj instanceof ByteBuffer) {
				return WsResponse.fromBytes(((ByteBuffer) obj).array());
			} else {
				log.error("{} {}.{}()方法，只允许返回byte[]、ByteBuffer、WsResponse或null，但是程序返回了{}", channelContext, this.getClass().getName(), methodName, obj.getClass().getName());
				return null;
			}
		}
	}

	/**
	 * 本方法改编自baseio: https://gitee.com/generallycloud/baseio<br>
	 * 感谢开源作者的付出
	 *
	 * @param request HttpRequest
	 * @return HttpResponse
	 * @throws TioDecodeException TioDecodeException
	 * @author tanyaowu
	 */
	public HttpResponse updateWebSocketProtocol(HttpRequest request) throws TioDecodeException {
		Map<String, String> headers = request.getHeaders();
		String secWebSocketKey = headers.get(HttpConst.RequestHeaderKey.Sec_WebSocket_Key);
		if (StrUtil.isNotBlank(secWebSocketKey)) {
			byte[] secWebSocketKeyBytes;
			try {
				secWebSocketKeyBytes = secWebSocketKey.getBytes(request.getCharset());
			} catch (UnsupportedEncodingException e) {
				throw new TioDecodeException(e);
			}
			byte[] allBs = new byte[secWebSocketKeyBytes.length + SEC_WEBSOCKET_KEY_SUFFIX_BYTES.length];
			System.arraycopy(secWebSocketKeyBytes, 0, allBs, 0, secWebSocketKeyBytes.length);
			System.arraycopy(SEC_WEBSOCKET_KEY_SUFFIX_BYTES, 0, allBs, secWebSocketKeyBytes.length, SEC_WEBSOCKET_KEY_SUFFIX_BYTES.length);
			byte[] keyArray = SHA1Util.SHA1(allBs);
			String acceptKey = BASE64Util.byteArrayToBase64(keyArray);
			HttpResponse httpResponse = new HttpResponse(request);
			// 101 协议转换
			httpResponse.setStatus(HttpResponseStatus.C101);
			Map<HeaderName, HeaderValue> respHeaders = new HashMap<>();
			respHeaders.put(HeaderName.Connection, HeaderValue.Connection.Upgrade);
			respHeaders.put(HeaderName.Upgrade, HeaderValue.Upgrade.WebSocket);
			respHeaders.put(HeaderName.Sec_WebSocket_Accept, HeaderValue.from(acceptKey));
			// websocket 子协议协商
			String[] supportedSubProtocols = wsMsgHandler.getSupportedSubProtocols();
			if (supportedSubProtocols != null && supportedSubProtocols.length > 0) {
				String requestedSubProtocols = headers.get(SEC_WEBSOCKET_PROTOCOL);
				String selectSubProtocol = selectSubProtocol(requestedSubProtocols, supportedSubProtocols);
				if (selectSubProtocol != null) {
					respHeaders.put(HEADER_NAME_SEC_WEBSOCKET_PROTOCOL, HeaderValue.from(selectSubProtocol));
				}
			}
			httpResponse.addHeaders(respHeaders);
			return httpResponse;
		}
		return null;
	}

	/**
	 * Selects the first matching supported sub protocol
	 *
	 * @param requestedSubProtocols 请求中的子协议
	 * @param subProtocols          系统支持得子协议，注意：不支持 * 通配
	 * @return First matching supported sub protocol. Null if not found.
	 */
	private static String selectSubProtocol(String requestedSubProtocols, String[] subProtocols) {
		if (requestedSubProtocols == null || subProtocols == null || subProtocols.length == 0) {
			return null;
		}
		String[] requestedSubProtocolArray = requestedSubProtocols.split(",");
		for (String p : requestedSubProtocolArray) {
			String requestedSubProtocol = p.trim();
			for (String supportedSubProtocol : subProtocols) {
				if (requestedSubProtocol.equals(supportedSubProtocol)) {
					return requestedSubProtocol;
				}
			}
		}
		// No match found
		return null;
	}

}
