/*
	Apache License
	Version 2.0, January 2004
	http://www.apache.org/licenses/
	
	TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
	
	1. Definitions.
	
	"License" shall mean the terms and conditions for use, reproduction, and
	distribution as defined by Sections 1 through 9 of this document.
	
	"Licensor" shall mean the copyright owner or entity authorized by the copyright
	owner that is granting the License.
	
	"Legal Entity" shall mean the union of the acting entity and all other entities
	that control, are controlled by, or are under common control with that entity.
	For the purposes of this definition, "control" means (i) the power, direct or
	indirect, to cause the direction or management of such entity, whether by
	contract or otherwise, or (ii) ownership of fifty percent (50%) or more of the
	outstanding shares, or (iii) beneficial ownership of such entity.
	
	"You" (or "Your") shall mean an individual or Legal Entity exercising
	permissions granted by this License.
	
	"Source" form shall mean the preferred form for making modifications, including
	but not limited to software source code, documentation source, and configuration
	files.
	
	"Object" form shall mean any form resulting from mechanical transformation or
	translation of a Source form, including but not limited to compiled object code,
	generated documentation, and conversions to other media types.
	
	"Work" shall mean the work of authorship, whether in Source or Object form, made
	available under the License, as indicated by a copyright notice that is included
	in or attached to the work (an example is provided in the Appendix below).
	
	"Derivative Works" shall mean any work, whether in Source or Object form, that
	is based on (or derived from) the Work and for which the editorial revisions,
	annotations, elaborations, or other modifications represent, as a whole, an
	original work of authorship. For the purposes of this License, Derivative Works
	shall not include works that remain separable from, or merely link (or bind by
	name) to the interfaces of, the Work and Derivative Works thereof.
	
	"Contribution" shall mean any work of authorship, including the original version
	of the Work and any modifications or additions to that Work or Derivative Works
	thereof, that is intentionally submitted to Licensor for inclusion in the Work
	by the copyright owner or by an individual or Legal Entity authorized to submit
	on behalf of the copyright owner. For the purposes of this definition,
	"submitted" means any form of electronic, verbal, or written communication sent
	to the Licensor or its representatives, including but not limited to
	communication on electronic mailing lists, source code control systems, and
	issue tracking systems that are managed by, or on behalf of, the Licensor for
	the purpose of discussing and improving the Work, but excluding communication
	that is conspicuously marked or otherwise designated in writing by the copyright
	owner as "Not a Contribution."
	
	"Contributor" shall mean Licensor and any individual or Legal Entity on behalf
	of whom a Contribution has been received by Licensor and subsequently
	incorporated within the Work.
	
	2. Grant of Copyright License.
	
	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable copyright license to reproduce, prepare Derivative Works of,
	publicly display, publicly perform, sublicense, and distribute the Work and such
	Derivative Works in Source or Object form.
	
	3. Grant of Patent License.
	
	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable (except as stated in this section) patent license to make, have
	made, use, offer to sell, sell, import, and otherwise transfer the Work, where
	such license applies only to those patent claims licensable by such Contributor
	that are necessarily infringed by their Contribution(s) alone or by combination
	of their Contribution(s) with the Work to which such Contribution(s) was
	submitted. If You institute patent litigation against any entity (including a
	cross-claim or counterclaim in a lawsuit) alleging that the Work or a
	Contribution incorporated within the Work constitutes direct or contributory
	patent infringement, then any patent licenses granted to You under this License
	for that Work shall terminate as of the date such litigation is filed.
	
	4. Redistribution.
	
	You may reproduce and distribute copies of the Work or Derivative Works thereof
	in any medium, with or without modifications, and in Source or Object form,
	provided that You meet the following conditions:
	
	You must give any other recipients of the Work or Derivative Works a copy of
	this License; and
	You must cause any modified files to carry prominent notices stating that You
	changed the files; and
	You must retain, in the Source form of any Derivative Works that You distribute,
	all copyright, patent, trademark, and attribution notices from the Source form
	of the Work, excluding those notices that do not pertain to any part of the
	Derivative Works; and
	If the Work includes a "NOTICE" text file as part of its distribution, then any
	Derivative Works that You distribute must include a readable copy of the
	attribution notices contained within such NOTICE file, excluding those notices
	that do not pertain to any part of the Derivative Works, in at least one of the
	following places: within a NOTICE text file distributed as part of the
	Derivative Works; within the Source form or documentation, if provided along
	with the Derivative Works; or, within a display generated by the Derivative
	Works, if and wherever such third-party notices normally appear. The contents of
	the NOTICE file are for informational purposes only and do not modify the
	License. You may add Your own attribution notices within Derivative Works that
	You distribute, alongside or as an addendum to the NOTICE text from the Work,
	provided that such additional attribution notices cannot be construed as
	modifying the License.
	You may add Your own copyright statement to Your modifications and may provide
	additional or different license terms and conditions for use, reproduction, or
	distribution of Your modifications, or for any such Derivative Works as a whole,
	provided Your use, reproduction, and distribution of the Work otherwise complies
	with the conditions stated in this License.
	
	5. Submission of Contributions.
	
	Unless You explicitly state otherwise, any Contribution intentionally submitted
	for inclusion in the Work by You to the Licensor shall be under the terms and
	conditions of this License, without any additional terms or conditions.
	Notwithstanding the above, nothing herein shall supersede or modify the terms of
	any separate license agreement you may have executed with Licensor regarding
	such Contributions.
	
	6. Trademarks.
	
	This License does not grant permission to use the trade names, trademarks,
	service marks, or product names of the Licensor, except as required for
	reasonable and customary use in describing the origin of the Work and
	reproducing the content of the NOTICE file.
	
	7. Disclaimer of Warranty.
	
	Unless required by applicable law or agreed to in writing, Licensor provides the
	Work (and each Contributor provides its Contributions) on an "AS IS" BASIS,
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied,
	including, without limitation, any warranties or conditions of TITLE,
	NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A PARTICULAR PURPOSE. You are
	solely responsible for determining the appropriateness of using or
	redistributing the Work and assume any risks associated with Your exercise of
	permissions under this License.
	
	8. Limitation of Liability.
	
	In no event and under no legal theory, whether in tort (including negligence),
	contract, or otherwise, unless required by applicable law (such as deliberate
	and grossly negligent acts) or agreed to in writing, shall any Contributor be
	liable to You for damages, including any direct, indirect, special, incidental,
	or consequential damages of any character arising as a result of this License or
	out of the use or inability to use the Work (including but not limited to
	damages for loss of goodwill, work stoppage, computer failure or malfunction, or
	any and all other commercial damages or losses), even if such Contributor has
	been advised of the possibility of such damages.
	
	9. Accepting Warranty or Additional Liability.
	
	While redistributing the Work or Derivative Works thereof, You may choose to
	offer, and charge a fee for, acceptance of support, warranty, indemnity, or
	other liability obligations and/or rights consistent with this License. However,
	in accepting such obligations, You may act only on Your own behalf and on Your
	sole responsibility, not on behalf of any other Contributor, and only if You
	agree to indemnify, defend, and hold each Contributor harmless for any liability
	incurred by, or claims asserted against, such Contributor by reason of your
	accepting any such warranty or additional liability.
	
	END OF TERMS AND CONDITIONS
	
	APPENDIX: How to apply the Apache License to your work
	
	To apply the Apache License to your work, attach the following boilerplate
	notice, with the fields enclosed by brackets "{}" replaced with your own
	identifying information. (Don't include the brackets!) The text should be
	enclosed in the appropriate comment syntax for the file format. We also
	recommend that a file or class name and description of purpose be included on
	the same "printed page" as the copyright notice for easier identification within
	third-party archives.
	
	   Copyright 2020 t-io
	
	   Licensed under the Apache License, Version 2.0 (the "License");
	   you may not use this file except in compliance with the License.
	   You may obtain a copy of the License at
	
	     http://www.apache.org/licenses/LICENSE-2.0
	
	   Unless required by applicable law or agreed to in writing, software
	   distributed under the License is distributed on an "AS IS" BASIS,
	   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	   See the License for the specific language governing permissions and
	   limitations under the License.
*/
package org.tio.utils.hutool;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author tanyaowu 
 * 2018年8月5日 下午8:34:06
 */
public class ReUtil {

	/** 正则表达式匹配中文汉字 */
	public final static String	RE_CHINESE	= "[\u4E00-\u9FFF]";
	/** 正则表达式匹配中文字符串 */
	public final static String	RE_CHINESES	= RE_CHINESE + "+";

	/** 正则中需要被转义的关键字 */
	public final static Set<Character> RE_KEYS = new HashSet<>();
	static {
		RE_KEYS.add('$');

		RE_KEYS.add('(');
		RE_KEYS.add(')');
		RE_KEYS.add('*');
		RE_KEYS.add('+');
		RE_KEYS.add('.');
		RE_KEYS.add('[');
		RE_KEYS.add(']');
		RE_KEYS.add('?');
		RE_KEYS.add('\\');
		RE_KEYS.add('^');
		RE_KEYS.add('{');
		RE_KEYS.add('}');
		RE_KEYS.add('|');
	}

	/**
	 * 获得匹配的字符串，，获得正则中分组0的内容
	 * 
	 * @param pattern 编译后的正则模式
	 * @param content 被匹配的内容
	 * @return 匹配后得到的字符串，未匹配返回null
	 * @since 3.1.2
	 */
	public static String getGroup0(Pattern pattern, String content) {
		return get(pattern, content, 0);
	}

	/**
	 * 获得匹配的字符串，，获得正则中分组1的内容
	 * 
	 * @param pattern 编译后的正则模式
	 * @param content 被匹配的内容
	 * @return 匹配后得到的字符串，未匹配返回null
	 * @since 3.1.2
	 */
	public static String getGroup1(Pattern pattern, String content) {
		return get(pattern, content, 1);
	}

	/**
	 * 获得匹配的字符串
	 * 
	 * @param pattern 编译后的正则模式
	 * @param content 被匹配的内容
	 * @param groupIndex 匹配正则的分组序号
	 * @return 匹配后得到的字符串，未匹配返回null
	 */
	public static String get(Pattern pattern, String content, int groupIndex) {
		if (null == content || null == pattern) {
			return null;
		}

		final Matcher matcher = pattern.matcher(content);
		if (matcher.find()) {
			return matcher.group(groupIndex);
		}
		return null;
	}

	/**
	 * 获得匹配的字符串匹配到的所有分组
	 * 
	 * @param pattern 编译后的正则模式
	 * @param content 被匹配的内容
	 * @return 匹配后得到的字符串数组，按照分组顺序依次列出，未匹配到返回空列表，任何一个参数为null返回null
	 * @since 3.1.0
	 */
	public static List<String> getAllGroups(Pattern pattern, String content) {
		return getAllGroups(pattern, content, true);
	}

	/**
	 * 获得匹配的字符串匹配到的所有分组
	 * 
	 * @param pattern 编译后的正则模式
	 * @param content 被匹配的内容
	 * @param withGroup0 是否包括分组0，此分组表示全匹配的信息
	 * @return 匹配后得到的字符串数组，按照分组顺序依次列出，未匹配到返回空列表，任何一个参数为null返回null
	 * @since 4.0.13
	 */
	public static List<String> getAllGroups(Pattern pattern, String content, boolean withGroup0) {
		if (null == content || null == pattern) {
			return null;
		}

		ArrayList<String> result = new ArrayList<>();
		final Matcher matcher = pattern.matcher(content);
		if (matcher.find()) {
			final int startGroup = withGroup0 ? 0 : 1;
			final int groupCount = matcher.groupCount();
			for (int i = startGroup; i <= groupCount; i++) {
				result.add(matcher.group(i));
			}
		}
		return result;
	}

	/**
	 * 删除匹配的第一个内容
	 * 
	 * @param pattern 正则
	 * @param content 被匹配的内容
	 * @return 删除后剩余的内容
	 */
	public static String delFirst(Pattern pattern, String content) {
		if (null == pattern || StrUtil.isBlank(content)) {
			return content;
		}

		return pattern.matcher(content).replaceFirst(StrUtil.EMPTY);
	}

	/**
	 * 删除匹配的全部内容
	 * 
	 * @param pattern 正则
	 * @param content 被匹配的内容
	 * @return 删除后剩余的内容
	 */
	public static String delAll(Pattern pattern, String content) {
		if (null == pattern || StrUtil.isBlank(content)) {
			return content;
		}

		return pattern.matcher(content).replaceAll(StrUtil.EMPTY);
	}

	/**
	 * 取得内容中匹配的所有结果，获得匹配的所有结果中正则对应分组0的内容
	 * 
	 * @param regex 正则
	 * @param content 被查找的内容
	 * @return 结果列表
	 * @since 3.1.2
	 */
	public static List<String> findAllGroup0(String regex, String content) {
		return findAll(regex, content, 0);
	}

	/**
	 * 取得内容中匹配的所有结果，获得匹配的所有结果中正则对应分组1的内容
	 * 
	 * @param regex 正则
	 * @param content 被查找的内容
	 * @return 结果列表
	 * @since 3.1.2
	 */
	public static List<String> findAllGroup1(String regex, String content) {
		return findAll(regex, content, 1);
	}

	/**
	 * 取得内容中匹配的所有结果
	 * 
	 * @param regex 正则
	 * @param content 被查找的内容
	 * @param group 正则的分组
	 * @return 结果列表
	 * @since 3.0.6
	 */
	public static List<String> findAll(String regex, String content, int group) {
		return findAll(regex, content, group, new ArrayList<String>());
	}

	/**
	 * 取得内容中匹配的所有结果
	 * 
	 * @param <T> 集合类型
	 * @param regex 正则
	 * @param content 被查找的内容
	 * @param group 正则的分组
	 * @param collection 返回的集合类型
	 * @return 结果集
	 */
	public static <T extends Collection<String>> T findAll(String regex, String content, int group, T collection) {
		if (null == regex) {
			return null;
		}

		Pattern pattern = Pattern.compile(regex, Pattern.DOTALL);
		return findAll(pattern, content, group, collection);
	}

	/**
	 * 取得内容中匹配的所有结果，获得匹配的所有结果中正则对应分组0的内容
	 * 
	 * @param pattern 编译后的正则模式
	 * @param content 被查找的内容
	 * @return 结果列表
	 * @since 3.1.2
	 */
	public static List<String> findAllGroup0(Pattern pattern, String content) {
		return findAll(pattern, content, 0);
	}

	/**
	 * 取得内容中匹配的所有结果，获得匹配的所有结果中正则对应分组1的内容
	 * 
	 * @param pattern 编译后的正则模式
	 * @param content 被查找的内容
	 * @return 结果列表
	 * @since 3.1.2
	 */
	public static List<String> findAllGroup1(Pattern pattern, String content) {
		return findAll(pattern, content, 1);
	}

	/**
	 * 取得内容中匹配的所有结果
	 * 
	 * @param pattern 编译后的正则模式
	 * @param content 被查找的内容
	 * @param group 正则的分组
	 * @return 结果列表
	 * @since 3.0.6
	 */
	public static List<String> findAll(Pattern pattern, String content, int group) {
		return findAll(pattern, content, group, new ArrayList<String>());
	}

	/**
	 * 取得内容中匹配的所有结果
	 * 
	 * @param <T> 集合类型
	 * @param pattern 编译后的正则模式
	 * @param content 被查找的内容
	 * @param group 正则的分组
	 * @param collection 返回的集合类型
	 * @return 结果集
	 */
	public static <T extends Collection<String>> T findAll(Pattern pattern, String content, int group, T collection) {
		if (null == pattern || null == content) {
			return null;
		}

		if (null == collection) {
			throw new NullPointerException("Null collection param provided!");
		}

		Matcher matcher = pattern.matcher(content);
		while (matcher.find()) {
			collection.add(matcher.group(group));
		}
		return collection;
	}

	/**
	 * 计算指定字符串中，匹配pattern的个数
	 * 
	 * @param pattern 编译后的正则模式
	 * @param content 被查找的内容
	 * @return 匹配个数
	 */
	public static int count(Pattern pattern, String content) {
		if (null == pattern || null == content) {
			return 0;
		}

		int count = 0;
		Matcher matcher = pattern.matcher(content);
		while (matcher.find()) {
			count++;
		}

		return count;
	}

	/**
	 * 指定内容中是否有表达式匹配的内容
	 * 
	 * @param pattern 编译后的正则模式
	 * @param content 被查找的内容
	 * @return 指定内容中是否有表达式匹配的内容
	 * @since 3.3.1
	 */
	public static boolean contains(Pattern pattern, String content) {
		if (null == pattern || null == content) {
			return false;
		}
		return pattern.matcher(content).find();
	}

	/**
	 * 给定内容是否匹配正则
	 * 
	 * @param regex 正则
	 * @param content 内容
	 * @return 正则为null或者""则不检查，返回true，内容为null返回false
	 */
	public static boolean isMatch(String regex, String content) {
		if (content == null) {
			// 提供null的字符串为不匹配
			return false;
		}

		if (StrUtil.isEmpty(regex)) {
			// 正则不存在则为全匹配
			return true;
		}

		// Pattern pattern = Pattern.compile(regex, Pattern.DOTALL);
		final Pattern pattern = Pattern.compile(regex, Pattern.DOTALL);//;PatternPool.get(regex, Pattern.DOTALL);
		return isMatch(pattern, content);
	}

	/**
	 * 给定内容是否匹配正则
	 * 
	 * @param pattern 模式
	 * @param content 内容
	 * @return 正则为null或者""则不检查，返回true，内容为null返回false
	 */
	public static boolean isMatch(Pattern pattern, String content) {
		if (content == null || pattern == null) {
			// 提供null的字符串为不匹配
			return false;
		}
		return pattern.matcher(content).matches();
	}

	/**
	 * 正则替换指定值<br>
	 * 通过正则查找到字符串，然后把匹配到的字符串加入到replacementTemplate中，$1表示分组1的字符串
	 * 
	 * @param content 文本
	 * @param regex 正则
	 * @param replacementTemplate 替换的文本模板，可以使用$1类似的变量提取正则匹配出的内容
	 * @return 处理后的文本
	 */
	public static String replaceAll(String content, String regex, String replacementTemplate) {
		final Pattern pattern = Pattern.compile(regex, Pattern.DOTALL);
		return replaceAll(content, pattern, replacementTemplate);
	}

	/**
	 * 正则替换指定值<br>
	 * 通过正则查找到字符串，然后把匹配到的字符串加入到replacementTemplate中，$1表示分组1的字符串
	 * 
	 * @param content 文本
	 * @param pattern {@link Pattern}
	 * @param replacementTemplate 替换的文本模板，可以使用$1类似的变量提取正则匹配出的内容
	 * @return 处理后的文本
	 * @since 3.0.4
	 */
	public static String replaceAll(String content, Pattern pattern, String replacementTemplate) {
		if (StrUtil.isEmpty(content)) {
			return content;
		}

		final Matcher matcher = pattern.matcher(content);
		boolean result = matcher.find();
		if (result) {
			final Set<String> varNums = findAll(PatternPool.GROUP_VAR, replacementTemplate, 1, new HashSet<String>());
			final StringBuffer sb = new StringBuffer();
			do {
				String replacement = replacementTemplate;
				for (String var : varNums) {
					int group = Integer.parseInt(var);
					replacement = replacement.replace("$" + var, matcher.group(group));
				}
				matcher.appendReplacement(sb, escape(replacement));
				result = matcher.find();
			} while (result);
			matcher.appendTail(sb);
			return sb.toString();
		}
		return content;
	}

	/**
	 * 转义字符，将正则的关键字转义
	 * 
	 * @param c 字符
	 * @return 转义后的文本
	 */
	public static String escape(char c) {
		final StringBuilder builder = new StringBuilder();
		if (RE_KEYS.contains(c)) {
			builder.append('\\');
		}
		builder.append(c);
		return builder.toString();
	}

	/**
	 * 转义字符串，将正则的关键字转义
	 * 
	 * @param content 文本
	 * @return 转义后的文本
	 */
	public static String escape(String content) {
		if (StrUtil.isBlank(content)) {
			return content;
		}

		final StringBuilder builder = new StringBuilder();
		int len = content.length();
		char current;
		for (int i = 0; i < len; i++) {
			current = content.charAt(i);
			if (RE_KEYS.contains(current)) {
				builder.append('\\');
			}
			builder.append(current);
		}
		return builder.toString();
	}
}
