/*
 * Decompiled with CFR 0.152.
 */
package net.e175.klaus.solarpositioning;

import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.Map;
import net.e175.klaus.solarpositioning.JulianDate;
import net.e175.klaus.solarpositioning.MathUtil;
import net.e175.klaus.solarpositioning.SolarPosition;
import net.e175.klaus.solarpositioning.SunriseResult;

public final class SPA {
    private static final int MS_PER_DAY = 86400000;
    private static final double SUNRISE_SUNSET = -0.83337;
    private static final double[][][] TERMS_L = new double[][][]{new double[][]{{1.75347046E8, 0.0, 0.0}, {3341656.0, 4.6692568, 6283.07585}, {34894.0, 4.6261, 12566.1517}, {3497.0, 2.7441, 5753.3849}, {3418.0, 2.8289, 3.5231}, {3136.0, 3.6277, 77713.7715}, {2676.0, 4.4181, 7860.4194}, {2343.0, 6.1352, 3930.2097}, {1324.0, 0.7425, 11506.7698}, {1273.0, 2.0371, 529.691}, {1199.0, 1.1096, 1577.3435}, {990.0, 5.233, 5884.927}, {902.0, 2.045, 26.298}, {857.0, 3.508, 398.149}, {780.0, 1.179, 5223.694}, {753.0, 2.533, 5507.553}, {505.0, 4.583, 18849.228}, {492.0, 4.205, 775.523}, {357.0, 2.92, 0.067}, {317.0, 5.849, 11790.629}, {284.0, 1.899, 796.298}, {271.0, 0.315, 10977.079}, {243.0, 0.345, 5486.778}, {206.0, 4.806, 2544.314}, {205.0, 1.869, 5573.143}, {202.0, 2.458, 6069.777}, {156.0, 0.833, 213.299}, {132.0, 3.411, 2942.463}, {126.0, 1.083, 20.775}, {115.0, 0.645, 0.98}, {103.0, 0.636, 4694.003}, {102.0, 0.976, 15720.839}, {102.0, 4.267, 7.114}, {99.0, 6.21, 2146.17}, {98.0, 0.68, 155.42}, {86.0, 5.98, 161000.69}, {85.0, 1.3, 6275.96}, {85.0, 3.67, 71430.7}, {80.0, 1.81, 17260.15}, {79.0, 3.04, 12036.46}, {75.0, 1.76, 5088.63}, {74.0, 3.5, 3154.69}, {74.0, 4.68, 801.82}, {70.0, 0.83, 9437.76}, {62.0, 3.98, 8827.39}, {61.0, 1.82, 7084.9}, {57.0, 2.78, 6286.6}, {56.0, 4.39, 14143.5}, {56.0, 3.47, 6279.55}, {52.0, 0.19, 12139.55}, {52.0, 1.33, 1748.02}, {51.0, 0.28, 5856.48}, {49.0, 0.49, 1194.45}, {41.0, 5.37, 8429.24}, {41.0, 2.4, 19651.05}, {39.0, 6.17, 10447.39}, {37.0, 6.04, 10213.29}, {37.0, 2.57, 1059.38}, {36.0, 1.71, 2352.87}, {36.0, 1.78, 6812.77}, {33.0, 0.59, 17789.85}, {30.0, 0.44, 83996.85}, {30.0, 2.74, 1349.87}, {25.0, 3.16, 4690.48}}, new double[][]{{6.28331966747E11, 0.0, 0.0}, {206059.0, 2.678235, 6283.07585}, {4303.0, 2.6351, 12566.1517}, {425.0, 1.59, 3.523}, {119.0, 5.796, 26.298}, {109.0, 2.966, 1577.344}, {93.0, 2.59, 18849.23}, {72.0, 1.14, 529.69}, {68.0, 1.87, 398.15}, {67.0, 4.41, 5507.55}, {59.0, 2.89, 5223.69}, {56.0, 2.17, 155.42}, {45.0, 0.4, 796.3}, {36.0, 0.47, 775.52}, {29.0, 2.65, 7.11}, {21.0, 5.34, 0.98}, {19.0, 1.85, 5486.78}, {19.0, 4.97, 213.3}, {17.0, 2.99, 6275.96}, {16.0, 0.03, 2544.31}, {16.0, 1.43, 2146.17}, {15.0, 1.21, 10977.08}, {12.0, 2.83, 1748.02}, {12.0, 3.26, 5088.63}, {12.0, 5.27, 1194.45}, {12.0, 2.08, 4694.0}, {11.0, 0.77, 553.57}, {10.0, 1.3, 6286.6}, {10.0, 4.24, 1349.87}, {9.0, 2.7, 242.73}, {9.0, 5.64, 951.72}, {8.0, 5.3, 2352.87}, {6.0, 2.65, 9437.76}, {6.0, 4.67, 4690.48}}, new double[][]{{52919.0, 0.0, 0.0}, {8720.0, 1.0721, 6283.0758}, {309.0, 0.867, 12566.152}, {27.0, 0.05, 3.52}, {16.0, 5.19, 26.3}, {16.0, 3.68, 155.42}, {10.0, 0.76, 18849.23}, {9.0, 2.06, 77713.77}, {7.0, 0.83, 775.52}, {5.0, 4.66, 1577.34}, {4.0, 1.03, 7.11}, {4.0, 3.44, 5573.14}, {3.0, 5.14, 796.3}, {3.0, 6.05, 5507.55}, {3.0, 1.19, 242.73}, {3.0, 6.12, 529.69}, {3.0, 0.31, 398.15}, {3.0, 2.28, 553.57}, {2.0, 4.38, 5223.69}, {2.0, 3.75, 0.98}}, new double[][]{{289.0, 5.844, 6283.076}, {35.0, 0.0, 0.0}, {17.0, 5.49, 12566.15}, {3.0, 5.2, 155.42}, {1.0, 4.72, 3.52}, {1.0, 5.3, 18849.23}, {1.0, 5.97, 242.73}}, new double[][]{{114.0, 3.142, 0.0}, {8.0, 4.13, 6283.08}, {1.0, 3.84, 12566.15}}, new double[][]{{1.0, 3.14, 0.0}}};
    private static final double[][][] TERMS_B = new double[][][]{new double[][]{{280.0, 3.199, 84334.662}, {102.0, 5.422, 5507.553}, {80.0, 3.88, 5223.69}, {44.0, 3.7, 2352.87}, {32.0, 4.0, 1577.34}}, new double[][]{{9.0, 3.9, 5507.55}, {6.0, 1.73, 5223.69}}};
    private static final double[][][] TERMS_R = new double[][][]{new double[][]{{1.00013989E8, 0.0, 0.0}, {1670700.0, 3.0984635, 6283.07585}, {13956.0, 3.05525, 12566.1517}, {3084.0, 5.1985, 77713.7715}, {1628.0, 1.1739, 5753.3849}, {1576.0, 2.8469, 7860.4194}, {925.0, 5.453, 11506.77}, {542.0, 4.564, 3930.21}, {472.0, 3.661, 5884.927}, {346.0, 0.964, 5507.553}, {329.0, 5.9, 5223.694}, {307.0, 0.299, 5573.143}, {243.0, 4.273, 11790.629}, {212.0, 5.847, 1577.344}, {186.0, 5.022, 10977.079}, {175.0, 3.012, 18849.228}, {110.0, 5.055, 5486.778}, {98.0, 0.89, 6069.78}, {86.0, 5.69, 15720.84}, {86.0, 1.27, 161000.69}, {65.0, 0.27, 17260.15}, {63.0, 0.92, 529.69}, {57.0, 2.01, 83996.85}, {56.0, 5.24, 71430.7}, {49.0, 3.25, 2544.31}, {47.0, 2.58, 775.52}, {45.0, 5.54, 9437.76}, {43.0, 6.01, 6275.96}, {39.0, 5.36, 4694.0}, {38.0, 2.39, 8827.39}, {37.0, 0.83, 19651.05}, {37.0, 4.9, 12139.55}, {36.0, 1.67, 12036.46}, {35.0, 1.84, 2942.46}, {33.0, 0.24, 7084.9}, {32.0, 0.18, 5088.63}, {32.0, 1.78, 398.15}, {28.0, 1.21, 6286.6}, {28.0, 1.9, 6279.55}, {26.0, 4.59, 10447.39}}, new double[][]{{103019.0, 1.10749, 6283.07585}, {1721.0, 1.0644, 12566.1517}, {702.0, 3.142, 0.0}, {32.0, 1.02, 18849.23}, {31.0, 2.84, 5507.55}, {25.0, 1.32, 5223.69}, {18.0, 1.42, 1577.34}, {10.0, 5.91, 10977.08}, {9.0, 1.42, 6275.96}, {9.0, 0.27, 5486.78}}, new double[][]{{4359.0, 5.7846, 6283.0758}, {124.0, 5.579, 12566.152}, {12.0, 3.14, 0.0}, {9.0, 3.63, 77713.77}, {6.0, 1.87, 5573.14}, {3.0, 5.47, 18849.23}}, new double[][]{{145.0, 4.273, 6283.076}, {7.0, 3.92, 12566.15}}, new double[][]{{4.0, 2.56, 6283.08}}};
    private static final double[][] NUTATION_COEFFS = new double[][]{{297.85036, 445267.11148, -0.0019142, 5.277768981496142E-6}, {357.52772, 35999.05034, -1.603E-4, -3.3333333333333333E-6}, {134.96298, 477198.867398, 0.0086972, 1.7777777777777777E-5}, {93.27191, 483202.017538, -0.0036825, 3.0555810187307116E-6}, {125.04452, -1934.136261, 0.0020708, 2.222222222222222E-6}};
    private static final double[][] TERMS_Y = new double[][]{{0.0, 0.0, 0.0, 0.0, 1.0}, {-2.0, 0.0, 0.0, 2.0, 2.0}, {0.0, 0.0, 0.0, 2.0, 2.0}, {0.0, 0.0, 0.0, 0.0, 2.0}, {0.0, 1.0, 0.0, 0.0, 0.0}, {0.0, 0.0, 1.0, 0.0, 0.0}, {-2.0, 1.0, 0.0, 2.0, 2.0}, {0.0, 0.0, 0.0, 2.0, 1.0}, {0.0, 0.0, 1.0, 2.0, 2.0}, {-2.0, -1.0, 0.0, 2.0, 2.0}, {-2.0, 0.0, 1.0, 0.0, 0.0}, {-2.0, 0.0, 0.0, 2.0, 1.0}, {0.0, 0.0, -1.0, 2.0, 2.0}, {2.0, 0.0, 0.0, 0.0, 0.0}, {0.0, 0.0, 1.0, 0.0, 1.0}, {2.0, 0.0, -1.0, 2.0, 2.0}, {0.0, 0.0, -1.0, 0.0, 1.0}, {0.0, 0.0, 1.0, 2.0, 1.0}, {-2.0, 0.0, 2.0, 0.0, 0.0}, {0.0, 0.0, -2.0, 2.0, 1.0}, {2.0, 0.0, 0.0, 2.0, 2.0}, {0.0, 0.0, 2.0, 2.0, 2.0}, {0.0, 0.0, 2.0, 0.0, 0.0}, {-2.0, 0.0, 1.0, 2.0, 2.0}, {0.0, 0.0, 0.0, 2.0, 0.0}, {-2.0, 0.0, 0.0, 2.0, 0.0}, {0.0, 0.0, -1.0, 2.0, 1.0}, {0.0, 2.0, 0.0, 0.0, 0.0}, {2.0, 0.0, -1.0, 0.0, 1.0}, {-2.0, 2.0, 0.0, 2.0, 2.0}, {0.0, 1.0, 0.0, 0.0, 1.0}, {-2.0, 0.0, 1.0, 0.0, 1.0}, {0.0, -1.0, 0.0, 0.0, 1.0}, {0.0, 0.0, 2.0, -2.0, 0.0}, {2.0, 0.0, -1.0, 2.0, 1.0}, {2.0, 0.0, 1.0, 2.0, 2.0}, {0.0, 1.0, 0.0, 2.0, 2.0}, {-2.0, 1.0, 1.0, 0.0, 0.0}, {0.0, -1.0, 0.0, 2.0, 2.0}, {2.0, 0.0, 0.0, 2.0, 1.0}, {2.0, 0.0, 1.0, 0.0, 0.0}, {-2.0, 0.0, 2.0, 2.0, 2.0}, {-2.0, 0.0, 1.0, 2.0, 1.0}, {2.0, 0.0, -2.0, 0.0, 1.0}, {2.0, 0.0, 0.0, 0.0, 1.0}, {0.0, -1.0, 1.0, 0.0, 0.0}, {-2.0, -1.0, 0.0, 2.0, 1.0}, {-2.0, 0.0, 0.0, 0.0, 1.0}, {0.0, 0.0, 2.0, 2.0, 1.0}, {-2.0, 0.0, 2.0, 0.0, 1.0}, {-2.0, 1.0, 0.0, 2.0, 1.0}, {0.0, 0.0, 1.0, -2.0, 0.0}, {-1.0, 0.0, 1.0, 0.0, 0.0}, {-2.0, 1.0, 0.0, 0.0, 0.0}, {1.0, 0.0, 0.0, 0.0, 0.0}, {0.0, 0.0, 1.0, 2.0, 0.0}, {0.0, 0.0, -2.0, 2.0, 2.0}, {-1.0, -1.0, 1.0, 0.0, 0.0}, {0.0, 1.0, 1.0, 0.0, 0.0}, {0.0, -1.0, 1.0, 2.0, 2.0}, {2.0, -1.0, -1.0, 2.0, 2.0}, {0.0, 0.0, 3.0, 2.0, 2.0}, {2.0, -1.0, 0.0, 2.0, 2.0}};
    private static final double[][] TERMS_PE = new double[][]{{-171996.0, -174.2, 92025.0, 8.9}, {-13187.0, -1.6, 5736.0, -3.1}, {-2274.0, -0.2, 977.0, -0.5}, {2062.0, 0.2, -895.0, 0.5}, {1426.0, -3.4, 54.0, -0.1}, {712.0, 0.1, -7.0, 0.0}, {-517.0, 1.2, 224.0, -0.6}, {-386.0, -0.4, 200.0, 0.0}, {-301.0, 0.0, 129.0, -0.1}, {217.0, -0.5, -95.0, 0.3}, {-158.0, 0.0, 0.0, 0.0}, {129.0, 0.1, -70.0, 0.0}, {123.0, 0.0, -53.0, 0.0}, {63.0, 0.0, 0.0, 0.0}, {63.0, 0.1, -33.0, 0.0}, {-59.0, 0.0, 26.0, 0.0}, {-58.0, -0.1, 32.0, 0.0}, {-51.0, 0.0, 27.0, 0.0}, {48.0, 0.0, 0.0, 0.0}, {46.0, 0.0, -24.0, 0.0}, {-38.0, 0.0, 16.0, 0.0}, {-31.0, 0.0, 13.0, 0.0}, {29.0, 0.0, 0.0, 0.0}, {29.0, 0.0, -12.0, 0.0}, {26.0, 0.0, 0.0, 0.0}, {-22.0, 0.0, 0.0, 0.0}, {21.0, 0.0, -10.0, 0.0}, {17.0, -0.1, 0.0, 0.0}, {16.0, 0.0, -8.0, 0.0}, {-16.0, 0.1, 7.0, 0.0}, {-15.0, 0.0, 9.0, 0.0}, {-13.0, 0.0, 7.0, 0.0}, {-12.0, 0.0, 6.0, 0.0}, {11.0, 0.0, 0.0, 0.0}, {-10.0, 0.0, 5.0, 0.0}, {-8.0, 0.0, 3.0, 0.0}, {7.0, 0.0, -3.0, 0.0}, {-7.0, 0.0, 0.0, 0.0}, {-7.0, 0.0, 3.0, 0.0}, {-7.0, 0.0, 3.0, 0.0}, {6.0, 0.0, 0.0, 0.0}, {6.0, 0.0, -3.0, 0.0}, {6.0, 0.0, -3.0, 0.0}, {-6.0, 0.0, 3.0, 0.0}, {-6.0, 0.0, 3.0, 0.0}, {5.0, 0.0, 0.0, 0.0}, {-5.0, 0.0, 3.0, 0.0}, {-5.0, 0.0, 3.0, 0.0}, {-5.0, 0.0, 3.0, 0.0}, {4.0, 0.0, 0.0, 0.0}, {4.0, 0.0, 0.0, 0.0}, {4.0, 0.0, 0.0, 0.0}, {-4.0, 0.0, 0.0, 0.0}, {-4.0, 0.0, 0.0, 0.0}, {-4.0, 0.0, 0.0, 0.0}, {3.0, 0.0, 0.0, 0.0}, {-3.0, 0.0, 0.0, 0.0}, {-3.0, 0.0, 0.0, 0.0}, {-3.0, 0.0, 0.0, 0.0}, {-3.0, 0.0, 0.0, 0.0}, {-3.0, 0.0, 0.0, 0.0}, {-3.0, 0.0, 0.0, 0.0}, {-3.0, 0.0, 0.0, 0.0}};
    private static final double[] OBLIQUITY_COEFFS = new double[]{84381.448, -4680.93, -1.55, 1999.25, 51.38, -249.67, -39.05, 7.12, 27.87, 5.79, 2.45};

    private SPA() {
    }

    public static SolarPosition calculateSolarPosition(ZonedDateTime date, double latitude, double longitude, double elevation, double deltaT, double pressure, double temperature) {
        MathUtil.checkLatLonRange(latitude, longitude);
        JulianDate jd = new JulianDate(date, deltaT);
        double jme = jd.julianEphemerisMillennium();
        double jce = jd.julianEphemerisCentury();
        double[] lTerms = SPA.calculateLBRTerms(jme, TERMS_L);
        double lDegrees = SPA.limitDegreesTo360(Math.toDegrees(SPA.calculateLBRPolynomial(jme, lTerms)));
        double[] bTerms = SPA.calculateLBRTerms(jme, TERMS_B);
        double bDegrees = SPA.limitDegreesTo360(Math.toDegrees(SPA.calculateLBRPolynomial(jme, bTerms)));
        double[] rTerms = SPA.calculateLBRTerms(jme, TERMS_R);
        double r = SPA.calculateLBRPolynomial(jme, rTerms);
        assert (r != 0.0);
        double thetaDegrees = SPA.limitDegreesTo360(lDegrees + 180.0);
        double betaDegrees = -bDegrees;
        double beta = Math.toRadians(betaDegrees);
        double[] xTerms = SPA.calculateNutationTerms(jce);
        double[] deltaPsiI = SPA.calculateDeltaPsiI(jce, xTerms);
        double[] deltaEpsilonI = SPA.calculateDeltaEpsilonI(jce, xTerms);
        double deltaPsi = SPA.calculateDeltaPsiEpsilon(deltaPsiI);
        double deltaEpsilon = SPA.calculateDeltaPsiEpsilon(deltaEpsilonI);
        double epsilonDegrees = SPA.calculateTrueObliquityOfEcliptic(jd, deltaEpsilon);
        double epsilon = Math.toRadians(epsilonDegrees);
        double deltaTau = -20.4898 / (3600.0 * r);
        double lambdaDegrees = thetaDegrees + deltaPsi + deltaTau;
        double lambda = Math.toRadians(lambdaDegrees);
        double nuDegrees = SPA.calculateApparentSiderealTimeAtGreenwich(jd, deltaPsi, epsilonDegrees);
        double alphaDegrees = SPA.calculateGeocentricSunRightAscension(beta, epsilon, lambda);
        double deltaDegrees = Math.toDegrees(SPA.calculateGeocentricSunDeclination(beta, epsilon, lambda));
        double hDegrees = SPA.limitDegreesTo360(nuDegrees + longitude - alphaDegrees);
        double h = Math.toRadians(hDegrees);
        double xiDegrees = 8.794 / (3600.0 * r);
        double xi = Math.toRadians(xiDegrees);
        double phi = Math.toRadians(latitude);
        double delta = Math.toRadians(deltaDegrees);
        double u = Math.atan(0.99664719 * Math.tan(phi));
        double x = Math.cos(u) + elevation * Math.cos(phi) / 6378140.0;
        double y = 0.99664719 * Math.sin(u) + elevation * Math.sin(phi) / 6378140.0;
        double x1 = Math.cos(delta) - x * Math.sin(xi) * Math.cos(h);
        double deltaAlphaDegrees = Math.toDegrees(Math.atan2(-x * Math.sin(xi) * Math.sin(h), x1));
        double deltaPrime = Math.atan2((Math.sin(delta) - y * Math.sin(xi)) * Math.cos(Math.toRadians(deltaAlphaDegrees)), x1);
        double hPrimeDegrees = hDegrees - deltaAlphaDegrees;
        double hPrime = Math.toRadians(hPrimeDegrees);
        return SPA.calculateTopocentricSolarPosition(pressure, temperature, phi, deltaPrime, hPrime);
    }

    public static SolarPosition calculateSolarPosition(ZonedDateTime date, double latitude, double longitude, double elevation, double deltaT) {
        return SPA.calculateSolarPosition(date, latitude, longitude, elevation, deltaT, Double.MIN_VALUE, Double.MIN_VALUE);
    }

    public static SunriseResult calculateSunriseTransitSet(ZonedDateTime day, double latitude, double longitude, double deltaT) {
        return SPA.calculateSunriseTransitSet(day, latitude, longitude, deltaT, Horizon.SUNRISE_SUNSET);
    }

    public static SunriseResult calculateSunriseTransitSet(ZonedDateTime day, double latitude, double longitude, double deltaT, Horizon horizon) {
        RiseSetParams params = SPA.calcRiseSetParams(day, latitude, longitude);
        return SPA.calcRiseAndSet(day, longitude, deltaT, horizon, Math.toRadians(latitude), params.nuDegrees, params.alphaDeltas, params.m);
    }

    public static Map<Horizon, SunriseResult> calculateSunriseTransitSet(ZonedDateTime day, double latitude, double longitude, double deltaT, Horizon ... horizons) {
        RiseSetParams params = SPA.calcRiseSetParams(day, latitude, longitude);
        HashMap<Horizon, SunriseResult> result = new HashMap<Horizon, SunriseResult>(horizons.length + 1, 1.0f);
        for (Horizon horizon : horizons) {
            result.put(horizon, SPA.calcRiseAndSet(day, longitude, deltaT, horizon, Math.toRadians(latitude), params.nuDegrees, params.alphaDeltas, params.m));
        }
        return result;
    }

    private static RiseSetParams calcRiseSetParams(ZonedDateTime day, double latitude, double longitude) {
        MathUtil.checkLatLonRange(latitude, longitude);
        ZonedDateTime dayStart = SPA.startOfDayUT(day);
        JulianDate jd = new JulianDate(dayStart, 0.0);
        double jce = jd.julianEphemerisCentury();
        double[] xTerms = SPA.calculateNutationTerms(jce);
        double[] deltaPsiI = SPA.calculateDeltaPsiI(jce, xTerms);
        double[] deltaEpsilonI = SPA.calculateDeltaEpsilonI(jce, xTerms);
        double deltaPsi = SPA.calculateDeltaPsiEpsilon(deltaPsiI);
        double deltaEpsilon = SPA.calculateDeltaPsiEpsilon(deltaEpsilonI);
        double epsilonDegrees = SPA.calculateTrueObliquityOfEcliptic(jd, deltaEpsilon);
        double nuDegrees = SPA.calculateApparentSiderealTimeAtGreenwich(jd, deltaPsi, epsilonDegrees);
        AlphaDelta[] alphaDeltas = new AlphaDelta[3];
        for (int i = 0; i < alphaDeltas.length; ++i) {
            AlphaDelta ad;
            JulianDate currentJd = new JulianDate(jd.julianDate() + (double)i - 1.0, 0.0);
            double currentJme = currentJd.julianEphemerisMillennium();
            alphaDeltas[i] = ad = SPA.calculateAlphaDelta(currentJme, deltaPsi, epsilonDegrees);
        }
        double[] m = new double[3];
        m[0] = (alphaDeltas[1].alpha - longitude - nuDegrees) / 360.0;
        return new RiseSetParams(nuDegrees, alphaDeltas, m);
    }

    private static SunriseResult calcRiseAndSet(ZonedDateTime day, double longitude, double deltaT, Horizon horizon, double phi, double nuDegrees, AlphaDelta[] alphaDeltas, double[] m) {
        double acosArg = (Math.sin(Math.toRadians(horizon.elevation())) - Math.sin(phi) * Math.sin(Math.toRadians(alphaDeltas[1].delta))) / (Math.cos(phi) * Math.cos(Math.toRadians(alphaDeltas[1].delta)));
        Type type = acosArg < -1.0 ? Type.ALL_DAY : (acosArg > 1.0 ? Type.ALL_NIGHT : Type.NORMAL);
        double h0 = Math.acos(acosArg);
        double h0Degrees = SPA.limitTo(Math.toDegrees(h0), 180.0);
        m[1] = SPA.limitTo(m[0] - h0Degrees / 360.0, 1.0);
        m[2] = SPA.limitTo(m[0] + h0Degrees / 360.0, 1.0);
        m[0] = SPA.limitTo(m[0], 1.0);
        double[] nu = new double[3];
        for (int i = 0; i < m.length; ++i) {
            nu[i] = nuDegrees + 360.985647 * m[i];
        }
        double[] n = new double[3];
        for (int i = 0; i < m.length; ++i) {
            n[i] = m[i] + deltaT / 86400.0;
        }
        double a = SPA.limitIfNecessary(alphaDeltas[1].alpha - alphaDeltas[0].alpha);
        double aPrime = SPA.limitIfNecessary(alphaDeltas[1].delta - alphaDeltas[0].delta);
        double b = SPA.limitIfNecessary(alphaDeltas[2].alpha - alphaDeltas[1].alpha);
        double bPrime = SPA.limitIfNecessary(alphaDeltas[2].delta - alphaDeltas[1].delta);
        double c = b - a;
        double cPrime = bPrime - aPrime;
        AlphaDelta[] alphaDeltaPrimes = new AlphaDelta[3];
        for (int i = 0; i < alphaDeltaPrimes.length; ++i) {
            double alphaPrimeI = alphaDeltas[1].alpha + n[i] * (a + b + c * n[i]) / 2.0;
            double deltaPrimeI = alphaDeltas[1].delta + n[i] * (aPrime + bPrime + cPrime * n[i]) / 2.0;
            alphaDeltaPrimes[i] = new AlphaDelta(alphaPrimeI, deltaPrimeI);
        }
        double[] hPrime = new double[3];
        for (int i = 0; i < hPrime.length; ++i) {
            double hPrimeI = nu[i] + longitude - alphaDeltaPrimes[i].alpha;
            hPrime[i] = SPA.limitHprime(hPrimeI);
        }
        double[] h = new double[3];
        for (int i = 0; i < h.length; ++i) {
            double deltaPrimeRad = Math.toRadians(alphaDeltaPrimes[i].delta);
            h[i] = Math.toDegrees(Math.asin(Math.sin(phi) * Math.sin(deltaPrimeRad) + Math.cos(phi) * Math.cos(deltaPrimeRad) * Math.cos(Math.toRadians(hPrime[i]))));
        }
        double t = m[0] - hPrime[0] / 360.0;
        double r = m[1] + (h[1] - horizon.elevation()) / (360.0 * Math.cos(Math.toRadians(alphaDeltaPrimes[1].delta)) * Math.cos(phi) * Math.sin(Math.toRadians(hPrime[1])));
        double s = m[2] + (h[2] - horizon.elevation()) / (360.0 * Math.cos(Math.toRadians(alphaDeltaPrimes[2].delta)) * Math.cos(phi) * Math.sin(Math.toRadians(hPrime[2])));
        return switch (type) {
            default -> throw new IncompatibleClassChangeError();
            case Type.NORMAL -> new SunriseResult.RegularDay(SPA.addFractionOfDay(day, r), SPA.addFractionOfDay(day, t), SPA.addFractionOfDay(day, s));
            case Type.ALL_DAY -> new SunriseResult.AllDay(SPA.addFractionOfDay(day, t));
            case Type.ALL_NIGHT -> new SunriseResult.AllNight(SPA.addFractionOfDay(day, t));
        };
    }

    private static ZonedDateTime addFractionOfDay(ZonedDateTime day, double fraction) {
        int millisPlus = (int)(8.64E7 * fraction);
        return day.truncatedTo(ChronoUnit.DAYS).plus(millisPlus, ChronoUnit.MILLIS);
    }

    private static double limitHprime(double hPrime) {
        double limited;
        if ((limited = 360.0 * ((hPrime /= 360.0) - Math.floor(hPrime))) < -180.0) {
            return limited + 360.0;
        }
        if (limited > 180.0) {
            return limited - 360.0;
        }
        return limited;
    }

    private static double limitIfNecessary(double val) {
        return Math.abs(val) > 2.0 ? SPA.limitTo(val, 1.0) : val;
    }

    private static AlphaDelta calculateAlphaDelta(double jme, double deltaPsi, double epsilonDegrees) {
        double[] bTerms = SPA.calculateLBRTerms(jme, TERMS_B);
        double bDegrees = SPA.limitDegreesTo360(Math.toDegrees(SPA.calculateLBRPolynomial(jme, bTerms)));
        double[] rTerms = SPA.calculateLBRTerms(jme, TERMS_R);
        double r = SPA.calculateLBRPolynomial(jme, rTerms);
        assert (r != 0.0);
        double[] lTerms = SPA.calculateLBRTerms(jme, TERMS_L);
        double lDegrees = SPA.limitDegreesTo360(Math.toDegrees(SPA.calculateLBRPolynomial(jme, lTerms)));
        double thetaDegrees = SPA.limitDegreesTo360(lDegrees + 180.0);
        double betaDegrees = -bDegrees;
        double beta = Math.toRadians(betaDegrees);
        double epsilon = Math.toRadians(epsilonDegrees);
        double deltaTau = -20.4898 / (3600.0 * r);
        double lambdaDegrees = thetaDegrees + deltaPsi + deltaTau;
        double lambda = Math.toRadians(lambdaDegrees);
        double alphaDegrees = SPA.calculateGeocentricSunRightAscension(beta, epsilon, lambda);
        double deltaDegrees = Math.toDegrees(SPA.calculateGeocentricSunDeclination(beta, epsilon, lambda));
        return new AlphaDelta(alphaDegrees, deltaDegrees);
    }

    private static ZonedDateTime startOfDayUT(ZonedDateTime day) {
        return day.truncatedTo(ChronoUnit.DAYS);
    }

    private static SolarPosition calculateTopocentricSolarPosition(double p, double t, double phi, double deltaPrime, double hPrime) {
        double eZero = Math.asin(Math.sin(phi) * Math.sin(deltaPrime) + Math.cos(phi) * Math.cos(deltaPrime) * Math.cos(hPrime));
        double topocentricZenithAngle = SPA.calculateTopocentricZenithAngle(p, t, eZero);
        double gamma = Math.atan2(Math.sin(hPrime), Math.cos(hPrime) * Math.sin(phi) - Math.tan(deltaPrime) * Math.cos(phi));
        double gammaDegrees = SPA.limitDegreesTo360(Math.toDegrees(gamma));
        double topocentricAzimuthAngle = SPA.limitDegreesTo360(gammaDegrees + 180.0);
        return new SolarPosition(topocentricAzimuthAngle, topocentricZenithAngle);
    }

    private static double calculateTopocentricZenithAngle(double p, double t, double eZero) {
        double eZeroDegrees = Math.toDegrees(eZero);
        double deltaEdegrees = 0.0;
        if (p > 0.0 && p < 3000.0 && t > -273.0 && t < 273.0 && eZeroDegrees > -0.83337) {
            deltaEdegrees = p / 1010.0 * (283.0 / (273.0 + t)) * 1.02 / (60.0 * Math.tan(Math.toRadians(eZeroDegrees + 10.3 / (eZeroDegrees + 5.11))));
        }
        double correctedEZeroDegrees = eZeroDegrees + deltaEdegrees;
        return 90.0 - correctedEZeroDegrees;
    }

    private static double calculateGeocentricSunDeclination(double betaRad, double epsilonRad, double lambdaRad) {
        return Math.asin(Math.sin(betaRad) * Math.cos(epsilonRad) + Math.cos(betaRad) * Math.sin(epsilonRad) * Math.sin(lambdaRad));
    }

    private static double calculateGeocentricSunRightAscension(double betaRad, double epsilonRad, double lambdaRad) {
        double alpha = Math.atan2(Math.sin(lambdaRad) * Math.cos(epsilonRad) - Math.tan(betaRad) * Math.sin(epsilonRad), Math.cos(lambdaRad));
        return SPA.limitDegreesTo360(Math.toDegrees(alpha));
    }

    private static double calculateTrueObliquityOfEcliptic(JulianDate jd, double deltaEpsilon) {
        double epsilon0 = MathUtil.polynomial(jd.julianEphemerisMillennium() / 10.0, OBLIQUITY_COEFFS);
        return epsilon0 / 3600.0 + deltaEpsilon;
    }

    private static double calculateApparentSiderealTimeAtGreenwich(JulianDate jd, double deltaPsi, double epsilonDegrees) {
        double nu0degrees = SPA.limitDegreesTo360(280.46061837 + 360.98564736629 * (jd.julianDate() - 2451545.0) + Math.pow(jd.julianCentury(), 2.0) * (3.87933E-4 - jd.julianCentury() / 3.871E7));
        return nu0degrees + deltaPsi * Math.cos(Math.toRadians(epsilonDegrees));
    }

    private static double calculateDeltaPsiEpsilon(double[] deltaPsiOrEpsilonI) {
        double sum = 0.0;
        for (double element : deltaPsiOrEpsilonI) {
            sum += element;
        }
        return sum / 3.6E7;
    }

    private static double[] calculateDeltaPsiI(double jce, double[] x) {
        double[] deltaPsiI = new double[TERMS_PE.length];
        for (int i = 0; i < TERMS_PE.length; ++i) {
            double a = TERMS_PE[i][0];
            double b = TERMS_PE[i][1];
            deltaPsiI[i] = (a + b * jce) * Math.sin(Math.toRadians(SPA.calculateXjYtermSum(i, x)));
        }
        return deltaPsiI;
    }

    private static double[] calculateDeltaEpsilonI(double jce, double[] x) {
        double[] deltaEpsilonI = new double[TERMS_PE.length];
        for (int i = 0; i < TERMS_PE.length; ++i) {
            double c = TERMS_PE[i][2];
            double d = TERMS_PE[i][3];
            deltaEpsilonI[i] = (c + d * jce) * Math.cos(Math.toRadians(SPA.calculateXjYtermSum(i, x)));
        }
        return deltaEpsilonI;
    }

    private static double calculateXjYtermSum(int i, double[] x) {
        double sum = 0.0;
        for (int j = 0; j < x.length; ++j) {
            sum += x[j] * TERMS_Y[i][j];
        }
        return sum;
    }

    private static double[] calculateNutationTerms(double jce) {
        double[] x = new double[NUTATION_COEFFS.length];
        for (int i = 0; i < x.length; ++i) {
            x[i] = MathUtil.polynomial(jce, NUTATION_COEFFS[i]);
        }
        return x;
    }

    private static double limitDegreesTo360(double degrees) {
        return SPA.limitTo(degrees, 360.0);
    }

    private static double limitTo(double degrees, double max) {
        double dividedDegrees = degrees / max;
        double limited = max * (dividedDegrees - Math.floor(dividedDegrees));
        return limited < 0.0 ? limited + max : limited;
    }

    private static double calculateLBRPolynomial(double jme, double[] terms) {
        return MathUtil.polynomial(jme, terms) / 1.0E8;
    }

    private static double[] calculateLBRTerms(double jme, double[][][] termCoeffs) {
        double[] lbrTerms = new double[]{0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
        for (int i = 0; i < termCoeffs.length; ++i) {
            double lbrSum = 0.0;
            for (int v = 0; v < termCoeffs[i].length; ++v) {
                double a = termCoeffs[i][v][0];
                double b = termCoeffs[i][v][1];
                double c = termCoeffs[i][v][2];
                lbrSum += a * Math.cos(b + c * jme);
            }
            lbrTerms[i] = lbrSum;
        }
        return lbrTerms;
    }

    public static enum Horizon {
        SUNRISE_SUNSET(-0.83337),
        CIVIL_TWILIGHT(-6.0),
        NAUTICAL_TWILIGHT(-12.0),
        ASTRONOMICAL_TWILIGHT(-18.0);

        private final double elevation;

        public double elevation() {
            return this.elevation;
        }

        private Horizon(double elevation) {
            this.elevation = elevation;
        }
    }

    private record RiseSetParams(double nuDegrees, AlphaDelta[] alphaDeltas, double[] m) {
    }

    private record AlphaDelta(double alpha, double delta) {
    }

    private static enum Type {
        NORMAL,
        ALL_DAY,
        ALL_NIGHT;

    }
}

