/*
 * Decompiled with CFR 0.152.
 */
package net.finmath.interpolation;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.util.Arrays;
import java.util.function.DoubleUnaryOperator;
import net.finmath.functions.LinearAlgebra;

public class RationalFunctionInterpolation
implements DoubleUnaryOperator,
Serializable {
    private static final long serialVersionUID = -3214160594013393575L;
    private final double[] points;
    private final double[] values;
    private InterpolationMethod interpolationMethod = InterpolationMethod.LINEAR;
    private ExtrapolationMethod extrapolationMethod = ExtrapolationMethod.DEFAULT;
    private RationalFunction[] interpolatingRationalFunctions;
    private transient Object interpolatingRationalFunctionsLazyInitLock = new Object();

    public RationalFunctionInterpolation(double[] points, double[] values) {
        this.points = points;
        this.values = values;
    }

    public RationalFunctionInterpolation(double[] points, double[] values, InterpolationMethod interpolationMethod, ExtrapolationMethod extrapolationMethod) {
        this.points = points;
        this.values = values;
        this.interpolationMethod = interpolationMethod;
        this.extrapolationMethod = extrapolationMethod;
    }

    public InterpolationMethod getInterpolationMethod() {
        return this.interpolationMethod;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public double getValue(double x) {
        Object object = this.interpolatingRationalFunctionsLazyInitLock;
        synchronized (object) {
            if (this.interpolatingRationalFunctions == null) {
                this.doCreateRationalFunctions();
            }
        }
        int pointIndex = Arrays.binarySearch(this.points, x);
        if (pointIndex >= 0) {
            return this.values[pointIndex];
        }
        int intervalIndex = -pointIndex - 2;
        if (intervalIndex < 0) {
            if (this.extrapolationMethod == ExtrapolationMethod.CONSTANT) {
                return this.values[0];
            }
            if (this.extrapolationMethod == ExtrapolationMethod.LINEAR) {
                return this.values[0] + (this.values[1] - this.values[0]) / (this.points[1] - this.points[0]) * (x - this.points[0]);
            }
            intervalIndex = 0;
        } else if (intervalIndex > this.points.length - 2) {
            if (this.extrapolationMethod == ExtrapolationMethod.CONSTANT) {
                return this.values[this.points.length - 1];
            }
            if (this.extrapolationMethod == ExtrapolationMethod.LINEAR) {
                return this.values[this.points.length - 1] + (this.values[this.points.length - 2] - this.values[this.points.length - 1]) / (this.points[this.points.length - 2] - this.points[this.points.length - 1]) * (x - this.points[this.points.length - 1]);
            }
            intervalIndex = this.points.length - 2;
        }
        RationalFunction rationalFunction = this.interpolatingRationalFunctions[intervalIndex];
        return rationalFunction.getValue(x - this.points[intervalIndex]);
    }

    private void doCreateRationalFunctions() {
        switch (this.interpolationMethod) {
            case PIECEWISE_CONSTANT: 
            case PIECEWISE_CONSTANT_LEFTPOINT: 
            case PIECEWISE_CONSTANT_RIGHTPOINT: {
                this.doCreateRationalFunctionsForPiecewiseConstantInterpolation();
                break;
            }
            default: {
                this.doCreateRationalFunctionsForLinearInterpolation();
                break;
            }
            case CUBIC_SPLINE: {
                this.doCreateRationalFunctionsForCubicSplineInterpolation();
                break;
            }
            case AKIMA: {
                this.doCreateRationalFunctionsForAkimaInterpolation();
                break;
            }
            case AKIMA_CONTINUOUS: {
                this.doCreateRationalFunctionsForAkimaInterpolation(0.01);
                break;
            }
            case HARMONIC_SPLINE: {
                this.doCreateRationalFunctionsForHarmonicSplineInterpolation();
                break;
            }
            case HARMONIC_SPLINE_WITH_MONOTONIC_FILTERING: {
                this.doCreateRationalFunctionsForHarmonicSplineInterpolation();
            }
        }
    }

    private void doCreateRationalFunctionsForPiecewiseConstantInterpolation() {
        this.interpolatingRationalFunctions = new RationalFunction[this.points.length - 1];
        for (int pointIndex = 0; pointIndex < this.points.length - 1; ++pointIndex) {
            double[] numeratorPolynomCoeff = this.interpolationMethod == InterpolationMethod.PIECEWISE_CONSTANT_RIGHTPOINT ? new double[]{this.values[pointIndex + 1]} : new double[]{this.values[pointIndex]};
            this.interpolatingRationalFunctions[pointIndex] = new RationalFunction(numeratorPolynomCoeff);
        }
    }

    private void doCreateRationalFunctionsForLinearInterpolation() {
        if (this.points.length == 0) {
            throw new IllegalArgumentException("Interpolation requested on curve with no points.");
        }
        this.interpolatingRationalFunctions = new RationalFunction[this.points.length - 1];
        for (int pointIndex = 0; pointIndex < this.points.length - 1; ++pointIndex) {
            double[] numeratorPolynomCoeff = new double[2];
            double xl = this.points[pointIndex];
            double xr = this.points[pointIndex + 1];
            double fl = this.values[pointIndex];
            double fr = this.values[pointIndex + 1];
            numeratorPolynomCoeff[1] = (fr - fl) / (xr - xl);
            numeratorPolynomCoeff[0] = fl;
            this.interpolatingRationalFunctions[pointIndex] = new RationalFunction(numeratorPolynomCoeff);
        }
    }

    private void doCreateRationalFunctionsForCubicSplineInterpolation() {
        int numberOfPoints = this.points.length;
        double[] step = new double[numberOfPoints - 1];
        for (int i = 0; i < numberOfPoints - 1; ++i) {
            step[i] = this.points[i + 1] - this.points[i];
        }
        double[] secondDerivativeVector = new double[numberOfPoints];
        double[][] secondDerivativeMarix = new double[numberOfPoints][numberOfPoints];
        double[] v = new double[numberOfPoints];
        secondDerivativeMarix[0][0] = 1.0;
        secondDerivativeMarix[numberOfPoints - 1][numberOfPoints - 1] = 1.0;
        v[0] = 0.0;
        v[numberOfPoints - 1] = 0.0;
        for (int intervalIndex = 1; intervalIndex < numberOfPoints - 1; ++intervalIndex) {
            v[intervalIndex] = 6.0 * ((this.values[intervalIndex + 1] - this.values[intervalIndex]) / step[intervalIndex] - (this.values[intervalIndex] - this.values[intervalIndex - 1]) / step[intervalIndex - 1]);
            secondDerivativeMarix[intervalIndex][intervalIndex - 1] = step[intervalIndex - 1];
            secondDerivativeMarix[intervalIndex][intervalIndex] = 2.0 * (step[intervalIndex - 1] + step[intervalIndex]);
            secondDerivativeMarix[intervalIndex][intervalIndex + 1] = step[intervalIndex];
        }
        if (numberOfPoints > 1) {
            secondDerivativeMarix[0][1] = secondDerivativeMarix[1][0];
            secondDerivativeMarix[numberOfPoints - 2][numberOfPoints - 1] = secondDerivativeMarix[numberOfPoints - 1][numberOfPoints - 2];
        }
        secondDerivativeVector = LinearAlgebra.solveLinearEquationSymmetric(secondDerivativeMarix, v);
        this.interpolatingRationalFunctions = new RationalFunction[numberOfPoints - 1];
        for (int i = 0; i < numberOfPoints - 1; ++i) {
            double[] numeratortorPolynomCoeff = new double[]{this.values[i], (this.values[i + 1] - this.values[i]) / step[i] - (secondDerivativeVector[i + 1] + 2.0 * secondDerivativeVector[i]) * step[i] / 6.0, secondDerivativeVector[i] / 2.0, (secondDerivativeVector[i + 1] - secondDerivativeVector[i]) / (6.0 * step[i])};
            this.interpolatingRationalFunctions[i] = new RationalFunction(numeratortorPolynomCoeff);
        }
    }

    private void doCreateRationalFunctionsForAkimaInterpolation() {
        this.doCreateRationalFunctionsForAkimaInterpolation(0.0);
    }

    private void doCreateRationalFunctionsForAkimaInterpolation(double minSlopeDifferenceWeight) {
        int numberOfPoints = this.points.length;
        if (numberOfPoints < 4) {
            this.doCreateRationalFunctionsForCubicSplineInterpolation();
        } else {
            int i;
            double[] step = new double[numberOfPoints - 1];
            double[] slope = new double[numberOfPoints - 1];
            double[] absSlopeDifference = new double[numberOfPoints - 2];
            for (int i2 = 0; i2 < numberOfPoints - 1; ++i2) {
                step[i2] = this.points[i2 + 1] - this.points[i2];
                slope[i2] = (this.values[i2 + 1] - this.values[i2]) / step[i2];
                if (i2 <= 0) continue;
                absSlopeDifference[i2 - 1] = Math.abs(slope[i2] - slope[i2 - 1]) + minSlopeDifferenceWeight;
            }
            double[] derivative = new double[numberOfPoints];
            derivative[0] = 0.5 * (3.0 * slope[0] - slope[1]);
            derivative[1] = absSlopeDifference[1] == 0.0 && absSlopeDifference[0] == 0.0 ? (step[1] * slope[0] + step[0] * slope[1]) / (step[0] + step[1]) : (absSlopeDifference[1] * slope[0] + absSlopeDifference[0] * slope[1]) / (absSlopeDifference[1] + absSlopeDifference[0]);
            derivative[numberOfPoints - 2] = absSlopeDifference[numberOfPoints - 3] == 0.0 && absSlopeDifference[numberOfPoints - 4] == 0.0 ? (step[numberOfPoints - 2] * slope[numberOfPoints - 3] + step[numberOfPoints - 3] * slope[numberOfPoints - 2]) / (step[numberOfPoints - 3] + step[numberOfPoints - 2]) : (absSlopeDifference[numberOfPoints - 3] * slope[numberOfPoints - 3] + absSlopeDifference[numberOfPoints - 4] * slope[numberOfPoints - 2]) / (absSlopeDifference[numberOfPoints - 3] + absSlopeDifference[numberOfPoints - 4]);
            derivative[numberOfPoints - 1] = 0.5 * (3.0 * slope[numberOfPoints - 2] - slope[numberOfPoints - 3]);
            for (i = 2; i < numberOfPoints - 2; ++i) {
                derivative[i] = absSlopeDifference[i] == 0.0 && absSlopeDifference[i - 2] == 0.0 ? (step[i] * slope[i - 1] + step[i - 1] * slope[i]) / (step[i - 1] + step[i]) : (absSlopeDifference[i] * slope[i - 1] + absSlopeDifference[i - 2] * slope[i]) / (absSlopeDifference[i] + absSlopeDifference[i - 2]);
            }
            this.interpolatingRationalFunctions = new RationalFunction[numberOfPoints - 1];
            for (i = 0; i < numberOfPoints - 1; ++i) {
                double[] numeratorPolynomCoeff = new double[]{this.values[i], derivative[i], (3.0 * slope[i] - 2.0 * derivative[i] - derivative[i + 1]) / step[i], (derivative[i] + derivative[i + 1] - 2.0 * slope[i]) / (step[i] * step[i])};
                this.interpolatingRationalFunctions[i] = new RationalFunction(numeratorPolynomCoeff);
            }
        }
    }

    private void doCreateRationalFunctionsForHarmonicSplineInterpolation() {
        int i;
        int numberOfPoints = this.points.length;
        double[] step = new double[numberOfPoints - 1];
        double[] slope = new double[numberOfPoints - 1];
        double[] doubleStep = new double[numberOfPoints - 2];
        for (int i2 = 0; i2 < numberOfPoints - 1; ++i2) {
            step[i2] = this.points[i2 + 1] - this.points[i2];
            slope[i2] = (this.values[i2 + 1] - this.values[i2]) / step[i2];
            if (i2 <= 0) continue;
            doubleStep[i2 - 1] = this.points[i2 + 1] - this.points[i2 - 1];
        }
        double[] derivative = new double[numberOfPoints];
        derivative[0] = (2.0 * step[0] + step[1]) / doubleStep[0] * slope[0] - step[0] / doubleStep[0] * slope[1];
        derivative[numberOfPoints - 1] = (2.0 * step[numberOfPoints - 2] + step[numberOfPoints - 3]) / doubleStep[numberOfPoints - 3] * slope[numberOfPoints - 2] - step[numberOfPoints - 2] / doubleStep[numberOfPoints - 3] * slope[numberOfPoints - 3];
        if (this.interpolationMethod == InterpolationMethod.HARMONIC_SPLINE_WITH_MONOTONIC_FILTERING) {
            if (derivative[0] * slope[0] > 0.0 && slope[0] * slope[1] <= 0.0 && Math.abs(derivative[0]) < 3.0 * Math.abs(slope[0])) {
                derivative[0] = 3.0 * slope[0];
            }
            if (derivative[0] * slope[0] <= 0.0) {
                derivative[0] = 0.0;
            }
            if (derivative[numberOfPoints - 1] * slope[numberOfPoints - 2] > 0.0 && slope[numberOfPoints - 2] * slope[numberOfPoints - 3] <= 0.0 && Math.abs(derivative[numberOfPoints - 1]) < 3.0 * Math.abs(slope[numberOfPoints - 2])) {
                derivative[numberOfPoints - 1] = 3.0 * slope[numberOfPoints - 2];
            }
            if (derivative[numberOfPoints - 1] * slope[numberOfPoints - 2] <= 0.0) {
                derivative[numberOfPoints - 1] = 0.0;
            }
        }
        for (i = 1; i < numberOfPoints - 1; ++i) {
            if (slope[i - 1] * slope[i] <= 0.0) {
                derivative[i] = 0.0;
                continue;
            }
            double weightedHarmonicMean = (step[i - 1] + 2.0 * step[i]) / (3.0 * doubleStep[i - 1] * slope[i - 1]) + (2.0 * step[i - 1] + step[i]) / (3.0 * doubleStep[i - 1] * slope[i]);
            derivative[i] = 1.0 / weightedHarmonicMean;
        }
        this.interpolatingRationalFunctions = new RationalFunction[numberOfPoints - 1];
        for (i = 0; i < numberOfPoints - 1; ++i) {
            double[] numeratortorPolynomCoeff = new double[]{this.values[i], derivative[i], (3.0 * slope[i] - 2.0 * derivative[i] - derivative[i + 1]) / step[i], (derivative[i] + derivative[i + 1] - 2.0 * slope[i]) / (step[i] * step[i])};
            this.interpolatingRationalFunctions[i] = new RationalFunction(numeratortorPolynomCoeff);
        }
    }

    public static void main(String[] args) {
        double value;
        double point;
        int samplePointIndex;
        int samplePoints = 200;
        double[] givenPoints = new double[]{0.0, 1.0, 2.0, 3.0, 4.0, 5.0};
        double[] givenValues = new double[]{5.0, 6.0, 4.0, 7.0, 5.0, 6.0};
        System.out.println("Interplation of given input points (x,y):");
        System.out.println("  x: " + Arrays.toString(givenPoints));
        System.out.println("  y: " + Arrays.toString(givenValues));
        System.out.println();
        System.out.println("Default:");
        RationalFunctionInterpolation interpolation = new RationalFunctionInterpolation(givenPoints, givenValues);
        for (samplePointIndex = 0; samplePointIndex < 200; ++samplePointIndex) {
            point = givenPoints[0] + (double)samplePointIndex / 199.0 * givenPoints[givenPoints.length - 1] - givenPoints[0];
            value = interpolation.getValue(point);
            System.out.println(point + "\t" + value);
        }
        System.out.println();
        interpolation = new RationalFunctionInterpolation(givenPoints, givenValues, InterpolationMethod.AKIMA, ExtrapolationMethod.CONSTANT);
        System.out.println("AKIMA:");
        for (samplePointIndex = 0; samplePointIndex < 200; ++samplePointIndex) {
            point = givenPoints[0] + (double)samplePointIndex / 199.0 * givenPoints[givenPoints.length - 1] - givenPoints[0];
            value = interpolation.getValue(point);
            System.out.println(point + "\t" + value);
        }
        System.out.println();
        interpolation = new RationalFunctionInterpolation(givenPoints, givenValues, InterpolationMethod.CUBIC_SPLINE, ExtrapolationMethod.CONSTANT);
        System.out.println("CUBIC_SPLINE:");
        for (samplePointIndex = 0; samplePointIndex < 200; ++samplePointIndex) {
            point = givenPoints[0] + (double)samplePointIndex / 199.0 * givenPoints[givenPoints.length - 1] - givenPoints[0];
            value = interpolation.getValue(point);
            System.out.println(point + "\t" + value);
        }
        System.out.println();
        interpolation = new RationalFunctionInterpolation(givenPoints, givenValues, InterpolationMethod.PIECEWISE_CONSTANT, ExtrapolationMethod.CONSTANT);
        System.out.println("PIECEWISE_CONSTANT:");
        for (samplePointIndex = 0; samplePointIndex < 200; ++samplePointIndex) {
            point = givenPoints[0] + (double)samplePointIndex / 199.0 * givenPoints[givenPoints.length - 1] - givenPoints[0];
            value = interpolation.getValue(point);
            System.out.println(point + "\t" + value);
        }
        System.out.println();
        interpolation = new RationalFunctionInterpolation(givenPoints, givenValues, InterpolationMethod.HARMONIC_SPLINE, ExtrapolationMethod.CONSTANT);
        System.out.println("HARMONIC_SPLINE:");
        for (samplePointIndex = 0; samplePointIndex < 200; ++samplePointIndex) {
            point = givenPoints[0] + (double)samplePointIndex / 199.0 * givenPoints[givenPoints.length - 1] - givenPoints[0];
            value = interpolation.getValue(point);
            System.out.println(point + "\t" + value);
        }
        System.out.println();
    }

    @Override
    public double applyAsDouble(double operand) {
        return this.getValue(operand);
    }

    private void readObject(ObjectInputStream in) throws ClassNotFoundException, IOException {
        in.defaultReadObject();
        this.interpolatingRationalFunctionsLazyInitLock = new Object();
    }

    public static enum InterpolationMethod {
        PIECEWISE_CONSTANT,
        PIECEWISE_CONSTANT_LEFTPOINT,
        PIECEWISE_CONSTANT_RIGHTPOINT,
        LINEAR,
        CUBIC_SPLINE,
        AKIMA,
        AKIMA_CONTINUOUS,
        HARMONIC_SPLINE,
        HARMONIC_SPLINE_WITH_MONOTONIC_FILTERING;

    }

    public static enum ExtrapolationMethod {
        DEFAULT,
        CONSTANT,
        LINEAR;

    }

    private static class RationalFunction
    implements Serializable {
        private static final long serialVersionUID = -1596026703859403853L;
        private final double[] coefficientsNumerator;
        private final double[] coefficientsDenominator;

        RationalFunction(double[] coefficientsNumerator, double[] coefficientsDenominator) {
            this.coefficientsNumerator = coefficientsNumerator;
            this.coefficientsDenominator = coefficientsDenominator;
        }

        RationalFunction(double[] coefficients) {
            this(coefficients, null);
        }

        public double getValue(double x) {
            double valueNumerator = 0.0;
            double valueDenominator = 0.0;
            double powerOfX = 1.0;
            for (double polynomialCoefficient : this.coefficientsNumerator) {
                valueNumerator += polynomialCoefficient * powerOfX;
                powerOfX *= x;
            }
            if (this.coefficientsDenominator == null) {
                return valueNumerator;
            }
            powerOfX = 1.0;
            for (double polynomialCoefficient : this.coefficientsDenominator) {
                valueDenominator += polynomialCoefficient * powerOfX;
                powerOfX *= x;
            }
            return valueNumerator / valueDenominator;
        }
    }
}

