package net.guerlab.web.result;

import net.guerlab.commons.collection.CollectionUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 远端异常
 *
 * @author guer
 */
public class RemoteException extends RuntimeException {

    /**
     * 应用堆栈跟踪
     */
    private final ApplicationStackTrace applicationStackTrace;

    private RemoteException(String message, ApplicationStackTrace applicationStackTrace, RemoteException cause) {
        super(message, cause);
        this.applicationStackTrace = applicationStackTrace;
    }

    /**
     * 构造远端异常
     *
     * @param message
     *         异常信息
     * @param applicationStackTraces
     *         应用堆栈列表
     * @return 远端异常
     */
    public static RemoteException build(String message, List<ApplicationStackTrace> applicationStackTraces) {
        if (CollectionUtil.isEmpty(applicationStackTraces)) {
            return new RemoteException(message, new ApplicationStackTrace(), null);
        }

        RemoteException exception = null;
        for (int i = applicationStackTraces.size() - 1; i >= 0; i--) {
            exception = new RemoteException(message, applicationStackTraces.get(i), exception);
        }

        return exception;
    }

    private static void fillApplicationStackTrace(List<ApplicationStackTrace> list, Throwable cause) {
        if (!(cause instanceof RemoteException)) {
            return;
        }

        RemoteException e = (RemoteException) cause;

        list.add(e.applicationStackTrace);
        fillApplicationStackTrace(list, e.getCause());
    }

    @Override
    public StackTraceElement[] getStackTrace() {
        if (applicationStackTrace.getStackTrace() == null) {
            return new StackTraceElement[] {};
        }
        return applicationStackTrace.getStackTrace().stream().map(this::buildStackTraceElement)
                .collect(Collectors.toList()).toArray(new StackTraceElement[] {});
    }

    private StackTraceElement buildStackTraceElement(String stackTrace) {
        int lastPointIndex = stackTrace.lastIndexOf(".");
        int colonIndex = stackTrace.lastIndexOf(":");
        String declaringClass = stackTrace.substring(0, lastPointIndex);
        String methodName = stackTrace.substring(lastPointIndex + 1, colonIndex);
        int lineNumber = Integer.parseInt(stackTrace.substring(colonIndex + 1));

        return new StackTraceElement(declaringClass, methodName, "", lineNumber);
    }

    /**
     * 获取应用堆栈跟踪
     *
     * @return 应用堆栈跟踪
     */
    public ApplicationStackTrace getApplicationStackTrace() {
        return applicationStackTrace;
    }

    /**
     * 获取应用堆栈跟踪列表
     *
     * @return 应用堆栈跟踪列表
     */
    public List<ApplicationStackTrace> getApplicationStackTraces() {
        List<ApplicationStackTrace> list = new ArrayList<>();
        fillApplicationStackTrace(list, this);
        return list;
    }
}
