/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.web.result;

import java.util.ArrayList;
import java.util.List;

/**
 * 通用JSON返回结果集
 *
 * @param <T>
 *         数据类型
 * @author guer
 */
public class Result<T> {

    /**
     * 响应状态
     */
    protected boolean status;

    /**
     * 消息
     */
    protected String message;

    /**
     * 数据
     */
    protected T data;

    /**
     * 错误码
     */
    protected int errorCode;

    /**
     * 错误详情
     */
    protected String errorDetail;

    /**
     * 堆栈跟踪列表
     */
    protected List<ApplicationStackTrace> stackTraces = new ArrayList<>();

    /**
     * 无参构造
     */
    public Result() {
        this(null, null);
    }

    /**
     * 通过设置消息内容来初始化结果集
     *
     * @param message
     *         消息内容
     */
    public Result(String message) {
        this(false, message, null);
    }

    /**
     * 通过设置消息内容和数据来初始化结果集
     *
     * @param message
     *         消息内容
     * @param data
     *         数据
     */
    public Result(String message, T data) {
        this(false, message, data);
    }

    /**
     * 通过设置消息内容和数据来初始化结果集
     *
     * @param message
     *         消息内容
     * @param errorCode
     *         错误码
     */
    public Result(String message, int errorCode) {
        this(false, message, null, errorCode);
    }

    /**
     * 通过设置响应状态、消息内容和数据来初始化结果集
     *
     * @param status
     *         响应状态
     * @param message
     *         消息内容
     * @param data
     *         数据
     */
    public Result(boolean status, String message, T data) {
        this(status, message, data, 0);
    }

    /**
     * 通过设置响应状态、消息内容和数据来初始化结果集
     *
     * @param status
     *         响应状态
     * @param message
     *         消息内容
     * @param data
     *         数据
     * @param errorCode
     *         错误码
     */
    public Result(boolean status, String message, T data, int errorCode) {
        this.status = status;
        this.message = message;
        this.data = data;
        this.errorCode = errorCode;
    }

    /**
     * 获取响应状态
     *
     * @return 响应状态
     */
    public final boolean isStatus() {
        return status;
    }

    /**
     * 设置响应状态
     *
     * @param status
     *         响应状态
     */
    public final void setStatus(boolean status) {
        this.status = status;
    }

    /**
     * 获取消息内容
     *
     * @return 消息内容
     */
    public final String getMessage() {
        return message;
    }

    /**
     * 设置消息内容
     *
     * @param message
     *         消息内容
     */
    public final void setMessage(String message) {
        this.message = message;
    }

    /**
     * 获取数据
     *
     * @return 数据
     */
    public final T getData() {
        return data;
    }

    /**
     * 设置数据
     *
     * @param data
     *         数据
     */
    public final void setData(T data) {
        this.data = data;
    }

    /**
     * 返回错误码
     *
     * @return 错误码
     */
    public int getErrorCode() {
        return errorCode;
    }

    /**
     * 设置错误码
     *
     * @param errorCode
     *         错误码
     */
    public void setErrorCode(int errorCode) {
        this.errorCode = errorCode;
    }

    /**
     * 获取错误详情
     *
     * @return 错误详情
     */
    public String getErrorDetail() {
        return errorDetail;
    }

    /**
     * 设置错误详情
     *
     * @param errorDetail
     *         错误详情
     */
    public void setErrorDetail(String errorDetail) {
        this.errorDetail = errorDetail;
    }

    /**
     * 获取堆栈跟踪列表
     *
     * @return 堆栈跟踪列表
     */
    public List<ApplicationStackTrace> getStackTraces() {
        return stackTraces;
    }

    /**
     * 设置堆栈跟踪列表
     *
     * @param stackTraces
     *         堆栈跟踪列表
     */
    public void setStackTraces(List<ApplicationStackTrace> stackTraces) {
        this.stackTraces = stackTraces;
    }

    /**
     * 添加堆栈跟踪列表
     *
     * @param stackTraces
     *         堆栈跟踪列表
     * @return 通用JSON返回结果集
     */
    public Result<T> addStackTraces(List<ApplicationStackTrace> stackTraces) {
        if (stackTraces != null && this.stackTraces != null) {
            this.stackTraces.addAll(stackTraces);
        }
        return this;
    }

    /**
     * 添加堆栈跟踪
     *
     * @param stackTrace
     *         堆栈跟踪
     * @return 通用JSON返回结果集
     */
    public Result<T> addStackTraces(ApplicationStackTrace stackTrace) {
        if (stackTrace != null && stackTraces != null) {
            stackTraces.add(stackTrace);
        }
        return this;
    }

    @Override
    public String toString() {
        return "Result [status=" + status + ", message=" + message + ", data=" + data + ", errorCode=" + errorCode
                + ", errorDetail=" + errorDetail + "]";
    }
}
