/*
 * Decompiled with CFR 0.152.
 */
package net.i2p.router.crypto;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import net.i2p.crypto.CertUtil;
import net.i2p.crypto.KeyStoreUtil;
import net.i2p.crypto.SigType;
import net.i2p.crypto.SigUtil;
import net.i2p.data.Base64;
import net.i2p.data.DataHelper;
import net.i2p.data.Hash;
import net.i2p.data.Signature;
import net.i2p.data.SigningPrivateKey;
import net.i2p.data.SigningPublicKey;
import net.i2p.data.router.RouterInfo;
import net.i2p.router.RouterContext;
import net.i2p.util.ConcurrentHashSet;
import net.i2p.util.FileSuffixFilter;
import net.i2p.util.Log;
import net.i2p.util.SecureDirectory;

public class FamilyKeyCrypto {
    private final RouterContext _context;
    private final Log _log;
    private final Map<Hash, Verified> _verified;
    private final Map<String, SigningPublicKey> _knownKeys;
    private final Map<Hash, Result> _negativeCache;
    private final Set<Hash> _ourFamily;
    private final String _fname;
    private final SigningPrivateKey _privkey;
    private final SigningPublicKey _pubkey;
    public static final String PROP_KEYSTORE_PASSWORD = "netdb.family.keystorePassword";
    public static final String PROP_FAMILY_NAME = "netdb.family.name";
    public static final String PROP_KEY_PASSWORD = "netdb.family.keyPassword";
    public static final String CERT_SUFFIX = ".crt";
    public static final String CRL_SUFFIX = ".crl";
    public static final String KEYSTORE_PREFIX = "family-";
    public static final String KEYSTORE_SUFFIX = ".ks";
    public static final String CN_SUFFIX = ".family.i2p.net";
    private static final int DEFAULT_KEY_VALID_DAYS = 3652;
    private static final String DEFAULT_KEY_ALGORITHM = SigType.ECDSA_SHA256_P256.isAvailable() ? "EC" : "DSA";
    private static final int DEFAULT_KEY_SIZE = SigType.ECDSA_SHA256_P256.isAvailable() ? 256 : 1024;
    private static final String KS_DIR = "keystore";
    private static final String CERT_DIR = "certificates/family";
    private static final String CRL_DIR = "crls";
    public static final String OPT_NAME = "family";
    public static final String OPT_SIG = "family.sig";
    public static final String OPT_KEY = "family.key";

    public FamilyKeyCrypto(RouterContext context) throws GeneralSecurityException {
        this._context = context;
        this._log = this._context.logManager().getLog(FamilyKeyCrypto.class);
        this._fname = this._context.getProperty(PROP_FAMILY_NAME);
        if (this._fname != null && (this._fname.contains("/") || this._fname.contains("\\") || this._fname.contains("..") || new File(this._fname).isAbsolute() || this._fname.length() <= 0)) {
            throw new GeneralSecurityException("Illegal family name: " + this._fname);
        }
        this._privkey = this._fname != null ? this.initialize() : null;
        this._pubkey = this._privkey != null ? this._privkey.toPublic() : null;
        this._verified = new ConcurrentHashMap<Hash, Verified>(16);
        this._negativeCache = new ConcurrentHashMap<Hash, Result>(4);
        this._ourFamily = this._privkey != null ? new ConcurrentHashSet(4) : Collections.emptySet();
        this._knownKeys = new HashMap<String, SigningPublicKey>(8);
        this.loadCerts();
    }

    private SigningPrivateKey initialize() throws GeneralSecurityException {
        SecureDirectory dir = new SecureDirectory(this._context.getConfigDir(), KS_DIR);
        File keyStore = new File((File)dir, KEYSTORE_PREFIX + this._fname + KEYSTORE_SUFFIX);
        this.verifyKeyStore(keyStore);
        return this.getPrivKey(keyStore);
    }

    public void shutdown() {
        this._verified.clear();
        this._negativeCache.clear();
    }

    public Map<String, String> sign(String family, Hash h) throws GeneralSecurityException {
        if (this._privkey == null) {
            this._log.logAlways(30, "family name now set, must restart router to generate or load keys");
            throw new GeneralSecurityException("family name now set, must restart router to generate or load keys");
        }
        if (h == null) {
            throw new GeneralSecurityException("null router hash");
        }
        if (!this._fname.equals(family)) {
            this._log.logAlways(30, "family name changed, must restart router to generate or load new keys");
            throw new GeneralSecurityException("family name changed, must restart router to generate or load new keys");
        }
        byte[] nb = DataHelper.getUTF8((String)this._fname);
        int len = nb.length + 32;
        byte[] b = new byte[len];
        System.arraycopy(nb, 0, b, 0, nb.length);
        System.arraycopy(h.getData(), 0, b, nb.length, 32);
        Signature sig = this._context.dsa().sign(b, this._privkey);
        if (sig == null) {
            throw new GeneralSecurityException("sig failed");
        }
        HashMap<String, String> rv = new HashMap<String, String>(3);
        rv.put(OPT_NAME, family);
        rv.put(OPT_KEY, this._pubkey.getType().getCode() + ":" + this._pubkey.toBase64());
        rv.put(OPT_SIG, sig.toBase64());
        return rv;
    }

    public boolean hasFamily() {
        return this._pubkey != null;
    }

    public Set<Hash> getOurFamily() {
        return this._ourFamily;
    }

    public String getOurFamilyName() {
        return this._fname;
    }

    public Result verify(RouterInfo ri) {
        String name = ri.getOption(OPT_NAME);
        if (name == null) {
            return Result.NO_FAMILY;
        }
        Result rv = this.verify(ri, name);
        if (this._log.shouldInfo()) {
            this._log.info("Result: " + (Object)((Object)rv) + " for " + name + ' ' + ri.getHash());
        }
        return rv;
    }

    public boolean verifyOurFamily(RouterInfo ri) {
        boolean rv;
        if (this._pubkey == null) {
            return false;
        }
        String name = ri.getOption(OPT_NAME);
        if (!this._fname.equals(name)) {
            return false;
        }
        Hash h = ri.getHash();
        if (this._ourFamily.contains(h)) {
            return true;
        }
        if (h.equals((Object)this._context.routerHash())) {
            return false;
        }
        boolean bl = rv = this.verify(ri, name) == Result.STORED_KEY;
        if (rv) {
            this._ourFamily.add(h);
            this._log.logAlways(20, "Found and verified member of our family (" + this._fname + "): " + h);
        } else if (this._log.shouldWarn()) {
            this._log.warn("Found spoofed member of our family (" + this._fname + "): " + h);
        }
        return rv;
    }

    private Result verify(RouterInfo ri, String name) {
        Result rv;
        Signature sig;
        boolean isKnownKey;
        SigningPublicKey spk;
        Hash h = ri.getHash();
        String ssig = ri.getOption(OPT_SIG);
        if (ssig == null) {
            return Result.NO_SIG;
        }
        Verified v = this._verified.get(h);
        if (v != null) {
            if (!v.name.equals(name)) {
                return Result.NAME_CHANGED;
            }
            if (v.sig.equals(ssig)) {
                return v.result;
            }
            this._verified.remove(h);
        }
        if (name.equals(this._fname)) {
            spk = this._pubkey;
            isKnownKey = true;
        } else {
            Result r = this._negativeCache.get(h);
            if (r != null) {
                return r;
            }
            spk = this._knownKeys.get(name);
            boolean bl = isKnownKey = spk != null;
            if (!isKnownKey) {
                String skey = ri.getOption(OPT_KEY);
                if (skey != null) {
                    int colon = skey.indexOf(58);
                    if (colon < 0) {
                        colon = skey.indexOf(59);
                    }
                    if (colon > 0) {
                        try {
                            byte[] bkey;
                            int code = Integer.parseInt(skey.substring(0, colon));
                            SigType type = SigType.getByCode((int)code);
                            if (type != null && (bkey = Base64.decode((String)skey.substring(colon + 1))) != null) {
                                spk = new SigningPublicKey(type, bkey);
                            }
                        }
                        catch (NumberFormatException e) {
                            if (this._log.shouldInfo()) {
                                this._log.info("Bad b64 family key: " + (Object)((Object)ri), (Throwable)e);
                            }
                            this._negativeCache.put(h, Result.BAD_KEY);
                            return Result.BAD_KEY;
                        }
                        catch (IllegalArgumentException e) {
                            if (this._log.shouldInfo()) {
                                this._log.info("Bad b64 family key: " + (Object)((Object)ri), (Throwable)e);
                            }
                            this._negativeCache.put(h, Result.BAD_KEY);
                            return Result.BAD_KEY;
                        }
                        catch (ArrayIndexOutOfBoundsException e) {
                            if (this._log.shouldInfo()) {
                                this._log.info("Bad b64 family key: " + (Object)((Object)ri), (Throwable)e);
                            }
                            this._negativeCache.put(h, Result.BAD_KEY);
                            return Result.BAD_KEY;
                        }
                    }
                }
                if (spk == null) {
                    this._negativeCache.put(h, Result.NO_KEY);
                    return Result.NO_KEY;
                }
            }
        }
        if (!spk.getType().isAvailable()) {
            this._negativeCache.put(h, Result.UNSUPPORTED_SIG);
            return Result.UNSUPPORTED_SIG;
        }
        byte[] bsig = Base64.decode((String)ssig);
        if (bsig == null) {
            this._negativeCache.put(h, Result.INVALID_SIG);
            return Result.INVALID_SIG;
        }
        try {
            sig = new Signature(spk.getType(), bsig);
        }
        catch (IllegalArgumentException iae) {
            this._negativeCache.put(h, Result.INVALID_SIG);
            return Result.INVALID_SIG;
        }
        byte[] nb = DataHelper.getUTF8((String)name);
        byte[] b = new byte[nb.length + 32];
        System.arraycopy(nb, 0, b, 0, nb.length);
        System.arraycopy(ri.getHash().getData(), 0, b, nb.length, 32);
        boolean ok = this._context.dsa().verifySignature(sig, b, spk);
        if (ok) {
            rv = isKnownKey ? Result.STORED_KEY : Result.RI_KEY;
            this._verified.put(h, new Verified(name, ssig, rv));
        } else {
            rv = Result.BAD_SIG;
            this._negativeCache.put(h, rv);
        }
        return rv;
    }

    private void verifyKeyStore(File ks) throws GeneralSecurityException {
        SecureDirectory sdir;
        if (ks.exists()) {
            if (this._context.getProperty(PROP_KEY_PASSWORD) == null) {
                String s = "Family key error, must set netdb.family.keyPassword in " + new File(this._context.getConfigDir(), "router.config").getAbsolutePath();
                this._log.error(s);
                throw new GeneralSecurityException(s);
            }
            return;
        }
        File dir = ks.getParentFile();
        if (!dir.exists() && !(sdir = new SecureDirectory(dir.getAbsolutePath())).mkdirs()) {
            String s = "Family key error, must set netdb.family.keyPassword in " + new File(this._context.getConfigDir(), "router.config").getAbsolutePath();
            this._log.error(s);
            throw new GeneralSecurityException(s);
        }
        try {
            this.createKeyStore(ks);
        }
        catch (IOException ioe) {
            throw new GeneralSecurityException("Failed to create NetDb family keystore", ioe);
        }
    }

    private void createKeyStore(File ks) throws GeneralSecurityException, IOException {
        String keyPassword = KeyStoreUtil.randomString();
        String cname = this._fname + CN_SUFFIX;
        Object[] rv = KeyStoreUtil.createKeysAndCRL((File)ks, (String)"changeit", (String)this._fname, (String)cname, (String)OPT_NAME, (int)3652, (String)DEFAULT_KEY_ALGORITHM, (int)DEFAULT_KEY_SIZE, (String)keyPassword);
        HashMap<String, String> changes = new HashMap<String, String>();
        changes.put(PROP_KEYSTORE_PASSWORD, "changeit");
        changes.put(PROP_KEY_PASSWORD, keyPassword);
        changes.put(PROP_FAMILY_NAME, this._fname);
        this._context.router().saveConfig(changes, null);
        this._log.logAlways(20, "Created new private key for netdb family \"" + this._fname + "\" in keystore: " + ks.getAbsolutePath() + "\nCopy the keystore to the other routers in the family,\nand add the following entries to their router.config file:\n" + PROP_FAMILY_NAME + '=' + this._fname + '\n' + PROP_KEYSTORE_PASSWORD + '=' + "changeit" + '\n' + PROP_KEY_PASSWORD + '=' + keyPassword);
        X509Certificate cert = (X509Certificate)rv[2];
        this.exportCert(cert);
        X509CRL crl = (X509CRL)rv[3];
        this.exportCRL(ks.getParentFile(), crl);
    }

    private void exportCert(X509Certificate cert) {
        SecureDirectory sdir = new SecureDirectory(this._context.getConfigDir(), CERT_DIR);
        if (sdir.exists() || sdir.mkdirs()) {
            String name = this._fname.replace("@", "_at_") + CERT_SUFFIX;
            File out = new File((File)sdir, name);
            boolean success = CertUtil.saveCert((Certificate)cert, (File)out);
            if (success) {
                this._log.logAlways(20, "Created new public key certificate for netdb family \"" + this._fname + "\" in file: " + out.getAbsolutePath() + "\nThe certificate will be associated with your router identity.\nCopy the certificate to the directory $I2P/" + CERT_DIR + " for each of the other routers in the family.\nGive this certificate to an I2P developer for inclusion in the next I2P release.");
            } else {
                this._log.error("Error saving family key certificate");
            }
        } else {
            this._log.error("Error saving family key certificate");
        }
    }

    private void exportCRL(File ksdir, X509CRL crl) {
        SecureDirectory sdir = new SecureDirectory(ksdir, CRL_DIR);
        if (sdir.exists() || sdir.mkdirs()) {
            String name = KEYSTORE_PREFIX + this._fname.replace("@", "_at_") + '-' + System.currentTimeMillis() + CRL_SUFFIX;
            File out = new File((File)sdir, name);
            boolean success = CertUtil.saveCRL((X509CRL)crl, (File)out);
            if (success) {
                this._log.logAlways(20, "Created certificate revocation list (CRL) for netdb family \"" + this._fname + "\" in file: " + out.getAbsolutePath() + "\nBack up the keystore and CRL files and keep them secure.\nIf your private key is ever compromised, give the CRL to an I2P developer for publication.");
            } else {
                this._log.error("Error saving family key CRL");
            }
        } else {
            this._log.error("Error saving family key CRL");
        }
    }

    private void loadCerts() {
        File dir = new File(this._context.getBaseDir(), CERT_DIR);
        File[] files = dir.listFiles((FileFilter)new FileSuffixFilter(CERT_SUFFIX));
        if (files == null) {
            return;
        }
        for (File file : files) {
            String name = file.getName();
            name = name.substring(0, name.length() - CERT_SUFFIX.length());
            SigningPublicKey spk = this.loadCert(file);
            if (spk == null) continue;
            this._knownKeys.put(name, spk);
        }
        if (this._log.shouldInfo()) {
            this._log.info("Loaded " + this._knownKeys.size() + " keys");
        }
    }

    private SigningPublicKey loadCert(File file) {
        try {
            PublicKey pk = CertUtil.loadKey((File)file);
            return SigUtil.fromJavaKey((PublicKey)pk);
        }
        catch (GeneralSecurityException gse) {
            this._log.error("Error loading family key " + file, (Throwable)gse);
        }
        catch (IOException ioe) {
            this._log.error("Error loading family key " + file, (Throwable)ioe);
        }
        return null;
    }

    private SigningPrivateKey getPrivKey(File ks) throws GeneralSecurityException {
        String ksPass = this._context.getProperty(PROP_KEYSTORE_PASSWORD, "changeit");
        String keyPass = this._context.getProperty(PROP_KEY_PASSWORD);
        if (keyPass == null) {
            throw new GeneralSecurityException("No key password, set netdb.family.keyPassword in " + new File(this._context.getConfigDir(), "router.config").getAbsolutePath());
        }
        try {
            PrivateKey pk = KeyStoreUtil.getPrivateKey((File)ks, (String)ksPass, (String)this._fname, (String)keyPass);
            if (pk == null) {
                throw new GeneralSecurityException("Family key not found: " + this._fname);
            }
            String familyName = this._fname.replace("@", "_at_");
            File dir = new File(this._context.getBaseDir(), CERT_DIR);
            File file = new File(dir, familyName + CERT_SUFFIX);
            if (!file.exists()) {
                KeyStoreUtil.exportCert((File)ks, (String)ksPass, (String)this._fname, (File)file);
            }
            return SigUtil.fromJavaKey((PrivateKey)pk);
        }
        catch (IOException ioe) {
            throw new GeneralSecurityException("Error loading family key " + this._fname, ioe);
        }
    }

    public static void main(String[] args) {
        File ks;
        if (args.length != 2) {
            System.err.println("Usage: FamilyKeyCrypto keystore.ks familyname");
            System.exit(1);
        }
        if ((ks = new File(args[0])).exists()) {
            System.err.println("Keystore already exists: " + ks);
            System.exit(1);
        }
        String fname = args[1];
        String cname = fname + CN_SUFFIX;
        String keyPassword = KeyStoreUtil.randomString();
        try {
            KeyStoreUtil.createKeysAndCRL((File)ks, (String)"changeit", (String)fname, (String)cname, (String)OPT_NAME, (int)3652, (String)DEFAULT_KEY_ALGORITHM, (int)DEFAULT_KEY_SIZE, (String)keyPassword);
            System.out.println("Family keys generated and saved in " + ks + '\n' + "Copy to " + KS_DIR + '/' + KEYSTORE_PREFIX + fname + KEYSTORE_SUFFIX + " in the i2p configuration directory\nFamily key configuration for router.config:\n" + PROP_FAMILY_NAME + '=' + fname + '\n' + PROP_KEYSTORE_PASSWORD + '=' + "changeit" + '\n' + PROP_KEY_PASSWORD + '=' + keyPassword);
        }
        catch (Exception e) {
            System.err.println("Failed");
            e.printStackTrace();
            System.exit(1);
        }
    }

    private static class Verified {
        public final String name;
        public final String sig;
        public final Result result;

        public Verified(String n, String s, Result r) {
            this.name = n;
            this.sig = s;
            this.result = r;
        }
    }

    public static enum Result {
        NO_FAMILY,
        NO_KEY,
        NO_SIG,
        NAME_CHANGED,
        SIG_CHANGED,
        INVALID_SIG,
        UNSUPPORTED_SIG,
        BAD_KEY,
        BAD_SIG,
        RI_KEY,
        STORED_KEY;

    }
}

