/*
 * Decompiled with CFR 0.152.
 */
package net.jacobpeterson.alpaca.websocket;

import com.google.common.base.Preconditions;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.Future;
import net.jacobpeterson.alpaca.websocket.AlpacaWebsocketInterface;
import net.jacobpeterson.alpaca.websocket.AlpacaWebsocketStateListener;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AlpacaWebsocket
extends WebSocketListener
implements AlpacaWebsocketInterface {
    public static final int WEBSOCKET_NORMAL_CLOSURE_CODE = 1000;
    public static final String WEBSOCKET_NORMAL_CLOSURE_MESSAGE = "Normal closure";
    public static int MAX_RECONNECT_ATTEMPTS = 5;
    public static Duration RECONNECT_SLEEP_INTERVAL = Duration.ofSeconds(1L);
    private static final Logger LOGGER = LoggerFactory.getLogger(AlpacaWebsocket.class);
    protected final OkHttpClient okHttpClient;
    protected final HttpUrl websocketURL;
    protected final String websocketName;
    protected AlpacaWebsocketStateListener alpacaWebsocketStateListener;
    private WebSocket websocket;
    protected boolean connected;
    protected boolean authenticated;
    protected CompletableFuture<Boolean> authenticationMessageFuture;
    protected boolean intentionalClose;
    protected int reconnectAttempts;
    protected boolean automaticallyReconnect;

    protected AlpacaWebsocket(OkHttpClient okHttpClient, HttpUrl websocketURL, String websocketName) {
        Preconditions.checkNotNull((Object)okHttpClient);
        Preconditions.checkNotNull((Object)websocketURL);
        Preconditions.checkNotNull((Object)websocketName);
        this.okHttpClient = okHttpClient;
        this.websocketURL = websocketURL;
        this.websocketName = websocketName;
        this.automaticallyReconnect = true;
    }

    @Override
    public void connect() {
        if (!this.isConnected()) {
            Request websocketRequest = new Request.Builder().url(this.websocketURL).get().build();
            this.websocket = this.okHttpClient.newWebSocket(websocketRequest, (WebSocketListener)this);
        }
    }

    @Override
    public void disconnect() {
        if (this.websocket != null && this.isConnected()) {
            this.intentionalClose = true;
            this.websocket.close(1000, WEBSOCKET_NORMAL_CLOSURE_MESSAGE);
        } else {
            this.cleanupState();
        }
    }

    @Override
    public boolean isConnected() {
        return this.connected;
    }

    @Override
    public boolean isAuthenticated() {
        return this.authenticated;
    }

    public void onOpen(@NotNull WebSocket webSocket, @NotNull Response response) {
        this.connected = true;
        LOGGER.info("{} websocket response: response={}", (Object)this.websocketName, (Object)response);
        ForkJoinPool.commonPool().execute(() -> {
            if (this.reconnectAttempts > 0) {
                this.onReconnection();
            } else {
                this.onConnection();
            }
        });
        if (this.alpacaWebsocketStateListener != null) {
            this.alpacaWebsocketStateListener.onOpen(response);
        }
    }

    public void onClosed(@NotNull WebSocket webSocket, int code, @NotNull String reason) {
        this.connected = false;
        if (this.intentionalClose) {
            LOGGER.info("{} websocket closed. code={}, reason={}", new Object[]{this.websocketName, code, reason});
            this.cleanupState();
        } else {
            LOGGER.error("{} websocket closed unintentionally! code={}, reason={}", new Object[]{this.websocketName, code, reason});
            this.handleReconnectionAttempt();
        }
        if (this.alpacaWebsocketStateListener != null) {
            this.alpacaWebsocketStateListener.onClosed(code, reason);
        }
    }

    public void onFailure(@NotNull WebSocket webSocket, @NotNull Throwable cause, @Nullable Response response) {
        if (this.intentionalClose) {
            this.onClosed(webSocket, 1000, WEBSOCKET_NORMAL_CLOSURE_MESSAGE);
            return;
        }
        LOGGER.error("{} websocket failure!", (Object)this.websocketName, (Object)cause);
        this.connected = false;
        this.handleReconnectionAttempt();
        if (this.alpacaWebsocketStateListener != null) {
            this.alpacaWebsocketStateListener.onFailure(cause);
        }
    }

    private void handleReconnectionAttempt() {
        if (!this.automaticallyReconnect) {
            return;
        }
        if (this.reconnectAttempts < MAX_RECONNECT_ATTEMPTS) {
            LOGGER.info("Attempting to reconnect {} websocket in {} seconds... (attempt {} of {})", new Object[]{this.websocketName, RECONNECT_SLEEP_INTERVAL.toSeconds(), this.reconnectAttempts + 1, MAX_RECONNECT_ATTEMPTS});
            ++this.reconnectAttempts;
            ForkJoinPool.commonPool().execute(() -> {
                try {
                    Thread.sleep(RECONNECT_SLEEP_INTERVAL.toMillis());
                }
                catch (InterruptedException interruptedException) {
                    return;
                }
                this.connect();
            });
        } else {
            LOGGER.error("Exhausted {} reconnection attempts. Not attempting to reconnect.", (Object)MAX_RECONNECT_ATTEMPTS);
            this.cleanupState();
        }
    }

    protected void cleanupState() {
        this.websocket = null;
        this.connected = false;
        this.authenticated = false;
        if (this.authenticationMessageFuture != null && !this.authenticationMessageFuture.isDone()) {
            this.authenticationMessageFuture.complete(false);
        }
        this.authenticationMessageFuture = null;
        this.intentionalClose = false;
        this.reconnectAttempts = 0;
    }

    protected void sendWebsocketMessage(String message) {
        if (!this.isConnected()) {
            throw new IllegalStateException("This websocket must be connected before send a message!");
        }
        LOGGER.trace("Websocket message sent: {}", (Object)message);
        this.websocket.send(message);
    }

    protected abstract void onConnection();

    protected abstract void onReconnection();

    protected abstract void sendAuthenticationMessage();

    @Override
    public Future<Boolean> getAuthorizationFuture() {
        if (this.authenticationMessageFuture == null) {
            this.authenticationMessageFuture = new CompletableFuture();
        }
        return this.authenticationMessageFuture;
    }

    @Override
    public void setAlpacaWebsocketStateListener(AlpacaWebsocketStateListener alpacaWebsocketStateListener) {
        this.alpacaWebsocketStateListener = alpacaWebsocketStateListener;
    }

    @Override
    public boolean doesAutomaticallyReconnect() {
        return this.automaticallyReconnect;
    }

    @Override
    public void setAutomaticallyReconnect(boolean automaticallyReconnect) {
        this.automaticallyReconnect = automaticallyReconnect;
    }

    public OkHttpClient getOkHttpClient() {
        return this.okHttpClient;
    }

    public HttpUrl getWebsocketURL() {
        return this.websocketURL;
    }

    public WebSocket getWebsocket() {
        return this.websocket;
    }
}

