package ext {

[Native("Ext.ClassManager", require)]
/**
 * Ext.ClassManager manages all classes and handles mapping from string class name to
 * actual class objects throughout the whole framework. It is not generally accessed directly,
 * rather through these convenient shorthands:
 * <ul>
 * <li>→<code>ext.Ext.define()</code></li>
 * <li>→<code>ext.Ext.create()</code></li>
 * <li>→<code>ext.Ext.widget()</code></li>
 * <li>→<code>ext.Ext.getClass()</code></li>
 * <li>→<code>ext.Ext.getClassName()</code></li>
 * </ul>
 * <p><b>Basic syntax:</b></p>
 * <pre>
 * Ext.define(className, properties);
 * </pre>
 * <p>in which <code>properties</code> is an object represent a collection of properties that apply to the class.
 * See →<code>ext.ClassManager.create()</code> for more detailed instructions.</p>
 * <pre>
 * Ext.define('Person', {
 *      name: 'Unknown',
 *
 *      constructor: function(name) {
 *          if (name) {
 *              this.name = name;
 *          }
 *      },
 *
 *      eat: function(foodType) {
 *          alert("I'm eating: " + foodType);
 *
 *          return this;
 *      }
 * });
 *
 * var aaron = new Person("Aaron");
 * aaron.eat("Sandwich"); // alert("I'm eating: Sandwich");
 * </pre>
 * <p>Ext.Class has a powerful set of extensible <i>pre-processors</i> (→<code>ext.ExtClass.registerPreprocessor()</code>)
 * which takes care of everything related to class creation, including but not limited to
 * inheritance, mixins, configuration, statics, etc.</p>
 * <p><b>Inheritance:</b></p>
 * <pre>
 * Ext.define('Developer', {
 *      extend: 'Person',
 *
 *      constructor: function(name, isGeek) {
 *          this.isGeek = isGeek;
 *
 *          // Apply a method from the parent class' prototype
 *          this.callParent([name]);
 *      },
 *
 *      code: function(language) {
 *          alert("I'm coding in: " + language);
 *
 *          this.eat("Bugs");
 *
 *          return this;
 *      }
 * });
 *
 * var jacky = new Developer("Jacky", true);
 * jacky.code("JavaScript"); // alert("I'm coding in: JavaScript");
 *                           // alert("I'm eating: Bugs");
 * </pre>
 * <p>See →<code>ext.Base.callParent()</code> for more details on calling superclass' methods</p>
 * <p><b>Mixins:</b></p>
 * <pre>
 * Ext.define('CanPlayGuitar', {
 *      playGuitar: function() {
 *         alert("F#...G...D...A");
 *      }
 * });
 *
 * Ext.define('CanComposeSongs', {
 *      composeSongs: function() { ... }
 * });
 *
 * Ext.define('CanSing', {
 *      sing: function() {
 *          alert("For he's a jolly good fellow...")
 *      }
 * });
 *
 * Ext.define('Musician', {
 *      extend: 'Person',
 *
 *      mixins: {
 *          canPlayGuitar: 'CanPlayGuitar',
 *          canComposeSongs: 'CanComposeSongs',
 *          canSing: 'CanSing'
 *      }
 * })
 *
 * Ext.define('CoolPerson', {
 *      extend: 'Person',
 *
 *      mixins: {
 *          canPlayGuitar: 'CanPlayGuitar',
 *          canSing: 'CanSing'
 *      },
 *
 *      sing: function() {
 *          alert("Ahem....");
 *
 *          this.mixins.canSing.sing.call(this);
 *
 *          alert("[Playing guitar at the same time...]");
 *
 *          this.playGuitar();
 *      }
 * });
 *
 * var me = new CoolPerson("Jacky");
 *
 * me.sing(); // alert("Ahem...");
 *            // alert("For he's a jolly good fellow...");
 *            // alert("[Playing guitar at the same time...]");
 *            // alert("F#...G...D...A");
 * </pre>
 * <p><b>Config:</b></p>
 * <pre>
 * Ext.define('SmartPhone', {
 *      config: {
 *          hasTouchScreen: false,
 *          operatingSystem: 'Other',
 *          price: 500
 *      },
 *
 *      isExpensive: false,
 *
 *      constructor: function(config) {
 *          this.initConfig(config);
 *      },
 *
 *      applyPrice: function(price) {
 *          this.isExpensive = (price &gt; 500);
 *
 *          return price;
 *      },
 *
 *      applyOperatingSystem: function(operatingSystem) {
 *          if (!(/^(iOS|Android|BlackBerry)$/i).test(operatingSystem)) {
 *              return 'Other';
 *          }
 *
 *          return operatingSystem;
 *      }
 * });
 *
 * var iPhone = new SmartPhone({
 *      hasTouchScreen: true,
 *      operatingSystem: 'iOS'
 * });
 *
 * iPhone.getPrice(); // 500;
 * iPhone.getOperatingSystem(); // 'iOS'
 * iPhone.getHasTouchScreen(); // true;
 *
 * iPhone.isExpensive; // false;
 * iPhone.setPrice(600);
 * iPhone.getPrice(); // 600
 * iPhone.isExpensive; // true;
 *
 * iPhone.setOperatingSystem('AlienOS');
 * iPhone.getOperatingSystem(); // 'Other'
 * </pre>
 * <p><b>Statics:</b></p>
 * <pre>
 * Ext.define('Computer', {
 *      statics: {
 *          factory: function(brand) {
 *             // 'this' in static methods refer to the class itself
 *              return new this(brand);
 *          }
 *      },
 *
 *      constructor: function() { ... }
 * });
 *
 * var dellComputer = Computer.factory('Dell');
 * </pre>
 * <p>Also see →<code>ext.Base.statics()</code> and →<code>ext.Base.self</code> for more details on accessing
 * static properties within class methods</p>
 * @see ext.SExt#define() ext.Ext.define()
 * @see ext.SExt#create() ext.Ext.create()
 * @see ext.SExt#widget() ext.Ext.widget()
 * @see ext.SExt#getClass() ext.Ext.getClass()
 * @see ext.SExt#getClassName() ext.Ext.getClassName()
 * @see ext.SClassManager#create() ext.ClassManager.create()
 * @see ext.ExtClass#registerPreprocessor()
 * @see ext.Base#callParent()
 * @see ext.Base#statics()
 * @see ext.Base#self
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html Original Ext JS documentation of 'Ext.ClassManager'
 */
public const ClassManager:SClassManager = new SClassManager;
}