package ext {
import ext.promise.Deferred;

[Native("Ext.Deferred", require)]
/**
 * Deferreds are the mechanism used to create new Promises. A Deferred has a single
 * associated Promise that can be safely returned to external consumers to ensure they do
 * not interfere with the resolution or rejection of the deferred operation.
 * <p>This implementation of Promises is an extension of the ECMAScript 6 Promises API as
 * detailed [here][1]. For a compatible, though less full featured, API see <code>→ext.Promise</code>.</p>
 * <p>A Deferred is typically used within the body of a function that performs an asynchronous
 * operation. When that operation succeeds, the Deferred should be resolved; if that
 * operation fails, the Deferred should be rejected.</p>
 * <p>Each Deferred has an associated Promise. A Promise delegates <code>→then()</code> calls to its
 * Deferred's <code>→then()</code> method. In this way, access to Deferred operations are divided between
 * producer (Deferred) and consumer (Promise) roles.</p>
 * <p><b><i>Basic Usage</i></b></p>
 * <p>In it's most common form, a method will create and return a Promise like this:</p>
 * <pre>
 *  // A method in a service class which uses a Store and returns a Promise
 *  //
 *  loadCompanies: function () {
 *      var deferred = new Ext.Deferred(); // create the Ext.Deferred object
 *
 *      this.companyStore.load({
 *          callback: function (records, operation, success) {
 *              if (success) {
 *                  // Use "deferred" to drive the promise:
 *                  deferred.resolve(records);
 *              }
 *              else {
 *                  // Use "deferred" to drive the promise:
 *                  deferred.reject("Error loading Companies.");
 *              }
 *          }
 *      });
 *
 *      return deferred.promise;  // return the Promise to the caller
 *  }
 * </pre>
 * <p>You can see this method first creates a <code>→ext.Deferred</code> object. It then
 * returns its <code>Promise</code> object for use by the caller. Finally, in the asynchronous
 * callback, it resolves the <code>deferred</code> object if the call was successful, and rejects the
 * <code>deferred</code> if the call failed.</p>
 * <p>When a Deferred's <code>→resolve()</code> method is called, it fulfills with the optionally specified
 * value. If <code>→resolve()</code> is called with a then-able (i.e.a Function or Object with a <code>→then()</code>
 * function, such as another Promise) it assimilates the then-able's result; the Deferred
 * provides its own <code>→resolve()</code> and <code>→reject()</code> methods as the onFulfilled or onRejected
 * arguments in a call to that then-able's <code>→then()</code> function. If an error is thrown while
 * calling the then-able's <code>→then()</code> function (prior to any call back to the specified
 * <code>→resolve()</code> or <code>→reject()</code> methods), the Deferred rejects with that error. If a Deferred's
 * <code>→resolve()</code> method is called with its own Promise, it rejects with a TypeError.</p>
 * <p>When a Deferred's <code>→reject()</code> method is called, it rejects with the optionally specified
 * reason.</p>
 * <p>Each time a Deferred's <code>→then()</code> method is called, it captures a pair of optional
 * onFulfilled and onRejected callbacks and returns a Promise of the Deferred's future
 * value as transformed by those callbacks.</p>
 * <p>See <code>→ext.IPromise</code> for an example of using the returned Promise.</p>
 * @see ext.Promise
 * @see #then()
 * @see ext.Deferred
 * @see #resolve()
 * @see #reject()
 * @see ext.IPromise
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html Original Ext JS documentation of 'Ext.Deferred'
 */
public class Deferred extends ext.promise.Deferred {
  /**
   * Returns a new Promise that will only resolve once all the specified
   * <code>promisesOrValues</code> have resolved.
   * <p>The resolution value will be an Array containing the resolution value of each
   * of the <code>promisesOrValues</code>.</p>
   * @param promisesOrValues An
   * Array of values or Promises, or a Promise of an Array of values or Promises.
   * @return A Promise of an Array of the resolved values.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-all Original Ext JS documentation of 'all'
   */
  public static native function all(promisesOrValues:*):IPromise;

  /**
   * Initiates a competitive race, returning a new Promise that will resolve when
   * any one of the specified <code>promisesOrValues</code> have resolved, or will reject when
   * all <code>promisesOrValues</code> have rejected or cancelled.
   * <p>The resolution value will the first value of <code>promisesOrValues</code> to resolve.</p>
   * @param promisesOrValues An
   * Array of values or Promises, or a Promise of an Array of values or Promises.
   * @return A Promise of the first resolved value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-any Original Ext JS documentation of 'any'
   */
  public static native function any(promisesOrValues:*):IPromise;

  /**
   * Returns a new Promise that will automatically resolve with the specified
   * Promise or value after the specified delay (in milliseconds).
   * @param promiseOrValue A Promise or value.
   * @param milliseconds A delay duration (in milliseconds).
   * @return A Promise of the specified Promise or value that
   * will resolve after the specified delay.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-delay Original Ext JS documentation of 'delay'
   */
  public static native function delay(promiseOrValue:*, milliseconds:Number):IPromise;

  /**
   * Traditional map function, similar to <code>Array.prototype.map()</code>, that allows
   * input to contain promises and/or values.
   * <p>The specified map function may return either a value or a promise.</p>
   * @param promisesOrValues An
   * Array of values or Promises, or a Promise of an Array of values or Promises.
   * @param mapFn A Function to call to transform each resolved value in
   * the Array.
   * @return A Promise of an Array of the mapped resolved
   * values.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-map Original Ext JS documentation of 'map'
   */
  public static native function map(promisesOrValues:*, mapFn:Function):IPromise;

  /**
   * Returns a new function that wraps the specified function and caches the
   * results for previously processed inputs.
   * <p>Similar to →<code>ext.FunctionUtil.memoize()</code>, except it
   * allows for parameters that are Promises and/or values.</p>
   * @param fn A Function to wrap.
   * @param scope An optional scope in which to execute the wrapped function.
   * @param hashFn An optional function used to compute a hash key for
   * storing the result, based on the arguments to the original function.
   * @return The new wrapper function.
   * @see ext.SFunctionUtil#memoize() ext.FunctionUtil.memoize()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-memoize Original Ext JS documentation of 'memoize'
   */
  public static native function memoize(fn:Function, scope:Object = null, hashFn:Function = null):Function;

  /**
   * Execute an Array (or →<code>ext.IPromise</code> of an Array) of
   * functions in parallel.
   * <p>The specified functions may optionally return their results as
   * <i>Promises</i> (→<code>ext.IPromise</code>).</p>
   * @param fns The Array (or Promise of an Array)
   * of functions to execute.
   * @param scope Optional scope in which to execute the specified functions.
   * @return Promise of an Array of results for each function
   * call (in the same order).
   * @see ext.IPromise
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-parallel Original Ext JS documentation of 'parallel'
   */
  public static native function parallel(fns:*, scope:Object = null):IPromise;

  /**
   * Execute an Array (or →<code>ext.IPromise</code> of an Array) of
   * functions as a pipeline, where each function's result is passed to the
   * subsequent function as input.
   * <p>The specified functions may optionally return their results as
   * <i>Promises</i> (→<code>ext.IPromise</code>).</p>
   * @param fns The Array (or Promise of an Array)
   * of functions to execute.
   * @param initialValue Initial value to be passed to the first function
   * in the pipeline.
   * @param scope Optional scope in which to execute the specified functions.
   * @return Promise of the result value for the final
   * function in the pipeline.
   * @see ext.IPromise
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-pipeline Original Ext JS documentation of 'pipeline'
   */
  public static native function pipeline(fns:*, initialValue:Object, scope:Object = null):IPromise;

  /**
   * Returns a promise that resolves or rejects as soon as one of the promises
   * in the array resolves or rejects, with the value or reason from that promise.
   * @since 6.5.0
   * @param promises The promises.
   * @return The promise to be resolved when the race completes.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-race Original Ext JS documentation of 'race'
   */
  public static native function race(promises:Array):IPromise;

  /**
   * Traditional reduce function, similar to <code>Array.reduce()</code>, that allows input to
   * contain promises and/or values.
   * @param values An
   * Array of values or Promises, or a Promise of an Array of values or Promises.
   * @param reduceFn A Function to call to transform each successive
   * item in the Array into the final reduced value.
   * @param initialValue An initial Promise or value.
   * @return A Promise of the reduced value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-reduce Original Ext JS documentation of 'reduce'
   */
  public static native function reduce(values:*, reduceFn:Function, initialValue:*):IPromise;

  /**
   * Convenience method that returns a new Promise rejected with the specified
   * reason.
   * @param reason Rejection reason.
   * @return The rejected Promise.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-rejected Original Ext JS documentation of 'rejected'
   */
  public static native function rejected(reason:Error):IPromise;

  /**
   * Returns a new Promise that either...
   * <ul>
   * <li>Resolves immediately for the specified value, or</li>
   * <li>Resolves or rejects when the specified promise (or third-party Promise or
   * then()-able) is resolved or rejected.</li>
   * </ul>
   * @param promiseOrValue A Promise (or third-party Promise or then()-able)
   * or value.
   * @return A Promise of the specified Promise or value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-resolved Original Ext JS documentation of 'resolved'
   */
  public static native function resolved(promiseOrValue:*):IPromise;

  /**
   * Execute an Array (or →<code>ext.IPromise</code> of an Array) of
   * functions sequentially.
   * <p>The specified functions may optionally return their results as <i>Promises</i> (→<code>ext.IPromise</code>).</p>
   * @param fns The Array (or Promise of an Array)
   * of functions to execute.
   * @param scope Optional scope in which to execute the specified functions.
   * @return Promise of an Array of results for each function
   * call (in the same order).
   * @see ext.IPromise
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-sequence Original Ext JS documentation of 'sequence'
   */
  public static native function sequence(fns:*, scope:Object = null):IPromise;

  /**
   * Initiates a competitive race, returning a new Promise that will resolve when
   * <code>howMany</code> of the specified <code>promisesOrValues</code> have resolved, or will reject
   * when it becomes impossible for <code>howMany</code> to resolve.
   * <p>The resolution value will be an Array of the first <code>howMany</code> values of
   * <code>promisesOrValues</code> to resolve.</p>
   * @param promisesOrValues An
   * Array of values or Promises, or a Promise of an Array of values or Promises.
   * @param howMany The expected number of resolved values.
   * @return A Promise of the expected number of resolved
   * values.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-some Original Ext JS documentation of 'some'
   */
  public static native function some(promisesOrValues:*, howMany:Number):IPromise;

  /**
   * Returns a new Promise that will automatically reject after the specified
   * timeout (in milliseconds) if the specified promise has not resolved or
   * rejected.
   * @param promiseOrValue A Promise or value.
   * @param milliseconds A timeout duration (in milliseconds).
   * @return A Promise of the specified Promise or value that
   * enforces the specified timeout.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Deferred.html#static-method-timeout Original Ext JS documentation of 'timeout'
   */
  public static native function timeout(promiseOrValue:*, milliseconds:Number):IPromise;
}
}