package ext {

[Native("Ext.Error", require)]
/**
 * A helper class for the native JavaScript Error object that adds a few useful capabilities for handling
 * errors in an application. When you use Ext.Error to →<code>raise()</code> an error from within any class that
 * uses the Class System, the Error class can automatically add the source class and method from which
 * the error was raised. It also includes logic to automatically log the error to the console, if available,
 * with additional metadata about the error. In all cases, the error will always be thrown at the end so that
 * execution will halt.
 * <p>Ext.Error also offers a global error <i>handling</i> (→<code>handle()</code>) method that can be overridden in order to
 * handle application-wide errors in a single spot. You can optionally →<code>ignore</code> errors altogether,
 * although in a real application it's usually a better idea to override the handling function and perform
 * logging or some other method of reporting the errors in a way that is meaningful to the application.</p>
 * <p>At its simplest you can simply raise an error as a simple string from within any code:</p>
 * <p>Example usage:</p>
 * <pre>
 * Ext.raise('Something bad happened!');
 * </pre>
 * <p>If raised from plain JavaScript code, the error will be logged to the console (if available) and the message
 * displayed. In most cases however you'll be raising errors from within a class, and it may often be useful to add
 * additional metadata about the error being raised. The →<code>raise()</code> method can also take a config object.
 * In this form the <code>msg</code> attribute becomes the error description, and any other data added to the config gets
 * added to the error object and, if the console is available, logged to the console for inspection.</p>
 * <p>Example usage:</p>
 * <pre>
 * Ext.define('Ext.Foo', {
 *     doSomething: function(option){
 *         if (someCondition === false) {
 *             Ext.raise({
 *                 msg: 'You cannot do that!',
 *                 option: option,   // whatever was passed into the method
 *                 'error code': 100 // other arbitrary info
 *             });
 *         }
 *     }
 * });
 * </pre>
 * <p>If a console is available (that supports the <code>console.dir</code> function) you'll see console output like:</p>
 * <pre>
 * An error was raised with the following data:
 * option:         Object { foo: "bar"}
 *     foo:        "bar"
 * error code:     100
 * msg:            "You cannot do that!"
 * sourceClass:   "Ext.Foo"
 * sourceMethod:  "doSomething"
 *
 * uncaught exception: You cannot do that!
 * </pre>
 * <p>As you can see, the error will report exactly where it was raised and will include as much information as the
 * raising code can usefully provide.</p>
 * <p>If you want to handle all application errors globally you can simply override the static →<code>handle()</code> method
 * and provide whatever handling logic you need. If the method returns true then the error is considered handled
 * and will not be thrown to the browser. If anything but true is returned then the error will be thrown normally.</p>
 * <p>Example usage:</p>
 * <pre>
 * Ext.Error.handle = function(err) {
 *     if (err.someProperty == 'NotReallyAnError') {
 *         // maybe log something to the application here if applicable
 *         return true;
 *     }
 *     // any non-true return value (including none) will cause the error to be thrown
 * }
 * </pre>
 * @see #raise()
 * @see #handle()
 * @see #ignore
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Error.html Original Ext JS documentation of 'Ext.Error'
 */
public class ExtError extends Object {
  /**
   * Static flag that can be used to globally disable error reporting to the browser if set
   * to true (defaults to false). Note that if you ignore Ext errors it's likely that some
   * other code may fail and throw a native JavaScript error thereafter, so use with caution.
   * In most cases it will probably be preferable to supply a custom error
   * <i>handling</i> (→<code>handle()</code>) function instead.
   * <p>Example usage:</p>
   * <pre>
   * Ext.Error.ignore = true;
   * </pre>
   * @default false
   * @see #handle()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Error.html#static-property-ignore Original Ext JS documentation of 'ignore'
   */
  public static native function get ignore():Boolean;

  /**
   * @private
   */
  public static native function set ignore(value:Boolean):void;

  /**
   * Globally handle any Ext errors that may be raised, optionally providing custom logic
   * to handle different errors individually. Return true from the function to bypass
   * throwing the error to the browser, otherwise the error will be thrown and execution
   * will halt.
   * <p>Example usage:</p>
   * <pre>
   * Ext.Error.handle = function(err) {
   *     if (err.someProperty == 'NotReallyAnError') {
   *         // maybe log something to the application here if applicable
   *         return true;
   *     }
   *     // any non-true return value (including none) will cause the error to be thrown
   * }
   * </pre>
   * @param err The error being raised. It will contain any attributes that were
   * originally raised with it, plus properties about the method and class from which
   * the error originated (if raised from a class that uses the Class System).
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Error.html#static-method-handle Original Ext JS documentation of 'handle'
   */
  public static native function handle(err:Object):void;

  [Deprecated(replacement="raise", since="6.0.0")]
  /**
   * This method is called internally by →<code>ext.Ext.raise()</code>. Application code should
   * call →<code>ext.Ext.raise()</code> instead of calling this method directly.
   * @see ext.SExt#raise() ext.Ext.raise()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Error.html#static-method-raise Original Ext JS documentation of 'raise'
   */
  public static native function raise(err:*):void;
}
}