package ext {

[Native("Ext.Factory", require)]
/**
 * Manages factories for families of classes (classes with a common <code>alias</code> prefix). The
 * factory for a class family is a function stored as a <code>static</code> on <code>→ext.Factory</code>. These
 * are created either by directly calling <code>Ext.Factory.define</code> or by using the
 * <code>→ext.mixin.Factoryable</code> interface.
 * <p>To illustrate, consider the layout system's use of aliases. The <code>hbox</code> layout maps to
 * the <code>"layout.hbox"</code> alias that one typically provides via the <code>layout</code> config on a
 * Container.</p>
 * <p>Under the covers this maps to a call like this:</p>
 * <pre>
 *  Ext.Factory.layout('hbox');
 * </pre>
 * <p>Or possibly:</p>
 * <pre>
 *  Ext.Factory.layout({
 *      type: 'hbox'
 *  });
 * </pre>
 * <p>The value of the <code>layout</code> config is passed to the <code>Ext.Factory.layout</code> function. The
 * exact signature of a factory method matches <code>→ext.Factory.create()</code>.</p>
 * <p>To define this factory directly, one could call <code>Ext.Factory.define</code> like so:</p>
 * <pre>
 *  Ext.Factory.define('layout', 'auto');  // "layout.auto" is the default type
 * </pre>
 * @see ext.Factory
 * @see ext.mixin.Factoryable
 * @see ext.Factory#create()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Factory.html Original Ext JS documentation of 'Ext.Factory'
 */
public class Factory extends Object {
  /**
   * For example, the layout alias family could be defined like this...
   * <pre>
   *  Ext.Factory.define('layout', {
   *      defaultType: 'auto'
   *  });
   * </pre>
   * <p>To define multiple families at once:</p>
   * <pre>
   *  Ext.Factory.define({
   *      layout: {
   *          defaultType: 'auto'
   *      }
   *  });
   * </pre>
   * @since 5.0.0
   * @param type The alias family (e.g., "layout").
   * @param config An object specifying the config for the <code>→ext.Factory</code>
   * to be created. If a string is passed it is treated as the <code>→defaultType</code>.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Factory.html#static-method-define Original Ext JS documentation of 'define'
   * @see ext.Factory
   * @see #defaultType
   */
  public static native function define(type:String, config:* = undefined):Function;

  /**
   * @param config @inheritDoc
   */
  public native function Factory(config:Factory = null);

  [ExtConfig]
  /**
   * The prefix to apply to <code>type</code> values to form a complete alias. This defaults to the
   * proper value in most all cases and should not need to be specified.
   * @since 5.0.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Factory.html#cfg-aliasPrefix Original Ext JS documentation of 'aliasPrefix'
   */
  public native function get aliasPrefix():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set aliasPrefix(value:String):void;

  [ExtConfig]
  /**
   * The name of the method used to prepare config objects for creation. This defaults
   * to <code>'create'</code> plus the capitalized name (e.g., <code>'createLayout'</code> for the 'laoyut'
   * alias family).
   * @default 'create' + Ext.String.capitalize(me.name)
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Factory.html#cfg-creator Original Ext JS documentation of 'creator'
   */
  public native function get creator():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set creator(value:String):void;

  [ExtConfig]
  /**
   * The config property to set when the factory is given a config that is a string.
   * @default "type"
   * @since 5.0.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Factory.html#cfg-defaultProperty Original Ext JS documentation of 'defaultProperty'
   */
  public native function get defaultProperty():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set defaultProperty(value:String):void;

  [ExtConfig]
  /**
   * An optional type to use if none is given to the factory at invocation. This is a
   * suffix added to the <code>→aliasPrefix</code>. For example, if <code>aliasPrefix="layout."</code> and
   * <code>defaultType="hbox"</code> the default alias is <code>"layout.hbox"</code>. This is an alternative
   * to <code>→xclass</code> so only one should be provided.
   * @since 5.0.0
   * @see #aliasPrefix
   * @see #xclass
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Factory.html#cfg-defaultType Original Ext JS documentation of 'defaultType'
   */
  public native function get defaultType():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set defaultType(value:String):void;

  [ExtConfig]
  /**
   * The property that identifies an object as instance vs a config.
   * @default "isInstance"
   * @since 5.0.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Factory.html#cfg-instanceProp Original Ext JS documentation of 'instanceProp'
   */
  public native function get instanceProp():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set instanceProp(value:String):void;

  [ExtConfig]
  /**
   * The property from which to read the type alias suffix.
   * @default "type"
   * @since 6.5.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Factory.html#cfg-typeProperty Original Ext JS documentation of 'typeProperty'
   */
  public native function get typeProperty():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set typeProperty(value:String):void;

  [ExtConfig]
  /**
   * The full classname of the type of instance to create when none is provided to the
   * factory. This is an alternative to <code>→defaultType</code> so only one should be specified.
   * @since 5.0.0
   * @see #defaultType
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Factory.html#cfg-xclass Original Ext JS documentation of 'xclass'
   */
  public native function get xclass():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set xclass(value:String):void;

  /**
   * Creates an instance of this class family given configuration options.
   * @since 5.0.0
   * @param config The configuration or instance (if an Object) or
   * just the type (if a String) describing the instance to create.
   * <ul>
   * <li><code>xclass:String</code> (optional) —
   * The full class name of the class to create.
   * </li>
   * <li><code>type:String</code> (optional) —
   * The type string to add to the alias prefix for this
   * factory.
   * </li>
   * </ul>
   * @param defaultType The type to create if no type is contained in the
   * <code>config</code>, or an object containing a default set of configs.
   * @return The newly created instance.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Factory.html#method-create Original Ext JS documentation of 'create'
   */
  public native function create(config:* = undefined, defaultType:* = undefined):Object;

  /**
   * Returns an instance of an ID generator based on the ID you pass in.
   * @param config The config object or <code>id</code> to lookup.
   * @return Ext.data.identifier.&#42; The data identifier
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Factory.html#method-dataIdentifier Original Ext JS documentation of 'dataIdentifier'
   */
  public native function dataIdentifier(config:*):Object;

  /**
   * This method accepts a <code>config</code> object and an existing <code>instance</code> if one exists
   * (can be <code>null</code>).
   * <p>The details are best explained by example:</p>
   * <pre>
   *  config: {
   *      header: {
   *          xtype: 'itemheader'
   *      }
   *  },
   *
   *  applyHeader: function (header, oldHeader) {
   *      return Ext.Factory.widget.update(oldHeader, header,
   *          this, 'createHeader');
   *  },
   *
   *  createHeader: function (header) {
   *      return Ext.apply({
   *          xtype: 'itemheader',
   *          ownerCmp: this
   *      }, header);
   *  }
   * </pre>
   * <p>Normally the <code>applyHeader</code> method would have to coordinate potential reuse of
   * the <code>oldHeader</code> and perhaps call <code>setConfig</code> on it with the new <code>header</code> config
   * options. If there was no <code>oldHeader</code>, of course, a new instance must be created
   * instead. These details are handled by this method. If the <code>oldHeader</code> is not
   * reused, it will be <i>destroyed</i> (→<code>ext.Base.destroy()</code>).</p>
   * <p>For derived class flexibility, the pattern of calling out to a "creator" method
   * that only returns the config object has become widely used in many components.
   * This pattern is also covered in this method. The goal is to allow the derived
   * class to <code>callParent</code> and yet not end up with an instantiated component (since
   * the type may not yet be known).</p>
   * <p>This mechanism should be used in favor of <code>Ext.factory()</code>.</p>
   * @since 6.5.0
   * @param instance
   * @param config The configuration (see →<code>create()</code>).
   * @param creator If passed, this object must provide the <code>→creator</code>
   * method or the <code>creatorMethod</code> parameter.
   * @param creatorMethod The name of a creation wrapper method on the
   * given <code>→creator</code> instance that "upgrades" the raw <code>config</code> object into a final
   * form for creation.
   * @param defaultsConfig The name of a config property (on the provided
   * <code>→creator</code> instance) that contains defaults to be used to create instances. These
   * defaults are present in the config object passed to the <code>creatorMethod</code>.
   * @return The reconfigured <code>instance</code> or a newly created one.
   * @see ext.Base#destroy()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Factory.html#method-update Original Ext JS documentation of 'update'
   * @see #create()
   * @see #creator
   */
  public native function update(instance:Base, config:*, creator:Object = null, creatorMethod:String = null, defaultsConfig:String = null):Object;
}
}