package ext {

[Mixin("ext.Mixin")]
/**
 * This class is a base class for mixins. These are classes that extend this class and are
 * designed to be used as a <code>→mixin()</code> by user code.
 * <p>It provides mixins with the ability to "hook" class methods of the classes in to which
 * they are mixed. For example, consider the <code>→destroy()</code> method pattern. If a mixin class
 * had cleanup requirements, it would need to be called as part of <code>→destroy()</code>.</p>
 * <p>Starting with a basic class we might have:</p>
 * <pre>
 *  Ext.define('Foo.bar.Base', {
 *      destroy: function() {
 *          console.log('B');
 *          // cleanup
 *      }
 *  });
 * </pre>
 * <p>A derived class would look like this:</p>
 * <pre>
 *  Ext.define('Foo.bar.Derived', {
 *      extend: 'Foo.bar.Base',
 *
 *      destroy: function() {
 *          console.log('D');
 *          // more cleanup
 *
 *          this.callParent(); // let Foo.bar.Base cleanup as well
 *      }
 *  });
 * </pre>
 * <p>To see how using this class help, start with a "normal" mixin class that also needs to
 * cleanup its resources. These mixins must be called explicitly by the classes that use
 * them. For example:</p>
 * <pre>
 *  Ext.define('Foo.bar.Util', {
 *      destroy: function() {
 *          console.log('U');
 *      }
 *  });
 *
 *  Ext.define('Foo.bar.Derived', {
 *      extend: 'Foo.bar.Base',
 *
 *      mixins: {
 *          util: 'Foo.bar.Util'
 *      },
 *
 *      destroy: function() {
 *          console.log('D');
 *          // more cleanup
 *
 *          this.mixins.util.destroy.call(this);
 *
 *          this.callParent(); // let Foo.bar.Base cleanup as well
 *      }
 *  });
 *
 *  var obj = new Foo.bar.Derived();
 *
 *  obj.destroy();
 *  // logs D then U then B
 * </pre>
 * <p>This class is designed to solve the above in simpler and more reliable way.</p>
 * <p><b><i>mixinConfig</i></b></p>
 * <p>Using <code>mixinConfig</code> the mixin class can provide "before" or "after" hooks that do not
 * involve the derived class implementation. This also means the derived class cannot
 * adjust parameters to the hook methods.</p>
 * <pre>
 *  Ext.define('Foo.bar.Util', {
 *      extend: 'Ext.Mixin',
 *
 *      mixinConfig: {
 *          after: {
 *              destroy: 'destroyUtil'
 *          }
 *      },
 *
 *      destroyUtil: function() {
 *          console.log('U');
 *      }
 *  });
 *
 *  Ext.define('Foo.bar.Class', {
 *      mixins: {
 *          util: 'Foo.bar.Util'
 *      },
 *
 *      destroy: function() {
 *          console.log('D');
 *      }
 *  });
 *
 *  var obj = new Foo.bar.Derived();
 *
 *  obj.destroy();
 *  // logs D then U
 * </pre>
 * <p>If the destruction should occur in the other order, you can use <code>before</code>:</p>
 * <pre>
 *  Ext.define('Foo.bar.Util', {
 *      extend: 'Ext.Mixin',
 *
 *      mixinConfig: {
 *          before: {
 *              destroy: 'destroyUtil'
 *          }
 *      },
 *
 *      destroyUtil: function() {
 *          console.log('U');
 *      }
 *  });
 *
 *  Ext.define('Foo.bar.Class', {
 *      mixins: {
 *          util: 'Foo.bar.Util'
 *      },
 *
 *      destroy: function() {
 *          console.log('D');
 *      }
 *  });
 *
 *  var obj = new Foo.bar.Derived();
 *
 *  obj.destroy();
 *  // logs U then D
 * </pre>
 * <p><i>Configs</i></p>
 * <p>Normally when a mixin defines <code>config</code> properties and the class into which the mixin is
 * initially mixed needs to specify values for those configs, the class processor does not
 * yet recognize these config and instead retains the properties on te target class
 * prototype.</p>
 * <p>Changing the above behavior would potentially break application code, so this class
 * provides a way to remedy this:</p>
 * <pre>
 *  Ext.define('Foo.bar.Util', {
 *      extend: 'Ext.Mixin',
 *
 *      mixinConfig: {
 *          configs: true
 *      },
 *
 *      config: {
 *          foo: null
 *      }
 *  });
 * </pre>
 * <p>Now the direct user class can correctly specify <code>foo</code> config properties:</p>
 * <pre>
 *  Ext.define('Some.other.Class', {
 *      mixins: [
 *          'Foo.bar.Util'
 *      ],
 *
 *      foo: 'bar'  // recognized as the foo config form Foo.bar.Util
 *  });
 * </pre>
 * <p><i>Chaining</i></p>
 * <p>One way for a mixin to provide methods that act more like normal inherited methods is
 * to use an <code>on</code> declaration. These methods will be injected into the <code>callParent</code> chain
 * between the derived and superclass. For example:</p>
 * <pre>
 *  Ext.define('Foo.bar.Util', {
 *      extend: 'Ext.Mixin',
 *
 *      mixinConfig: {
 *          on: {
 *              destroy: function() {
 *                  console.log('M');
 *              }
 *          }
 *      }
 *  });
 *
 *  Ext.define('Foo.bar.Base', {
 *      destroy: function() {
 *          console.log('B');
 *      }
 *  });
 *
 *  Ext.define('Foo.bar.Derived', {
 *      extend: 'Foo.bar.Base',
 *
 *      mixins: {
 *          util: 'Foo.bar.Util'
 *      },
 *
 *      destroy: function() {
 *          this.callParent();
 *          console.log('D');
 *      }
 *  });
 *
 *  var obj = new Foo.bar.Derived();
 *
 *  obj.destroy();
 *  // logs M then B then D
 * </pre>
 * <p>As with <code>before</code> and <code>after</code>, the value of <code>on</code> can be a method name.</p>
 * <pre>
 *  Ext.define('Foo.bar.Util', {
 *      extend: 'Ext.Mixin',
 *
 *      mixinConfig: {
 *          on: {
 *              destroy: 'onDestroy'
 *          }
 *      }
 *
 *      onDestroy: function() {
 *          console.log('M');
 *      }
 *  });
 * </pre>
 * <p>Because this technique leverages <code>callParent</code>, the derived class controls the time and
 * parameters for the call to all of its bases (be they <code>→extend()</code> or <code>→mixin()</code> flavor).</p>
 * <p><i>Derivations</i></p>
 * <p>Some mixins need to process class extensions of their target class. To do this you can
 * define an <code>extended</code> method like so:</p>
 * <pre>
 *  Ext.define('Foo.bar.Util', {
 *      extend: 'Ext.Mixin',
 *
 *      mixinConfig: {
 *          extended: function(baseClass, derivedClass, classBody) {
 *              // This function is called whenever a new "derivedClass" is created
 *              // that extends a "baseClass" in to which this mixin was mixed.
 *          }
 *      }
 *  });
 * </pre>
 * @see ext.Mixin#mixin()
 * @see ext.Mixin#destroy()
 * @see ext.Mixin#extend()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Mixin.html Original Ext JS documentation of 'Ext.Mixin'
 * @see ext.Mixin
 */
public interface IMixin {
}
}