package ext {

[Native("Ext.Promise", require)]
/**
 * This class provides an API compatible implementation of the ECMAScript 6 Promises API
 * (providing an implementation as necessary for browsers that do not natively support the
 * <code>Promise</code> class).
 * <p>This class will use the native <code>Promise</code> implementation if one is available. The
 * native implementation, while standard, does not provide all of the features of the
 * Ext JS Promises implementation.</p>
 * <p>To use the Ext JS enhanced Promises implementation, see <code>→ext.Deferred</code> for
 * creating enhanced promises and additional static utility methods.</p>
 * <p>Typical usage:</p>
 * <pre>
 *  function getAjax (url) {
 *      // The function passed to Ext.Promise() is called immediately to start
 *      // the asynchronous action.
 *      //
 *      return new Ext.Promise(function (resolve, reject) {
 *          Ext.Ajax.request({
 *              url: url,
 *
 *              success: function (response) {
 *                  // Use the provided "resolve" method to deliver the result.
 *                  //
 *                  resolve(response.responseText);
 *              },
 *
 *              failure: function (response) {
 *                  // Use the provided "reject" method to deliver error message.
 *                  //
 *                  reject(response.status);
 *              }
 *          });
 *      });
 *  }
 *
 *  getAjax('http://stuff').then(function (content) {
 *      // content is responseText of ajax response
 *  });
 * </pre>
 * <p>To adapt the Ext JS <code>store (→ext.data.Store)</code> to use a Promise, you might do
 * something like this:</p>
 * <pre>
 *  loadCompanies: function() {
 *      var companyStore = this.companyStore;
 *
 *      return new Ext.Promise(function (resolve, reject) {
 *          companyStore.load({
 *              callback: function(records, operation, success) {
 *                  if (success) {
 *                      // Use the provided "resolve" method  to drive the promise:
 *                      resolve(records);
 *                  }
 *                  else {
 *                      // Use the provided "reject" method  to drive the promise:
 *                      reject("Error loading Companies.");
 *                  }
 *              }
 *          });
 *      });
 *  }
 * </pre>
 * @see ext.Deferred
 * @see ext.data.Store
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Promise.html Original Ext JS documentation of 'Ext.Promise'
 */
public class Promise extends Base implements IPromise {
  /**
   * Returns a new Promise that will only resolve once all the specified
   * <code>promisesOrValues</code> have resolved.
   * <p>The resolution value will be an Array containing the resolution value of each
   * of the <code>promisesOrValues</code>.</p>
   * @param promisesOrValues An Array of values
   * or Promises, or a Promise of an Array of values or Promises.
   * @return A Promise of an Array of the resolved values.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Promise.html#static-method-all Original Ext JS documentation of 'all'
   */
  public static native function all(promisesOrValues:*):IPromise;

  /**
   * Returns a promise that resolves or rejects as soon as one of the promises in the
   * array resolves or rejects, with the value or reason from that promise.
   * @since 6.5.0
   * @param promises The promises.
   * @return The promise to be resolved when the race completes.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Promise.html#static-method-race Original Ext JS documentation of 'race'
   */
  public static native function race(promises:Array):IPromise;

  /**
   * Convenience method that returns a new Promise rejected with the specified
   * reason.
   * @param reason Rejection reason.
   * @return The rejected Promise.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Promise.html#static-method-reject Original Ext JS documentation of 'reject'
   */
  public static native function reject(reason:Error):IPromise;

  /**
   * Returns a new Promise that either...
   * <ul>
   * <li>Resolves immediately for the specified value, or</li>
   * <li>Resolves or rejects when the specified promise (or third-party Promise or
   * then()-able) is resolved or rejected.</li>
   * </ul>
   * @param value A Promise (or third-party Promise or then()-able)
   * or value.
   * @return A Promise of the specified Promise or value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Promise.html#static-method-resolve Original Ext JS documentation of 'resolve'
   */
  public static native function resolve(value:*):IPromise;

  /**
   * @inheritDoc
   */
  public native function then(onFulfilled:Function = null, onRejected:Function = null, onProgress:Function = null):IPromise;
}
}