package ext {

[Native]
/**
 * Ext.ClassManager manages all classes and handles mapping from string class name to
 * actual class objects throughout the whole framework. It is not generally accessed directly,
 * rather through these convenient shorthands:
 * <ul>
 * <li>→<code>ext.Ext.define()</code></li>
 * <li>→<code>ext.Ext.create()</code></li>
 * <li>→<code>ext.Ext.widget()</code></li>
 * <li>→<code>ext.Ext.getClass()</code></li>
 * <li>→<code>ext.Ext.getClassName()</code></li>
 * </ul>
 * <p><b>Basic syntax:</b></p>
 * <pre>
 * Ext.define(className, properties);
 * </pre>
 * <p>in which <code>properties</code> is an object represent a collection of properties that apply to the class.
 * See →<code>ext.ClassManager.create()</code> for more detailed instructions.</p>
 * <pre>
 * Ext.define('Person', {
 *      name: 'Unknown',
 *
 *      constructor: function(name) {
 *          if (name) {
 *              this.name = name;
 *          }
 *      },
 *
 *      eat: function(foodType) {
 *          alert("I'm eating: " + foodType);
 *
 *          return this;
 *      }
 * });
 *
 * var aaron = new Person("Aaron");
 * aaron.eat("Sandwich"); // alert("I'm eating: Sandwich");
 * </pre>
 * <p>Ext.Class has a powerful set of extensible <i>pre-processors</i> (→<code>ext.ExtClass.registerPreprocessor()</code>)
 * which takes care of everything related to class creation, including but not limited to
 * inheritance, mixins, configuration, statics, etc.</p>
 * <p><b>Inheritance:</b></p>
 * <pre>
 * Ext.define('Developer', {
 *      extend: 'Person',
 *
 *      constructor: function(name, isGeek) {
 *          this.isGeek = isGeek;
 *
 *          // Apply a method from the parent class' prototype
 *          this.callParent([name]);
 *      },
 *
 *      code: function(language) {
 *          alert("I'm coding in: " + language);
 *
 *          this.eat("Bugs");
 *
 *          return this;
 *      }
 * });
 *
 * var jacky = new Developer("Jacky", true);
 * jacky.code("JavaScript"); // alert("I'm coding in: JavaScript");
 *                           // alert("I'm eating: Bugs");
 * </pre>
 * <p>See →<code>ext.Base.callParent()</code> for more details on calling superclass' methods</p>
 * <p><b>Mixins:</b></p>
 * <pre>
 * Ext.define('CanPlayGuitar', {
 *      playGuitar: function() {
 *         alert("F#...G...D...A");
 *      }
 * });
 *
 * Ext.define('CanComposeSongs', {
 *      composeSongs: function() { ... }
 * });
 *
 * Ext.define('CanSing', {
 *      sing: function() {
 *          alert("For he's a jolly good fellow...")
 *      }
 * });
 *
 * Ext.define('Musician', {
 *      extend: 'Person',
 *
 *      mixins: {
 *          canPlayGuitar: 'CanPlayGuitar',
 *          canComposeSongs: 'CanComposeSongs',
 *          canSing: 'CanSing'
 *      }
 * })
 *
 * Ext.define('CoolPerson', {
 *      extend: 'Person',
 *
 *      mixins: {
 *          canPlayGuitar: 'CanPlayGuitar',
 *          canSing: 'CanSing'
 *      },
 *
 *      sing: function() {
 *          alert("Ahem....");
 *
 *          this.mixins.canSing.sing.call(this);
 *
 *          alert("[Playing guitar at the same time...]");
 *
 *          this.playGuitar();
 *      }
 * });
 *
 * var me = new CoolPerson("Jacky");
 *
 * me.sing(); // alert("Ahem...");
 *            // alert("For he's a jolly good fellow...");
 *            // alert("[Playing guitar at the same time...]");
 *            // alert("F#...G...D...A");
 * </pre>
 * <p><b>Config:</b></p>
 * <pre>
 * Ext.define('SmartPhone', {
 *      config: {
 *          hasTouchScreen: false,
 *          operatingSystem: 'Other',
 *          price: 500
 *      },
 *
 *      isExpensive: false,
 *
 *      constructor: function(config) {
 *          this.initConfig(config);
 *      },
 *
 *      applyPrice: function(price) {
 *          this.isExpensive = (price &gt; 500);
 *
 *          return price;
 *      },
 *
 *      applyOperatingSystem: function(operatingSystem) {
 *          if (!(/^(iOS|Android|BlackBerry)$/i).test(operatingSystem)) {
 *              return 'Other';
 *          }
 *
 *          return operatingSystem;
 *      }
 * });
 *
 * var iPhone = new SmartPhone({
 *      hasTouchScreen: true,
 *      operatingSystem: 'iOS'
 * });
 *
 * iPhone.getPrice(); // 500;
 * iPhone.getOperatingSystem(); // 'iOS'
 * iPhone.getHasTouchScreen(); // true;
 *
 * iPhone.isExpensive; // false;
 * iPhone.setPrice(600);
 * iPhone.getPrice(); // 600
 * iPhone.isExpensive; // true;
 *
 * iPhone.setOperatingSystem('AlienOS');
 * iPhone.getOperatingSystem(); // 'Other'
 * </pre>
 * <p><b>Statics:</b></p>
 * <pre>
 * Ext.define('Computer', {
 *      statics: {
 *          factory: function(brand) {
 *             // 'this' in static methods refer to the class itself
 *              return new this(brand);
 *          }
 *      },
 *
 *      constructor: function() { ... }
 * });
 *
 * var dellComputer = Computer.factory('Dell');
 * </pre>
 * <p>Also see →<code>ext.Base.statics()</code> and →<code>ext.Base.self</code> for more details on accessing
 * static properties within class methods</p>
 * <p>Type of singleton ClassManager.</p>
 * @see ext.SExt#define() ext.Ext.define()
 * @see ext.SExt#create() ext.Ext.create()
 * @see ext.SExt#widget() ext.Ext.widget()
 * @see ext.SExt#getClass() ext.Ext.getClass()
 * @see ext.SExt#getClassName() ext.Ext.getClassName()
 * @see ext.SClassManager#create() ext.ClassManager.create()
 * @see ext.ExtClass#registerPreprocessor()
 * @see ext.Base#callParent()
 * @see ext.Base#statics()
 * @see ext.Base#self
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html Original Ext JS documentation of 'Ext.ClassManager'
 * @see ext.#ClassManager ext.ClassManager
 */
public class SClassManager extends Object {
  [ExtConfig]
  /**
   * A collection of diagnostic methods to decorate the real methods of the class. These
   * methods are applied as an <code>override</code> if this class has debug enabled as defined by
   * <code>Ext.isDebugEnabled</code>.
   * <p>These will be automatically removed by the Sencha Cmd compiler for production builds.</p>
   * <p>Example usage:</p>
   * <pre>
   *  Ext.define('Foo.bar.Class', {
   *      foo: function(a, b, c) {
   *          ...
   *      },
   *
   *      bar: function(a, b) {
   *          ...
   *          return 42;
   *      },
   *
   *      debugHooks: {
   *          foo: function(a, b, c) {
   *              // check arguments...
   *              return this.callParent(arguments);
   *          }
   *      }
   *  });
   * </pre>
   * <p>If you specify a <code>$enabled</code> property in the <code>→debugHooks</code> object that will be used
   * as the default enabled state for the hooks. If the <code>→ext.Ext.manifest</code> contains
   * a <code>debug</code> object of if <code>→ext.Ext.debugConfig</code> is specified, the <code>$enabled</code> flag
   * will override its "&#42;" value.</p>
   * @see SClassManager#debugHooks
   * @see ext.SExt#manifest ext.Ext.manifest
   * @see ext.SExt#debugConfig ext.Ext.debugConfig
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#cfg-debugHooks Original Ext JS documentation of 'debugHooks'
   */
  public native function get debugHooks():Object;

  [ExtConfig]
  /**
   * @private
   */
  public native function set debugHooks(value:Object):void;

  [ExtConfig]
  /**
   * The object given has properties that describe the versions at which the deprecations
   * apply.
   * <p>The purpose of the <code>→deprecated</code> declaration is to enable development mode to give
   * suitable error messages when deprecated methods or properties are used. Methods can
   * always be injected to provide this feedback, but properties can only be handled on
   * some browsers (those that support <code>Object.defineProperty</code>).</p>
   * <p>In some cases, deprecated methods can be restored to their previous behavior or
   * added back if they have been removed.</p>
   * <p>The structure of a <code>→deprecated</code> declaration is this:</p>
   * <pre>
   *  Ext.define('Foo.bar.Class', {
   *      ...
   *
   *      deprecated: {
   *          // Optional package name - default is the framework (ext or touch)
   *          name: 'foobar',
   *
   *          '5.0': {
   *              methods: {
   *                  // Throws: '"removedMethod" is deprecated.'
   *                  removedMethod: null,
   *
   *                  // Throws: '"oldMethod" is deprecated. Please use "newMethod" instead.'
   *                  oldMethod: 'newMethod',
   *
   *                  // When this block is enabled, this method is applied as an
   *                  // override. Otherwise you get same as "removeMethod".
   *                  method: function() {
   *                      // Do what v5 "method" did. If "method" exists in newer
   *                      // versions callParent can call it. If 5.1 has "method"
   *                      // then it would be next in line, otherwise 5.2 and last
   *                      // would be the current class.
   *                  },
   *
   *                  moreHelpful: {
   *                      message: 'Something helpful to do instead.',
   *                      fn: function() {
   *                          // The v5 "moreHelpful" method to use when enabled.
   *                      }
   *                  }
   *              },
   *              properties: {
   *                  // Throws: '"removedProp" is deprecated.'
   *                  removedProp: null,
   *
   *                  // Throws: '"oldProp" is deprecated. Please use "newProp" instead.'
   *                  oldProp: 'newProp',
   *
   *                  helpful: {
   *                      message: 'Something helpful message about what to do.'
   *                  }
   *                  ...
   *              },
   *              statics: {
   *                  methods: {
   *                      ...
   *                  },
   *                  properties: {
   *                      ...
   *                  },
   *              }
   *          },
   *
   *          '5.1': {
   *              ...
   *          },
   *
   *          '5.2': {
   *              ...
   *          }
   *      }
   *  });
   * </pre>
   * <p>The primary content of <code>→deprecated</code> are the version number keys. These indicate
   * a version number where methods or properties were deprecated. These versions are
   * compared to the version reported by <code>Ext.getCompatVersion</code> to determine the action
   * to take for each "block".</p>
   * <p>When the compatibility version is set to a value less than a version number key,
   * that block is said to be "enabled". For example, if a method was deprecated in
   * version 5.0 but the desired compatibility level is 4.2 then the block is used to
   * patch methods and (to some degree) restore pre-5.0 compatibility.</p>
   * <p>When multiple active blocks have the same method name, each method is applied as
   * an override in reverse order of version. In the above example, if a method appears
   * in the "5.0", "5.1" and "5.2" blocks then the "5.2" method is applied as an override
   * first, followed by the "5.1" method and finally the "5.0" method. This means that
   * the <code>callParent</code> from the "5.0" method calls the "5.1" method which calls the
   * "5.2" method which can (if applicable) call the current version.</p>
   * @see SClassManager#deprecated
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#cfg-deprecated Original Ext JS documentation of 'deprecated'
   */
  public native function get deprecated():Object;

  [ExtConfig]
  /**
   * @private
   */
  public native function set deprecated(value:Object):void;

  [ExtConfig]
  /**
   * Allows setting config values for a class based on specific platforms. The value
   * of this config is an object whose properties are "rules" and whose values are
   * objects containing config values.
   * <p>For example:</p>
   * <pre>
   *  Ext.define('App.view.Foo', {
   *      extend: 'Ext.panel.Panel',
   *
   *      platformConfig: {
   *          desktop: {
   *              title: 'Some Rather Descriptive Title'
   *          },
   *
   *          '!desktop': {
   *              title: 'Short Title'
   *          }
   *      }
   *  });
   * </pre>
   * <p>In the above, "desktop" and "!desktop" are (mutually exclusive) rules. Whichever
   * evaluates to <code>true</code> will have its configs applied to the class. In this case, only
   * the "title" property, but the object can contain any number of config properties.
   * In this case, the <code>→platformConfig</code> is evaluated as part of the class and there is
   * no cost for each instance created.</p>
   * <p>The rules are evaluated expressions in the context of the platform tags contained
   * in <code>→ext.Ext.platformTags</code>. Any properties of that object are
   * implicitly usable (as shown above).</p>
   * <p>If a <code>→platformConfig</code> specifies a config value, it will replace any values declared
   * on the class itself.</p>
   * <p>Use of <code>→platformConfig</code> on instances is handled by the config system when classes
   * call <code>→ext.Base.initConfig()</code>. For example:</p>
   * <pre>
   *  Ext.create({
   *      xtype: 'panel',
   *
   *      platformConfig: {
   *          desktop: {
   *              title: 'Some Rather Descriptive Title'
   *          },
   *
   *          '!desktop': {
   *              title: 'Short Title'
   *          }
   *      }
   *  });
   * </pre>
   * <p>The following is equivalent to the above:</p>
   * <pre>
   *  if (Ext.platformTags.desktop) {
   *      Ext.create({
   *          xtype: 'panel',
   *          title: 'Some Rather Descriptive Title'
   *      });
   *  } else {
   *      Ext.create({
   *          xtype: 'panel',
   *          title: 'Short Title'
   *      });
   *  }
   * </pre>
   * <p>To adjust configs based on dynamic conditions, see <code>→ext.mixin.Responsive</code>.</p>
   * @see SClassManager#platformConfig
   * @see ext.SExt#platformTags ext.Ext.platformTags
   * @see ext.Base#initConfig()
   * @see ext.mixin.Responsive
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#cfg-platformConfig Original Ext JS documentation of 'platformConfig'
   */
  public native function get platformConfig():Object;

  [ExtConfig]
  /**
   * @private
   */
  public native function set platformConfig(value:Object):void;

  /**
   * Adds a batch of class name to alias mappings.
   * @param aliases The set of mappings of the form.
   * className : [values...]
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#method-addNameAliasMappings Original Ext JS documentation of 'addNameAliasMappings'
   */
  public native function addNameAliasMappings(aliases:Object):void;

  /**
   * @param alternates The set of mappings of the form
   * className : [values...]
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#method-addNameAlternateMappings Original Ext JS documentation of 'addNameAlternateMappings'
   */
  public native function addNameAlternateMappings(alternates:Object):void;

  /**
   * Retrieve a class by its name.
   * @return class
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#method-get Original Ext JS documentation of 'get'
   */
  public native function get(name:String):ExtClass;

  /**
   * Get a reference to the class by its alias.
   * @return class
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#method-getByAlias Original Ext JS documentation of 'getByAlias'
   */
  public native function getByAlias(alias:String):ExtClass;

  /**
   * Get the class of the provided object; returns null if it's not an instance
   * of any class created with Ext.define. This is usually invoked by the
   * shorthand →<code>ext.Ext.getClass()</code>.
   * <pre>
   * var component = new Ext.Component();
   *
   * Ext.getClass(component); // returns Ext.Component
   * </pre>
   * @return class
   * @see ext.SExt#getClass() ext.Ext.getClass()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#method-getClass Original Ext JS documentation of 'getClass'
   */
  public native function getClass(object:Object):ExtClass;

  /**
   * Returns the displayName property or className or object. When all else fails,
   * returns "Anonymous".
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#method-getDisplayName Original Ext JS documentation of 'getDisplayName'
   */
  public native function getDisplayName(object:Object):String;

  /**
   * Get the name of the class by its reference or its instance. This is
   * usually invoked by the shorthand →<code>ext.Ext.getClassName()</code>.
   * <pre>
   * Ext.ClassManager.getName(Ext.Action); // returns "Ext.Action"
   * </pre>
   * @return className
   * @see ext.SExt#getClassName() ext.Ext.getClassName()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#method-getName Original Ext JS documentation of 'getName'
   */
  public native function getName(object:*):String;

  [Deprecated(message="Use Ext.create() instead.\n", since="5.0")]
  /**
   * Instantiate a class by either full name, alias or alternate name
   * @param name
   * @param args Additional arguments after the name will be passed to the class'
   * constructor.
   * @return instance
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#method-instantiate Original Ext JS documentation of 'instantiate'
   */
  public native function instantiate(name:String, args:*):Object;

  /**
   * Instantiate a class by its alias. This is usually invoked by the
   * shorthand →<code>ext.Ext.createByAlias()</code>.
   * <p>If →<code>ext.Loader</code> is <i>enabled</i> (→<code>ext.Loader.setConfig()</code>) and the class
   * has not been defined yet, it will attempt to load the class via synchronous
   * loading.</p>
   * <pre>
   * var window = Ext.createByAlias('widget.window', { width: 600, height: 800 });
   * </pre>
   * @param alias
   * @param args Additional arguments after the alias will be passed to the
   * class constructor.
   * @return instance
   * @see ext.SExt#createByAlias() ext.Ext.createByAlias()
   * @see ext.#Loader ext.Loader
   * @see ext.SLoader#setConfig() ext.Loader.setConfig()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#method-instantiateByAlias Original Ext JS documentation of 'instantiateByAlias'
   */
  public native function instantiateByAlias(alias:String, ...args):Object;

  /**
   * Checks if a class has already been created.
   * @return exist
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#method-isCreated Original Ext JS documentation of 'isCreated'
   */
  public native function isCreated(className:String):Boolean;

  /**
   * Sets a name reference to a class.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#method-set Original Ext JS documentation of 'set'
   */
  public native function set(name:String, value:Object):SClassManager;

  /**
   * Creates a namespace and assign the <code>value</code> to the created object.
   * <pre>
   * Ext.ClassManager.setNamespace('MyCompany.pkg.Example', someObject);
   *
   * alert(MyCompany.pkg.Example === someObject); // alerts true
   * </pre>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ClassManager.html#method-setNamespace Original Ext JS documentation of 'setNamespace'
   */
  public native function setNamespace(namespace:String, value:Object):void;
}
}