package ext {
import ext.container.Container;

[Native]
/**
 * Provides searching of Components within Ext.ComponentManager (globally) or a specific
 * Ext.container.Container on the document with a similar syntax to a CSS selector.
 * Returns Array of matching Components, or empty Array.
 * <p><b><i>Basic Component lookup</i></b></p>
 * <p>Components can be retrieved by using their <i>xtype</i> (→<code>ext.Component</code>):</p>
 * <ul>
 * <li><code>component</code></li>
 * <li><code>gridpanel</code></li>
 * </ul>
 * <p>Matching by <code>xtype</code> matches inherited types, so in the following code, the previous field
 * <i>of any type which inherits from <code>TextField</code></i> will be found:</p>
 * <pre>
 * prevField = myField.previousNode('textfield');
 * </pre>
 * <p>To match only the exact type, pass the "shallow" flag by adding <code>(true)</code> to xtype
 * (See Component's →<code>ext.Component.isXType()</code> method):</p>
 * <pre>
 * prevTextField = myField.previousNode('textfield(true)');
 * </pre>
 * <p>You can search Components by their <code>id</code> or <code>itemId</code> property, prefixed with a #:</p>
 * <pre>
 * #myContainer
 * </pre>
 * <p>Component <code>xtype</code> and <code>id</code> or <code>itemId</code> can be used together to avoid possible
 * id collisions between Components of different types:</p>
 * <pre>
 * panel#myPanel
 * </pre>
 * <p>When Component's <code>id</code> or <code>xtype</code> contains dots, you can escape them in your selector:</p>
 * <pre>
 * my\.panel#myPanel
 * </pre>
 * <p>Keep in mind that JavaScript treats the backslash character in a special way, so you
 * need to escape it, too, in the actual code:</p>
 * <pre>
 * var myPanel = Ext.ComponentQuery.query('my\\.panel#myPanel');
 * </pre>
 * <p><b><i>Traversing Component tree</i></b></p>
 * <p>Components can be found by their relation to other Components. There are several
 * relationship operators, mostly taken from CSS selectors:</p>
 * <ul>
 * <li><b><code>E F</code></b> All descendant Components of E that match F</li>
 * <li><b><code>E &gt; F</code></b> All direct children Components of E that match F</li>
 * <li><b><code>E ^ F</code></b> All parent Components of E that match F</li>
 * </ul>
 * <p>Expressions between relationship operators are matched left to right, i.e. leftmost
 * selector is applied first, then if one or more matches are found, relationship operator
 * itself is applied, then next selector expression, etc. It is possible to combine
 * relationship operators in complex selectors:</p>
 * <pre>
 * window[title="Input form"] textfield[name=login] ^ form &gt; button[action=submit]
 * </pre>
 * <p>That selector can be read this way: Find a window with title "Input form", in that
 * window find a TextField with name "login" at any depth (including subpanels and/or
 * FieldSets), then find an <code>→ext.form.FormPanel</code> that is a parent of the TextField, and in
 * that form find a direct child that is a button with custom property <code>action</code> set to
 * value "submit".</p>
 * <p>Whitespace on both sides of <code>^</code> and <code>&gt;</code> operators is non-significant, i.e. can be
 * omitted, but usually is used for clarity.</p>
 * <p><b><i>Searching by Component attributes</i></b></p>
 * <p>Components can be searched by their object property values (attributes). To do that,
 * use attribute matching expression in square brackets:</p>
 * <ul>
 * <li><code>component[disabled]</code> - matches any Component that has <code>disabled</code> property with
 * any truthy (non-empty, not <code>false</code>) value.</li>
 * <li><code>panel[title="Test"]</code> - matches any Component that has <code>title</code> property set to
 * "Test". Note that if the value does not contain spaces, the quotes are optional.</li>
 * </ul>
 * <p>Attributes can use any of the following operators to compare values:
 * <code>=</code>, <code>!=</code>, <code>^=</code>, <code>$=</code>, <code>&#42;=</code>, <code>%=</code>, <code>|=</code> and <code>~=</code>.</p>
 * <p>Prefixing the attribute name with an at sign <code>&#64;</code> means that the property must be
 * the object's <code>ownProperty</code>, not a property from the prototype chain.</p>
 * <p>Specifications like <code>[propName]</code> check that the property is a truthy value. To check
 * that the object has an <code>ownProperty</code> of a certain name, regardless of the value use
 * the form <code>[?propName]</code>.</p>
 * <p>The specified value is coerced to match the type of the property found in the
 * candidate Component using →<code>ext.Ext.coerce()</code>.</p>
 * <p>If you need to find Components by their <code>itemId</code> property, use the <code>#id</code> form; it will
 * do the same as <code>[itemId=id]</code> but is easier to read.</p>
 * <p>If you need to include a metacharacter like (, ), [, ], etc., in the query, escape it
 * by prefixing it with a backslash:</p>
 * <pre>
 *  var component = Ext.ComponentQuery.query('[myProperty=\\[foo\\]]');
 * </pre>
 * <p><b><i>Attribute matching operators</i></b></p>
 * <p>The '=' operator will return the results that <b>exactly</b> match the
 * specified object property (attribute):</p>
 * <pre>
 * Ext.ComponentQuery.query('panel[cls=my-cls]');
 * </pre>
 * <p>Will match the following Component:</p>
 * <pre>
 * Ext.create('Ext.window.Window', {
 *     cls: 'my-cls'
 * });
 * </pre>
 * <p>But will not match the following Component, because 'my-cls' is one value
 * among others:</p>
 * <pre>
 *  Ext.create('Ext.panel.Panel', {
 *      cls: 'foo-cls my-cls bar-cls'
 *  });
 * </pre>
 * <p>You can use the '~=' operator instead, it will return Components with
 * the property that <b>exactly</b> matches one of the whitespace-separated
 * values. This is also true for properties that only have <i>one</i> value:</p>
 * <pre>
 * Ext.ComponentQuery.query('panel[cls~=my-cls]');
 * </pre>
 * <p>Will match both Components:</p>
 * <pre>
 * Ext.create('Ext.panel.Panel', {
 *     cls: 'foo-cls my-cls bar-cls'
 * });
 *
 * Ext.create('Ext.window.Window', {
 *     cls: 'my-cls'
 * });
 * </pre>
 * <p>Generally, '=' operator is more suited for object properties other than
 * CSS classes, while '~=' operator will work best with properties that
 * hold lists of whitespace-separated CSS classes.</p>
 * <p>The '^=' operator will return Components with specified attribute that
 * start with the passed value:</p>
 * <pre>
 * Ext.ComponentQuery.query('panel[title^=Sales]');
 * </pre>
 * <p>Will match the following Component:</p>
 * <pre>
 * Ext.create('Ext.panel.Panel', {
 *     title: 'Sales estimate for Q4'
 * });
 * </pre>
 * <p>The '$=' operator will return Components with specified properties that
 * end with the passed value:</p>
 * <pre>
 * Ext.ComponentQuery.query('field[fieldLabel$=name]');
 * </pre>
 * <p>Will match the following Component:</p>
 * <pre>
 * Ext.create('Ext.form.field.Text', {
 *     fieldLabel: 'Enter your name'
 * });
 * </pre>
 * <p>The '/=' operator will return Components with specified properties that
 * match the passed regular expression:</p>
 * <pre>
 * Ext.ComponentQuery.query('button[action/="edit|save"]');
 * </pre>
 * <p>Will match the following Components with a custom <code>action</code> property:</p>
 * <pre>
 * Ext.create('Ext.button.Button', {
 *      action: 'edit'
 * });
 *
 * Ext.create('Ext.button.Button', {
 *      action: 'save'
 * });
 * </pre>
 * <p>When you need to use meta characters like [], (), etc. in your query, make sure
 * to escape them with back slashes:</p>
 * <pre>
 * Ext.ComponentQuery.query('panel[title="^Sales for Q\\[1-4\\]"]');
 * </pre>
 * <p>The following test will find panels with their <code>ownProperty</code> collapsed being equal to
 * <code>false</code>. It will <b>not</b> match a collapsed property from the prototype chain.</p>
 * <pre>
 * Ext.ComponentQuery.query('panel[&#64;collapsed=false]');
 * </pre>
 * <p>Member expressions from candidate Components may be tested. If the expression returns
 * a <i>truthy</i> value, the candidate Component will be included in the query:</p>
 * <pre>
 * var disabledFields = myFormPanel.query("{isDisabled()}");
 * </pre>
 * <p>Such expressions are executed in Component's context, and the above expression is
 * similar to running this snippet for every Component in your application:</p>
 * <pre>
 *  if (component.isDisabled()) {
 *      matches.push(component);
 *  }
 * </pre>
 * <p>It is important to use only methods that are available in <b>every</b> Component instance
 * to avoid run time exceptions. If you need to match your Components with a custom
 * condition formula, you can augment <code>→ext.Component</code> to provide custom matcher that
 * will return <code>false</code> by default, and override it in your custom classes:</p>
 * <pre>
 *  Ext.define('My.Component', {
 *      override: 'Ext.Component',
 *      myMatcher: function() { return false; }
 *  });
 *
 *  Ext.define('My.Panel', {
 *      extend: 'Ext.panel.Panel',
 *      requires: ['My.Component'],     // Ensure that Component override is applied
 *      myMatcher: function(selector) {
 *          return selector === 'myPanel';
 *      }
 *  });
 * </pre>
 * <p>After that you can use a selector with your custom matcher to find all instances
 * of <code>My.Panel</code>:</p>
 * <pre>
 *  Ext.ComponentQuery.query("{myMatcher('myPanel')}");
 * </pre>
 * <p>However if you really need to use a custom matcher, you may find it easier to implement
 * a custom Pseudo class instead (see below).</p>
 * <p><b><i>Conditional matching</i></b></p>
 * <p>Attribute matchers can be combined to select only Components that match <b>all</b>
 * conditions (logical AND operator):</p>
 * <pre>
 * Ext.ComponentQuery.query('panel[cls~=my-cls][floating=true][title$="sales data"]');
 * </pre>
 * <p>E.g., the query above will match only a Panel-descended Component that has 'my-cls'
 * CSS class <i>and</i> is floating <i>and</i> with a title that ends with "sales data".</p>
 * <p>Expressions separated with commas will match any Component that satisfies
 * <i>either</i> expression (logical OR operator):</p>
 * <pre>
 * Ext.ComponentQuery.query('field[fieldLabel^=User], field[fieldLabel&#42;=password]');
 * </pre>
 * <p>E.g., the query above will match any field with field label starting with "User",
 * <i>or</i> any field that has "password" in its label.</p>
 * <p>If you need to include a comma in an attribute matching expression, escape it with a
 * backslash:</p>
 * <pre>
 * Ext.ComponentQuery.query('field[fieldLabel^="User\\, foo"], field[fieldLabel&#42;=password]');
 * </pre>
 * <p><b><i>Pseudo classes</i></b></p>
 * <p>Pseudo classes may be used to filter results in the same way as in
 * →<code>ext.dom.DomQuery</code>. There are five default pseudo classes:</p>
 * <ul>
 * <li><code>not</code> Negates a selector.</li>
 * <li><code>first</code> Filters out all except the first matching item for a selector.</li>
 * <li><code>last</code> Filters out all except the last matching item for a selector.</li>
 * <li><code>focusable</code> Filters out all except Components which by definition and configuration are
 * potentially able to receieve focus, and can be focused at this time. Component can be
 * focused when it is rendered, visible, and not disabled. Some Components can be focusable
 * even when disabled (e.g. Menu items) via their parent Container configuration.
 * Containers such as Panels generally are not focusable by themselves but can use
 * focus delegation (<code>defaultFocus</code> config). Some Containers such as Menus and Windows
 * are focusable by default.</li>
 * <li><code>canfocus</code> Filters out all except Components which are curently able to receieve focus.
 * That is, they are defined and configured focusable, and they are also visible and enabled.
 * Note that this selector intentionally bypasses some checks done by <code>focusable</code> selector
 * and works in a subtly different way. It is used internally by the framework and is not
 * a replacement for <code>:focusable</code> selector.</li>
 * <li><code>nth-child</code> Filters Components by ordinal position in the selection.</li>
 * <li><code>scrollable</code> Filters out all except Components which are scrollable.</li>
 * <li><code>visible</code> Filters out hidden Components. May test deep visibility using <code>':visible(true)'</code></li>
 * </ul>
 * <p>These pseudo classes can be used with other matchers or without them:</p>
 * <pre>
 *  // Select first direct child button in any panel
 *  Ext.ComponentQuery.query('panel &gt; button:first');
 *
 *  // Select last field in Profile form
 *  Ext.ComponentQuery.query('form[title=Profile] field:last');
 *
 *  // Find first focusable Component in a panel and focus it
 *  panel.down(':canfocus').focus();
 *
 *  // Select any field that is not hidden in a form
 *  form.query('field:not(hiddenfield)');
 *
 *  // Find last scrollable Component and reset its scroll positions.
 *  tabpanel.down(':scrollable[hideMode=display]:last').getScrollable().scrollTo(0, 0);
 * </pre>
 * <p>Pseudo class <code>nth-child</code> can be used to find any child Component by its
 * position relative to its siblings. This class' handler takes one argument
 * that specifies the selection formula as <code>Xn</code> or <code>Xn+Y</code>:</p>
 * <pre>
 *  // Find every odd field in a form
 *  form.query('field:nth-child(2n+1)'); // or use shortcut: :nth-child(odd)
 *
 *  // Find every even field in a form
 *  form.query('field:nth-child(2n)');   // or use shortcut: :nth-child(even)
 *
 *  // Find every 3rd field in a form
 *  form.query('field:nth-child(3n)');
 * </pre>
 * <p><b>Note:</b> The <code>nth-child</code> selector returns 1-based result sets.</p>
 * <p>Pseudo classes can be combined to further filter the results, e.g., in the
 * form example above we can modify the query to exclude hidden fields:</p>
 * <pre>
 *  // Find every 3rd non-hidden field in a form
 *  form.query('field:not(hiddenfield):nth-child(3n)');
 * </pre>
 * <p>Note that when combining pseudo classes, whitespace is significant, i.e.
 * there should be no spaces between pseudo classes. This is a common mistake;
 * if you accidentally type a space between <code>field</code> and <code>:not</code>, the query
 * will not return any result because it will mean "find <i>field's children
 * Components</i> that are not hidden fields...".</p>
 * <p><b><i>Custom pseudo classes</i></b></p>
 * <p>It is possible to define your own custom pseudo classes. In fact, a
 * pseudo class is just a property in <code>Ext.ComponentQuery.pseudos</code> object
 * that defines pseudo class name (property name) and pseudo class handler
 * (property value):</p>
 * <pre>
 * // Function receives array and returns a filtered array.
 * Ext.ComponentQuery.pseudos.invalid = function(items) {
 *     var i = 0, l = items.length, c, result = [];
 *     for (; i &lt; l; i++) {
 *         if (!(c = items[i]).isValid()) {
 *             result.push(c);
 *         }
 *     }
 *     return result;
 * };
 *
 * var invalidFields = myFormPanel.query('field:invalid');
 * if (invalidFields.length) {
 *     invalidFields[0].getEl().scrollIntoView(myFormPanel.body);
 *     for (var i = 0, l = invalidFields.length; i &lt; l; i++) {
 *         invalidFields[i].getEl().frame("red");
 *     }
 * }
 * </pre>
 * <p>Pseudo class handlers can be even more flexible, with a selector
 * argument used to define the logic:</p>
 * <pre>
 *  // Handler receives array of itmes and selector in parentheses
 *  Ext.ComponentQuery.pseudos.titleRegex = function(components, selector) {
 *      var i = 0, l = components.length, c, result = [], regex = new RegExp(selector);
 *      for (; i &lt; l; i++) {
 *          c = components[i];
 *          if (c.title &amp;&amp; regex.test(c.title)) {
 *              result.push(c);
 *          }
 *      }
 *      return result;
 *  }
 *
 *  var salesTabs = tabPanel.query('panel:titleRegex("sales\\s+for\\s+201[123]")');
 * </pre>
 * <p>Be careful when using custom pseudo classes with MVC Controllers: when
 * you use a pseudo class in Controller's <code>control</code> or <code>listen</code> component
 * selectors, the pseudo class' handler function will be called very often
 * and may slow down your application significantly. A good rule of thumb
 * is to always specify Component xtype with the pseudo class so that the
 * handlers are only called on Components that you need, and try to make
 * the condition checks as cheap in terms of execution time as possible.
 * Note how in the example above, handler function checks that Component
 * <i>has</i> a title first, before running regex test on it.</p>
 * <p><b><i>Query examples</i></b></p>
 * <p>Queries return an array of Components. Here are some example queries:</p>
 * <pre>
 * // retrieve all Ext.Panels in the document by xtype
 * var panelsArray = Ext.ComponentQuery.query('panel');
 *
 * // retrieve all Ext.Panels within the container with an id myCt
 * var panelsWithinmyCt = Ext.ComponentQuery.query('#myCt panel');
 *
 * // retrieve all direct children which are Ext.Panels within myCt
 * var directChildPanel = Ext.ComponentQuery.query('#myCt &gt; panel');
 *
 * // retrieve all grids or trees
 * var gridsAndTrees = Ext.ComponentQuery.query('gridpanel, treepanel');
 *
 * // Focus first Component
 * myFormPanel.child(':canfocus').focus();
 *
 * // Retrieve every odd text field in a form
 * myFormPanel.query('textfield:nth-child(odd)');
 *
 * // Retrieve every even field in a form, excluding hidden fields
 * myFormPanel.query('field:not(hiddenfield):nth-child(even)');
 *
 * // Retrieve every scrollable in a tabpanel
 * tabpanel.query(':scrollable');
 * </pre>
 * <p>For easy access to queries based from a particular Container see the
 * →<code>ext.container.Container.query()</code>, →<code>ext.container.Container.down()</code> and
 * →<code>ext.container.Container.child()</code> methods. Also see
 * →<code>ext.Component.up()</code>.</p>
 * <p>Type of singleton ComponentQuery.</p>
 * @see ext.Component
 * @see ext.Component#isXType()
 * @see ext.form.FormPanel
 * @see ext.SExt#coerce() ext.Ext.coerce()
 * @see ext.dom.#DomQuery ext.dom.DomQuery
 * @see ext.container.Container#query()
 * @see ext.container.Container#down()
 * @see ext.container.Container#child()
 * @see ext.Component#up()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ComponentQuery.html Original Ext JS documentation of 'Ext.ComponentQuery'
 * @see ext.#ComponentQuery ext.ComponentQuery
 */
public class SComponentQuery extends Base {
  /**
   * Tests whether the passed Component matches the selector string.
   * An empty selector will always match.
   * @param component The Component to test
   * @param selector The selector string to test against.
   * Or a filter function which returns <code>true</code> if the component matches.
   * @param root The root component.
   * @return True if the Component matches the selector.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ComponentQuery.html#method-is Original Ext JS documentation of 'is'
   */
  public native function matches(component:Component, selector:*, root:Component = null):Boolean;

  [ArrayElementType("ext.Component")]
  /**
   * Returns an array of matched Components from within the passed root object.
   * <p>This method filters returned Components in a similar way to how CSS selector based DOM
   * queries work using a textual selector string.</p>
   * <p>See class summary for details.</p>
   * @param selector The selector string to filter returned Components.
   * @param root The Container within which to perform the query.
   * If omitted, all Components within the document are included in the search.
   * <p>This parameter may also be an array of Components to filter according to the selector.</p>
   * @return The matched Components.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ComponentQuery.html#method-query Original Ext JS documentation of 'query'
   */
  public native function query(selector:String, root:Container = null):Array;

  /**
   * Traverses the tree rooted at the passed root in post-order mode, calling the passed
   * function on the nodes at each level. That is the function is called upon each node
   * <b>after</b> being called on its children).
   * <p>For an object to be queryable, it must implement the <code>getRefItems</code> method which returns
   * all immediate child items.</p>
   * <p>This method is used at each level down the cascade. Currently
   * →<code>ext.Component</code>s and →<code>ext.data.TreeModel</code>s are queryable.</p>
   * <p>If you have tree-structured data, you can make your nodes queryable, and use
   * ComponentQuery on them.</p>
   * @param selector A ComponentQuery selector used to filter candidate nodes
   * before calling the function. An empty string matches any node.
   * @param root The root queryable object to start from.
   * @param fn The function to call. Return <code>false</code> to abort the traverse.
   * <ul>
   * <li><code>node:Object</code> (optional) —
   * The node being visited.
   * </li>
   * </ul>
   * @param scope The context (<code>this</code> reference) in which the function is executed.
   * @param extraArgs A set of arguments to be appended to the function's argument
   * list to pass down extra data known to the caller <b>after</b> the node being visited.
   * @see ext.Component
   * @see ext.data.TreeModel
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ComponentQuery.html#method-visitPostOrder Original Ext JS documentation of 'visitPostOrder'
   */
  public native function visitPostOrder(selector:Object, root:String, fn:Function, scope:Object = null, extraArgs:Array = null):void;

  /**
   * Traverses the tree rooted at the passed root in pre-order mode, calling the passed
   * function on the nodes at each level. That is the function is called upon each node
   * <b>before</b> being called on its children).
   * <p>For an object to be queryable, it must implement the <code>getRefItems</code> method which returns
   * all immediate child items.</p>
   * <p>This method is used at each level down the cascade. Currently
   * →<code>ext.Component</code>s and →<code>ext.data.TreeModel</code>s are queryable.</p>
   * <p>If you have tree-structured data, you can make your nodes queryable, and use
   * ComponentQuery on them.</p>
   * @param selector A ComponentQuery selector used to filter candidate nodes before
   * calling the function. An empty string matches any node.
   * @param root The root queryable object to start from.
   * @param fn The function to call. Return <code>false</code> to abort the traverse.
   * <ul>
   * <li><code>node:Object</code> (optional) —
   * The node being visited.
   * </li>
   * </ul>
   * @param scope The context (<code>this</code> reference) in which the function is executed.
   * @param extraArgs A set of arguments to be appended to the function's argument
   * list to pass down extra data known to the caller <b>after</b> the node being visited.
   * @see ext.Component
   * @see ext.data.TreeModel
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.ComponentQuery.html#method-visitPreOrder Original Ext JS documentation of 'visitPreOrder'
   */
  public native function visitPreOrder(selector:Object, root:String, fn:Function, scope:Object = null, extraArgs:Array = null):void;
}
}