package ext {

[Native]
/**
 * This class defines some basic methods for handling dates.
 * <p>The date parsing and formatting syntax contains a subset of
 * <a href="http://www.php.net/date">PHP's <code>date()</code> function</a>, and the formats that are
 * supported will provide results equivalent to their PHP versions.</p>
 * <p>The following is a list of all currently supported formats:</p>
 * <pre>
 *  Format      Description                                                               Example returned values
 *  ------      -----------------------------------------------------------------------   -----------------------
 *    d         Day of the month, 2 digits with leading zeros                             01 to 31
 *    D         A short textual representation of the day of the week                     Mon to Sun
 *    j         Day of the month without leading zeros                                    1 to 31
 *    l         A full textual representation of the day of the week                      Sunday to Saturday
 *    N         ISO-8601 numeric representation of the day of the week                    1 (for Monday) through 7 (for Sunday)
 *    S         English ordinal suffix for the day of the month, 2 characters             st, nd, rd or th. Works well with j
 *    w         Numeric representation of the day of the week                             0 (for Sunday) to 6 (for Saturday)
 *    z         The day of the year (starting from 0)                                     0 to 364 (365 in leap years)
 *    W         ISO-8601 week number of year, weeks starting on Monday                    01 to 53
 *    F         A full textual representation of a month, such as January or March        January to December
 *    m         Numeric representation of a month, with leading zeros                     01 to 12
 *    M         A short textual representation of a month                                 Jan to Dec
 *    n         Numeric representation of a month, without leading zeros                  1 to 12
 *    t         Number of days in the given month                                         28 to 31
 *    L         Whether it&amp;#39;s a leap year                                                  1 if it is a leap year, 0 otherwise.
 *    o         ISO-8601 year number (identical to (Y), but if the ISO week number (W)    Examples: 1998 or 2004
 *              belongs to the previous or next year, that year is used instead)
 *    Y         A full numeric representation of a year, 4 digits                         Examples: 1999 or 2003
 *    y         A two digit representation of a year                                      Examples: 99 or 03
 *    a         Lowercase Ante meridiem and Post meridiem                                 am or pm
 *    A         Uppercase Ante meridiem and Post meridiem                                 AM or PM
 *    g         12-hour format of an hour without leading zeros                           1 to 12
 *    G         24-hour format of an hour without leading zeros                           0 to 23
 *    h         12-hour format of an hour with leading zeros                              01 to 12
 *    H         24-hour format of an hour with leading zeros                              00 to 23
 *    i         Minutes, with leading zeros                                               00 to 59
 *    s         Seconds, with leading zeros                                               00 to 59
 *    u         Decimal fraction of a second                                              Examples:
 *              (minimum 1 digit, arbitrary number of digits allowed)                     001 (i.e. 0.001s) or
 *                                                                                        100 (i.e. 0.100s) or
 *                                                                                        999 (i.e. 0.999s) or
 *                                                                                        999876543210 (i.e. 0.999876543210s)
 *    O         Difference to Greenwich time (GMT) in hours and minutes                   Example: +1030
 *    P         Difference to Greenwich time (GMT) with colon between hours and minutes   Example: -08:00
 *    T         Timezone abbreviation of the machine running the code                     Examples: EST, MDT, PDT ...
 *    Z         Timezone offset in seconds (negative if west of UTC, positive if east)    -43200 to 50400
 *    c         ISO 8601 date represented as the local time with an offset to UTC appended.
 *              Notes:                                                                    Examples:
 *              1) If unspecified, the month / day defaults to the current month / day,   1991 or
 *                 the time defaults to midnight, while the timezone defaults to the      1992-10 or
 *                 browser's timezone. If a time is specified, it must include both hours 1993-09-20 or
 *                 and minutes. The "T" delimiter, seconds, milliseconds and timezone     1994-08-19T16:20+01:00 or
 *                 are optional.                                                          1995-07-18T17:21:28-02:00 or
 *              2) The decimal fraction of a second, if specified, must contain at        1996-06-17T18:22:29.98765+03:00 or
 *                 least 1 digit (there is no limit to the maximum number                 1997-05-16T19:23:30,12345-0400 or
 *                 of digits allowed), and may be delimited by either a '.' or a ','      1998-04-15T20:24:31.2468Z or
 *              Refer to the examples on the right for the various levels of              1999-03-14T20:24:32Z or
 *              date-time granularity which are supported, or see                         2000-02-13T21:25:33
 *              http://www.w3.org/TR/NOTE-datetime for more info.                         2001-01-12 22:26:34
 *    C         An ISO date string as implemented by the native Date object's             1962-06-17T09:21:34.125Z
 *              [Date.toISOString](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Date/toISOString)
 *              method. This outputs the numeric part with &#42;UTC&#42; hour and minute
 *              values, and indicates this by appending the `'Z'` timezone
 *              identifier.
 *    U         Seconds since the Unix Epoch (January 1 1970 00:00:00 GMT)                1193432466 or -2138434463
 *    MS        Microsoft AJAX serialized dates                                           \/Date(1238606590509)\/ (i.e. UTC milliseconds since epoch) or
 *                                                                                        \/Date(1238606590509+0800)\/
 *    time      A javascript millisecond timestamp                                        1350024476440
 *    timestamp A UNIX timestamp (same as U)                                              1350024866
 * </pre>
 * <p>Example usage (note that you must escape format specifiers with '\' to render them as character literals):</p>
 * <pre>
 * // Sample date:
 * // 'Wed Jan 10 2007 15:05:01 GMT-0600 (Central Standard Time)'
 *
 * var dt = new Date('1/10/2007 03:05:01 PM GMT-0600');
 * console.log(Ext.Date.format(dt, 'Y-m-d'));                          // 2007-01-10
 * console.log(Ext.Date.format(dt, 'F j, Y, g:i a'));                  // January 10, 2007, 3:05 pm
 * console.log(Ext.Date.format(dt, 'l, \\t\\he jS \\of F Y h:i:s A')); // Wednesday, the 10th of January 2007 03:05:01 PM
 * </pre>
 * <p>Here are some standard date/time patterns that you might find helpful. They
 * are not part of the source of Ext.Date, but to use them you can simply copy this
 * block of code into any script that is included after Ext.Date and they will also become
 * globally available on the Date object. Feel free to add or remove patterns as needed in your code.</p>
 * <pre>
 * Ext.Date.patterns = {
 *     ISO8601Long:"Y-m-d H:i:s",
 *     ISO8601Short:"Y-m-d",
 *     ShortDate: "n/j/Y",
 *     LongDate: "l, F d, Y",
 *     FullDateTime: "l, F d, Y g:i:s A",
 *     MonthDay: "F d",
 *     ShortTime: "g:i A",
 *     LongTime: "g:i:s A",
 *     SortableDateTime: "Y-m-d\\TH:i:s",
 *     UniversalSortableDateTime: "Y-m-d H:i:sO",
 *     YearMonth: "F, Y"
 * };
 * </pre>
 * <p>Example usage:</p>
 * <pre>
 * var dt = new Date();
 * console.log(Ext.Date.format(dt, Ext.Date.patterns.ShortDate));
 * </pre>
 * <p>Developer-written, custom formats may be used by supplying both a formatting and a parsing function
 * which perform to specialized requirements. The functions are stored in →<code>parseFunctions</code> and →<code>formatFunctions</code>.</p>
 * <p>Type of singleton DateUtil.</p>
 * @see SDateUtil#parseFunctions
 * @see SDateUtil#formatFunctions
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html Original Ext JS documentation of 'Ext.Date'
 * @see ext.#DateUtil ext.DateUtil
 */
public class SDateUtil extends Object {
  /**
   * Date interval constant.
   * @default "d"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-DAY Original Ext JS documentation of 'DAY'
   */
  public native function get DAY():String;

  /**
   * The number of days in a week.
   * @default 7
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-DAYS_IN_WEEK Original Ext JS documentation of 'DAYS_IN_WEEK'
   */
  public native function get DAYS_IN_WEEK():Number;

  /**
   * Date interval constant.
   * @default "h"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-HOUR Original Ext JS documentation of 'HOUR'
   */
  public native function get HOUR():String;

  /**
   * The maximum number of days in a month.
   * @default 31
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-MAX_DAYS_IN_MONTH Original Ext JS documentation of 'MAX_DAYS_IN_MONTH'
   */
  public native function get MAX_DAYS_IN_MONTH():Number;

  /**
   * Date interval constant.
   * @default "ms"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-MILLI Original Ext JS documentation of 'MILLI'
   */
  public native function get MILLI():String;

  /**
   * Date interval constant.
   * @default "mi"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-MINUTE Original Ext JS documentation of 'MINUTE'
   */
  public native function get MINUTE():String;

  /**
   * Date interval constant.
   * @default "mo"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-MONTH Original Ext JS documentation of 'MONTH'
   */
  public native function get MONTH():String;

  /**
   * The number of months in a year.
   * @default 12
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-MONTHS_IN_YEAR Original Ext JS documentation of 'MONTHS_IN_YEAR'
   */
  public native function get MONTHS_IN_YEAR():Number;

  /**
   * Date interval constant.
   * @default "s"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-SECOND Original Ext JS documentation of 'SECOND'
   */
  public native function get SECOND():String;

  /**
   * Date interval constant.
   * @default "y"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-YEAR Original Ext JS documentation of 'YEAR'
   */
  public native function get YEAR():String;

  [ArrayElementType("String")]
  /**
   * An array of textual day names.
   * Override these values for international dates.
   * <p>Example:</p>
   * <pre>
   * Ext.Date.dayNames = [
   *     'SundayInYourLang',
   *     'MondayInYourLang'
   *     // ...
   * ];
   * </pre>
   * @default [
   *     "Sunday",
   *     "Monday",
   *     "Tuesday",
   *     "Wednesday",
   *     "Thursday",
   *     "Friday",
   *     "Saturday"
   * ]
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-dayNames Original Ext JS documentation of 'dayNames'
   */
  public native function get dayNames():Array;

  /**
   * @private
   */
  public native function set dayNames(value:Array):void;

  /**
   * The date format string that the →<code>ext.util.Format.dateRenderer()</code>
   * and →<code>ext.util.Format.date()</code> functions use. See →<code>ext.DateUtil</code> for details.
   * <p>This is the format that →<code>flexParse()</code> uses to disambiguate all-numeric
   * input dates.</p>
   * <p>This may be overridden in a locale file.</p>
   * @default 'm/d/Y'
   * @see ext.util.SFormat#dateRenderer() ext.util.Format.dateRenderer()
   * @see ext.util.SFormat#date() ext.util.Format.date()
   * @see ext.#DateUtil ext.DateUtil
   * @see SDateUtil#flexParse()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-defaultFormat Original Ext JS documentation of 'defaultFormat'
   */
  public native function get defaultFormat():String;

  /**
   * @private
   */
  public native function set defaultFormat(value:String):void;

  /**
   * The default time format.
   * <p>This may be overridden in a locale file.</p>
   * @default 'h:i A'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-defaultTimeFormat Original Ext JS documentation of 'defaultTimeFormat'
   */
  public native function get defaultTimeFormat():String;

  /**
   * @private
   */
  public native function set defaultTimeFormat(value:String):void;

  /**
   * An object hash containing default date values used during date parsing.
   * <p>The following properties are available:</p>
   * <ul>
   * <li><code>y</code>: →<code>Number</code> - The default year value. Defaults to <code>undefined</code>.</li>
   * <li><code>m</code>: →<code>Number</code> - The default 1-based month value. Defaults to <code>undefined</code>.</li>
   * <li><code>d</code>: →<code>Number</code> - The default day value. Defaults to <code>undefined</code>.</li>
   * <li><code>h</code>: →<code>Number</code> - The default hour value. Defaults to <code>undefined</code>.</li>
   * <li><code>i</code>: →<code>Number</code> - The default minute value. Defaults to <code>undefined</code>.</li>
   * <li><code>s</code>: →<code>Number</code> - The default second value. Defaults to <code>undefined</code>.</li>
   * <li><code>ms</code>: →<code>Number</code> - The default millisecond value. Defaults to <code>undefined</code>.</li>
   * </ul>
   * <p>Override these properties to customize the default date values used by the →<code>parse()</code>
   * method.</p>
   * <p><b>Note:</b> In countries which experience Daylight Saving Time (i.e. DST), the <code>h</code>, <code>i</code>, <code>s</code>
   * and <code>ms</code> properties may coincide with the exact time in which DST takes effect.
   * It is the responsibility of the developer to account for this.</p>
   * <p>Example Usage:</p>
   * <pre>
   * // set default day value to the first day of the month
   * Ext.Date.defaults.d = 1;
   *
   * // parse a February date string containing only year and month values.
   * // setting the default day value to 1 prevents weird date rollover issues
   * // when attempting to parse the following date string on, for example, March 31st 2009
   * Ext.Date.parse('2009-02', 'Y-m'); // returns a Date object representing February 1st 2009
   * </pre>
   * @default {}
   * @see Number
   * @see SDateUtil#parse()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-defaults Original Ext JS documentation of 'defaults'
   */
  public native function get defaults():Object;

  /**
   * @private
   */
  public native function set defaults(value:Object):void;

  /**
   * The day on which the week starts. <code>0</code> being Sunday, through <code>6</code> being Saturday.
   * <p>This may be overridden in a locale file.</p>
   * @default 0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-firstDayOfWeek Original Ext JS documentation of 'firstDayOfWeek'
   */
  public native function get firstDayOfWeek():Number;

  /**
   * @private
   */
  public native function set firstDayOfWeek(value:Number):void;

  /**
   * The base format-code to formatting-function hashmap used by the →<code>format()</code> method.
   * Formatting functions are strings (or functions which return strings) which
   * will return the appropriate value when evaluated in the context of the Date object
   * from which the →<code>format()</code> method is called.
   * Add to / override these mappings for custom date formatting.
   * <p><b>Note:</b> <code>Ext.Date.format()</code> treats characters as literals if an appropriate mapping
   * cannot be found.</p>
   * <p>Example:</p>
   * <pre>
   * Ext.Date.formatCodes.x = "Ext.util.Format.leftPad(this.getDate(), 2, '0')";
   * console.log(Ext.Date.format(new Date(), 'X'); // returns the current day of the month
   * </pre>
   * @default {
   *     /&#42; eslint-disable max-len &#42;/
   *     d: "Ext.String.leftPad(m.getDate(), 2, '0')",
   *     D: "Ext.Date.getShortDayName(m.getDay())",
   *     // get localized short day name
   *     j: "m.getDate()",
   *     l: "Ext.Date.dayNames[m.getDay()]",
   *     N: "(m.getDay() ? m.getDay() : 7)",
   *     S: "Ext.Date.getSuffix(m)",
   *     w: "m.getDay()",
   *     z: "Ext.Date.getDayOfYear(m)",
   *     W: "Ext.String.leftPad(Ext.Date.getWeekOfYear(m), 2, '0')",
   *     F: "Ext.Date.monthNames[m.getMonth()]",
   *     m: "Ext.String.leftPad(m.getMonth() + 1, 2, '0')",
   *     M: "Ext.Date.getShortMonthName(m.getMonth())",
   *     // get localized short month name
   *     n: "(m.getMonth() + 1)",
   *     t: "Ext.Date.getDaysInMonth(m)",
   *     L: "(Ext.Date.isLeapYear(m) ? 1 : 0)",
   *     o: "(m.getFullYear() + (Ext.Date.getWeekOfYear(m) == 1 && m.getMonth() > 0 ? +1 : (Ext.Date.getWeekOfYear(m) >= 52 && m.getMonth() < 11 ? -1 : 0)))",
   *     Y: "Ext.String.leftPad(m.getFullYear(), 4, '0')",
   *     y: "('' + m.getFullYear()).substring(2, 4)",
   *     a: "(m.getHours() < 12 ? 'am' : 'pm')",
   *     A: "(m.getHours() < 12 ? 'AM' : 'PM')",
   *     g: "((m.getHours() % 12) ? m.getHours() % 12 : 12)",
   *     G: "m.getHours()",
   *     h: "Ext.String.leftPad((m.getHours() % 12) ? m.getHours() % 12 : 12, 2, '0')",
   *     H: "Ext.String.leftPad(m.getHours(), 2, '0')",
   *     i: "Ext.String.leftPad(m.getMinutes(), 2, '0')",
   *     s: "Ext.String.leftPad(m.getSeconds(), 2, '0')",
   *     u: "Ext.String.leftPad(m.getMilliseconds(), 3, '0')",
   *     O: "Ext.Date.getGMTOffset(m)",
   *     P: "Ext.Date.getGMTOffset(m, true)",
   *     T: "Ext.Date.getTimezone(m)",
   *     Z: "(m.getTimezoneOffset() &#42; -60)",
   *     /&#42; eslint-enable max-len &#42;/
   *     c: function() {
   *         // ISO-8601 -- GMT format
   *         var c = "Y-m-dTH:i:sP",
   *             code = [],
   *             l = c.length,
   *             i, e;
   *         for (i = 0; i < l; ++i) {
   *             e = c.charAt(i);
   *             // treat T as a character literal
   *             code.push(e === "T" ? "'T'" : utilDate.getFormatCode(e));
   *         }
   *         return code.join(" + ");
   *     },
   *     C: function() {
   *         // ISO-1601 -- browser format. UTC numerics with the 'Z' TZ id.
   *         return 'm.toISOString()';
   *     },
   *     U: "Math.round(m.getTime() / 1000)"
   * }
   * @see SDateUtil#format()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-formatCodes Original Ext JS documentation of 'formatCodes'
   */
  public native function get formatCodes():Object;

  /**
   * @private
   */
  public native function set formatCodes(value:Object):void;

  /**
   * An object hash in which each property is a date formatting function. The property name is the
   * format string which corresponds to the produced formatted date string.
   * <p>This object is automatically populated with date formatting functions as
   * date formats are requested for Ext standard formatting strings.</p>
   * <p>Custom formatting functions may be inserted into this object, keyed by a name which
   * from then on may be used as a format string to →<code>format()</code>.</p>
   * <p>Example:</p>
   * <pre>
   * Ext.Date.formatFunctions['x-date-format'] = myDateFormatter;
   * </pre>
   * <p>A formatting function should return a string representation of the Date object which
   * is the scope (this) of the function.</p>
   * <p>To enable date strings to also be <i>parsed</i> according to that format, a corresponding
   * parsing function must be placed into the →<code>parseFunctions</code> property.</p>
   * @default {
   *     "MS": function() {
   *         // UTC milliseconds since Unix epoch (MS-AJAX serialized date format (MRSF))
   *         return '\\/Date(' + this.getTime() + ')\\/';
   *     },
   *     "time": function() {
   *         return this.getTime().toString();
   *     },
   *     "timestamp": function() {
   *         return utilDate.format(this, 'U');
   *     }
   * }
   * @see SDateUtil#format()
   * @see SDateUtil#parseFunctions
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-formatFunctions Original Ext JS documentation of 'formatFunctions'
   */
  public native function get formatFunctions():Object;

  /**
   * @private
   */
  public native function set formatFunctions(value:Object):void;

  [ArrayElementType("String")]
  /**
   * An array of textual month names.
   * Override these values for international dates.
   * <p>Example:</p>
   * <pre>
   * Ext.Date.monthNames = [
   *     'JanInYourLang',
   *     'FebInYourLang'
   *     // ...
   * ];
   * </pre>
   * @default [
   *     "January",
   *     "February",
   *     "March",
   *     "April",
   *     "May",
   *     "June",
   *     "July",
   *     "August",
   *     "September",
   *     "October",
   *     "November",
   *     "December"
   * ]
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-monthNames Original Ext JS documentation of 'monthNames'
   */
  public native function get monthNames():Array;

  /**
   * @private
   */
  public native function set monthNames(value:Array):void;

  /**
   * An object hash of zero-based JavaScript month numbers (with short month names as keys).
   * <p><b>Note:</b> keys are case-sensitive.</p>
   * <p>Override these values for international dates.</p>
   * <p>Example:</p>
   * <pre>
   * Ext.Date.monthNumbers = {
   *     'LongJanNameInYourLang': 0,
   *     'ShortJanNameInYourLang':0,
   *     'LongFebNameInYourLang':1,
   *     'ShortFebNameInYourLang':1
   *     // ...
   * };
   * </pre>
   * @default {
   *     January: 0,
   *     Jan: 0,
   *     February: 1,
   *     Feb: 1,
   *     March: 2,
   *     Mar: 2,
   *     April: 3,
   *     Apr: 3,
   *     May: 4,
   *     June: 5,
   *     Jun: 5,
   *     July: 6,
   *     Jul: 6,
   *     August: 7,
   *     Aug: 7,
   *     September: 8,
   *     Sep: 8,
   *     October: 9,
   *     Oct: 9,
   *     November: 10,
   *     Nov: 10,
   *     December: 11,
   *     Dec: 11
   * }
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-monthNumbers Original Ext JS documentation of 'monthNumbers'
   */
  public native function get monthNumbers():Object;

  /**
   * @private
   */
  public native function set monthNumbers(value:Object):void;

  /**
   * @default nativeDate.now
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-now Original Ext JS documentation of 'now'
   */
  public native function get now():*;

  /**
   * @private
   */
  public native function set now(value:*):void;

  /**
   * An object hash in which each property is a date parsing function. The property name is the
   * format string which that function parses.
   * <p>This object is automatically populated with date parsing functions as
   * date formats are requested for Ext standard formatting strings.</p>
   * <p>Custom parsing functions may be inserted into this object, keyed by a name which from then on
   * may be used as a format string to →<code>parse()</code>.</p>
   * <p>Example:</p>
   * <pre>
   * Ext.Date.parseFunctions['x-date-format'] = myDateParser;
   * </pre>
   * <p>A parsing function should return a Date object, and is passed the following parameters:</p>
   * <ul>
   * <li><code>date</code>: →<code>String</code> - The date string to parse.</li>
   * <li><code>strict</code>: →<code>Boolean</code> - <code>true</code> to validate date strings while parsing
   * (i.e. prevent JavaScript Date "rollover"). <b>The default must be <code>false</code>.</b>
   * Invalid date strings should return <code>null</code> when parsed.</li>
   * </ul>
   * <p>To enable Dates to also be <i>formatted</i> according to that format, a corresponding
   * formatting function must be placed into the →<code>formatFunctions</code> property.</p>
   * @default {
   *     "MS": function(input, strict) {
   *         // note: the timezone offset is ignored since the MS Ajax server sends
   *         // a UTC milliseconds-since-Unix-epoch value (negative values are allowed)
   *         var r = (input || '').match(MSFormatRe);
   *         return r ? new nativeDate(((r[1] || '') + r[2]) &#42; 1) : null;
   *     },
   *     "time": function(input, strict) {
   *         var num = parseInt(input, 10);
   *         if (num || num === 0) {
   *             return new nativeDate(num);
   *         }
   *         return null;
   *     },
   *     "timestamp": function(input, strict) {
   *         var num = parseInt(input, 10);
   *         if (num || num === 0) {
   *             return new nativeDate(num &#42; 1000);
   *         }
   *         return null;
   *     }
   * }
   * @see SDateUtil#parse()
   * @see String
   * @see Boolean
   * @see SDateUtil#formatFunctions
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-parseFunctions Original Ext JS documentation of 'parseFunctions'
   */
  public native function get parseFunctions():Object;

  /**
   * @private
   */
  public native function set parseFunctions(value:Object):void;

  /**
   * Global flag which determines if strict date parsing should be used.
   * Strict date parsing will not roll-over invalid dates, which is the
   * default behavior of JavaScript Date objects.
   * (see →<code>parse()</code> for more information)
   * @default false
   * @see SDateUtil#parse()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-useStrict Original Ext JS documentation of 'useStrict'
   */
  public native function get useStrict():Boolean;

  /**
   * @private
   */
  public native function set useStrict(value:Boolean):void;

  [ArrayElementType("Number")]
  /**
   * The days on which weekend falls. <code>0</code> being Sunday, through <code>6</code> being Saturday.
   * <p>This may be overridden in a locale file.</p>
   * @default [
   *     0,
   *     6
   * ]
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#property-weekendDays Original Ext JS documentation of 'weekendDays'
   */
  public native function get weekendDays():Array;

  /**
   * @private
   */
  public native function set weekendDays(value:Array):void;

  /**
   * Provides a convenient method for performing basic date arithmetic. This method
   * does not modify the Date instance being called - it creates and returns
   * a new Date instance containing the resulting date value.
   * <p>Examples:</p>
   * <pre>
   * // Basic usage:
   * var dt = Ext.Date.add(new Date('10/29/2006'), Ext.Date.DAY, 5);
   * console.log(dt); // returns 'Fri Nov 03 2006 00:00:00'
   *
   * // Negative values will be subtracted:
   * var dt2 = Ext.Date.add(new Date('10/1/2006'), Ext.Date.DAY, -5);
   * console.log(dt2); // returns 'Tue Sep 26 2006 00:00:00'
   *
   *  // Decimal values can be used:
   * var dt3 = Ext.Date.add(new Date('10/1/2006'), Ext.Date.DAY, 1.25);
   * console.log(dt3); // returns 'Mon Oct 02 2006 06:00:00'
   * </pre>
   * @param date The date to modify
   * @param interval A valid date interval enum value.
   * @param value The amount to add to the current date.
   * @param preventDstAdjust <code>true</code> to prevent adjustments when crossing
   * daylight savings boundaries.
   * @default false
   * @return The new Date instance.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-add Original Ext JS documentation of 'add'
   */
  public native function add(date:Date, interval:String, value:Number, preventDstAdjust:Boolean = false):Date;

  /**
   * Align the date to <code>unit</code>.
   * @param date The date to be aligned.
   * @param unit The unit. This unit is compatible with the date interval constants.
   * @return The aligned date.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-align Original Ext JS documentation of 'align'
   */
  public native function align(date:Date, unit:String, step:Number):Date;

  /**
   * Checks if a date falls on or between the given start and end dates.
   * @param date The date to check
   * @param start Start date
   * @param end End date
   * @return <code>true</code> if this date falls on or between the given start and end dates.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-between Original Ext JS documentation of 'between'
   */
  public native function between(date:Date, start:Date, end:Date):Boolean;

  /**
   * Attempts to clear all time information from this Date by setting the time to midnight
   * of the same day, automatically adjusting for Daylight Saving Time (DST) where applicable.
   * <p><b>Note:</b> DST timezone information for the browser's host operating system is assumed to be
   * up-to-date.</p>
   * @param date The date
   * @param clone <code>true</code> to create a clone of this date, clear the time and
   * return it.
   * @default false
   * @return this or the clone.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-clearTime Original Ext JS documentation of 'clearTime'
   */
  public native function clearTime(date:Date, clone:Boolean = false):Date;

  /**
   * Creates and returns a new Date instance with the exact same date value as the called
   * instance. Dates are copied and passed by reference, so if a copied date variable is modified
   * later, the original variable will also be changed. When the intention is to create a new
   * variable that will not modify the original instance, you should create a clone.
   * <p>Example of correctly cloning a date:</p>
   * <pre>
   * //wrong way:
   * var orig = new Date('10/1/2006');
   * var copy = orig;
   * copy.setDate(5);
   * console.log(orig);  // returns 'Thu Oct 05 2006'!
   *
   * //correct way:
   * var orig = new Date('10/1/2006'),
   *     copy = Ext.Date.clone(orig);
   * copy.setDate(5);
   * console.log(orig);  // returns 'Thu Oct 01 2006'
   * </pre>
   * @param date The date.
   * @return The new Date instance.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-clone Original Ext JS documentation of 'clone'
   */
  public native function clone(date:Date):Date;

  /**
   * Calculate how many units are there between two time.
   * @param min The first time.
   * @param max The second time.
   * @param unit The unit. This unit is compatible with the date interval constants.
   * @return The maximum number n of units that min + n &#42; unit &lt;= max.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-diff Original Ext JS documentation of 'diff'
   */
  public native function diff(min:Date, max:Date, unit:String):Number;

  /**
   * Formats a date given the supplied format string.
   * @param date The date to format
   * @param format The format string
   * @return The formatted date or an empty string if date parameter is not
   * a JavaScript Date object
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-format Original Ext JS documentation of 'format'
   */
  public native function format(date:Date, format:String):String;

  /**
   * Checks if the specified format contains information about
   * anything other than the time.
   * @param format The format to check
   * @return True if the format contains information about
   * date/day information.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-formatContainsDateInfo Original Ext JS documentation of 'formatContainsDateInfo'
   */
  public native function formatContainsDateInfo(format:String):Boolean;

  /**
   * Checks if the specified format contains hour information
   * @param format The format to check
   * @return True if the format contains hour information
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-formatContainsHourInfo Original Ext JS documentation of 'formatContainsHourInfo'
   */
  public native function formatContainsHourInfo(format:String):Boolean;

  /**
   * Get the numeric day number of the year, adjusted for leap year.
   * <pre>
   * var dt = new Date('9/17/2011');
   * console.log(Ext.Date.getDayOfYear(dt)); // 259
   * </pre>
   * @param date The date
   * @return 0 to 364 (365 in leap years).
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getDayOfYear Original Ext JS documentation of 'getDayOfYear'
   */
  public native function getDayOfYear(date:Date):Number;

  /**
   * Get the number of days in the current month, adjusted for leap year.
   * @param date The date
   * @return The number of days in the month.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getDaysInMonth Original Ext JS documentation of 'getDaysInMonth'
   */
  public native function getDaysInMonth(date:Date):Number;

  /**
   * Returns the number of milliseconds between two dates.
   * @param dateA The first date.
   * @param dateB (Default new Date()) The second date.
   * @default new Date()
   * @return The difference in milliseconds
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getElapsed Original Ext JS documentation of 'getElapsed'
   */
  public native function getElapsed(dateA:Date, dateB:Date = null):Number;

  /**
   * Get the date of the first day of the month in which this date resides.
   * @param date The date
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getFirstDateOfMonth Original Ext JS documentation of 'getFirstDateOfMonth'
   */
  public native function getFirstDateOfMonth(date:Date):Date;

  /**
   * Get the first day of the current month, adjusted for leap year. The returned value
   * is the numeric day index within the week (0-6) which can be used in conjunction with
   * the →<code>monthNames</code> array to retrieve the textual day name.
   * <pre>
   * var dt = new Date('1/10/2007'),
   *     firstDay = Ext.Date.getFirstDayOfMonth(dt);
   *
   * console.log(Ext.Date.dayNames[firstDay]); // output: 'Monday'
   * </pre>
   * @param date The date
   * @return The day number (0-6).
   * @see SDateUtil#monthNames
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getFirstDayOfMonth Original Ext JS documentation of 'getFirstDayOfMonth'
   */
  public native function getFirstDayOfMonth(date:Date):Number;

  /**
   * Get the offset from GMT of the current date (equivalent to the format specifier 'O').
   * <pre>
   * var dt = new Date('9/17/2011');
   * console.log(Ext.Date.getGMTOffset(dt));
   * </pre>
   * @param date The date
   * @param colon <code>true</code> to separate the hours and minutes with a colon.
   * @default false
   * @return The 4-character offset string prefixed with + or - (e.g. '-0600').
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getGMTOffset Original Ext JS documentation of 'getGMTOffset'
   */
  public native function getGMTOffset(date:Date, colon:Boolean = false):String;

  /**
   * Get the date of the last day of the month in which this date resides.
   * @param date The date
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getLastDateOfMonth Original Ext JS documentation of 'getLastDateOfMonth'
   */
  public native function getLastDateOfMonth(date:Date):Date;

  /**
   * Get the last day of the current month, adjusted for leap year. The returned value
   * is the numeric day index within the week (0-6) which can be used in conjunction with
   * the →<code>monthNames</code> array to retrieve the textual day name.
   * <pre>
   * var dt = new Date('1/10/2007'),
   *     lastDay = Ext.Date.getLastDayOfMonth(dt);
   *
   * console.log(Ext.Date.dayNames[lastDay]); // output: 'Wednesday'
   * </pre>
   * @param date The date
   * @return The day number (0-6).
   * @see SDateUtil#monthNames
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getLastDayOfMonth Original Ext JS documentation of 'getLastDayOfMonth'
   */
  public native function getLastDayOfMonth(date:Date):Number;

  /**
   * Get the zero-based JavaScript month number for the given short/full month name.
   * Override this function for international dates.
   * @param name The short/full month name.
   * @return The zero-based JavaScript month number.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getMonthNumber Original Ext JS documentation of 'getMonthNumber'
   */
  public native function getMonthNumber(name:String):Number;

  /**
   * Get the short day name for the given day number.
   * Override this function for international dates.
   * @param day A zero-based JavaScript day number.
   * @return The short day name.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getShortDayName Original Ext JS documentation of 'getShortDayName'
   */
  public native function getShortDayName(day:Number):String;

  /**
   * Get the short month name for the given month number.
   * Override this function for international dates.
   * @param month A zero-based JavaScript month number.
   * @return The short month name.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getShortMonthName Original Ext JS documentation of 'getShortMonthName'
   */
  public native function getShortMonthName(month:Number):String;

  /**
   * Get the English ordinal suffix of the current day (equivalent to the format specifier 'S').
   * @param date The date
   * @return 'st, 'nd', 'rd' or 'th'.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getSuffix Original Ext JS documentation of 'getSuffix'
   */
  public native function getSuffix(date:Date):String;

  /**
   * Get the timezone abbreviation of the current date (equivalent to the format specifier 'T').
   * <p><b>Note:</b> The date string returned by the JavaScript Date object's <code>toString()</code> method varies
   * between browsers (e.g. FF vs IE) and system region settings (e.g. IE in Asia vs IE in
   * America). For a given date string e.g. "Thu Oct 25 2007 22:55:35 GMT+0800 (Malay Peninsula
   * Standard Time)", <code>getTimezone()</code> first tries to get the timezone abbreviation from between
   * a pair of parentheses (which may or may not be present), failing which it proceeds to get
   * the timezone abbreviation from the GMT offset portion of the date string.</p>
   * <pre>
   * var dt = new Date('9/17/2011');
   * console.log(Ext.Date.getTimezone(dt));
   * </pre>
   * @param date The date
   * @return The abbreviated timezone name (e.g. 'CST', 'PDT', 'EDT', 'MPST' ...).
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getTimezone Original Ext JS documentation of 'getTimezone'
   */
  public native function getTimezone(date:Date):String;

  /**
   * Get the numeric ISO-8601 week number of the year.
   * (equivalent to the format specifier 'W', but without a leading zero).
   * <pre>
   * var dt = new Date('9/17/2011');
   * console.log(Ext.Date.getWeekOfYear(dt)); // 37
   * </pre>
   * @param date The date.
   * @return 1 to 53.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-getWeekOfYear Original Ext JS documentation of 'getWeekOfYear'
   */
  public native function getWeekOfYear(date:Date):Number;

  /**
   * Checks if the current date is affected by Daylight Saving Time (DST).
   * @param date The date
   * @return <code>true</code> if the current date is affected by DST.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-isDST Original Ext JS documentation of 'isDST'
   */
  public native function isDST(date:Date):Boolean;

  /**
   * Compares if two dates are equal by comparing their values.
   * @return <code>true</code> if the date values are equal
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-isEqual Original Ext JS documentation of 'isEqual'
   */
  public native function isEqual(date1:Date, date2:Date):Boolean;

  /**
   * Checks if the current date falls within a leap year.
   * <pre>
   * var dt = new Date('1/10/2011');
   * console.log(Ext.Date.isLeapYear(dt)); // false
   * </pre>
   * @param date The date
   * @return <code>true</code> if the current date falls within a leap year, <code>false</code> otherwise.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-isLeapYear Original Ext JS documentation of 'isLeapYear'
   */
  public native function isLeapYear(date:Date):Boolean;

  /**
   * Checks if the passed Date parameters will cause a JavaScript Date "rollover".
   * @param year 4-digit year.
   * @param month 1-based month-of-year.
   * @param day Day of month.
   * @param hour Hour.
   * @param minute Minute.
   * @param second Second.
   * @param millisecond Millisecond.
   * @return <code>true</code> if the passed parameters do not cause a Date "rollover",
   * <code>false</code> otherwise.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-isValid Original Ext JS documentation of 'isValid'
   */
  public native function isValid(year:Number, month:Number, day:Number, hour:Number = NaN, minute:Number = NaN, second:Number = NaN, millisecond:Number = NaN):Boolean;

  /**
   * Checks if the date is a weekend day. Uses →<code>weekendDays</code>.
   * @since 6.2.0
   * @param date The date.
   * @return <code>true</code> if the day falls on a weekend.
   * @see SDateUtil#weekendDays
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-isWeekend Original Ext JS documentation of 'isWeekend'
   */
  public native function isWeekend(date:Date):Boolean;

  /**
   * Converts the passed local date into a UTC date.
   * For example, if the passed date is:
   * <code>Wed Jun 01 2016 00:00:00 GMT+1000 (AUS Eastern Standard Time)</code>, then
   * the returned date will be <code>Wed Jun 01 2016 10:00:00 GMT+1000 (AUS Eastern Standard Time)</code>.
   * @since 6.2.0
   * @param d The date to convert.
   * @return The date as UTC. Does not modify the passed date.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-localToUtc Original Ext JS documentation of 'localToUtc'
   */
  public native function localToUtc(d:Date):Date;

  /**
   * Parses the passed string using the specified date format.
   * Note that this function expects normal calendar dates, meaning that months are 1-based
   * (i.e. 1 = January). The →<code>defaults</code> hash will be used for any date value (i.e. year,
   * month, day, hour, minute, second or millisecond) which cannot be found in the passed string.
   * If a corresponding default date value has not been specified in the →<code>defaults</code> hash,
   * the current date's year, month, day or DST-adjusted zero-hour time value will be used
   * instead. Keep in mind that the input date string must precisely match the specified format
   * string in order for the parse operation to be successful (failed parse operations return a
   * <code>null</code> value).
   * <p>Example:</p>
   * <pre>
   * //dt = Fri May 25 2007 (current date)
   * var dt = new Date();
   *
   * //dt = Thu May 25 2006 (today&amp;#39;s month/day in 2006)
   * dt = Ext.Date.parse("2006", "Y");
   *
   * //dt = Sun Jan 15 2006 (all date parts specified)
   * dt = Ext.Date.parse("2006-01-15", "Y-m-d");
   *
   * //dt = Sun Jan 15 2006 15:20:01
   * dt = Ext.Date.parse("2006-01-15 3:20:01 PM", "Y-m-d g:i:s A");
   *
   * // attempt to parse Sun Feb 29 2006 03:20:01 in strict mode
   * dt = Ext.Date.parse("2006-02-29 03:20:01", "Y-m-d H:i:s", true); // returns null
   * </pre>
   * <p><b><i>Heuristic Parsing</i></b></p>
   * <p>When no <code>→format()</code> is specified, this method parses the date in a flexible way allowing
   * for different delimiters and textual month names to infer the position of the other
   * parts.</p>
   * <p>Supported inferred date orders when alphabetic month names are used are:</p>
   * <pre>
   * - `D,M,Y`
   * - `M,D,Y`
   * - `Y,M,D`
   * </pre>
   * <p>If the passed in date consists of all numeric tokens then the relative magnitude of
   * the first two tokens is used to make an inference about the user's intention.
   * If one token is less than 13 and the other is greater than 12, then the user's
   * intention is known.</p>
   * <p>Failing this, the →<code>defaultFormat</code> is used to determine the input order for
   * the current locale.</p>
   * <p>Part delimiters may be any of these:</p>
   * <pre>
   * - `'/'`
   * - `'-'`
   * - `'.'`
   * - `'\'`
   * - `' '` (space)
   * </pre>
   * <p>For example, the inputs <code>"Jun 1 62"</code> and <code>"1 Jun 62"</code> would be understood as the
   * first of June, 1962 in all English locales regardless of the locale's default date
   * ordering.</p>
   * <p>If <code>"25/1/62"</code> was passed in, it's obvious that the user means the twenty fifth
   * of January.</p>
   * <p>If, however, <code>"1/6/62"</code> was passed in, the →<code>defaultFormat</code> would be consulted
   * to disambiguate the meaning of those first two tokens.</p>
   * @param input The date string to parse.
   * @param format The expected date string format. If not passed, the date
   * string will be parsed heuristically as described above.
   * @param strict Pass <code>true</code> to validate date strings while parsing
   * (i.e. prevents JavaScript Date "rollover"). Invalid date strings will return <code>null</code>
   * when parsed.
   * @default false
   * @return The parsed Date, or <code>null</code> if an invalid date string.
   * @see SDateUtil#defaults
   * @see SDateUtil#format()
   * @see SDateUtil#defaultFormat
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-parse Original Ext JS documentation of 'parse'
   */
  public native function parse(input:String, format:String = null, strict:Boolean = false):Date;

  /**
   * Provides a convenient method for performing basic date arithmetic. This method
   * does not modify the Date instance being called - it creates and returns
   * a new Date instance containing the resulting date value.
   * <p>Examples:</p>
   * <pre>
   * // Basic usage:
   * var dt = Ext.Date.subtract(new Date('10/29/2006'), Ext.Date.DAY, 5);
   * console.log(dt); // returns 'Tue Oct 24 2006 00:00:00'
   *
   * // Negative values will be added:
   * var dt2 = Ext.Date.subtract(new Date('10/1/2006'), Ext.Date.DAY, -5);
   * console.log(dt2); // returns 'Fri Oct 6 2006 00:00:00'
   *
   *  // Decimal values can be used:
   * var dt3 = Ext.Date.subtract(new Date('10/1/2006'), Ext.Date.DAY, 1.25);
   * console.log(dt3); // returns 'Fri Sep 29 2006 06:00:00'
   * </pre>
   * @param date The date to modify
   * @param interval A valid date interval enum value.
   * @param value The amount to subtract from the current date.
   * @param preventDstAdjust <code>true</code> to prevent adjustments when crossing
   * daylight savings boundaries.
   * @default false
   * @return The new Date instance.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-subtract Original Ext JS documentation of 'subtract'
   */
  public native function subtract(date:Date, interval:String, value:Number, preventDstAdjust:Boolean = false):Date;

  /**
   * Removes all escaping for a date format string. In date formats,
   * using a '' can be used to escape special characters.
   * @param format The format to unescape
   * @return The unescaped format
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-unescapeFormat Original Ext JS documentation of 'unescapeFormat'
   */
  public native function unescapeFormat(format:String):String;

  /**
   * Create a UTC date.
   * @since 6.2.0
   * @param year The year.
   * @param month The month.
   * @param day The day.
   * @param hour The hour.
   * @default 0
   * @param min The minutes.
   * @default 0
   * @param s The seconds.
   * @default 0
   * @param ms The milliseconds.
   * @default 0
   * @return The UTC date.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-utc Original Ext JS documentation of 'utc'
   */
  public native function utc(year:Number, month:Number, day:Number, hour:Number = 0, min:Number = 0, s:Number = 0, ms:Number = 0):Date;

  /**
   * Converts the passed UTC date into a local date.
   * For example, if the passed date is:
   * <code>Wed Jun 01 2016 00:10:00 GMT+1000 (AUS Eastern Standard Time)</code>, then
   * the returned date will be <code>Wed Jun 01 2016 00:00:00 GMT+1000 (AUS Eastern Standard Time)</code>.
   * @since 6.2.0
   * @param d The date to convert.
   * @return The date as a local. Does not modify the passed date.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Date.html#method-utcToLocal Original Ext JS documentation of 'utcToLocal'
   */
  public native function utcToLocal(d:Date):Date;
}
}