package ext {

[Native]
/**
 * A collection of useful static methods to deal with function callbacks.
 * <p>Type of singleton FunctionUtil.</p>
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html Original Ext JS documentation of 'Ext.Function'
 * @see ext.#FunctionUtil ext.FunctionUtil
 */
public class SFunctionUtil extends Object {
  /**
   * Create an alias to the provided method property with name <code>methodName</code> of <code>object</code>.
   * Note that the execution scope will still be bound to the provided <code>object</code> itself.
   * @return aliasFn
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-alias Original Ext JS documentation of 'alias'
   */
  public native function alias(object:*, methodName:String):Function;

  /**
   * Create a new function from the provided <code>fn</code>, change <code>this</code> to the provided scope,
   * optionally overrides arguments for the call. Defaults to the arguments passed by
   * the caller.
   * <p>→<code>ext.Ext.bind()</code> is alias for →<code>ext.FunctionUtil.bind()</code></p>
   * <p><b>NOTE:</b> This method is similar to the native <code>bind()</code> method. The major difference
   * is in the way the parameters are passed. This method expects an array of parameters,
   * and if supplied, it does not automatically pass forward parameters from the bound
   * function:</p>
   * <pre>
   *  function foo (a, b, c) {
   *      console.log(a, b, c);
   *  }
   *
   *  var nativeFn = foo.bind(this, 1, 2);
   *  var extFn = Ext.Function.bind(foo, this, [1, 2]);
   *
   *  nativeFn(3); // 1, 2, 3
   *  extFn(3); // 1, 2, undefined
   * </pre>
   * <p>This method is unavailable natively on IE8 and IE/Quirks but Ext JS provides a
   * "polyfill" to emulate the important features of the standard <code>→bind()</code> method. In
   * particular, the polyfill only provides binding of "this" and optional arguments.</p>
   * @param fn The function to delegate.
   * @param scope The scope (<code>this</code> reference) in which the function
   * is executed.
   * <b>If omitted, defaults to the global environment object (usually the browser <code>window</code>).</b>
   * @param args Overrides arguments for the call. (Defaults to
   * the arguments passed by the caller).
   * @param appendArgs if <code>true</code> the <code>args</code> are appended to the
   * arguments passed to the returned wrapper (by default these arguments are ignored).
   * If a number then the <code>args</code> are inserted at the specified position.
   * @return The bound wrapper function.
   * @see ext.SExt#bind() ext.Ext.bind()
   * @see ext.SFunctionUtil#bind() ext.FunctionUtil.bind()
   * @see SFunctionUtil#bind()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-bind Original Ext JS documentation of 'bind'
   */
  public native function bind(fn:Function, scope:Object = null, args:Array = null, appendArgs:* = undefined):Function;

  /**
   * Captures the given parameters for a later call to <code>Ext.callback</code>. This binding is
   * most useful for resolving scopes for example to an <code>→ext.app.ViewController</code>.
   * <p>The arguments match that of <code>Ext.callback</code> except for the <code>args</code> which, if provided
   * to this method, are prepended to any arguments supplied by the eventual caller of
   * the returned function.</p>
   * @since 5.0.0
   * @return A function that, when called, uses <code>Ext.callback</code> to call the
   * captured <code>callback</code>.
   * @see ext.app.ViewController
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-bindCallback Original Ext JS documentation of 'bindCallback'
   */
  public native function bindCallback(callback:*, scope:*, args:*, delay:*, caller:*):Function;

  /**
   * Create a "clone" of the provided method. The returned method will call the given
   * method passing along all arguments and the "this" pointer and return its result.
   * @return cloneFn
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-clone Original Ext JS documentation of 'clone'
   */
  public native function clone(method:Function):Function;

  /**
   * Wraps the passed function in a barrier function which will call the passed function
   * after the passed number of invocations.
   * @param count The number of invocations which will result in the calling
   * of the passed function.
   * @param fn The function to call after the required number of invocations.
   * @param scope The scope (<code>this</code> reference) in which the function will be called.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-createBarrier Original Ext JS documentation of 'createBarrier'
   */
  public native function createBarrier(count:Number, fn:Function, scope:Object):void;

  /**
   * Creates a delegate function, optionally with a bound scope which, when called, buffers
   * the execution of the passed function for the configured number of milliseconds.
   * If called again within that period, the impending invocation will be canceled, and the
   * timeout period will begin again.
   * @param fn The function to invoke on a buffered timer.
   * @param buffer The number of milliseconds by which to buffer the invocation
   * of the function.
   * @param scope The scope (<code>this</code> reference) in which.
   * the passed function is executed. If omitted, defaults to the scope specified
   * by the caller.
   * @param args Override arguments for the call. Defaults to the arguments
   * passed by the caller.
   * @return A function which invokes the passed function after buffering
   * for the specified time.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-createBuffered Original Ext JS documentation of 'createBuffered'
   */
  public native function createBuffered(fn:Function, buffer:Number, scope:Object = null, args:Array = null):Function;

  /**
   * Creates a delegate (callback) which, when called, executes after a specific delay.
   * @param fn The function which will be called on a delay when the returned
   * function is called. Optionally, a replacement (or additional) argument list
   * may be specified.
   * @param delay The number of milliseconds to defer execution by whenever called.
   * @param scope The scope (<code>this</code> reference) used by the function
   * at execution time.
   * @param args Override arguments for the call.
   * (Defaults to the arguments passed by the caller)
   * @param appendArgs if True args are appended to call args
   * instead of overriding, if a number the args are inserted at the specified position.
   * @return A function which, when called, executes the original function
   * after the specified delay.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-createDelayed Original Ext JS documentation of 'createDelayed'
   */
  public native function createDelayed(fn:Function, delay:Number, scope:Object = null, args:Array = null, appendArgs:* = undefined):Function;

  /**
   * Creates an interceptor function. The passed function is called before the original one.
   * If it returns false, the original one is not called. The resulting function returns
   * the results of the original function. The passed function is called with the parameters
   * of the original function. Example usage:
   * <pre>
   * var sayHi = function(name){
   *     alert('Hi, ' + name);
   * };
   *
   * sayHi('Fred'); // alerts "Hi, Fred"
   *
   * // create a new function that validates input without
   * // directly modifying the original function:
   * var sayHiToFriend = Ext.Function.createInterceptor(sayHi, function(name){
   *     return name === 'Brian';
   * });
   *
   * sayHiToFriend('Fred');  // no alert
   * sayHiToFriend('Brian'); // alerts "Hi, Brian"
   * </pre>
   * @param origFn The original function.
   * @param newFn The function to call before the original.
   * @param scope The scope (<code>this</code> reference) in which the passed function
   * is executed. <b>If omitted, defaults to the scope in which the original function
   * is called or the browser window.</b>
   * @param returnValue The value to return if the passed function return
   * <code>false</code>.
   * @return The new function.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-createInterceptor Original Ext JS documentation of 'createInterceptor'
   */
  public native function createInterceptor(origFn:Function, newFn:Function, scope:Object = null, returnValue:Object = null):Function;

  /**
   * Create a combined function call sequence of the original function + the passed function.
   * The resulting function returns the results of the original function.
   * The passed function is called with the parameters of the original function.
   * Example usage:
   * <pre>
   * var sayHi = function(name){
   *     alert('Hi, ' + name);
   * };
   *
   * sayHi('Fred'); // alerts "Hi, Fred"
   *
   * var sayGoodbye = Ext.Function.createSequence(sayHi, function(name){
   *     alert('Bye, ' + name);
   * });
   *
   * sayGoodbye('Fred'); // both alerts show
   * </pre>
   * @param originalFn The original function.
   * @param newFn The function to sequence.
   * @param scope The scope (<code>this</code> reference) in which the passed function
   * is executed. If omitted, defaults to the scope in which the original function is called
   * or the default global environment object (usually the browser window).
   * @return The new function.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-createSequence Original Ext JS documentation of 'createSequence'
   */
  public native function createSequence(originalFn:Function, newFn:Function, scope:Object = null):Function;

  /**
   * Creates a throttled version of the passed function which, when called repeatedly and
   * rapidly, invokes the passed function only after a certain interval has elapsed since the
   * previous invocation.
   * <p>This is useful for wrapping functions which may be called repeatedly, such as
   * a handler of a mouse move event when the processing is expensive.</p>
   * @param fn The function to execute at a regular time interval.
   * @param interval The interval in milliseconds on which the passed function
   * is executed.
   * @param scope The scope (<code>this</code> reference) in which
   * the passed function is executed. If omitted, defaults to the scope specified
   * by the caller.
   * @return A function which invokes the passed function at the specified
   * interval.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-createThrottled Original Ext JS documentation of 'createThrottled'
   */
  public native function createThrottled(fn:Function, interval:Number, scope:Object = null):Function;

  /**
   * Calls function <code>fn</code> after the number of milliseconds specified, optionally with
   * a specific <code>scope</code> (<code>this</code> pointer).
   * <p>Example usage:</p>
   * <pre>
   * var sayHi = function(name) {
   *     alert('Hi, ' + name);
   * }
   *
   * // executes immediately:
   * sayHi('Fred');
   *
   * // executes after 2 seconds:
   * Ext.defer(sayHi, 2000, this, ['Fred']);
   * </pre>
   * <p>The following syntax is useful for scheduling anonymous functions:</p>
   * <pre>
   * Ext.defer(function() {
   *     alert('Anonymous');
   * }, 100);
   * </pre>
   * <p>NOTE: The <code>Ext.Function.defer()</code> method is an alias for <code>Ext.defer()</code>.</p>
   * @param fn The function to defer.
   * @param millis The number of milliseconds for the <code>setTimeout</code> call
   * (if less than or equal to 0 the function is executed immediately).
   * @param scope The scope (<code>this</code> reference) in which the function
   * is executed. <b>If omitted, defaults to the browser window.</b>
   * @param args Overrides arguments for the call. Defaults to the arguments passed
   * by the caller.
   * @param appendArgs If <code>true</code> args are appended to call args
   * instead of overriding, or, if a number, then the args are inserted at the specified
   * position.
   * @default false
   * @return The timeout id that can be used with <code>Ext.undefer</code>.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-defer Original Ext JS documentation of 'defer'
   */
  public native function defer(fn:Function, millis:Number, scope:Object = null, args:Array = null, appendArgs:* = false):Number;

  /**
   * A very commonly used method throughout the framework. It acts as a wrapper around
   * another method which originally accepts 2 arguments for <code>name</code> and <code>value</code>.
   * The wrapped function then allows "flexible" value setting of either:
   * <ul>
   * <li><code>name</code> and <code>value</code> as 2 arguments</li>
   * <li>one single object argument with multiple key - value pairs</li>
   * </ul>
   * <p>For example:</p>
   * <pre>
   * var setValue = Ext.Function.flexSetter(function(name, value) {
   *     this[name] = value;
   * });
   *
   * // Afterwards
   * // Setting a single name - value
   * setValue('name1', 'value1');
   *
   * // Settings multiple name - value pairs
   * setValue({
   *     name1: 'value1',
   *     name2: 'value2',
   *     name3: 'value3'
   * });
   * </pre>
   * @param setter The single value setter method.
   * <ul>
   * <li><code>name:String</code> (optional) —
   * The name of the value being set.
   * </li>
   * <li><code>value:Object</code> (optional) —
   * The value being set.
   * </li>
   * </ul>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-flexSetter Original Ext JS documentation of 'flexSetter'
   */
  public native function flexSetter(setter:Function):Function;

  /**
   * Adds behavior to an existing method that is executed after the
   * original behavior of the function. For example:
   * <pre>
   * var soup = {
   *     contents: [],
   *     add: function(ingredient) {
   *         this.contents.push(ingredient);
   *     }
   * };
   * Ext.Function.interceptAfter(soup, "add", function(ingredient){
   *     // Always add a bit of extra salt
   *     this.contents.push("salt");
   * });
   * soup.add("water");
   * soup.add("onions");
   * soup.contents; // will contain: water, salt, onions, salt
   * </pre>
   * @param object The target object
   * @param methodName Name of the method to override
   * @param fn Function with the new behavior. It will
   * be called with the same arguments as the original method. The
   * return value of this function will be the return value of the
   * new method.
   * @param scope The scope to execute the interceptor function.
   * Defaults to the object.
   * @return The new function just created.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-interceptAfter Original Ext JS documentation of 'interceptAfter'
   */
  public native function interceptAfter(object:Object, methodName:String, fn:Function, scope:Object = null):Function;

  /**
   * Adds behavior to an existing method that is executed before the
   * original behavior of the function. For example:
   * <pre>
   * var soup = {
   *     contents: [],
   *     add: function(ingredient) {
   *         this.contents.push(ingredient);
   *     }
   * };
   * Ext.Function.interceptBefore(soup, "add", function(ingredient){
   *     if (!this.contents.length &amp;&amp; ingredient !== "water") {
   *         // Always add water to start with
   *         this.contents.push("water");
   *     }
   * });
   * soup.add("onions");
   * soup.add("salt");
   * soup.contents; // will contain: water, onions, salt
   * </pre>
   * @param object The target object
   * @param methodName Name of the method to override
   * @param fn Function with the new behavior. It will
   * be called with the same arguments as the original method. The
   * return value of this function will be the return value of the
   * new method.
   * @param scope The scope to execute the interceptor function.
   * Defaults to the object.
   * @return The new function just created.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-interceptBefore Original Ext JS documentation of 'interceptBefore'
   */
  public native function interceptBefore(object:Object, methodName:String, fn:Function, scope:Object = null):Function;

  /**
   * Calls the function <code>fn</code> repeatedly at a given interval, optionally with a
   * specific <code>scope</code> (<code>this</code> pointer).
   * <pre>
   * var sayHi = function(name) {
   *     console.log('Hi, ' + name);
   * }
   *
   * // executes every 2 seconds:
   * var timerId = Ext.interval(sayHi, 2000, this, ['Fred']);
   * </pre>
   * <p>The timer is stopped by:</p>
   * <pre>
   * Ext.uninterval(timerId);
   * </pre>
   * <p>NOTE: The <code>Ext.Function.interval()</code> method is an alias for <code>Ext.interval()</code>.</p>
   * @param fn The function to defer.
   * @param millis The number of milliseconds for the <code>setInterval</code> call
   * @param scope The scope (<code>this</code> reference) in which the function
   * is executed. <b>If omitted, defaults to the browser window.</b>
   * @param args Overrides arguments for the call. Defaults to the arguments
   * passed by the caller.
   * @param appendArgs If <code>true</code> args are appended to call args
   * instead of overriding, or, if a number, then the args are inserted at the specified
   * position.
   * @default false
   * @return The interval id that can be used with <code>Ext.uninterval</code>.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-interval Original Ext JS documentation of 'interval'
   */
  public native function interval(fn:Function, millis:Number, scope:Object = null, args:Array = null, appendArgs:* = false):Number;

  /**
   * Returns a wrapper function that caches the return value for previously
   * processed function argument(s).
   * <p>For example:</p>
   * <pre>
   *  function factorial (value) {
   *      var ret = value;
   *
   *      while (--value &gt; 1) {
   *          ret &#42;= value;
   *      }
   *
   *      return ret;
   *  }
   * </pre>
   * <p>Each call to <code>factorial</code> will loop and multiply to produce the answer. Using
   * this function we can wrap the above and cache its answers:</p>
   * <pre>
   *  factorial = Ext.Function.memoize(factorial);
   * </pre>
   * <p>The returned function operates in the same manner as before, but results are
   * stored in a cache to avoid calling the wrapped function when given the same
   * arguments.</p>
   * <pre>
   *  var x = factorial(20);  // first time; call real factorial()
   *  var y = factorial(20);  // second time; return value from first call
   * </pre>
   * <p>To support multi-argument methods, you will need to provide a <code>hashFn</code>.</p>
   * <pre>
   *  function permutation (n, k) {
   *      return factorial(n) / factorial(n - k);
   *  }
   *
   *  permutation = Ext.Function.memoize(permutation, null, function(n, k) {
   *      n + '-' + k;
   *  });
   * </pre>
   * <p>In this case, the <code>→memoize()</code> of <code>factorial</code> is sufficient optimization, but the
   * example is simply to illustrate how to generate a unique key for an expensive,
   * multi-argument method.</p>
   * <p><b>IMPORTANT</b>: This cache is unbounded so be cautious of memory leaks if the
   * <code>→memoize()</code>d function is kept indefinitely or is given an unbounded set of
   * possible arguments.</p>
   * @since 6.0.0
   * @param fn Function to wrap.
   * @param scope Optional scope in which to execute the wrapped function.
   * @param hashFn Optional function used to compute a hash key for
   * storing the result, based on the arguments to the original function.
   * @return The caching wrapper function.
   * @see SFunctionUtil#memoize()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-memoize Original Ext JS documentation of 'memoize'
   */
  public native function memoize(fn:Function, scope:Object, hashFn:Function):Function;

  /**
   * Create a new function from the provided <code>fn</code>, the arguments of which are pre-set
   * to <code>args</code>. New arguments passed to the newly created callback when it's invoked
   * are appended after the pre-set ones.
   * This is especially useful when creating callbacks.
   * <p>For example:</p>
   * <pre>
   * var originalFunction = function(){
   *     alert(Ext.Array.from(arguments).join(' '));
   * };
   *
   * var callback = Ext.Function.pass(originalFunction, ['Hello', 'World']);
   *
   * callback(); // alerts 'Hello World'
   * callback('by Me'); // alerts 'Hello World by Me'
   * </pre>
   * <p>→<code>ext.Ext.pass()</code> is alias for →<code>ext.FunctionUtil.pass()</code></p>
   * @param fn The original function.
   * @param args The arguments to pass to new callback.
   * @param scope The scope (<code>this</code> reference) in which the function
   * is executed.
   * @return The new callback function.
   * @see ext.SExt#pass() ext.Ext.pass()
   * @see ext.SFunctionUtil#pass() ext.FunctionUtil.pass()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Function.html#method-pass Original Ext JS documentation of 'pass'
   */
  public native function pass(fn:Function, args:Array, scope:Object = null):Function;
}
}