package ext {

[Native("Ext.Version", require)]
/**
 * A utility class that wraps around a version number string and provides convenient methods
 * to perform comparisons. A version number is expressed in the following general format:
 * <pre>
 * major[.minor[.patch[.build[release]]]]
 * </pre>
 * <p>The <code>Version</code> instance holds various readonly properties that contain the digested form
 * of the version string. The numeric componnets of <code>→major</code>, <code>→minor</code>, <code>→patch</code> and <code>→build</code>
 * as well as the textual suffix called <code>→release</code>.</p>
 * <p>Not depicted in the above syntax are three possible prefixes used to control partial
 * matching. These are '^' (the default), '&gt;' and '~'. These are discussed below.</p>
 * <p>Examples:</p>
 * <pre>
 * var version = new Ext.Version('1.0.2beta'); // or maybe "1.0" or "1.2.3.4RC"
 * console.log("Version is " + version); // Version is 1.0.2beta
 *
 * console.log(version.getMajor()); // 1
 * console.log(version.getMinor()); // 0
 * console.log(version.getPatch()); // 2
 * console.log(version.getBuild()); // 0
 * console.log(version.getRelease()); // beta
 * </pre>
 * <p>The understood values of <code>→release</code> are assigned numberic equivalents for the sake of
 * comparsion. The order of these from smallest to largest is as follows:</p>
 * <ul>
 * <li><code>"dev"</code></li>
 * <li><code>"alpha"</code> or <code>"a"</code></li>
 * <li><code>"beta"</code> or <code>"b"</code></li>
 * <li><code>"RC"</code> or <code>"rc"</code></li>
 * <li><code>"#"</code></li>
 * <li><code>"pl"</code> or <code>"p"</code></li>
 * </ul>
 * <p>Any other (unrecognized) suffix is consider greater than any of these.</p>
 * <p><b><i>Comparisons</i></b></p>
 * <p>There are two forms of comparison that are commonly needed: full and partial. Full
 * comparison is simpler and is also the default.</p>
 * <p>Example:</p>
 * <pre>
 * var version = new Ext.Version('1.0.2beta');
 *
 * console.log(version.isGreaterThan('1.0.1')); // True
 * console.log(version.isGreaterThan('1.0.2alpha')); // True
 * console.log(version.isGreaterThan('1.0.2RC')); // False
 * console.log(version.isGreaterThan('1.0.2')); // False
 * console.log(version.isLessThan('1.0.2')); // True
 *
 * console.log(version.match(1.0)); // True (using a Number)
 * console.log(version.match('1.0.2')); // True (using a String)
 * </pre>
 * <p>These comparisons are ultimately implemented by →<code>ext.Version.compareTo()</code>
 * which returns -1, 0 or 1 depending on whether the `Version' instance is less than, equal
 * to, or greater than the given "other" version.</p>
 * <p>For example:</p>
 * <pre>
 *  var n = version.compareTo('1.0.1');  // == 1  (because 1.0.2beta &gt; 1.0.1)
 *
 *  n = version.compareTo('1.1');  // == -1
 *  n = version.compareTo(version); // == 0
 * </pre>
 * <p><i>Partial Comparisons</i></p>
 * <p>By default, unspecified version number fields are filled with 0. In other words, the
 * version number fields are 0-padded on the right or a "lower bound". This produces the
 * most commonly used forms of comparsion:</p>
 * <pre>
 *  var ver = new Version('4.2');
 *
 *  n = ver.compareTo('4.2.1'); // == -1  (4.2 promotes to 4.2.0 and is less than 4.2.1)
 * </pre>
 * <p>There are two other ways to interpret comparisons of versions of different length. The
 * first of these is to change the padding on the right to be a large number (scuh as
 * Infinity) instead of 0. This has the effect of making the version an upper bound. For
 * example:</p>
 * <pre>
 *  var ver = new Version('^4.2'); // NOTE: the '^' prefix used
 *
 *  n = ver.compareTo('4.3'); // == -1  (less than 4.3)
 *
 *  n = ver.compareTo('4.2'); // == 1   (greater than all 4.2's)
 *  n = ver.compareTo('4.2.1'); // == 1
 *  n = ver.compareTo('4.2.9'); // == 1
 * </pre>
 * <p>The second way to interpret this comparison is to ignore the extra digits, making the
 * match a prefix match. For example:</p>
 * <pre>
 *  var ver = new Version('~4.2'); // NOTE: the '~' prefix used
 *
 *  n = ver.compareTo('4.3'); // == -1
 *
 *  n = ver.compareTo('4.2'); // == 0
 *  n = ver.compareTo('4.2.1'); // == 0
 * </pre>
 * <p>This final form can be useful when version numbers contain more components than are
 * important for certain comparisons. For example, the full version of Ext JS 4.2.1 is
 * "4.2.1.883" where 883 is the <code>→build</code> number.</p>
 * <p>This is how to create a "partial" <code>Version</code> and compare versions to it:</p>
 * <pre>
 *  var version421ish = new Version('~4.2.1');
 *
 *  n = version421ish.compareTo('4.2.1.883'); // == 0
 *  n = version421ish.compareTo('4.2.1.2'); // == 0
 *  n = version421ish.compareTo('4.2.1'); // == 0
 *
 *  n = version421ish.compareTo('4.2'); // == 1
 * </pre>
 * <p>In the above example, '4.2.1.2' compares as equal to '4.2.1' because digits beyond the
 * given "4.2.1" are ignored. However, '4.2' is less than the '4.2.1' prefix; its missing
 * digit is filled with 0.</p>
 * @see #major
 * @see #minor
 * @see #patch
 * @see #build
 * @see #release
 * @see ext.Version#compareTo()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html Original Ext JS documentation of 'Ext.Version'
 */
public class Version extends Object {
  /**
   * Compare 2 specified versions by ensuring the first parameter is a <code>Version</code>
   * instance and then calling the <code>→compareTo()</code> method.
   * @param current The current version to compare to
   * @param target The target version to compare to
   * @return Returns -1 if the current version is smaller than the target version,
   * 1 if greater, and 0 if they're equivalent
   * @see #compareTo()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#static-method-compare Original Ext JS documentation of 'compare'
   */
  public static native function compare(current:String, target:String):Number;

  /**
   * Converts a version component to a comparable value
   * @param value The value to convert
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#static-method-getComponentValue Original Ext JS documentation of 'getComponentValue'
   */
  public static native function getComponentValue(value:Object):Object;

  /**
   * The fourth numeric part of the version number string.
   * @default parts[3] || pad
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#property-build Original Ext JS documentation of 'build'
   */
  public native function get build():Number;

  /**
   * The first numeric part of the version number string.
   * @default parts[0] || pad
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#property-major Original Ext JS documentation of 'major'
   */
  public native function get major():Number;

  /**
   * The second numeric part of the version number string.
   * @default parts[1] || pad
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#property-minor Original Ext JS documentation of 'minor'
   */
  public native function get minor():Number;

  /**
   * The third numeric part of the version number string.
   * @default parts[2] || pad
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#property-patch Original Ext JS documentation of 'patch'
   */
  public native function get patch():Number;

  /**
   * The release level. The following values are understood:
   * <ul>
   * <li><code>"dev"</code></li>
   * <li><code>"alpha"</code> or <code>"a"</code></li>
   * <li><code>"beta"</code> or <code>"b"</code></li>
   * <li><code>"RC"</code> or <code>"rc"</code></li>
   * <li><code>"#"</code></li>
   * <li><code>"pl"</code> or <code>"p"</code></li>
   * </ul>
   * @default ""
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#property-release Original Ext JS documentation of 'release'
   */
  public native function get release():String;

  /**
   * Compares this version instance to the specified <code>other</code> version.
   * @param other The other version to which to compare.
   * @return -1 if this version is less than the target version, 1 if this
   * version is greater, and 0 if they are equal.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-compareTo Original Ext JS documentation of 'compareTo'
   */
  public native function compareTo(other:*):Number;

  /**
   * Returns whether this version equals to the supplied argument
   * @param target The version to compare with
   * @return <code>true</code> if this version equals to the target, <code>false</code> otherwise
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-equals Original Ext JS documentation of 'equals'
   */
  public native function equals(target:*):Boolean;

  /**
   * Returns the build component value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-getBuild Original Ext JS documentation of 'getBuild'
   */
  public native function getBuild():Number;

  /**
   * Returns the major component value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-getMajor Original Ext JS documentation of 'getMajor'
   */
  public native function getMajor():Number;

  /**
   * Returns the minor component value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-getMinor Original Ext JS documentation of 'getMinor'
   */
  public native function getMinor():Number;

  /**
   * Returns the patch component value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-getPatch Original Ext JS documentation of 'getPatch'
   */
  public native function getPatch():Number;

  /**
   * Returns the release component text (e.g., "beta").
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-getRelease Original Ext JS documentation of 'getRelease'
   */
  public native function getRelease():String;

  /**
   * Returns the release component value for comparison purposes.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-getReleaseValue Original Ext JS documentation of 'getReleaseValue'
   */
  public native function getReleaseValue():*;

  /**
   * Returns shortVersion version without dots and release
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-getShortVersion Original Ext JS documentation of 'getShortVersion'
   */
  public native function getShortVersion():String;

  /**
   * Convenient alias to →<code>ext.Version.isGreaterThan()</code>
   * @see ext.Version#isGreaterThan()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-gt Original Ext JS documentation of 'gt'
   */
  public native function gt(target:*):Boolean;

  /**
   * Convenient alias to →<code>ext.Version.isGreaterThanOrEqual()</code>
   * @see ext.Version#isGreaterThanOrEqual()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-gtEq Original Ext JS documentation of 'gtEq'
   */
  public native function gtEq(target:*):Boolean;

  /**
   * Returns whether this version if greater than the supplied argument
   * @param target The version to compare with
   * @return <code>true</code> if this version if greater than the target, <code>false</code> otherwise
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-isGreaterThan Original Ext JS documentation of 'isGreaterThan'
   */
  public native function isGreaterThan(target:*):Boolean;

  /**
   * Returns whether this version if greater than or equal to the supplied argument
   * @param target The version to compare with
   * @return <code>true</code> if this version if greater than or equal to the target,
   * <code>false</code> otherwise
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-isGreaterThanOrEqual Original Ext JS documentation of 'isGreaterThanOrEqual'
   */
  public native function isGreaterThanOrEqual(target:*):Boolean;

  /**
   * Returns whether this version if smaller than the supplied argument
   * @param target The version to compare with
   * @return <code>true</code> if this version if smaller than the target, <code>false</code> otherwise
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-isLessThan Original Ext JS documentation of 'isLessThan'
   */
  public native function isLessThan(target:*):Boolean;

  /**
   * Returns whether this version if less than or equal to the supplied argument
   * @param target The version to compare with
   * @return <code>true</code> if this version if less than or equal to the target,
   * <code>false</code> otherwise
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-isLessThanOrEqual Original Ext JS documentation of 'isLessThanOrEqual'
   */
  public native function isLessThanOrEqual(target:*):Boolean;

  /**
   * Convenient alias to →<code>ext.Version.isLessThan()</code>
   * @see ext.Version#isLessThan()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-lt Original Ext JS documentation of 'lt'
   */
  public native function lt(target:*):Boolean;

  /**
   * Convenient alias to →<code>ext.Version.isLessThanOrEqual()</code>
   * @see ext.Version#isLessThanOrEqual()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-ltEq Original Ext JS documentation of 'ltEq'
   */
  public native function ltEq(target:*):Boolean;

  /**
   * Returns whether this version matches the supplied argument. Example:
   * <pre>
   * var version = new Ext.Version('1.0.2beta');
   * console.log(version.match(1)); // true
   * console.log(version.match(1.0)); // true
   * console.log(version.match('1.0.2')); // true
   * console.log(version.match('1.0.2RC')); // false
   * </pre>
   * @param target The version to compare with
   * @return <code>true</code> if this version matches the target, <code>false</code> otherwise
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-match Original Ext JS documentation of 'match'
   */
  public native function match(target:*):Boolean;

  [ArrayElementType("Number")]
  /**
   * Returns this format: [major, minor, patch, build, release]. Useful for comparison.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.Version.html#method-toArray Original Ext JS documentation of 'toArray'
   */
  public native function toArray():Array;
}
}