package ext {

[Native("Ext.XTemplate", require)]
/**
 * A template class that supports advanced functionality like...
 * <ul>
 * <li>Auto-filling arrays using templates and sub-templates</li>
 * <li>Conditional processing with basic comparison operators</li>
 * <li>Basic math function support</li>
 * <li>Execute arbitrary inline code with special built-in template variables</li>
 * <li>Custom member functions</li>
 * <li>Many special tags and built-in operators that aren't defined as part of the API, but are
 * supported in the templates that can be created</li>
 * </ul>
 * <p>XTemplate provides the templating mechanism built into →<code>ext.view.DataView</code>.</p>
 * <p>The →<code>ext.Template</code> describes the acceptable parameters to pass to the constructor.
 * The following examples demonstrate all of the supported features.</p>
 * <p><b>Sample Data</b></p>
 * <p>This is the data object used for reference in each code example:</p>
 * <pre>
 * var data = {
 *     name: 'Don Griffin',
 *     title: 'Senior Technomage',
 *     company: 'Sencha Inc.',
 *     drinks: ['Coffee', 'Water', 'More Coffee'],
 *     kids: [
 *         { name: 'Aubrey',  age: 17 },
 *         { name: 'Joshua',  age: 13 },
 *         { name: 'Cale',    age: 10 },
 *         { name: 'Nikol',   age: 5 },
 *         { name: 'Solomon', age: 0 }
 *     ]
 * };
 * </pre>
 * <p><b>Auto filling of arrays</b></p>
 * <p>The <b>tpl</b> tag and the <b>for</b> operator are used to process the provided data object:</p>
 * <ul>
 * <li>If the value specified in for is an array, it will auto-fill, repeating the template block
 * inside the tpl tag for each item in the array.</li>
 * <li>If for="." is specified, the data object provided is examined.</li>
 * <li>If between="..." is specified, the provided value will be inserted between the items.
 * This is also supported in the "foreach" looping template.</li>
 * <li>While processing an array, the special variable {#} will provide the current array index + 1
 * (starts at 1, not 0).</li>
 * </ul>
 * <p>Examples:</p>
 * <pre>
 * // loop through array at root node
 * &lt;tpl for="."&gt;...&lt;/tpl&gt;
 *
 * // loop through array at foo node
 * &lt;tpl for="foo"&gt;...&lt;/tpl&gt;
 *
 * // loop through array at foo.bar node
 * &lt;tpl for="foo.bar"&gt;...&lt;/tpl&gt;
 *
 * // loop through array at root node and insert ',' between each item
 * &lt;tpl for="." between=","&gt;...&lt;/tpl&gt;
 * </pre>
 * <p>Using the sample data above:</p>
 * <pre>
 * var tpl = new Ext.XTemplate(
 *     '&lt;p&gt;Kids: ',
 *     '&lt;tpl for="."&gt;',       // process the data.kids node
 *         '&lt;p&gt;{#}. {name}&lt;/p&gt;',  // use current array index to autonumber
 *     '&lt;/tpl&gt;&lt;/p&gt;'
 * );
 * tpl.overwrite(panel.body, data.kids); // pass the kids property of the data object
 * </pre>
 * <p>An example illustrating how the <b>for</b> property can be leveraged to access specified members
 * of the provided data object to populate the template:</p>
 * <pre>
 * var tpl = new Ext.XTemplate(
 *     '&lt;p&gt;Name: {name}&lt;/p&gt;',
 *     '&lt;p&gt;Title: {title}&lt;/p&gt;',
 *     '&lt;p&gt;Company: {company}&lt;/p&gt;',
 *     '&lt;p&gt;Kids: ',
 *     '&lt;tpl for="kids"&gt;',     // interrogate the kids property within the data
 *         '&lt;p&gt;{name}&lt;/p&gt;',
 *     '&lt;/tpl&gt;&lt;/p&gt;'
 * );
 * tpl.overwrite(panel.body, data);  // pass the root node of the data object
 * </pre>
 * <p>Flat arrays that contain values (and not objects) can be auto-rendered using the special
 * <b><code>{.}</code></b> variable inside a loop. This variable will represent the value of the array
 * at the current index:</p>
 * <pre>
 * var tpl = new Ext.XTemplate(
 *     '&lt;p&gt;{name}\'s favorite beverages:&lt;/p&gt;',
 *     '&lt;tpl for="drinks"&gt;',
 *         '&lt;div&gt; - {.}&lt;/div&gt;',
 *     '&lt;/tpl&gt;'
 * );
 * tpl.overwrite(panel.body, data);
 * </pre>
 * <p>When processing a sub-template, for example while looping through a child array, you can access
 * the parent object's members via the <b>parent</b> object:</p>
 * <pre>
 * var tpl = new Ext.XTemplate(
 *     '&lt;p&gt;Name: {name}&lt;/p&gt;',
 *     '&lt;p&gt;Kids: ',
 *     '&lt;tpl for="kids"&gt;',
 *         '&lt;tpl if="age &amp;gt; 1"&gt;',
 *             '&lt;p&gt;{name}&lt;/p&gt;',
 *             '&lt;p&gt;Dad: {parent.name}&lt;/p&gt;',
 *         '&lt;/tpl&gt;',
 *     '&lt;/tpl&gt;&lt;/p&gt;'
 * );
 * tpl.overwrite(panel.body, data);
 * </pre>
 * <p>The <b>foreach</b> operator is used to loop over an object's properties. The following
 * example demonstrates looping over the main data object's properties:</p>
 * <pre>
 * var tpl = new Ext.XTemplate(
 *     '&lt;dl&gt;',
 *         '&lt;tpl foreach="."&gt;',
 *             // the special &#42;&#42;`{$}`&#42;&#42; variable contains the property name
 *             '&lt;dt&gt;{$}&lt;/dt&gt;',
 *             // within the loop, the &#42;&#42;`{.}`&#42;&#42; variable is set to the property value
 *             '&lt;dd&gt;{.}&lt;/dd&gt;',
 *         '&lt;/tpl&gt;',
 *     '&lt;/dl&gt;'
 * );
 * tpl.overwrite(panel.body, data);
 * </pre>
 * <p><b>Conditional processing with basic comparison operators</b></p>
 * <p>The <b>tpl</b> tag and the <b>if</b> operator are used to provide conditional checks for deciding
 * whether or not to render specific parts of the template.</p>
 * <p>Using the sample data above:</p>
 * <pre>
 * var tpl = new Ext.XTemplate(
 *     '&lt;p&gt;Name: {name}&lt;/p&gt;',
 *     '&lt;p&gt;Kids: ',
 *     '&lt;tpl for="kids"&gt;',
 *         '&lt;tpl if="age &amp;gt; 1"&gt;',
 *             '&lt;p&gt;{name}&lt;/p&gt;',
 *         '&lt;/tpl&gt;',
 *     '&lt;/tpl&gt;&lt;/p&gt;'
 * );
 * tpl.overwrite(panel.body, data);
 * </pre>
 * <p>More advanced conditionals are also supported:</p>
 * <pre>
 * var tpl = new Ext.XTemplate(
 *     '&lt;p&gt;Name: {name}&lt;/p&gt;',
 *     '&lt;p&gt;Kids: ',
 *     '&lt;tpl for="kids"&gt;',
 *         '&lt;p&gt;{name} is a ',
 *         '&lt;tpl if="age &amp;gt;= 13"&gt;',
 *             '&lt;p&gt;teenager&lt;/p&gt;',
 *         '&lt;tpl elseif="age &amp;gt;= 2"&gt;',
 *             '&lt;p&gt;kid&lt;/p&gt;',
 *         '&lt;tpl else&gt;',
 *             '&lt;p&gt;baby&lt;/p&gt;',
 *         '&lt;/tpl&gt;',
 *     '&lt;/tpl&gt;&lt;/p&gt;'
 * );
 *
 * var tpl = new Ext.XTemplate(
 *     '&lt;p&gt;Name: {name}&lt;/p&gt;',
 *     '&lt;p&gt;Kids: ',
 *     '&lt;tpl for="kids"&gt;',
 *         '&lt;p&gt;{name} is a ',
 *         '&lt;tpl switch="name"&gt;',
 *             '&lt;tpl case="Aubrey" case="Nikol"&gt;',
 *                 '&lt;p&gt;girl&lt;/p&gt;',
 *             '&lt;tpl default&gt;',
 *                 '&lt;p&gt;boy&lt;/p&gt;',
 *         '&lt;/tpl&gt;',
 *     '&lt;/tpl&gt;&lt;/p&gt;'
 * );
 * </pre>
 * <p>A <code>break</code> is implied between each case and default, however, multiple cases can be listed
 * in a single &lt;tpl&gt; tag.</p>
 * <p><b>Using double quotes</b></p>
 * <p>Examples:</p>
 * <pre>
 * var tpl = new Ext.XTemplate(
 *     "&lt;tpl if='age &amp;gt; 1 &amp;&amp; age &amp;lt; 10'&gt;Child&lt;/tpl&gt;",
 *     "&lt;tpl if='age &amp;gt;= 10 &amp;&amp; age &amp;lt; 18'&gt;Teenager&lt;/tpl&gt;",
 *     "&lt;tpl if='this.isGirl(name)'&gt;...&lt;/tpl&gt;",
 *     '&lt;tpl if="id == \'download\'"&gt;...&lt;/tpl&gt;',
 *     "&lt;tpl if='needsIcon'&gt;&lt;img src='{icon}' class='{iconCls}'/&gt;&lt;/tpl&gt;",
 *     "&lt;tpl if='name == \"Don\"'&gt;Hello&lt;/tpl&gt;"
 * );
 * </pre>
 * <p><b>Basic math support</b></p>
 * <p>The following basic math operators may be applied directly on numeric data values:</p>
 * <pre>
 * + - &#42; /
 * </pre>
 * <p>For example:</p>
 * <pre>
 * var tpl = new Ext.XTemplate(
 *     '&lt;p&gt;Name: {name}&lt;/p&gt;',
 *     '&lt;p&gt;Kids: ',
 *     '&lt;tpl for="kids"&gt;',
 *         '&lt;tpl if="age &amp;gt; 1"&gt;',  // &lt;-- Note that the &gt; is encoded
 *             '&lt;p&gt;{#}: {name}&lt;/p&gt;',  // &lt;-- Auto-number each item
 *             '&lt;p&gt;In 5 Years: {age+5}&lt;/p&gt;',  // &lt;-- Basic math
 *             '&lt;p&gt;Dad: {parent.name}&lt;/p&gt;',
 *         '&lt;/tpl&gt;',
 *     '&lt;/tpl&gt;&lt;/p&gt;'
 * );
 * tpl.overwrite(panel.body, data);
 * </pre>
 * <p><b>Execute arbitrary inline code with special built-in template variables</b></p>
 * <p>Anything between <code>{[ ... ]}</code> is considered code to be executed in the scope of the template.
 * The expression is evaluated and the result is included in the generated result. There are
 * some special variables available in that code:</p>
 * <ul>
 * <li><b>out</b>: The output array into which the template is being appended (using <code>push</code> to later
 * <code>join</code>).</li>
 * <li><b>values</b>: The values in the current scope. If you are using scope changing sub-templates,
 * you can change what values is.</li>
 * <li><b>parent</b>: The scope (values) of the ancestor template.</li>
 * <li><b>xindex</b>: If you are in a "for" or "foreach" looping template, the index of the loop you
 * are in (1-based).</li>
 * <li><b>xcount</b>: If you are in a "for" looping template, the total length of the array you are
 * looping.</li>
 * <li><b>xkey</b>: If you are in a "foreach" looping template, the key of the current property
 * being examined.</li>
 * </ul>
 * <p>This example demonstrates basic row striping using an inline code block and the xindex variable:</p>
 * <pre>
 * var tpl = new Ext.XTemplate(
 *     '&lt;p&gt;Name: {name}&lt;/p&gt;',
 *     '&lt;p&gt;Company: {[values.company.toUpperCase() + ", " + values.title]}&lt;/p&gt;',
 *     '&lt;p&gt;Kids: ',
 *     '&lt;tpl for="kids"&gt;',
 *         '&lt;div class="{[xindex % 2 === 0 ? "even" : "odd"]}"&gt;',
 *         '{name}',
 *         '&lt;/div&gt;',
 *     '&lt;/tpl&gt;&lt;/p&gt;'
 *  );
 * </pre>
 * <p>Any code contained in "verbatim" blocks (using "{% ... %}") will be inserted directly in
 * the generated code for the template. These blocks are not included in the output. This
 * can be used for simple things like break/continue in a loop, or control structures or
 * method calls (when they don't produce output). The <code>this</code> references the template instance.</p>
 * <pre>
 * var tpl = new Ext.XTemplate(
 *     '&lt;p&gt;Name: {name}&lt;/p&gt;',
 *     '&lt;p&gt;Company: {[values.company.toUpperCase() + ", " + values.title]}&lt;/p&gt;',
 *     '&lt;p&gt;Kids: ',
 *     '&lt;tpl for="kids"&gt;',
 *         '{% if (xindex % 2 === 0) continue; %}',
 *         '{name}',
 *         '{% if (xindex &gt; 100) break; %}',
 *         '&lt;/div&gt;',
 *     '&lt;/tpl&gt;&lt;/p&gt;'
 *  );
 * </pre>
 * <p><b>Template member functions</b></p>
 * <p>One or more member functions can be specified in a configuration object passed into the
 * XTemplate constructor for more complex processing:</p>
 * <pre>
 * var tpl = new Ext.XTemplate(
 *     '&lt;p&gt;Name: {name}&lt;/p&gt;',
 *     '&lt;p&gt;Kids: ',
 *     '&lt;tpl for="kids"&gt;',
 *         '&lt;tpl if="this.isGirl(name)"&gt;',
 *             '&lt;p&gt;Girl: {name} - {age}&lt;/p&gt;',
 *         '&lt;tpl else&gt;',
 *             '&lt;p&gt;Boy: {name} - {age}&lt;/p&gt;',
 *         '&lt;/tpl&gt;',
 *         '&lt;tpl if="this.isBaby(age)"&gt;',
 *             '&lt;p&gt;{name} is a baby!&lt;/p&gt;',
 *         '&lt;/tpl&gt;',
 *     '&lt;/tpl&gt;&lt;/p&gt;',
 *     {
 *         // XTemplate configuration:
 *         disableFormats: true,
 *         // member functions:
 *         isGirl: function(name){
 *            return name == 'Aubrey' || name == 'Nikol';
 *         },
 *         isBaby: function(age){
 *            return age &lt; 1;
 *         }
 *     }
 * );
 * tpl.overwrite(panel.body, data);
 * </pre>
 * @see ext.view.DataView
 * @see ext.Template
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.XTemplate.html Original Ext JS documentation of 'Ext.XTemplate'
 */
public class XTemplate extends Template {
  /**
   * Gets an <code>XTemplate</code> from an object (an instance of an →<code>ext.Ext.define()</code>'d class).
   * Many times, templates are configured high in the class hierarchy and are to be
   * shared by all classes that derive from that base. To further complicate matters,
   * these templates are seldom actual instances but are rather configurations. For
   * example:
   * <pre>
   *  Ext.define('MyApp.Class', {
   *      extraCls: 'extra-class',
   *
   *      someTpl: [
   *          '&lt;div class="{%this.emitClass(out)%}"&gt;&lt;/div&gt;',
   *      {
   *          // Member fn - outputs the owing class's extra CSS class
   *          emitClass: function(out) {
   *              out.push(this.owner.extraCls);
   *          }
   *      }]
   *  });
   * </pre>
   * <p>The goal being to share that template definition with all instances and even
   * instances of derived classes, until <code>someTpl</code> is overridden. This method will
   * "upgrade" these configurations to be real <code>XTemplate</code> instances <i>in place</i> (to
   * avoid creating one instance per object).</p>
   * <p>The resulting XTemplate will have an <code>owner</code> reference injected which refers back
   * to the owning object whether that is an object which has an <i>own instance</i>, or a
   * class prototype. Through this link, XTemplate member functions will be able to access
   * prototype properties of its owning class.</p>
   * @param instance The object from which to get the <code>XTemplate</code> (must be
   * an instance of an →<code>ext.Ext.define()</code>'d class).
   * @param name The name of the property by which to get the <code>XTemplate</code>.
   * @return The <code>XTemplate</code> instance or null if not found.
   * @see ext.SExt#define() ext.Ext.define()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.XTemplate.html#static-method-getTpl Original Ext JS documentation of 'getTpl'
   */
  protected static native function getTpl(instance:Object, name:String):XTemplate;

  /**
   * Creates new x-template.
   * @param config List of strings to be concatenated into template or x-template Configs.
   */
  public function XTemplate(...config) {
    super(config);
  }

  [ExtConfig]
  /**
   * Only applies to →<code>ext.Template</code>, XTemplates are compiled automatically on the
   * first call to →<code>apply()</code> or →<code>applyOut()</code>.
   * @see ext.Template
   * @see #apply()
   * @see #applyOut()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.XTemplate.html#cfg-compiled Original Ext JS documentation of 'compiled'
   */
  override public native function get compiled():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  override public native function set compiled(value:Boolean):void;

  [ExtConfig]
  /**
   * Optional. A statement, or array of statements which set up <code>var</code>s which may then
   * be accessed within the scope of the generated function.
   * <pre>
   * var data = {
   *     name: 'Don Griffin',
   *     isWizard: true,
   *     title: 'Senior Technomage',
   *     company: 'Sencha Inc.'
   * };
   *
   * var tpl = new Ext.XTemplate('{[values.isWizard ? wizard : notSoWizard]}' +
   *     ' {name}', {
   *     definitions: 'var wizard = "Wizard", notSoWizard = "Townsperson";'
   * });
   *
   * console.log(tpl.apply(data));
   * // LOGS: Wizard Don Griffin
   * </pre>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.XTemplate.html#cfg-definitions Original Ext JS documentation of 'definitions'
   */
  public native function get definitions():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set definitions(value:*):void;

  [ExtConfig]
  /**
   * Expressions in templates that traverse "dot paths" and fail (due to <code>null</code> at some
   * stage) have always been expanded as empty strings. This is convenient in most cases
   * but doing so can also mask errors in the template. Setting this to <code>true</code> changes
   * this default so that any expression errors will be thrown as exceptions.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.XTemplate.html#cfg-strict Original Ext JS documentation of 'strict'
   */
  public native function get strict():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set strict(value:Boolean):void;

  /**
   * Does nothing. XTemplates are compiled automatically, so this function simply returns this.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.XTemplate.html#method-compile Original Ext JS documentation of 'compile'
   */
  override public native function compile():Template;
}
}