package ext.app {
import ext.container.Container;

[Native("Ext.app.Application", require)]
/**
 * Represents an Ext JS application, which is typically a single page app using a
 * →<code>ext.container.Viewport</code>.
 * <p>An application consists of one or more Views. The behavior of a View is managed by its
 * corresponding →<code>ext.app.ViewController</code> and →<code>ext.app.ViewModel</code>.</p>
 * <p>Global activities are coordinated by <i>Controllers</i> (→<code>ext.app.Controller</code>) which are
 * ultimately instantiated by an instance of this (or a derived) class.</p>
 * <pre>
 * Ext.application({
 *     name: 'MyApp',
 *
 *     // The name of the initial view to create. With the classic toolkit this class
 *     // will gain a "viewport" plugin if it does not extend Ext.Viewport. With the
 *     // modern toolkit, the main view will be added to the Viewport.
 *     //
 *     //mainView: 'Main.view.main.Main'
 * });
 * </pre>
 * <p>This does several things. First it creates a global variable called 'MyApp' - all of
 * your Application's classes (such as its Models, Views and Controllers) will reside under
 * this single namespace, which drastically lowers the chances of colliding global variables.</p>
 * <p>The MyApp global will also have a getApplication method to get a reference to the current
 * application:</p>
 * <pre>
 * var app = MyApp.getApplication();
 * </pre>
 * <p><b>Telling Application about the rest of the app</b></p>
 * <p>Because an Ext.app.Application represents an entire app, we should tell it about the other
 * parts of the app - namely the Models, Views and Controllers that are bundled with the
 * application. Let's say we have a blog management app; we might have Models and Controllers for
 * Posts and Comments, and Views for listing, adding and editing Posts and Comments. Here's how
 * we'd tell our Application about all these things:</p>
 * <pre>
 * Ext.application({
 *     name: 'Blog',
 *
 *     models: ['Post', 'Comment'],
 *
 *     controllers: ['Posts', 'Comments'],
 *
 *     launch: function() {
 *         ...
 *     }
 * });
 * </pre>
 * <p>Note that we didn't actually list the Views directly in the Application itself. This is because
 * Views are managed by Controllers, so it makes sense to keep those dependencies there.
 * The Application will load each of the specified Controllers using the pathing conventions
 * laid out in the [application architecture guide][1]</p>
 * <ul>
 * <li>
 * <p>in this case expecting the controllers to reside in app/controller/Posts.js and
 * app/controller/Comments.js. In turn, each Controller simply needs to list the Views it uses
 * and they will be automatically loaded. Here's how our Posts controller like be defined:</p>
 * <p>Ext.define('MyApp.controller.Posts', {
 * extend: 'Ext.app.Controller',
 * views: ['posts.List', 'posts.Edit'],</p>
 * <pre>
 *   // the rest of the Controller here
 * </pre>
 * <p>});</p>
 * </li>
 * </ul>
 * <p>Because we told our Application about our Models and Controllers, and our Controllers about
 * their Views, Ext JS will automatically load all of our app files for us. This means we don't
 * have to manually add script tags into our html files whenever we add a new class, but more
 * importantly it enables us to create a minimized build of our entire application using Sencha Cmd.</p>
 * <p><b>Deriving from Ext.app.Application</b></p>
 * <p>Typically, applications do not derive directly from Ext.app.Application. Rather, the
 * configuration passed to <code>Ext.application</code> mimics what you might do in a derived class.
 * In some cases, however, it can be desirable to share logic by using a derived class
 * from <code>→ext.app.Application</code>.</p>
 * <p>Derivation works as you would expect, but using the derived class should still be the
 * job of the <code>Ext.application</code> method.</p>
 * <pre>
 * Ext.define('MyApp.Application', {
 *     extend: 'Ext.app.Application',
 *     name: 'MyApp',
 *     ...
 * });
 *
 * Ext.application('MyApp.Application');
 * </pre>
 * <p>For more information about writing Ext JS applications, please see
 * the [application architecture guide][1].
 * [1]: ../guides/application_architecture/application_architecture.html</p>
 * @see ext.container.Viewport
 * @see ext.app.ViewController
 * @see ext.app.ViewModel
 * @see ext.app.Controller
 * @see ext.app.Application
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html Original Ext JS documentation of 'Ext.app.Application'
 */
public class Application extends Controller {

  /**
   * The singleton application instance.
   */
  public static native function get instance():Application;

  /**
   * Creates new Application.
   * @param config Config object.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#method-constructor Original Ext JS documentation of 'constructor'
   */
  public function Application(config:Application = null) {
    super();
  }

  [ExtConfig]
  /**
   * The path to the directory which contains all application's classes.
   * This path will be registered via →<code>ext.Loader.setPath()</code> for the namespace specified
   * in the →<code>name</code> config.
   * @default "app"
   * @see ext.SLoader#setPath() ext.Loader.setPath()
   * @see #name
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-appFolder Original Ext JS documentation of 'appFolder'
   */
  public native function get appFolder():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set appFolder(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * The name of a property to be assigned to the main namespace to gain a reference to
   * this application. Can be set to an empty value to prevent the reference from
   * being created
   * <pre>
   * Ext.application({
   *     name: 'MyApp',
   *     appProperty: 'myProp',
   *
   *     launch: function() {
   *         console.log(MyApp.myProp === this);
   *     }
   * });
   * </pre>
   * @default 'app'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-appProperty Original Ext JS documentation of 'appProperty'
   * @see #getAppProperty()
   * @see #setAppProperty()
   */
  public native function get appProperty():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set appProperty(value:String):void;

  [Deprecated(replacement="mainView", since="5.1")]
  [ExtConfig]
  /**
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-autoCreateViewport Original Ext JS documentation of 'autoCreateViewport'
   */
  public native function get autoCreateViewport():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set autoCreateViewport(value:*):void;

  [ExtConfig]
  /**
   * Names of <i>controllers</i> (→<code>ext.app.Controller</code>) that the app uses. By default,
   * the framework will look for the controllers in the "controller" folder within the
   * →<code>appFolder</code>. Controller classes should be named using the syntax of
   * "{appName}.controller.{ClassName}" with additional sub-folders under the
   * "controller" folder specified within the class name following "controller.".
   * <pre>
   * // by default, the following controller class would be located at:
   * // app/controller/Main.js
   * controllers: '.Main' // or 'MyApp.controller.Main'
   *
   * // while the following would be located at:
   * // app/controller/customer/Main.js
   * controllers: 'customer.Main' // or 'MyApp.controller.customer.Main'
   * </pre>
   * <p><b>Note:</b> If the controller has a different namespace than that of the
   * application you will need to specify the full class name as well as define a path
   * in the <i>Loader's paths</i> (→<code>ext.Loader.paths</code>) config or
   * →<code>ext.Loader.setPath()</code> method.</p>
   * @see ext.app.Controller
   * @see #appFolder
   * @see ext.SLoader#paths ext.Loader.paths
   * @see ext.SLoader#setPath() ext.Loader.setPath()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-controllers Original Ext JS documentation of 'controllers'
   */
  public native function get controllers():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set controllers(value:*):void;

  [ExtConfig]
  [Bindable]
  /**
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-currentProfile Original Ext JS documentation of 'currentProfile'
   * @see #getCurrentProfile()
   * @see #setCurrentProfile()
   */
  public native function get currentProfile():Profile;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set currentProfile(value:Profile):void;

  [ExtConfig]
  [Bindable]
  /**
   * The default token to be used at application launch
   * if one is not present. Often this is set to something like <code>'home'</code>.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-defaultToken Original Ext JS documentation of 'defaultToken'
   * @see #getDefaultToken()
   * @see #setDefaultToken()
   */
  public native function get defaultToken():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set defaultToken(value:String):void;

  [Deprecated(replacement="quickTips", since="6.2.0")]
  [ExtConfig]
  [Bindable]
  /**
   * @default null
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-enableQuickTips Original Ext JS documentation of 'enableQuickTips'
   * @see #getEnableQuickTips()
   * @see #setEnableQuickTips()
   */
  public native function get enableQuickTips():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set enableQuickTips(value:Boolean):void;

  [ExtConfig]
  /**
   * A class name to use with the <code>Ext.application</code> call. The class must
   * also extend →<code>ext.app.Application</code>.
   * <pre>
   * Ext.define('MyApp.Application', {
   *     extend: 'Ext.app.Application',
   *
   *     launch: function() {
   *         Ext.direct.Manager.addProvider(Ext.REMOTING_API);
   *     }
   * });
   *
   * Ext.application({
   *     extend: 'MyApp.Application'
   * });
   * </pre>
   * @see ext.app.Application
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-extend Original Ext JS documentation of 'extend'
   */
  public native function get extend():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set extend(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * The glyphFontFamily to use for this application. Used as the default font-family
   * for all components that support a <code>glyph</code> config.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-glyphFontFamily Original Ext JS documentation of 'glyphFontFamily'
   * @see #getGlyphFontFamily()
   * @see #setGlyphFontFamily()
   */
  public native function get glyphFontFamily():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set glyphFontFamily(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * The application class to be used as the main viewport view for the
   * application. The view will be configured with the
   * <i>viewport plugin</i> (→<code>ext.plugin.ViewportPlugin</code>) to ensure the view takes up all
   * available space in the browser viewport. The main view will be created after
   * the application's →<code>init()</code> method is called and before the
   * →<code>launch()</code> method. The main view should be an application class type and
   * not a class from the framework.
   * <p>The main view value may be:</p>
   * <ul>
   * <li>
   * <p>string representing the full class name of the main view or the partial class name
   * following "AppName.view." (provided your main view class follows that convention).</p>
   * </li>
   * <li>
   * <p>config object for the main view</p>
   * </li>
   * <li>
   * <p>main view class instance</p>
   * <p>Ext.define('MyApp.view.main.Main', {
   * extend: 'Ext.panel.Panel',
   * xtype: 'mainview',
   * title: 'Main Viewport View'
   * });</p>
   * <p>Ext.application({
   * name : 'MyApp',</p>
   * <pre>
   *  mainView: 'MyApp.view.main.Main'
   *  // mainView: 'main.Main'
   *  // mainView: new MyApp.view.main.Main()
   *  // mainView: { xtype: 'mainview' }
   * </pre>
   * <p>});</p>
   * </li>
   * </ul>
   * <p><b>Note:</b> You may also call →<code>setMainView()</code> at runtime if you require
   * logic within the application's →<code>launch()</code> method to be processed prior to
   * the creation of the main view.</p>
   * @default null
   * @see ext.plugin.ViewportPlugin
   * @see #init()
   * @see #launch()
   * @see #setMainView()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-mainView Original Ext JS documentation of 'mainView'
   * @see #getMainView()
   */
  public native function get mainView():Container;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set mainView(value:Container):void;

  [ExtConfig]
  [Bindable]
  /**
   * The name of your application. This will also be the namespace for your views, controllers
   * models and stores. Don't use spaces or special characters in the name. <b>Application name
   * is mandatory</b>.
   * @default ''
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-name Original Ext JS documentation of 'name'
   * @see #getName()
   * @see #setName()
   */
  public native function get name():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set name(value:String):void;

  [ExtConfig]
  /**
   * The list of namespace prefixes used in the application to resolve dependencies
   * like Views and Stores...
   * <pre>
   *  Ext.application({
   *      name: 'MyApp',
   *
   *      namespaces: ['Common.code'],
   *
   *      controllers: [ 'Common.code.controller.Foo', 'Bar' ]
   *  });
   *
   *  Ext.define('Common.code.controller.Foo', {
   *      extend: 'Ext.app.Controller',
   *
   *      models: ['Foo'],    // Loads Common.code.model.Foo
   *      views:  ['Bar']     // Loads Common.code.view.Bar
   *  });
   *
   *  Ext.define('MyApp.controller.Bar', {
   *      extend: 'Ext.app.Controller',
   *
   *      models: ['Foo'],    // Loads MyApp.model.Foo
   *      views:  ['Bar']     // Loads MyApp.view.Bar
   *  });
   * </pre>
   * <p>You don't need to include main namespace (MyApp), it will be added to the list
   * automatically.</p>
   * @default []
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-namespaces Original Ext JS documentation of 'namespaces'
   */
  public native function get namespaces():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set namespaces(value:*):void;

  [ExtConfig]
  /**
   * Additional load paths to add to Ext.Loader.
   * See →<code>ext.Loader.paths</code> config for more details.
   * @see ext.SLoader#paths ext.Loader.paths
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-paths Original Ext JS documentation of 'paths'
   */
  public native function get paths():Object;

  [ExtConfig]
  /**
   * @private
   */
  public native function set paths(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * Names of the profiles that the app uses.
   * @default []
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-profiles Original Ext JS documentation of 'profiles'
   * @see #getProfiles()
   * @see #setProfiles()
   */
  public native function get profiles():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set profiles(value:*):void;

  [ExtConfig]
  /**
   * True to automatically set up Ext.tip.QuickTip support.
   * @default true
   * @since 6.2.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-quickTips Original Ext JS documentation of 'quickTips'
   */
  public native function get quickTips():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set quickTips(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * A configuration object to apply onto the →<code>ext.route.Router</code>.
   * @since 6.5.0
   * @see ext.route.#Router ext.route.Router
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-router Original Ext JS documentation of 'router'
   * @see #getRouter()
   * @see #setRouter()
   */
  public native function get router():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set router(value:Object):void;

  [ExtConfig]
  /**
   * The scope to execute the →<code>launch()</code> function in. Defaults to the Application instance.
   * @default undefined
   * @see #launch()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#cfg-scope Original Ext JS documentation of 'scope'
   */
  public native function get scope():Object;

  [ExtConfig]
  /**
   * @private
   */
  public native function set scope(value:Object):void;

  /**
   * Create an instance of a controller by name.
   * @param name The name of the controller. For a controller with the
   * full class name <code>MyApp.controller.Foo</code>, the name parameter should be <code>Foo</code>.
   * If the controller already exists, it will be returned.
   * @return controller
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#method-createController Original Ext JS documentation of 'createController'
   */
  public native function createController(name:String):Controller;

  /**
   * Destroys a controller, any listeners are unbound.
   * @param controller The controller
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#method-destroyController Original Ext JS documentation of 'destroyController'
   */
  public native function destroyController(controller:*):void;

  /**
   * Returns the value of <code>appProperty</code>.
   * @see #appProperty
   */
  public native function getAppProperty():String;

  /**
   * Get an application's controller based on name or id. Generally, the controller id will be
   * the same as the name unless otherwise specified.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#method-getController Original Ext JS documentation of 'getController'
   */
  override public native function getController(id:String):Controller;

  /**
   * Returns the value of <code>currentProfile</code>.
   * @see #currentProfile
   */
  public native function getCurrentProfile():Profile;

  /**
   * Returns the value of <code>defaultToken</code>.
   * @see #defaultToken
   */
  public native function getDefaultToken():String;

  [Deprecated(replacement="quickTips", since="6.2.0")]
  /**
   * Returns the value of <code>enableQuickTips</code>.
   * @see #enableQuickTips
   */
  public native function getEnableQuickTips():Boolean;

  /**
   * Returns the value of <code>glyphFontFamily</code>.
   * @see #glyphFontFamily
   */
  public native function getGlyphFontFamily():String;

  /**
   * Returns the value of <code>mainView</code>.
   * @see #mainView
   */
  public native function getMainView():Container;

  /**
   * Returns the value of <code>name</code>.
   * @see #name
   */
  public native function getName():String;

  /**
   * Returns the value of <code>profiles</code>.
   * @see #profiles
   */
  public native function getProfiles():*;

  /**
   * Returns the value of <code>router</code>.
   * @see #router
   */
  public native function getRouter():Object;

  /**
   * Called automatically when the page has completely loaded. This is an empty function that
   * should be overridden by each application that needs to take action on page load.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @param profile The detected application profile
   * @return By default, the Application will dispatch to the configured startup
   * controller and action immediately after running the launch function. Return false
   * to prevent this behavior.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#method-launch Original Ext JS documentation of 'launch'
   */
  public native function launch(profile:String):Boolean;

  /**
   * Called automatically when an update to either the Application Cache or the Local Storage
   * Cache is detected.
   * This is mainly used during production builds.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @param updateInfo updateInfo Update information object contains properties for
   * checking which cache triggered the update
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Application.html#method-onAppUpdate Original Ext JS documentation of 'onAppUpdate'
   */
  public native function onAppUpdate(updateInfo:Object = null):void;

  /**
   * Sets the value of <code>appProperty</code>.
   * @param appProperty The new value.
   * @see #appProperty
   */
  public native function setAppProperty(appProperty:String):void;

  /**
   * Sets the value of <code>currentProfile</code>.
   * @param currentProfile The new value.
   * @see #currentProfile
   */
  public native function setCurrentProfile(currentProfile:Profile):void;

  /**
   * Sets the value of <code>defaultToken</code>.
   * @param defaultToken The new value.
   * @see #defaultToken
   */
  public native function setDefaultToken(defaultToken:String):void;

  [Deprecated(replacement="quickTips", since="6.2.0")]
  /**
   * Sets the value of <code>enableQuickTips</code>.
   * @param enableQuickTips The new value.
   * @see #enableQuickTips
   */
  public native function setEnableQuickTips(enableQuickTips:Boolean):void;

  /**
   * Sets the value of <code>glyphFontFamily</code>.
   * @param glyphFontFamily The new value.
   * @see #glyphFontFamily
   */
  public native function setGlyphFontFamily(glyphFontFamily:String):void;

  /**
   * Sets the value of <code>mainView</code>.
   * @param mainView The new value.
   * @see #mainView
   */
  public native function setMainView(mainView:Container):void;

  /**
   * Sets the value of <code>name</code>.
   * @param name The new value.
   * @see #name
   */
  public native function setName(name:String):void;

  /**
   * Sets the value of <code>profiles</code>.
   * @param profiles The new value.
   * @see #profiles
   */
  public native function setProfiles(profiles:*):void;

  /**
   * Sets the value of <code>router</code>.
   * @param router The new value.
   * @see #router
   */
  public native function setRouter(router:Object):void;
}
}