package ext.app {
import ext.Base;
import ext.ExtClass;
import ext.data.Store;

[Native("Ext.app.Controller", require)]
/**
 * Controllers are the glue that binds an application together. That said, their main
 * purpose is to listen for events (usually from views) and take some action. Here's how
 * we might create a Controller to manage Users:
 * <pre>
 *  Ext.define('MyApp.controller.Users', {
 *      extend: 'Ext.app.Controller',
 *
 *      init: function() {
 *          console.log('Initialized Users! This happens before ' +
 *                      'the Application launch() function is called');
 *      }
 *  });
 * </pre>
 * <p>The init function is a special method that is called when your application boots. It is
 * called before the →<code>ext.app.Application</code>'s launch function is executed.
 * This creates an area you can run code prior to Viewport creation.</p>
 * <p>The controller's →<code>control()</code> function
 * makes it easy to listen to events on your view classes and take some action with a
 * handler function. Let's update our Users controller to tell us when the panel is
 * rendered:</p>
 * <pre>
 *  Ext.define('MyApp.controller.Users', {
 *      extend: 'Ext.app.Controller',
 *
 *      control: {
 *          'viewport &gt; panel': {
 *              render: 'onPanelRendered'
 *          }
 *      }
 *
 *      onPanelRendered: function() {
 *          console.log('The panel was rendered');
 *      }
 *  });
 * </pre>
 * <p>The <i>control method</i> (→<code>ext.app.BaseController.control()</code>) has now set up
 * listeners on views in our application. The control method uses the ComponentQuery
 * engine to quickly and easily get references to components on the page. If you are not
 * familiar with ComponentQuery yet, be sure to check out the
 * <i>documentation</i> (→<code>ext.ComponentQuery</code>). In brief, it allows us to pass a
 * CSS-like selector that will find every matching component on the page.</p>
 * <p>In our init function above, we supplied 'viewport &gt; panel', which translates to "find me
 * every Panel that is a direct child of a Viewport". We then supplied an object that maps
 * event names (just 'render' in this case) to handler functions. In short, whenever
 * a component that matches our selector fires a 'render' event, our
 * onPanelRendered function is called.</p>
 * <p><b><i>Event domains</i></b></p>
 * <p>In Ext JS 4.2, we introduced the concept of event domains. In terms of MVC, an event
 * domain is one or more base classes that fire events to which a Controller wants to
 * listen. Besides Component event domain that encompass →<code>ext.Component</code>-descended
 * Views, Controllers now can listen to events from data Stores, Ext Direct Providers,
 * other Controllers, and Ext.GlobalEvents. This feature provides a way to communicate
 * between parts of the whole application without the need to bind controllers together
 * tightly, and allows to develop and test application parts in isolation.</p>
 * <p>See usage examples in →<code>listen()</code> method documentation.</p>
 * <p><b><i>Using refs</i></b></p>
 * <p>One of the most useful parts of Controllers is the ref system. These use
 * →<code>ext.ComponentQuery</code> to make it really easy to get references to Views on your
 * page. Let's look at an example of this now:</p>
 * <pre>
 *  Ext.define('MyApp.controller.Users', {
 *      extend: 'Ext.app.Controller',
 *
 *      refs: [{
 *          ref: 'list',
 *          selector: 'grid'
 *      }],
 *
 *      control: {
 *          'button': {
 *              click: 'refreshGrid'
 *          }
 *      },
 *
 *      refreshGrid: function() {
 *          this.getList().store.load();
 *      }
 *  });
 * </pre>
 * <p>This example assumes the existence of a <i>Grid</i> (→<code>ext.grid.GridPanel</code>) on the page, which
 * contains a single button to refresh the Grid when clicked. In our refs array, we set up
 * a reference to the grid. There are two parts to this - the 'selector', which is a
 * →<code>ext.ComponentQuery</code> selector which finds any grid on the page and
 * assigns it to the reference 'list'.</p>
 * <p>By giving the reference a name, we get a number of things for free. The first is the
 * getList function that we use in the refreshGrid method above. This is generated
 * automatically by the Controller based on the name of our ref, which was capitalized and
 * prepended with get to go from 'list' to 'getList'.</p>
 * <p>The way this works is that the first time getList is called by your code, the
 * ComponentQuery selector is run and the first component that matches the selector
 * ('grid' in this case) will be returned. All future calls to getList will use a cached
 * reference to that grid. Usually it is advised to use a specific ComponentQuery selector
 * that will only match a single View in your application (in the case above our selector
 * will match any grid on the page).</p>
 * <p>Bringing it all together, we configure control
 * to listen to any click on a <i>button</i> (→<code>ext.button.Button</code>) and call our refreshGrid
 * function (again, this will match any button on the page so we advise a more specific
 * selector than just 'button', but have left it this way for simplicity). When the button
 * is clicked we use out getList function to refresh the grid.</p>
 * <p>You can create any number of refs and control any number of components this way, simply
 * adding more functions to your Controller as you go. For an example of real-world usage
 * of Controllers see the Feed Viewer example in the examples/app/feed-viewer folder in
 * the SDK download.</p>
 * <p><b><i>Generated getter methods</i></b></p>
 * <p>Refs aren't the only thing that generate convenient getter methods. Controllers often
 * have to deal with Models and Stores so the framework offers a couple of easy ways to
 * get access to those too. Let's look at another example:</p>
 * <pre>
 *  Ext.define('MyApp.controller.Users', {
 *      extend: 'Ext.app.Controller',
 *
 *      models: ['User'],
 *      stores: ['AllUsers', 'AdminUsers'],
 *
 *      init: function() {
 *          var User, allUsers, ed;
 *
 *          User = this.getUserModel();
 *          allUsers = this.getAllUsersStore();
 *
 *          ed = new User({ name: 'Ed' });
 *          allUsers.add(ed);
 *      }
 *  });
 * </pre>
 * <p>By specifying Models and Stores that the Controller cares about, it again dynamically
 * loads them from the appropriate locations (app/model/User.js, app/store/AllUsers.js and
 * app/store/AdminUsers.js in this case) and creates getter functions for them all. The
 * example above will create a new User model instance and add it to the AllUsers Store.
 * Of course, you could do anything in this function but in this case we just did
 * something simple to demonstrate the functionality.</p>
 * <p><b><i>Further Reading</i></b></p>
 * <p>For more information about writing Ext JS 5 applications, please see the
 * <a href="https://docs.sencha.com/application_architecture/application_architecture.html">Application Architecture</a>.
 * Also see the →<code>ext.app.Application</code> documentation.</p>
 * @see ext.app.Application
 * @see #control()
 * @see ext.app.BaseController#control()
 * @see ext.#ComponentQuery ext.ComponentQuery
 * @see ext.Component
 * @see #listen()
 * @see ext.grid.GridPanel
 * @see ext.button.Button
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html Original Ext JS documentation of 'Ext.app.Controller'
 */
public class Controller extends BaseController {
  /**
   * Creates new Controller.
   * @param config Configuration object.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#method-constructor Original Ext JS documentation of 'constructor'
   */
  public function Controller(config:Controller = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * The →<code>ext.app.Application</code> for this controller accessible via the
   * getApplication method.
   * @see ext.app.Application
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#cfg-application Original Ext JS documentation of 'application'
   * @see #getApplication()
   */
  public native function get application():Application;

  [ExtConfig]
  /**
   * Array of models to require from AppName.model namespace. For example:
   * <pre>
   *  Ext.define("MyApp.controller.Foo", {
   *      extend: "Ext.app.Controller",
   *      models: ['User', 'Vehicle']
   *  });
   * </pre>
   * <p>This is equivalent to:</p>
   * <pre>
   *  Ext.define("MyApp.controller.Foo", {
   *      extend: "Ext.app.Controller",
   *      requires: ['MyApp.model.User', 'MyApp.model.Vehicle'],
   *
   *      getUserModel: function() {
   *          return this.getModel("User");
   *      },
   *
   *      getVehicleModel: function() {
   *          return this.getModel("Vehicle");
   *      }
   *  });
   * </pre>
   * <p><b>Note:</b> If the model has a different namespace than that of the
   * application you will need to specify the full class name as well as define a path
   * in the <i>Loader's paths</i> (→<code>ext.Loader.paths</code>) config or
   * →<code>ext.Loader.setPath()</code> method.</p>
   * @default null
   * @see ext.SLoader#paths ext.Loader.paths
   * @see ext.SLoader#setPath() ext.Loader.setPath()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#cfg-models Original Ext JS documentation of 'models'
   */
  public native function get models():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set models(value:*):void;

  [ExtConfig]
  [Bindable]
  /**
   * The refs config creates a getter method on the controller that internally
   * uses Ext.ComponentQuery to fetch the component instance using the configured
   * selector. The following example will add the <code>getList</code> method to
   * the controller and will return the first component in the application
   * hierarchy with an xtype of "grid". By default, <i>undefined</i> will be returned
   * when the query does not locate the target component.
   * <pre>
   * Ext.define('MyApp.controller.Foo', {
   *     extend: 'Ext.app.Controller',
   *
   *     refs: [{
   *         ref: 'list',
   *         selector: 'grid'
   *     }]
   * });
   * </pre>
   * <p>The following fields may be used in the ref definition:</p>
   * <ul>
   * <li><code>ref</code> - name of the reference.</li>
   * <li><code>selector</code> - Ext.ComponentQuery selector to access the component.</li>
   * <li><code>autoCreate</code> - True to create the component automatically if not found on
   * page.</li>
   * <li><code>forceCreate</code> - True to force the creation of the component every time
   * reference is accessed (when <code>get&lt;REFNAME&gt;</code> is called).</li>
   * <li><code>xtype</code> - Used to create the component by its xtype with <code>autoCreate</code> or
   * <code>forceCreate</code>. If you don't provide <code>xtype</code>, an Ext.Component instance will
   * be created.</li>
   * </ul>
   * <p>The following example will create a <code>getList</code> and <code>getUser</code> method on the
   * controller.</p>
   * <pre>
   * Ext.define('MyApp.controller.Foo', {
   *     extend: 'Ext.app.Controller',
   *
   *     refs: [{
   *         list: 'grid',
   *         user: {
   *             autoCreate: true,
   *             selector: 'form',
   *             xtype: 'form'
   *         }
   *     }]
   * });
   * </pre>
   * @default null
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#cfg-refs Original Ext JS documentation of 'refs'
   * @see #getRefs()
   * @see #setRefs()
   */
  public native function get refs():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set refs(value:*):void;

  [ExtConfig]
  /**
   * Array of stores to require from AppName.store namespace and to generate getter methods for.
   * For example:
   * <pre>
   *  Ext.define("MyApp.controller.Foo", {
   *      extend: "Ext.app.Controller",
   *      stores: ['Users', 'Vehicles']
   *  });
   * </pre>
   * <p>This is equivalent to:</p>
   * <pre>
   *  Ext.define("MyApp.controller.Foo", {
   *      extend: "Ext.app.Controller",
   *
   *      requires: [
   *          'MyApp.store.Users',
   *          'MyApp.store.Vehicles'
   *      ]
   *
   *      getUsersStore: function() {
   *          return this.getStore("Users");
   *      },
   *
   *      getVehiclesStore: function() {
   *          return this.getStore("Vehicles");
   *      }
   *  });
   * </pre>
   * <p><b>Note:</b> If the store has a different namespace than that of the
   * application you will need to specify the full class name as well as define a path
   * in the <i>Loader's paths</i> (→<code>ext.Loader.paths</code>) config or
   * →<code>ext.Loader.setPath()</code> method.</p>
   * @default null
   * @see ext.SLoader#paths ext.Loader.paths
   * @see ext.SLoader#setPath() ext.Loader.setPath()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#cfg-stores Original Ext JS documentation of 'stores'
   */
  public native function get stores():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set stores(value:*):void;

  [ExtConfig]
  /**
   * Array of views to require from AppName.view namespace and to generate getter methods for.
   * For example:
   * <pre>
   *  Ext.define("MyApp.controller.Foo", {
   *      extend: "Ext.app.Controller",
   *      views: ['List', 'Detail']
   *  });
   * </pre>
   * <p>This is equivalent to:</p>
   * <pre>
   *  Ext.define("MyApp.controller.Foo", {
   *      extend: "Ext.app.Controller",
   *      requires: ['MyApp.view.List', 'MyApp.view.Detail'],
   *
   *      getListView: function() {
   *          return this.getView("List");
   *      },
   *
   *      getDetailView: function() {
   *          return this.getView("Detail");
   *      }
   *  });
   * </pre>
   * <p><b>Note:</b> If the view has a different namespace than that of the
   * application you will need to specify the full class name as well as define a path
   * in the <i>Loader's paths</i> (→<code>ext.Loader.paths</code>) config or
   * →<code>ext.Loader.setPath()</code> method.</p>
   * @default null
   * @see ext.SLoader#paths ext.Loader.paths
   * @see ext.SLoader#setPath() ext.Loader.setPath()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#cfg-views Original Ext JS documentation of 'views'
   */
  public native function get views():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set views(value:*):void;

  /**
   * Allow the controller to resume receiving events from the event bus.
   * Routes will also be able to begin firing on this controller.
   * Also see →<code>deactivate()</code>.
   * @see #deactivate()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#method-activate Original Ext JS documentation of 'activate'
   */
  public native function activate():void;

  /**
   * Registers one or more <i>references</i> (→<code>refs</code>).
   * @see #refs
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#method-addRef Original Ext JS documentation of 'addRef'
   */
  public native function addRef(refs:*):void;

  /**
   * Prevent this controller from receiving events from the event bus.
   * Routes will also not be triggered on inactive controllers unless
   * the →<code>ext.route.Route.allowInactive</code> flag is set.
   * Also see →<code>activate()</code>.
   * @see ext.route.Route#allowInactive
   * @see #activate()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#method-deactivate Original Ext JS documentation of 'deactivate'
   */
  public native function deactivate():void;

  /**
   * Returns the value of <code>application</code>.
   * @see #application
   */
  public native function getApplication():Application;

  /**
   * Returns instance of a →<code>ext.app.Controller</code> with the given id.
   * When controller doesn't exist yet, it's created. Note that this method depends
   * on Application instance and will return undefined when Application is not
   * accessible. The only exception is when this Controller instance's id is requested;
   * in that case we always return the instance even if Application is no available.
   * @return controller instance or undefined.
   * @see ext.app.Controller
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#method-getController Original Ext JS documentation of 'getController'
   */
  public native function getController(id:String):Controller;

  /**
   * Returns a →<code>ext.data.Model</code> class with the given name.
   * @return A class ultimately derived from <code>→ext.data.Model</code>.
   * @see ext.data.Model
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#method-getModel Original Ext JS documentation of 'getModel'
   */
  public native function getModel(modelName:String):ExtClass;

  /**
   * Returns instance of a →<code>ext.app.Profile</code> with the given name.
   * @return a profile instance.
   * @see ext.app.Profile
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#method-getProfile Original Ext JS documentation of 'getProfile'
   */
  public native function getProfile(name:String):String;

  /**
   * Returns the value of <code>refs</code>.
   * @see #refs
   */
  public native function getRefs():*;

  /**
   * Returns instance of a →<code>ext.data.Store</code> with the given name.
   * When store doesn't exist yet, it's created.
   * @return a store instance.
   * @see ext.data.Store
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#method-getStore Original Ext JS documentation of 'getStore'
   */
  public native function getStore(name:String):Store;

  /**
   * Returns a View class with the given name. To create an instance of the view,
   * you can use it like it's used by Application to create the Viewport:
   * <pre>
   * this.getView('Viewport').create();
   * </pre>
   * @return a view class.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#method-getView Original Ext JS documentation of 'getView'
   */
  public native function getView(view:String):Base;

  /**
   * Returns <code>true</code> if a <i>reference</i> (→<code>refs</code>) is registered.
   * @param ref The name of the ref to check for.
   * @see #refs
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#method-hasRef Original Ext JS documentation of 'hasRef'
   */
  public native function hasRef(ref:String):Boolean;

  /**
   * A template method that is called when your application boots. It is called before the
   * →<code>ext.app.Application</code>'s launch function is executed so gives a hook point
   * to run any code before your Viewport is created.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @see ext.app.Application
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#method-init Original Ext JS documentation of 'init'
   */
  public native function init(application:Application):void;

  /**
   * Checks if this controller is active. See →<code>activate()</code> &amp;
   * →<code>deactivate()</code>.
   * @return <code>true</code> if this controller is active.
   * @see #activate()
   * @see #deactivate()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#method-isActive Original Ext JS documentation of 'isActive'
   */
  public native function isActive():Boolean;

  /**
   * A template method like →<code>init()</code>, but called after the viewport is created.
   * This is called after the →<code>ext.app.Application.launch()</code> method of Application
   * is executed.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @see #init()
   * @see ext.app.Application#launch()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.Controller.html#method-onLaunch Original Ext JS documentation of 'onLaunch'
   */
  public native function onLaunch(application:Application):void;

  /**
   * Sets the value of <code>refs</code>.
   * @param refs The new value.
   * @see #refs
   */
  public native function setRefs(refs:*):void;
}
}