package ext.app {
import ext.Base;
import ext.container.Container;
import ext.data.Session;
import ext.data.Store;
import ext.mixin.IFactoryable;
import ext.mixin.IIdentifiable;

[Native("Ext.app.ViewModel", require)]
[ExtConfig]
/**
 * This class manages arbitrary data and its relationship to data models. Instances of
 * <code>ViewModel</code> are associated with some <code>Component</code> and then used by their child items
 * for the purposes of Data Binding.
 * <p><b>Binding</b></p>
 * <p>The most commonly used aspect of a <code>ViewModel</code> is the <code>→bind()</code> method. This method takes
 * a "bind descriptor" (see below) and a callback to call when the data indicated by the
 * bind descriptor either becomes available or changes.</p>
 * <p>The <code>→bind()</code> method, based on the bind descriptor given, will return different types of
 * "binding" objects. These objects maintain the connection between the requested data and
 * the callback. Bindings ultimately derive from <code>→ext.app.bind.BaseBinding</code>
 * which provides several methods to help manage the binding.</p>
 * <p>Perhaps the most important method is <code>→destroy()</code>. When the binding is no longer needed
 * it is important to remember to <code>→destroy()</code> it. Leaking bindings can cause performance
 * problems or worse when callbacks are called at unexpected times.</p>
 * <p>The types of bindings produced by <code>→bind()</code> are:</p>
 * <ul>
 * <li><code>→ext.app.bind.Binding</code></li>
 * <li><code>→ext.app.bind.Multi</code></li>
 * <li><code>→ext.app.bind.TemplateBinding</code></li>
 * </ul>
 * <p><b><i>Bind Descriptors</i></b></p>
 * <p>A "bind descriptor" is a value (a String, an Object or an array of these) that describe
 * the desired data. Any piece of data in the <code>ViewModel</code> can be described by a bind
 * descriptor.</p>
 * <p><i>Textual Bind Descriptors</i></p>
 * <p>The simplest and most common form of bind descriptors are strings that look like an
 * <code>→ext.Template</code> containing text and tokens surrounded by "{}" with dot notation inside
 * to traverse objects and their properties.</p>
 * <p>For example:</p>
 * <ul>
 * <li><code>'Hello {user.name}!'</code></li>
 * <li><code>'You have selected "{selectedItem.text}".'</code></li>
 * <li><code>'{!isDisabled}'</code></li>
 * <li><code>'{a &gt; b ? "Bigger" : "Smaller"}'</code></li>
 * <li><code>'{user.groups}'</code></li>
 * </ul>
 * <p>All except the last are <code>template bindings (→ext.app.bind.TemplateBinding)</code>
 * which use the familiar <code>→ext.Template</code> syntax with some slight differences. For more on
 * templates see <code>→null</code>.</p>
 * <p>The last descriptor is called a "direct bind descriptor". This special form of
 * bind maps one-to-one to some piece of data in the <code>ViewModel</code> and is managed by the
 * <code>→ext.app.bind.Binding</code> class.</p>
 * <p><i>Two-Way Descriptors</i></p>
 * <p>A direct bind descriptor may be able to write back a value to the <code>ViewModel</code> as well
 * as retrieve one. When this is the case, they are said to be "two-way". For example:</p>
 * <pre>
 *  var binding = viewModel.bind('{s}', function(s) { console.log('s=' + s); });
 *
 *  binding.setValue('abc');
 * </pre>
 * <p>Direct use of <code>ViewModel</code> in this way is not commonly needed because <code>→ext.Component</code>
 * automates this process. For example, a <code>textfield</code> component understands when it is
 * given a "two-way" binding and automatically synchronizes its value bidirectionally using
 * the above technique. For example:</p>
 * <pre>
 *  Ext.widget({
 *      items: [{
 *          xtype: 'textfield',
 *          bind: '{s}'  // a two-way / direct bind descriptor
 *      }]
 *  });
 * </pre>
 * <p><i>Object and Array Descriptors / Multi-Bind</i></p>
 * <p>With two exceptions (see below) an Object is interpreted as a "shape" to produce by
 * treating each of its properties as individual bind descriptors. An object of the same
 * shape is passed as the value of the bind except that each property is populated with
 * the appropriate value. Of course, this definition is recursive, so these properties
 * may also be objects.</p>
 * <p>For example:</p>
 * <pre>
 *  viewModel.bind({
 *          x: '{x}',
 *          foo: {
 *              bar: 'Hello {foo.bar}'
 *          }
 *      },
 *      function (obj) {
 *          //  obj = {
 *          //      x: 42,
 *          //      foo: {
 *          //          bar: 'Hello foobar'
 *          //      }
 *          //  }
 *      });
 * </pre>
 * <p>Arrays are handled in the same way. Each element of the array is considered a bind
 * descriptor (recursively) and the value produced for the binding is an array with each
 * element set to the bound property.</p>
 * <p><i>Bind Options</i></p>
 * <p>One exception to the "object is a multi-bind" rule is when that object contains a
 * <code>bindTo</code> property. When an object contains a <code>bindTo</code> property the object is understood
 * to contain bind options and the value of <code>bindTo</code> is considered the actual bind
 * descriptor.</p>
 * <p>For example:</p>
 * <pre>
 *  viewModel.bind({
 *          bindTo: '{x}',
 *          single: true
 *      },
 *      function (x) {
 *          console.log('x: ' + x); // only called once
 *      });
 * </pre>
 * <p>The available bind options depend on the type of binding, but since all bindings
 * derive from <code>→ext.app.bind.BaseBinding</code> its options are always applicable.
 * For a list of the other types of bindings, see above.</p>
 * <p><i>Deep Binding</i></p>
 * <p>When a direct bind is made and the bound property is an object, by default the binding
 * callback is only called when that reference changes. This is the most efficient way to
 * understand a bind of this type, but sometimes you may need to be notified if any of the
 * properties of that object change.</p>
 * <p>To do this, we create a "deep bind":</p>
 * <pre>
 *  viewModel.bind({
 *          bindTo: '{someObject}',
 *          deep: true
 *      },
 *      function (someObject) {
 *          // called when reference changes or &#42;any&#42; property changes
 *      });
 * </pre>
 * <p><i>Binding Timings</i></p>
 * <p>The <code>ViewModel</code> has a →<code>scheduler</code> attached that is used to coordinate the firing of
 * bindings.
 * It serves 2 main purposes:</p>
 * <ul>
 * <li>To coordinate dependencies between bindings. This means bindings will be fired in an order
 * such that the any dependencies for a binding are fired before the binding itself.</li>
 * <li>To batch binding firings. The scheduler runs on a short timer, so the following code will
 * only trigger a single binding (the last), the changes in between will never be triggered.</li>
 * </ul>
 * <p>Example:</p>
 * <pre>
 * viewModel.bind('{val}', function(v) {
 *     console.log(v);
 * });
 * viewModel.set('val', 1);
 * viewModel.set('val', 2);
 * viewModel.set('val', 3);
 * viewModel.set('val', 4);
 * </pre>
 * <p>The <code>ViewModel</code> can be forced to process by calling <code>→notify()</code>, which will force the
 * scheduler to run immediately in the current state.</p>
 * <pre>
 * viewModel.bind('{val}', function(v) {
 *     console.log(v);
 * });
 * viewModel.set('val', 1);
 * viewModel.notify();
 * viewModel.set('val', 2);
 * viewModel.notify();
 * viewModel.set('val', 3);
 * viewModel.notify();
 * viewModel.set('val', 4);
 * viewModel.notify();
 * </pre>
 * <p><i>Models, Stores and Associations</i></p>
 * <p>A →<code>ext.data.Session</code> manages model instances and their associations.
 * The <code>ViewModel</code> may be used with or without a <code>Session</code>. When a <code>Session</code> is attached, the
 * <code>ViewModel</code> will always consult the <code>Session</code> to ask about records and stores. The <code>Session</code>
 * ensures that only a single instance of each model Type/Id combination is created. This is
 * important when tracking changes in models so that we always have the same reference.</p>
 * <p>A <code>ViewModel</code> provides functionality to easily consume the built in data package types
 * →<code>ext.data.Model</code> and →<code>ext.data.Store</code>, as well as their associations.</p>
 * <p><i>Model Links</i></p>
 * <p>A model can be described declaratively using →<code>links</code>. In the example code below,
 * We ask the <code>ViewModel</code> to construct a record of type <code>User</code> with <code>id: 17</code>. The model will be
 * loaded from the server and the bindings will trigger once the load has completed. Similarly,
 * we could also attach a model instance to the <code>ViewModel</code> data directly.</p>
 * <pre>
 * Ext.define('MyApp.model.User', {
 *     extend: 'Ext.data.Model',
 *     fields: ['name']
 * });
 *
 * var rec = new MyApp.model.User({
 *     id: 12,
 *     name: 'Foo'
 * });
 *
 * var viewModel = new Ext.app.ViewModel({
 *     links: {
 *         theUser: {
 *             type: 'User',
 *             id: 17
 *         }
 *     },
 *     data: {
 *         otherUser: rec
 *     }
 * });
 * viewModel.bind('{theUser.name}', function(v) {
 *     console.log(v);
 * });
 * viewModel.bind('{otherUser.name}', function(v) {
 *     console.log(v);
 * });
 * </pre>
 * <p><i>Model Fields</i></p>
 * <p>Bindings have the functionality to inspect the parent values and resolve the underlying
 * value dynamically. This behavior allows model fields to be interrogated as part of a binding.</p>
 * <pre>
 * Ext.define('MyApp.model.User', {
 *     extend: 'Ext.data.Model',
 *     fields: ['name', 'age']
 * });
 *
 * var viewModel = new Ext.app.ViewModel({
 *     links: {
 *         theUser: {
 *             type: 'User',
 *             id: 22
 *         }
 *     }
 * });
 *
 * // Server responds with:
 * {
 *     "id": 22,
 *     "name": "Foo",
 *     "age": 100
 * }
 *
 * viewModel.bind('Hello {name}, you are {age} years old', function(v) {
 *     console.log(v);
 * });
 * </pre>
 * <p><i>Record Properties</i></p>
 * <p>It is possible to bind to the certain state properties of a record. The available options are:</p>
 * <ul>
 * <li><code>→ext.data.Model.dirty</code></li>
 * <li><code>→ext.data.Model.phantom</code></li>
 * <li><code>valid (→ext.data.Model.isValid())</code></li>
 * </ul>
 * <p>Example usage:</p>
 * <pre>
 * Ext.define('MyApp.model.User', {
 *     extend: 'Ext.data.Model',
 *     fields: [{
 *         name: 'name',
 *         validators: 'presence'
 *     }, {
 *         name: 'age',
 *         validators: {
 *            type: 'range',
 *             min: 0
 *          }
 *     }]
 * });
 *
 * var rec = new MyApp.model.User();
 *
 * var viewModel = new Ext.app.ViewModel({
 *     data: {
 *         theUser: rec
 *     }
 * });
 *
 * viewModel.bind({
 *     dirty: '{theUser.dirty}',
 *     phantom: '{theUser.phantom}',
 *     valid: '{theUser.valid}'
 * }, function(v) {
 *     console.log(v.dirty, v.valid);
 * });
 *
 * rec.set('name', 'Foo');
 * viewModel.notify(); // dirty, not valid
 * rec.set('age', 20);
 * viewModel.notify(); // dirty, valid
 * rec.reject();
 * viewModel.notify(); // not dirty, not valid
 * </pre>
 * <p><i>Advanced Record Binding</i></p>
 * <p>For accessing other record information that is not exposed by the binding API, formulas
 * can be used to achieve more advanced operations:</p>
 * <pre>
 * Ext.define('MyApp.model.User', {
 *     extend: 'Ext.data.Model',
 *     fields: ['name', 'age']
 * });
 *
 * var rec = new MyApp.model.User();
 *
 * var viewModel = new Ext.app.ViewModel({
 *     formulas: {
 *         isNameModified: {
 *             bind: {
 *                 bindTo: '{theUser}',
 *                 deep: true
 *             },
 *             get: function(rec) {
 *                 return rec.isModified('name');
 *             }
 *         }
 *     },
 *     data: {
 *         theUser: rec
 *     }
 * });
 *
 * viewModel.bind('{isNameModified}', function(modified) {
 *     console.log(modified);
 * });
 * rec.set('name', 'Foo');
 * </pre>
 * <p><i>Associations</i></p>
 * <p>In the same way as fields, the bindings can also traverse associations in a bind statement.
 * The <code>ViewModel</code> will handle the asynchronous loading of data and only present the value once
 * the full path has been loaded. For more information on associations see
 * →<code>ext.data.schema.OneToOne</code> and →<code>ext.data.schema.ManyToOne</code>
 * associations.</p>
 * <pre>
 * Ext.define('User', {
 *     extend: 'Ext.data.Model',
 *     fields: ['name']
 * });
 *
 * Ext.define('Order', {
 *     extend: 'Ext.data.Model',
 *     fields: ['date', {
 *         name: 'userId',
 *         reference: 'User'
 *     }]
 * });
 *
 * Ext.define('OrderItem', {
 *     extend: 'Ext.data.Model',
 *     fields: ['price', 'qty', {
 *         name: 'orderId',
 *         reference: 'Order'
 *     }]
 * });
 *
 * var viewModel = new Ext.app.ViewModel({
 *     links: {
 *         orderItem: {
 *             type: 'OrderItem',
 *             id: 13
 *         }
 *     }
 * });
 * // The viewmodel will handle both ways of loading the data:
 * // a) If the data is loaded inline in a nested fashion it will
 * //    not make requests for extra data
 * // b) Only loading a single model at a time. So the Order will be loaded once
 * //    the OrderItem returns. The User will be loaded once the Order loads.
 * viewModel.bind('{orderItem.order.user.name}', function(name) {
 *     console.log(name);
 * });
 * </pre>
 * <p><i>Stores</i></p>
 * <p>Stores can be created as part of the <code>ViewModel</code> definition. The definitions are processed
 * like bindings which allows for very powerful dynamic functionality.</p>
 * <p>It is important to ensure that you name viewModel's data keys uniquely. If data is not named
 * uniquely, binds and formulas may receive information from an unintended data source.
 * This applies to keys in the viewModel's data block, stores, and links configs.</p>
 * <pre>
 * var viewModel = new Ext.app.ViewModel({
 *     stores: {
 *         users: {
 *             model: 'User',
 *             autoLoad: true,
 *             filters: [{
 *                 property: 'createdDate',
 *                 value: '{createdFilter}',
 *                 operator: '&gt;'
 *             }]
 *         }
 *     }
 * });
 * // Later on in our code, we set the date so that the store is created.
 * viewModel.set('createdFilter', Ext.Date.subtract(new Date(), Ext.Date.DAY, 7));
 * </pre>
 * <p>See →<code>stores</code> for more detail.</p>
 * <p><i>Store Properties</i></p>
 * <p>It is possible to bind to the certain state properties of the store. The available options are:</p>
 * <ul>
 * <li><code>count (→ext.data.Store.getCount())</code></li>
 * <li><code>→ext.data.Store.first()</code></li>
 * <li><code>→ext.data.Store.last()</code></li>
 * <li><code>loading (→ext.data.Store.hasPendingLoad())</code></li>
 * <li><code>totalCount (→ext.data.Store.getTotalCount())</code></li>
 * </ul>
 * <p>Example:</p>
 * <pre>
 * Ext.define('MyApp.model.User', {
 *     extend: 'Ext.data.Model',
 *     fields: ['name']
 * });
 *
 * var viewModel = new Ext.app.ViewModel({
 *     stores: {
 *         users: {
 *             model: 'MyApp.model.User',
 *             data: [{
 *                 name: 'Foo'
 *             }, {
 *                 name: 'Bar'
 *             }]
 *         }
 *     }
 * });
 *
 * viewModel.bind('{users.first}', function(first) {
 *     console.log(first ? first.get('name') : 'Nobody');
 * });
 *
 * var timer = Ext.interval(function() {
 *     var store = viewModel.getStore('users');
 *     if (store.getCount()) {
 *         store.removeAt(0);
 *     } else {
 *         Ext.uninterval(timer);
 *     }
 * }, 100);
 * </pre>
 * <p><i>Advanced Store Binding</i></p>
 * <p>For accessing other store information that is not exposed by the binding API, formulas
 * can be used to achieve more advanced operations:</p>
 * <pre>
 * Ext.define('MyApp.model.User', {
 *     extend: 'Ext.data.Model',
 *     fields: ['name', 'score']
 * });
 *
 * var viewModel = new Ext.app.ViewModel({
 *     stores: {
 *         users: {
 *             model: 'MyApp.model.User',
 *             data: [{
 *                 name: 'Foo',
 *                 score: 100
 *             }, {
 *                 name: 'Bar',
 *                 score: 350
 *             }]
 *         }
 *     },
 *     formulas: {
 *         totalScore: {
 *             bind: {
 *                 bindTo: '{users}',
 *                 deep: true
 *             },
 *             get: function(store) {
 *                 return store.sum('score');
 *             }
 *         }
 *     }
 * });
 *
 * viewModel.bind('{totalScore}', function(score) {
 *     console.log(score);
 * });
 *
 * viewModel.notify();
 * viewModel.getStore('users').removeAll();
 * </pre>
 * <p><i>Formulas</i></p>
 * <p>Formulas allow for calculated <code>ViewModel</code> data values. The dependencies for these formulas
 * are automatically determined so that the formula will not be processed until the required
 * data is present.</p>
 * <pre>
 * var viewModel = new Ext.app.ViewModel({
 *     formulas: {
 *         fullName: function(get) {
 *             return get('firstName') + ' ' + get('lastName');
 *         }
 *     },
 *     data: {firstName: 'John', lastName: 'Smith'}
 * });
 *
 * viewModel.bind('{fullName}', function(v) {
 *     console.log(v);
 * });
 * </pre>
 * <p>See →<code>formulas</code> for more detail.</p>
 * <p><i>Inheriting Data With Nesting</i></p>
 * <p>ViewModels can have a →<code>parent</code> which allows values to be consumed from
 * a shared base. These values that are available from the →<code>parent</code> are not copied,
 * rather they are "inherited" in a similar fashion to a javascript closure scope chain.
 * This is demonstrated in the example below:</p>
 * <pre>
 * var parent = new Ext.app.ViewModel({
 *     data: {
 *         foo: 3
 *     }
 * });
 * var child = new Ext.app.ViewModel({
 *     parent: parent
 * });
 * </pre>
 * <p>This is analogous to the following javascript closure:</p>
 * <pre>
 * var foo = 3;
 * Ext.Ajax.request({
 *     success: function() {
 *         // foo is available here
 *     }
 * });
 * </pre>
 * <p><i>Climbing/Inheriting</i></p>
 * <p>In line with the above, the default behaviour when setting the value of a child ViewModel
 * (either) through →<code>set()</code> or →<code>ext.app.bind.Binding.setValue()</code> is to climb to
 * where the value is "owned" and set the value there:</p>
 * <pre>
 * var parent = new Ext.app.ViewModel({
 *     data: {
 *         foo: 3
 *     }
 * });
 * var child = new Ext.app.ViewModel({
 *     parent: parent
 * });
 *
 * child.set('foo', 100); // Climbs to set the value on parent
 * console.log(parent.get('foo')); // 100
 * parent.set('foo', 200);
 * console.log(child.get('foo')); // 200, inherited from the parent
 * </pre>
 * <p>Any subsequent sets are also inherited in the same fashion. The inheriting/climbing behavior
 * occurs for any arbitrary depth, climbing/inherting can owned by a parent at any level above.</p>
 * <pre>
 * function log() {
 *     console.log([a, b, c, d, e].map(function(vm) {
 *         return vm.get('foo');
 *     }));
 * }
 *
 * var a = new Ext.app.ViewModel({data: {foo: 3}}),
 *     b = new Ext.app.ViewModel({parent: a}),
 *     c = new Ext.app.ViewModel({parent: b}),
 *     d = new Ext.app.ViewModel({parent: c}),
 *     e = new Ext.app.ViewModel({parent: d});
 *
 * log(); // [3, 3, 3, 3, 3]
 *
 * e.set('foo', 100);
 * log(); // [100, 100, 100, 100, 100]
 * </pre>
 * <p>This same climbing behavior applies when setting a value on a binding. The climbing begins from
 * the ViewModel where the binding was attached:</p>
 * <pre>
 * function log() {
 *     console.log([a, b, c].map(function(vm) {
 *         return vm.get('foo');
 *     }));
 * }
 *
 * var a = new Ext.app.ViewModel({data: {foo: 3}}),
 *     b = new Ext.app.ViewModel({parent: a}),
 *     c = new Ext.app.ViewModel({parent: b});
 *
 * var bind = c.bind('{foo}', function() {});
 *
 * bind.setValue(100);
 * log(); // [100, 100, 100]
 * </pre>
 * <p>The exception to this rule is when there is nothing above to climb to. If a value is set and
 * there is no parent above to hold it, then the value is set where it was called:</p>
 * <pre>
 * function log() {
 *     console.log([a, b, c].map(function(vm) {
 *         return vm.get('foo');
 *     }));
 * }
 *
 * var a = new Ext.app.ViewModel(),
 *     b = new Ext.app.ViewModel({parent: a}),
 *     c = new Ext.app.ViewModel({parent: b});
 *
 * c.set('foo', 3);
 * log(); // [null, null, 3]
 *
 * b.set('foo', 2);
 * log(); // [null, 2, 3]
 *
 * a.set('foo', 1);
 * log(); // [1, 2, 3]
 * </pre>
 * <p>These values are called local values, which are discussed below.</p>
 * <p><i>Local Values</i></p>
 * <p>If the child ViewModel is declared with top level data that also exists in the parent, then that
 * child is considered to own that local value, so no value is inherited from the parent, nor does
 * the climbing behaviour occur.</p>
 * <pre>
 * var parent = new Ext.app.ViewModel({
 *     data: {
 *         foo: 3
 *     }
 * });
 * var child = new Ext.app.ViewModel({
 *     parent: parent,
 *     data: {
 *         foo: 5
 *     }
 * });
 *
 * console.log(parent.get('foo'), child.get('foo')); // 3, 5
 * child.set('foo', 100);
 * console.log(parent.get('foo'), child.get('foo')); // 3, 100
 * parent.set('foo', 200);
 * console.log(parent.get('foo'), child.get('foo')); // 200, 100
 * </pre>
 * <p>The inheriting/climbing behavior is limited to local values:</p>
 * <pre>
 * function log() {
 *     console.log([a, b, c, d, e].map(function(vm) {
 *         return vm.get('foo');
 *     }));
 * }
 *
 * var a = new Ext.app.ViewModel({data: {foo: 1}}),
 *     b = new Ext.app.ViewModel({parent: a}),
 *     c = new Ext.app.ViewModel({parent: b, data: {foo: 2}}),
 *     d = new Ext.app.ViewModel({parent: c}),
 *     e = new Ext.app.ViewModel({parent: d, data: {foo: 3}});
 *
 * log(); // [1, 1, 2, 2, 3]
 *
 * e.set('foo', 100);
 * log(); // [1, 1, 2, 2, 100]
 *
 * d.set('foo', 200);
 * log(); // [1, 1, 200, 200, 100]
 *
 * c.set('foo', 201);
 * log(); // [1, 1, 201, 201, 100]
 *
 * b.set('foo', 300);
 * log(); // [300, 300, 201, 201, 100]
 *
 * a.set('foo', 301);
 * log(); // [301, 301, 201, 201, 100]
 * </pre>
 * <p><i>Attaching/Clearing Local Values Dynamically</i></p>
 * <p>To bypass the climbing behaviour and push a value into a particular point
 * in the hierarchy, the →<code>setData()</code> method should be used. Once a local value
 * is set, it will be used as such in the future.</p>
 * <pre>
 * function log() {
 *     console.log([a, b, c, d, e].map(function(vm) {
 *         return vm.get('foo');
 *     }));
 * }
 *
 * var a = new Ext.app.ViewModel({data: {foo: 3}}),
 *     b = new Ext.app.ViewModel({parent: a}),
 *     c = new Ext.app.ViewModel({parent: b}),
 *     d = new Ext.app.ViewModel({parent: c}),
 *     e = new Ext.app.ViewModel({parent: d});
 *
 * log(); // [3, 3, 3, 3, 3]
 *
 * c.setData({
 *     foo: 100
 * });
 *
 * log(); // [3, 3, 100, 100, 100]
 *
 * d.set('foo', 200); // Climbs to new local value
 * log(); // [3, 3, 200, 200, 200]
 * </pre>
 * <p>Similarly, data can be cleared from being a local value by setting the value to undefined:</p>
 * <pre>
 * function log() {
 *     console.log([a, b, c, d].map(function(vm) {
 *         return vm.get('foo');
 *     }));
 * }
 *
 * var a = new Ext.app.ViewModel({data: {foo: 3}}),
 *     b = new Ext.app.ViewModel({parent: a}),
 *     c = new Ext.app.ViewModel({parent: b, data: {foo: 100}}),
 *     d = new Ext.app.ViewModel({parent: c});
 *
 * log(); // [3, 3, 100, 100]
 *
 * c.setData({
 *     foo: undefined
 * });
 * log([3, 3, 3, 3]);
 * </pre>
 * @see #bind()
 * @see ext.app.bind.BaseBinding
 * @see #destroy()
 * @see ext.app.bind.Binding
 * @see ext.app.bind.Multi
 * @see ext.app.bind.TemplateBinding
 * @see ext.Template
 * @see null
 * @see ext.Component
 * @see #scheduler
 * @see #notify()
 * @see ext.data.Session
 * @see ext.data.Model
 * @see ext.data.Store
 * @see #links
 * @see ext.data.Model#dirty
 * @see ext.data.Model#phantom
 * @see ext.data.Model#isValid()
 * @see ext.data.schema.OneToOne
 * @see ext.data.schema.ManyToOne
 * @see #stores
 * @see ext.data.Store#getCount()
 * @see ext.data.Store#first()
 * @see ext.data.Store#last()
 * @see ext.data.Store#hasPendingLoad()
 * @see ext.data.Store#getTotalCount()
 * @see #formulas
 * @see #parent
 * @see #set()
 * @see ext.app.bind.Binding#setValue()
 * @see #setData()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html Original Ext JS documentation of 'Ext.app.ViewModel'
 */
public class ViewModel extends Base implements IFactoryable, IIdentifiable {
  /**
   * @param config @inheritDoc
   */
  public function ViewModel(config:ViewModel = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * This object holds the arbitrary data that populates the <code>ViewModel</code> and is
   * then available for binding.
   * @default true
   * @since 5.0.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#cfg-data Original Ext JS documentation of 'data'
   * @see #getData()
   * @see #setData()
   */
  public native function get data():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set data(value:Object):void;

  /**
   * @inheritDoc
   */
  public native function get factoryConfig():Object;

  /**
   * @inheritDoc
   */
  public native function set factoryConfig(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * An object that defines named values whose value is managed by function calls.
   * The names of the properties of this object are assigned as values in the
   * ViewModel.
   * <p>For example:</p>
   * <pre>
   *  formulas: {
   *      xy: function (get) { return get('x') &#42; get('y'); }
   *  }
   * </pre>
   * <p>For more details about defining a formula, see <code>→ext.app.bind.Formula</code>.</p>
   * @since 5.0.0
   * @see ext.app.bind.Formula
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#cfg-formulas Original Ext JS documentation of 'formulas'
   * @see #getFormulas()
   * @see #setFormulas()
   */
  public native function get formulas():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set formulas(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * Links provide a way to assign a simple name to a more complex bind. The primary
   * use for this is to assign names to records in the data model.
   * <pre>
   *  links: {
   *      theUser: {
   *          type: 'User',
   *          id: 12
   *      }
   *  }
   * </pre>
   * <p>It is also possible to force a new phantom record to be created by not specifying an
   * id but passing <code>create: true</code> as part of the descriptor. This is often useful when
   * creating a new record for a child session.</p>
   * <pre>
   * links: {
   *     newUser: {
   *         type: 'User',
   *         create: true
   *     }
   * }
   * </pre>
   * <p><code>→create()</code> can also be an object containing initial data for the record.</p>
   * <pre>
   * links: {
   *     newUser: {
   *         type: 'User',
   *         create: {
   *             firstName: 'John',
   *             lastName: 'Smith'
   *         }
   *     }
   * }
   * </pre>
   * <p>While that is the typical use, the value of each property in <code>→links</code> may also be
   * a bind descriptor (see <code>→bind()</code> for the various forms of bind
   * descriptors).</p>
   * @since 5.0.0
   * @see #create()
   * @see #links
   * @see #bind()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#cfg-links Original Ext JS documentation of 'links'
   * @see #getLinks()
   * @see #setLinks()
   */
  public native function get links():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set links(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * The parent <code>ViewModel</code> of this <code>ViewModel</code>. Once set, this cannot be changed.
   * @since 5.0.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#cfg-parent Original Ext JS documentation of 'parent'
   * @see #getParent()
   */
  public native function get parent():ViewModel;

  [ExtConfig]
  [Bindable]
  /**
   * The schema to use for getting information about entities.
   * @default 'default'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#cfg-schema Original Ext JS documentation of 'schema'
   * @see #getSchema()
   * @see #setSchema()
   */
  public native function get schema():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set schema(value:*):void;

  [ExtConfig]
  [Bindable]
  /**
   * The session used to manage the data model (records and stores).
   * @since 5.0.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#cfg-session Original Ext JS documentation of 'session'
   * @see #getSession()
   * @see #setSession()
   */
  public native function get session():Session;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set session(value:Session):void;

  [ExtConfig]
  [Bindable]
  /**
   * A declaration of <code>→ext.data.Store</code> configurations that are first processed as
   * binds to produce an effective store configuration.
   * <p>A simple store definition. We can reference this in our bind statements using the
   * <code>{users}</code> as we would with other data values.</p>
   * <pre>
   * new Ext.app.ViewModel({
   *     stores: {
   *         users: {
   *             model: 'User',
   *             autoLoad: true
   *         }
   *     }
   * });
   * </pre>
   * <p>This store definition contains a dynamic binding. The store will not be created until
   * the initial value for groupId is set. Once that occurs, the store is created with the
   * appropriate filter configuration. Subsequently, once we change the group value, the old
   * filter will be overwritten with the new value.</p>
   * <pre>
   * var viewModel = new Ext.app.ViewModel({
   *     stores: {
   *         users: {
   *             model: 'User',
   *             filters: [{
   *                 property: 'groupId',
   *                 value: '{groupId}'
   *             }]
   *         }
   *     }
   * });
   * viewModel.set('groupId', 1); // This will trigger the store creation with the filter.
   * viewModel.set('groupId', 2); // The filter value will be changed.
   * </pre>
   * <p>This store uses <i>store chaining</i> (→<code>ext.data.ChainedStore</code>) to create a store backed by
   * the data in another store. By specifying a string as the store, it will bind our creation
   * and backing to the other store. This functionality is especially useful when wanting to
   * display a different "view" of a store, for example a different sort order or different
   * filters.</p>
   * <pre>
   * var viewModel = new Ext.app.ViewModel({
   *     stores: {
   *         allUsers: {
   *             model: 'User',
   *             autoLoad: true
   *         },
   *         children: {
   *             source: '{allUsers}',
   *             filters: [{
   *                 property: 'age',
   *                 value: 18,
   *                 operator: '&lt;'
   *             }]
   *         }
   *     }
   * });
   * </pre>
   * @since 5.0.0
   * @see ext.data.Store
   * @see ext.data.ChainedStore
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#cfg-stores Original Ext JS documentation of 'stores'
   * @see #getStores()
   * @see #setStores()
   */
  public native function get stores():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set stores(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * The Container that owns this <code>ViewModel</code> instance.
   * @since 5.0.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#cfg-view Original Ext JS documentation of 'view'
   * @see #getView()
   * @see #setView()
   */
  public native function get view():Container;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set view(value:Container):void;

  /**
   * This method requests that data in this <code>ViewModel</code> be delivered to the specified
   * <code>callback</code>. The data desired is given in a "bind descriptor" which is the first
   * argument.
   * <p>A simple call might look like this:</p>
   * <pre>
   * var binding = vm.bind('{foo}', this.onFoo, this);
   *
   * binding.destroy();  // when done with the binding
   * </pre>
   * <p>Options for the binding can be provided in the last argument:</p>
   * <pre>
   * var binding = vm.bind('{foo}', this.onFoo, this, {
   *     deep: true
   * });
   * </pre>
   * <p>Alternatively, bind options can be combined with the bind descriptor using only
   * the first argument:</p>
   * <pre>
   * var binding = vm.bind({
   *     bindTo: '{foo}',  // the presence of bindTo identifies this form
   *     deep: true
   * }, this.onFoo, this);
   * </pre>
   * <p>See the class documentation for more details on Bind Descriptors and options.</p>
   * @param descriptor The bind descriptor. See class description
   * for details.
   * @param callback The function to call with the value of the bound property.
   * @param scope The scope (<code>this</code> pointer) for the <code>callback</code>.
   * @param options Additional options to configure the
   * <i>binding</i> (→<code>ext.app.bind.Binding</code>). If this parameter is provided, the <code>bindTo</code> form
   * of combining options and bind descriptor is not recognized.
   * @return The binding.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#method-bind Original Ext JS documentation of 'bind'
   * @see ext.app.bind.Binding
   */
  public native function bind(descriptor:*, callback:Function, scope:Object = null, options:Object = null):*;

  /**
   * Get a value from the data for this viewmodel.
   * @param path The path of the data to retrieve.
   * <pre>
   * var value = vm.get('theUser.address.city');
   * </pre>
   * @return The data stored at the passed path.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#method-get Original Ext JS documentation of 'get'
   */
  public native function get(path:String):*;

  /**
   * Returns the value of <code>data</code>.
   * @see #data
   */
  public native function getData():Object;

  /**
   * Returns the value of <code>formulas</code>.
   * @see #formulas
   */
  public native function getFormulas():Object;

  /**
   * @inheritDoc
   */
  public native function getId():String;

  /**
   * Returns the value of <code>links</code>.
   * @see #links
   */
  public native function getLinks():Object;

  /**
   * Returns the value of <code>parent</code>.
   * @see #parent
   */
  public native function getParent():ViewModel;

  /**
   * Returns the value of <code>schema</code>.
   * @see #schema
   */
  public native function getSchema():*;

  /**
   * Gets the session attached to this (or a parent) ViewModel. See the →<code>session</code>
   * configuration.
   * @return The session. <code>null</code> if no session exists.
   * @see #session
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#method-getSession Original Ext JS documentation of 'getSession'
   */
  public native function getSession():Session;

  /**
   * Gets a store configured via the →<code>stores</code> configuration.
   * @param key The name of the store.
   * @return The store. <code>null</code> if no store exists.
   * @see #stores
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#method-getStore Original Ext JS documentation of 'getStore'
   */
  public native function getStore(key:String):Store;

  /**
   * Returns the value of <code>stores</code>.
   * @see #stores
   */
  public native function getStores():Object;

  /**
   * Returns the value of <code>view</code>.
   * @see #view
   */
  public native function getView():Container;

  /**
   * Create a link to a reference. See the →<code>links</code> configuration.
   * @param key The name for the link.
   * @param reference The reference descriptor.
   * @see #links
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#method-linkTo Original Ext JS documentation of 'linkTo'
   */
  public native function linkTo(key:String, reference:Object):void;

  /**
   * Forces all bindings in this ViewModel hierarchy to evaluate immediately. Use this to do
   * a synchronous flush of all bindings.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#method-notify Original Ext JS documentation of 'notify'
   */
  public native function notify():void;

  /**
   * Set a value in the data for this viewmodel. This method will climb to set data on
   * a parent view model if appropriate. See "Inheriting Data" in the class introduction for
   * more information.
   * @param path The path of the value to set, or an object literal to set
   * at the root of the viewmodel.
   * @param value The data to set at the value. If the value is an object literal,
   * any required paths will be created.
   * <pre>
   * // Set a single property at the root level
   * viewModel.set('expiry', Ext.Date.add(new Date(), Ext.Date.DAY, 7));
   * console.log(viewModel.get('expiry'));
   * // Sets a single property in user.address, does not overwrite any hierarchy.
   * viewModel.set('user.address.city', 'London');
   * console.log(viewModel.get('user.address.city'));
   * // Sets 2 properties of "user". Overwrites any existing hierarchy.
   * viewModel.set('user', {firstName: 'Foo', lastName: 'Bar'});
   * console.log(viewModel.get('user.firstName'));
   * // Sets a single property at the root level. Overwrites any existing hierarchy.
   * viewModel.set({rootKey: 1});
   * console.log(viewModel.get('rootKey'));
   * </pre>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#method-set Original Ext JS documentation of 'set'
   */
  public native function set(path:*, value:*):void;

  /**
   * Sets data directly at the level of this viewmodel. This method does not climb
   * to set data on parent view models. Passing <code>undefined</code> will clear the value
   * in this viewmodel, which means that this viewmodel is free to inherit data
   * from a parent. See "Inheriting Data" in the class introduction for more information.
   * @param data The new data to set.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.ViewModel.html#method-setData Original Ext JS documentation of 'setData'
   */
  public native function setData(data:Object):void;

  /**
   * Sets the value of <code>formulas</code>.
   * @param formulas The new value.
   * @see #formulas
   */
  public native function setFormulas(formulas:Object):void;

  /**
   * Sets the value of <code>links</code>.
   * @param links The new value.
   * @see #links
   */
  public native function setLinks(links:Object):void;

  /**
   * Sets the value of <code>schema</code>.
   * @param schema The new value.
   * @see #schema
   */
  public native function setSchema(schema:*):void;

  /**
   * Sets the value of <code>session</code>.
   * @param session The new value.
   * @see #session
   */
  public native function setSession(session:Session):void;

  /**
   * Sets the value of <code>stores</code>.
   * @param stores The new value.
   * @see #stores
   */
  public native function setStores(stores:Object):void;

  /**
   * Sets the value of <code>view</code>.
   * @param view The new value.
   * @see #view
   */
  public native function setView(view:Container):void;
}
}