package ext.app.bind {
import ext.util.Schedulable;

[Native("Ext.app.bind.Formula", require)]
/**
 * This class manages a formula defined for an <code>→ext.app.ViewModel</code>.
 * <p><b><i>Formula Basics</i></b></p>
 * <p>Formulas in a <code>ViewModel</code> can be defined as simply as just a function:</p>
 * <pre>
 *  formulas: {
 *      xy: function (get) { return get('x') &#42; get('y'); }
 *  }
 * </pre>
 * <p>When you need to be more explicit, "xy" can become an object. The following means the
 * same thing as above:</p>
 * <pre>
 *  formulas: {
 *      xy: {
 *          get: function (get) { return get('x') &#42; get('y'); }
 *      }
 *  }
 * </pre>
 * <p><i>Data Dependencies</i></p>
 * <p>One of the important aspects of a <code>ViewModel</code> is notification of change. In order to
 * manage this, a <code>ViewModel</code> <i>must</i> know the dependencies between data. In the above case
 * this is accomplished by <b>parsing the text of the function</b>. While this is convenient
 * and reduces the maintenance/risk that would come from explicitly listing dependencies
 * separately, there are some rules to be aware of:</p>
 * <ul>
 * <li>All dependencies are resolved by matching the binding statements in the getter function.</li>
 * <li>If you need to use these values in other ways, cache them as a <code>var</code> (following
 * the first rule to capture the value) and use that <code>var</code>.</li>
 * </ul>
 * <p>In the above formulas, the "xy" formula depends on "x" and "y" in the <code>ViewModel</code>. As
 * these values change, the formula is called to produce the correct value for "xy". This
 * in turn can be used by other formulas. For example:</p>
 * <pre>
 *  formulas: {
 *      xy: function (get) {  // "get" is arbitrary but a good convention
 *          return get('x') &#42; get('y');
 *      },
 *
 *      xyz: function (get) {
 *          return get('xy') &#42; get('z');
 *      }
 *  }
 * </pre>
 * <p>In the above, "xyz" depends on "xy" and "z" values in the <code>ViewModel</code>.</p>
 * <p><i>The Getter Method</i></p>
 * <p>The argument passed to the formula is a function that allows you to retrieve
 * the matched bind statements.</p>
 * <pre>
 *  formulas: {
 *      foo: function (get) {
 *          return get('theUser.address.city');
 *      }
 *  }
 * </pre>
 * <p>In the above, the dependency is resolved to <code>theUser.address.city</code>. The formula will not
 * be triggered until the value for <code>city</code> is present.</p>
 * <p><i>Capturing Values</i></p>
 * <p>If values need to be used repeatedly, you can use a <code>var</code> as long as the Rules are not
 * broken.</p>
 * <pre>
 *  formulas: {
 *      x2y2: function (get) {
 *          // These are still "visible" as "get('x')" and "get('y')" so this is OK:
 *          var x = get('x'),
 *              y = get('y');
 *
 *          return x &#42; x &#42; y &#42; y;
 *      }
 *  }
 * </pre>
 * <p><b><i>Explicit Binding</i></b></p>
 * <p>While function parsing is convenient, there are times it is not the best solution. In
 * these cases, an explicit <code>→bind</code> can be given. To revisit the previous example with an
 * explicit binding:</p>
 * <pre>
 *  formulas: {
 *      zip: {
 *          bind: '{foo.bar.zip}',
 *
 *          get: function (zip) {
 *              // NOTE: the only thing we get is what our bind produces.
 *              return zip &#42; 2;
 *          }
 *      }
 *  }
 * </pre>
 * <p>In this case we have given the formula an explicit <code>→bind</code> value so it will no longer
 * parse the <code>→get</code> function. Instead, it will call <code>→ext.app.ViewModel.bind()</code> with
 * the value of the <code>→bind</code> property and pass the produced value to <code>→get</code> whenever it
 * changes.</p>
 * <p><b><i>Settable Formulas</i></b></p>
 * <p>When a formula is "reversible" it can be given a <code>→set</code> method to allow it to participate
 * in two-way binding. For example:</p>
 * <pre>
 *  formulas: {
 *         fullName: {
 *             get: function (get) {
 *                 var ret = get('firstName') || '';
 *
 *                 if (get('lastName')) {
 *                     ret += ' ' +  get('lastName');
 *                 }
 *
 *                 return ret;
 *             },
 *
 *             set: function (value) {
 *                 var space = value.indexOf(' '),
 *                     split = (space &lt; 0) ? value.length : space;
 *
 *                 this.set({
 *                     firstName: value.substring(0, split),
 *                     lastName: value.substring(split + 1)
 *                 });
 *             }
 *         }
 *     }
 * </pre>
 * <p>When the <code>→set</code> method is called the <code>this</code> reference is the <code>→ext.app.ViewModel</code> so it
 * just calls its <code>set method (→ext.app.ViewModel.set())</code>.</p>
 * <p><b><i>Single Run Formulas</i></b></p>
 * <p>If a formula only needs to produce an initial value, it can be marked as <code>→single</code>.</p>
 * <pre>
 *  formulas: {
 *      xy: {
 *          single: true,
 *
 *          get: function (get) {
 *              return get('x') &#42; get('y');
 *          }
 *      }
 *  }
 * </pre>
 * <p>This formulas <code>→get</code> method will be called with <code>x</code> and <code>y</code> once and then its binding
 * to these properties will be destroyed. This means the <code>→get</code> method (and hence the value
 * of <code>xy</code>) will only be executed/calculated once.</p>
 * @see ext.app.ViewModel
 * @see #bind
 * @see #get
 * @see ext.app.ViewModel#bind()
 * @see #set
 * @see ext.app.ViewModel#set()
 * @see #single
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.bind.Formula.html Original Ext JS documentation of 'Ext.app.bind.Formula'
 */
public class Formula extends Schedulable {
  /**
   * @param config @inheritDoc
   */
  public function Formula(config:Formula = null) {
    super();
  }

  [ExtConfig]
  /**
   * An explicit bind request to produce data to provide the <code>→get</code> function. If this is
   * specified, the result of this bind is the first argument to <code>→get</code>. If not given,
   * then <code>→get</code> receives a getter function that can retrieve bind expressions. For details
   * on what can be specified for this property see <code>→ext.app.ViewModel.bind()</code>.
   * @since 5.0.0
   * @see #get
   * @see ext.app.ViewModel#bind()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.bind.Formula.html#cfg-bind Original Ext JS documentation of 'bind'
   */
  public native function get bind():Object;

  [ExtConfig]
  /**
   * @private
   */
  public native function set bind(value:Object):void;

  [ExtConfig]
  /**
   * The function to call to calculate the formula's value. The <code>→get</code> method executes
   * with a <code>this</code> pointer of the <code>ViewModel</code> and receives a getter function or the result of
   * a configured <code>→bind</code>.
   * @since 5.0.0
   * @see #get
   * @see #bind
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.bind.Formula.html#cfg-get Original Ext JS documentation of 'get'
   */
  public native function get get():Function;

  [ExtConfig]
  /**
   * @private
   */
  public native function set get(value:Function):void;

  [ExtConfig]
  /**
   * If provided this method allows a formula to be set. This method is typically called
   * when <code>→ext.app.bind.Binding.setValue()</code> is called. The <code>→set</code> method executes
   * with a <code>this</code> pointer of the <code>ViewModel</code>. Whatever values need to be updated can
   * be set by calling <code>→ext.app.ViewModel.set()</code>.
   * @since 5.0.0
   * @see ext.app.bind.Binding#setValue()
   * @see #set
   * @see ext.app.ViewModel#set()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.bind.Formula.html#cfg-set Original Ext JS documentation of 'set'
   */
  public native function get set():Function;

  [ExtConfig]
  /**
   * @private
   */
  public native function set set(value:Function):void;

  [ExtConfig]
  /**
   * This option instructs the binding to call its <code>→destroy()</code> method immediately after
   * delivering the initial value.
   * @default false
   * @since 5.0.0
   * @see #destroy()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.app.bind.Formula.html#cfg-single Original Ext JS documentation of 'single'
   */
  public native function get single():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set single(value:Boolean):void;
}
}