package ext.container {
import ext.Component;

[Mixin("ext.container.DockingContainer")]
/**
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.container.DockingContainer.html Original Ext JS documentation of 'Ext.container.DockingContainer'
 * @see ext.container.DockingContainer
 */
public interface IDockingContainer {
  [ExtConfig] /**
   * This object holds the default weights applied to dockedItems that have no weight.
   * These start with a weight of 1, to allow negative weights to insert before top items
   * and are odd numbers so that even weights can be used to get between different dock orders.
   * <p>To make default docking order match border layout, do this:</p>
   * <pre>
   *  Ext.panel.Panel.prototype.defaultDockWeights = { top: 1, bottom: 3, left: 5, right: 7 };
   * </pre>
   * <p>Changing these defaults as above or individually on this object will effect all Panels.
   * To change the defaults on a single panel, you should replace the entire object:</p>
   * <pre>
   *  initComponent: function() {
   *      // NOTE: Don't change members of defaultDockWeights since the object is shared.
   *      this.defaultDockWeights = { top: 1, bottom: 3, left: 5, right: 7 };
   *
   *      this.callParent();
   *  }
   * </pre>
   * <p>To change only one of the default values, you do this:</p>
   * <pre>
   *  initComponent: function() {
   *      // NOTE: Don't change members of defaultDockWeights since the object is shared.
   *      this.defaultDockWeights = Ext.applyIf({ top: 10 }, this.defaultDockWeights);
   *
   *      this.callParent();
   *  }
   * </pre>
   * @default {
   *     top: {
   *         render: 1,
   *         visual: 1
   *     },
   *     left: {
   *         render: 3,
   *         visual: 5
   *     },
   *     right: {
   *         render: 5,
   *         visual: 7
   *     },
   *     bottom: {
   *         render: 7,
   *         visual: 3
   *     }
   * }
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.container.DockingContainer.html#cfg-defaultDockWeights Original Ext JS documentation of 'defaultDockWeights'
   */
  function get defaultDockWeights():Object;

  [ExtConfig] function set defaultDockWeights(value:Object):void;

  [ArrayElementType("ext.Component")] /**
   * Adds docked item(s) to the container.
   * @param items The Component or array of components to add. The components
   * must include a 'dock' parameter on each component to indicate where it should be docked
   * ('top', 'right', 'bottom', 'left').
   * @param pos The index at which the Component will be added
   * @return The added components.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.container.DockingContainer.html#method-addDocked Original Ext JS documentation of 'addDocked'
   */
  function addDocked(items:*, pos:Number = NaN):Array;

  /**
   * Finds a docked component by id, itemId or position. Also see →<code>getDockedItems()</code>
   * @param comp The id, itemId or position of the docked component
   * (see →<code>ext.container.Container.getComponent()</code> for details)
   * @return The docked component (if found)
   * @see ext.container.DockingContainer#getDockedItems()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.container.DockingContainer.html#method-getDockedComponent Original Ext JS documentation of 'getDockedComponent'
   * @see ext.container.Container#getComponent()
   */
  function getDockedComponent(comp:*):Component;

  [ArrayElementType("ext.Component")] /**
   * Retrieves an array of all currently docked Components.
   * <p>For example to find a toolbar that has been docked at top:</p>
   * <pre>
   * panel.getDockedItems('toolbar[dock="top"]');
   * </pre>
   * @param selector A →<code>ext.ComponentQuery</code> selector string
   * to filter the returned items.
   * @param beforeBody An optional flag to limit the set of items to only those
   * before the body (true) or after the body (false). All components are returned by default.
   * @return The array of docked components meeting the specified criteria.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.container.DockingContainer.html#method-getDockedItems Original Ext JS documentation of 'getDockedItems'
   * @see ext.#ComponentQuery ext.ComponentQuery
   */
  function getDockedItems(selector:String, beforeBody:Boolean = undefined):Array;

  /**
   * Inserts docked item(s) to the panel at the indicated position.
   * @param pos The index at which the Component will be inserted
   * @param items The Component or array of components to add. The components
   * must include a 'dock' paramater on each component to indicate where it should be docked
   * ('top', 'right', 'bottom', 'left').
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.container.DockingContainer.html#method-insertDocked Original Ext JS documentation of 'insertDocked'
   */
  function insertDocked(pos:Number, items:*):void;

  /**
   * Removes the docked item from the panel.
   * @param item The Component to remove.
   * @param flags Pass <code>true</code> to destroy the item or an
   * object with multiple options. If not passed, →<code>autoDestroy</code> will
   * determine if the item is destroyed.
   * <ul>
   * <li><code>destroy:Boolean</code> (optional) —
   * Pass <code>true</code> to destroy the item.
   * </li>
   * <li><code>detach:Boolean</code> (optional) —
   * Pass <code>true</code> to put the item in the
   * <i>detachedBody element</i> (→<code>ext.Ext.getDetachedBody()</code>).
   * </li>
   * </ul>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.container.DockingContainer.html#method-removeDocked Original Ext JS documentation of 'removeDocked'
   * @see ext.container.DockingContainer#autoDestroy
   * @see ext.SExt#getDetachedBody() ext.Ext.getDetachedBody()
   */
  function removeDocked(item:Component, flags:* = undefined):void;
}
}