package ext.data {
import ext.Base;
import ext.ExtClass;
import ext.data.operation.DestroyOperation;
import ext.data.operation.ReadOperation;
import ext.data.proxy.DataProxy;
import ext.data.schema.Schema;

[Native("Ext.data.Model", require)]
/**
 * A Model or Entity represents some object that your application manages. For example, one
 * might define a Model for Users, Products, Cars, or other real-world object that we want
 * to model in the system. Models are used by <i>stores</i> (→<code>ext.data.Store</code>), which are in
 * turn used by many of the data-bound components in Ext.
 * <p><b>Fields</b></p>
 * <p>Models are defined as a set of fields and any arbitrary methods and properties relevant
 * to the model. For example:</p>
 * <pre>
 * Ext.define('User', {
 *     extend: 'Ext.data.Model',
 *     fields: [
 *         {name: 'name',  type: 'string'},
 *         {name: 'age',   type: 'int', convert: null},
 *         {name: 'phone', type: 'string'},
 *         {name: 'alive', type: 'boolean', defaultValue: true, convert: null}
 *     ],
 *
 *     changeName: function() {
 *         var oldName = this.get('name'),
 *             newName = oldName + " The Barbarian";
 *
 *         this.set('name', newName);
 *     }
 * });
 * </pre>
 * <p>Now we can create instances of our User model and call any model logic we defined:</p>
 * <pre>
 * var user = Ext.create('User', {
 *     id   : 'ABCD12345',
 *     name : 'Conan',
 *     age  : 24,
 *     phone: '555-555-5555'
 * });
 *
 * user.changeName();
 * user.get('name'); //returns "Conan The Barbarian"
 * </pre>
 * <p>By default, the built in field types such as number and boolean coerce string values
 * in the raw data by virtue of their →<code>ext.data.field.DataField.convert()</code> method.
 * When the server can be relied upon to send data in a format that does not need to be
 * converted, disabling this can improve performance. The →<code>ext.data.reader.JsonReader</code>
 * and →<code>ext.data.reader.ArrayReader</code> readers are likely candidates for this
 * optimization. To disable field conversions you simply specify <code>null</code> for the field's
 * <i>convert config</i> (→<code>ext.data.field.DataField.convert</code>).</p>
 * <p><b><i>The "id" Field and <code>idProperty (→idProperty)</code></i></b></p>
 * <p>A Model definition always has an <i>identifying field</i> which should yield a unique key
 * for each instance. By default, a field named "id" will be created with a
 * →<code>ext.data.field.DataField.mapping</code> of "id". This happens because of the default
 * →<code>idProperty</code> provided in Model definitions.</p>
 * <p>To alter which field is the identifying field, use the →<code>idProperty</code> config.</p>
 * <p><b>Validators</b></p>
 * <p>Models have built-in support for field validators. Validators are added to models as in
 * the follow example:</p>
 * <pre>
 * Ext.define('User', {
 *     extend: 'Ext.data.Model',
 *     fields: [
 *         { name: 'name',     type: 'string' },
 *         { name: 'age',      type: 'int' },
 *         { name: 'phone',    type: 'string' },
 *         { name: 'gender',   type: 'string' },
 *         { name: 'username', type: 'string' },
 *         { name: 'alive',    type: 'boolean', defaultValue: true }
 *     ],
 *
 *     validators: {
 *         age: 'presence',
 *         name: { type: 'length', min: 2 },
 *         gender: { type: 'inclusion', list: ['Male', 'Female'] },
 *         username: [
 *             { type: 'exclusion', list: ['Admin', 'Operator'] },
 *             { type: 'format', matcher: /([a-z]+)[0-9]{2,3}/i }
 *         ]
 *     }
 * });
 * </pre>
 * <p>The derived type of <code>→ext.data.field.DataField</code> can also provide validation. If <code>→validators</code>
 * need to be duplicated on multiple fields, instead consider creating a custom field type.</p>
 * <p><b><i>Validation</i></b></p>
 * <p>The results of the validators can be retrieved via the "associated" validation record:</p>
 * <pre>
 * var instance = Ext.create('User', {
 *     name: 'Ed',
 *     gender: 'Male',
 *     username: 'edspencer'
 * });
 *
 * var validation = instance.getValidation();
 * </pre>
 * <p>The returned object is an instance of <code>→ext.data.Validation</code> and has as its fields the
 * result of the field <code>→validators</code>. The validation object is "dirty" if there are one or
 * more validation errors present.</p>
 * <p>This record is also available when using data binding as a "pseudo-association" called
 * "validation". This pseudo-association can be hidden by an explicitly declared
 * association by the same name (for compatibility reasons), but doing so is not
 * recommended.</p>
 * <p>The <code>→ext.Component.modelValidation</code> config can be used to enable automatic
 * binding from the "validation" of a record to the form fields that may be bound to its
 * values.</p>
 * <p><b>Associations</b></p>
 * <p>Models often have associations with other Models. These associations can be defined by
 * fields (often called "foreign keys") or by other data such as a many-to-many (or "matrix").
 * See →<code>ext.data.schema.Association</code> for information about configuring and using associations.</p>
 * <p><b>Using a Proxy</b></p>
 * <p>Models are great for representing types of data and relationships, but sooner or later we're
 * going to want to load or save that data somewhere. All loading and saving of data is handled
 * via a →<code>ext.data.proxy.DataProxy</code>, which can be set directly on the Model:</p>
 * <pre>
 * Ext.define('User', {
 *     extend: 'Ext.data.Model',
 *     fields: ['id', 'name', 'email'],
 *
 *     proxy: {
 *         type: 'rest',
 *         url : '/users'
 *     }
 * });
 * </pre>
 * <p>Here we've set up a <i>Rest Proxy</i> (→<code>ext.data.proxy.RestProxy</code>), which knows how to load and save
 * data to and from a RESTful backend. Let's see how this works:</p>
 * <pre>
 * var user = Ext.create('User', {name: 'Ed Spencer', email: 'ed&#64;sencha.com'});
 *
 * user.save(); //POST /users
 * </pre>
 * <p>Calling →<code>save()</code> on the new Model instance tells the configured RestProxy that we wish to
 * persist this Model's data onto our server. RestProxy figures out that this Model hasn't been
 * saved before because it doesn't have an id, and performs the appropriate action - in this case
 * issuing a POST request to the url we configured (/users). We configure any Proxy on any Model
 * and always follow this API - see →<code>ext.data.proxy.DataProxy</code> for a full list.</p>
 * <p>Loading data via the Proxy is accomplished with the static <code>load</code> method:</p>
 * <pre>
 * // Uses the configured RestProxy to make a GET request to /users/123
 * User.load(123, {
 *     success: function(user) {
 *         console.log(user.getId()); //logs 123
 *     }
 * });
 * </pre>
 * <p>Models can also be updated and destroyed easily:</p>
 * <pre>
 * // the user Model we loaded in the last snippet:
 * user.set('name', 'Edward Spencer');
 *
 * // tells the Proxy to save the Model. In this case it will perform a PUT request
 * // to /users/123 as this Model already has an id
 * user.save({
 *     success: function() {
 *         console.log('The User was updated');
 *     }
 * });
 *
 * // tells the Proxy to destroy the Model. Performs a DELETE request to /users/123
 * user.erase({
 *     success: function() {
 *         console.log('The User was destroyed!');
 *     }
 * });
 * </pre>
 * <p><b>HTTP Parameter names when using a <i>Ajax proxy</i> (→<code>ext.data.proxy.AjaxProxy</code>)</b></p>
 * <p>By default, the model ID is specified in an HTTP parameter named <code>id</code>. To change the
 * name of this parameter use the Proxy's →<code>ext.data.proxy.AjaxProxy.idParam</code>
 * configuration.</p>
 * <p>Parameters for other commonly passed values such as
 * <i>page number</i> (→<code>ext.data.proxy.AjaxProxy.pageParam</code>) or
 * <i>start row</i> (→<code>ext.data.proxy.AjaxProxy.startParam</code>) may also be configured.</p>
 * <p><b>Usage in Stores</b></p>
 * <p>It is very common to want to load a set of Model instances to be displayed and manipulated
 * in the UI. We do this by creating a →<code>ext.data.Store</code>:</p>
 * <pre>
 * var store = Ext.create('Ext.data.Store', {
 *     model: 'User'
 * });
 *
 * //uses the Proxy we set up on Model to load the Store data
 * store.load();
 * </pre>
 * <p>A Store is just a collection of Model instances - usually loaded from a server somewhere. Store
 * can also maintain a set of added, updated and removed Model instances to be synchronized with
 * the server via the Proxy. See the <i>Store docs</i> (→<code>ext.data.Store</code>) for more information
 * on Stores.</p>
 * @see ext.data.Store
 * @see ext.data.field.DataField#convert()
 * @see ext.data.reader.JsonReader
 * @see ext.data.reader.ArrayReader
 * @see ext.data.field.DataField#convert
 * @see #idProperty
 * @see ext.data.field.DataField#mapping
 * @see ext.data.field.DataField
 * @see #validators
 * @see ext.data.Validation
 * @see ext.Component#modelValidation
 * @see ext.data.schema.Association
 * @see ext.data.proxy.DataProxy
 * @see ext.data.proxy.RestProxy
 * @see #save()
 * @see ext.data.proxy.AjaxProxy
 * @see ext.data.proxy.AjaxProxy#idParam
 * @see ext.data.proxy.AjaxProxy#pageParam
 * @see ext.data.proxy.AjaxProxy#startParam
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html Original Ext JS documentation of 'Ext.data.Model'
 */
public class Model extends Base {
  /**
   * Record event type 'commit'.
   * @see http://docs.sencha.com/extjs/6.0/6.0.1-classic/#!/api/Ext.data.Model-static-property-COMMIT Sencha Docs Ext JS 6.0
   */
  public static const COMMIT:String = "commit";

  /**
   * The default proxy to use for instances of this Model when no proxy is configured
   * on the instance. When specified, the model will use this proxy instead of
   * requesting one from the →<code>ext.data.Session</code>.
   * <p>Can be a string "type", or a →<code>ext.data.proxy.DataProxy</code> config object.</p>
   * <p>This proxy is not inherited by subclasses.</p>
   * @default 'memory'
   * @see ext.data.Session
   * @see ext.data.proxy.DataProxy
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#static-property-defaultProxy Original Ext JS documentation of 'defaultProxy'
   */
  public static native function get defaultProxy():*;

  /**
   * @private
   */
  public static native function set defaultProxy(value:*):void;

  /**
   * This method adds the given set of fields to this model class.
   * @since 5.0.0
   * @param newFields The new fields to add. Based on the <code>name</code>
   * of a field this may replace a previous field definition.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#static-method-addFields Original Ext JS documentation of 'addFields'
   */
  protected static native function addFields(newFields:*):void;

  /**
   * Returns the configured Proxy for this Model.
   * @return The proxy
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#static-method-getProxy Original Ext JS documentation of 'getProxy'
   */
  public static native function getProxy():DataProxy;

  /**
   * Get the summary model type. If →<code>summary</code> is specified, it is
   * a new type that extends from this type. If not, then it is the same
   * model type.
   * @since 6.5.0
   * @return The summary model type.
   * @see #summary
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#static-method-getSummaryModel Original Ext JS documentation of 'getSummaryModel'
   */
  public static native function getSummaryModel():ExtClass;

  /**
   * Asynchronously loads a model instance by id. Any processing of the loaded
   * record should be done in a callback.
   * <p>Sample usage:</p>
   * <pre>
   * Ext.define('MyApp.User', {
   *     extend: 'Ext.data.Model',
   *     fields: [
   *         {name: 'id', type: 'int'},
   *         {name: 'name', type: 'string'}
   *     ]
   * });
   *
   * MyApp.User.load(10, {
   *     scope: this,
   *     failure: function(record, operation) {
   *         //do something if the load failed
   *     },
   *     success: function(record, operation) {
   *         //do something if the load succeeded
   *     },
   *     callback: function(record, operation, success) {
   *         //do something whether the load succeeded or failed
   *     }
   * });
   * </pre>
   * @param id The ID of the model to load.
   * <b>NOTE:</b> The model returned must have an ID matching the param in the load
   * request.
   * @param options The options param is an
   * →<code>ext.data.operation.ReadOperation</code> config object containing success, failure and
   * callback functions, plus optional scope.
   * <ul>
   * <li><code>success:Function</code> (optional) —
   * A function to be called when the
   * model is processed by the proxy successfully.
   * The callback is passed the following parameters:
   * <ul>
   * <li><code>record:ext.data.Model</code> (optional) —
   * The record.
   * </li>
   * <li><code>operation:ext.data.operation.Operation</code> (optional) —
   * The operation.
   * </li>
   * </ul>
   * </li>
   * <li><code>failure:Function</code> (optional) —
   * A function to be called when the
   * model is unable to be processed by the server.
   * The callback is passed the following parameters:
   * <ul>
   * <li><code>record:ext.data.Model</code> (optional) —
   * The record.
   * </li>
   * <li><code>operation:ext.data.operation.Operation</code> (optional) —
   * The operation.
   * </li>
   * </ul>
   * </li>
   * <li><code>callback:Function</code> (optional) —
   * A function to be called whether the proxy
   * transaction was successful or not.
   * The callback is passed the following parameters:
   * <ul>
   * <li><code>record:ext.data.Model</code> (optional) —
   * The record.
   * </li>
   * <li><code>operation:ext.data.operation.Operation</code> (optional) —
   * The
   * operation.
   * </li>
   * <li><code>success:Boolean</code> (optional) —
   * <code>true</code> if the operation was
   * successful.
   * </li>
   * </ul>
   * </li>
   * <li><code>scope:Object</code> (optional) —
   * The scope in which to execute the callback
   * functions. Defaults to the model instance.
   * </li>
   * </ul>
   * @param session The session for this record.
   * @return The newly created model. Note that the model will
   * (probably) still be loading once it is returned from this method. To do any
   * post-processing on the data, the appropriate place to do see is in the
   * callback.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#static-method-load Original Ext JS documentation of 'load'
   * @see ext.data.operation.ReadOperation
   */
  public static native function load(id:*, options:Object = null, session:Session = null):Model;

  /**
   * Create a model while also parsing any data for associations.
   * @since 6.5.0
   * @param data The model data, including any associated data if required.
   * The type of data should correspond to what the configured data reader would expect.
   * @param session The session.
   * @return The model.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#static-method-loadData Original Ext JS documentation of 'loadData'
   */
  public static native function loadData(data:Object, session:Session = null):Model;

  /**
   * Removes the given set of fields from this model.
   * @since 5.0.0
   * @param removeFields The names of fields to remove or <code>true</code>
   * to remove all existing fields. Removes are processed first followed by adds so
   * if a field name appears in <code>newFields</code> as well that field will effectively be
   * added (however, in that case there is no need to include the field in this
   * array).
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#static-method-removeFields Original Ext JS documentation of 'removeFields'
   */
  protected static native function removeFields(removeFields:*):void;

  /**
   * This method replaces the specified set of fields with a given set of new fields.
   * Fields should normally be considered immutable, but if the timing is right (that
   * is, before derived classes are declared), it is permissible to change the fields
   * collection.
   * @since 5.0.0
   * @param newFields The new fields to add. Based on the <code>name</code>
   * of a field this may replace a previous field definition.
   * @param removeFields The names of fields to remove or <code>true</code>
   * to remove all existing fields. Removes are processed first followed by adds so
   * if a field name appears in <code>newFields</code> as well that field will effectively be
   * added (however, in that case there is no need to include the field in this
   * array).
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#static-method-replaceFields Original Ext JS documentation of 'replaceFields'
   */
  protected static native function replaceFields(newFields:*, removeFields:*):void;

  /**
   * Sets the Proxy to use for this model. Accepts any options that can be accepted by
   * →<code>ext.Ext.createByAlias()</code>.
   * @param proxy The proxy
   * @see ext.SExt#createByAlias() ext.Ext.createByAlias()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#static-method-setProxy Original Ext JS documentation of 'setProxy'
   */
  public static native function setProxy(proxy:*):DataProxy;

  /**
   * @param data
   * @param session
   */
  public function Model(data:Object, session:Session = null) {
    super();
  }

  [Deprecated(message="Use `hasMany/hasOne/belongsTo`.\n", since="6.2.0")]
  [ExtConfig]
  /**
   * An array of <i>associations</i> (→<code>ext.data.schema.Association</code>) for this model.
   * <p>For further documentation, see →<code>ext.data.schema.Association</code>.</p>
   * @see ext.data.schema.Association
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-associations Original Ext JS documentation of 'associations'
   */
  public native function get associations():Array;

  [ExtConfig]
  [ArrayElementType("ext.data.schema.Association")]
  /**
   * @private
   */
  public native function set associations(value:Array):void;

  [ExtConfig]
  /**
   * One or more <code>→ext.data.schema.BelongsTo</code> associations for this model.
   * @see ext.data.schema.BelongsTo
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-belongsTo Original Ext JS documentation of 'belongsTo'
   */
  public native function get belongsTo():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set belongsTo(value:*):void;

  [ExtConfig]
  /**
   * The name of the property a server will use to send back a client-generated id in a
   * <code>→create()</code> or <code>update</code> <code>operation (→ext.data.operation.Operation)</code>.
   * <p>If specified, this property cannot have the same name as any other field.</p>
   * <p>For example:</p>
   * <pre>
   *  Ext.define('Person', {
   *      idProperty: 'id',  // this is the default value (for clarity)
   *
   *      clientIdProperty: 'clientId',
   *
   *      identifier: 'negative', // to generate -1, -2 etc on the client
   *
   *      fields: [ 'name' ]
   *  });
   *
   *  var person = new Person({
   *      // no id provided, so -1 is generated
   *      name: 'Clark Kent'
   *  });
   * </pre>
   * <p>The server is given this data during the <code>→create()</code>:</p>
   * <pre>
   *  {
   *      id: -1,
   *      name: 'Clark Kent'
   *  }
   * </pre>
   * <p>The server allocates a real id and responds like so:</p>
   * <pre>
   *  {
   *      id: 427,
   *      clientId: -1
   *  }
   * </pre>
   * <p>This property is most useful when creating multiple entities in a single call to
   * the server in a <code>create operation (→ext.data.operation.CreateOperation)</code>. Alternatively,
   * the server could respond with records that correspond one-to-one to those sent in
   * the <code>operation</code>.</p>
   * <p>For example the client could send a <code>→create()</code> with this data:</p>
   * <pre>
   *  [ { id: -1, name: 'Clark Kent' },
   *    { id: -2, name: 'Peter Parker' },
   *    { id: -3, name: 'Bruce Banner' } ]
   * </pre>
   * <p>And the server could respond in the same order:</p>
   * <pre>
   *  [ { id: 427 },      // updates id = -1
   *    { id: 428 },      // updates id = -2
   *    { id: 429 } ]     // updates id = -3
   * </pre>
   * <p>Or using <code>→clientIdProperty</code> the server could respond in arbitrary order:</p>
   * <pre>
   *  [ { id: 427, clientId: -3 },
   *    { id: 428, clientId: -1 },
   *    { id: 429, clientId: -2 } ]
   * </pre>
   * <p><b>IMPORTANT:</b> When upgrading from previous versions be aware that this property
   * used to perform the role of <code>→ext.data.writer.DataWriter.clientIdProperty</code> as
   * well as that described above. To continue send a client-generated id as other than
   * the <code>→idProperty</code>, set <code>→clientIdProperty</code> on the <code>writer</code>. A better solution, however,
   * is most likely a properly configured <code>→identifier</code> as that would work better with
   * associations.</p>
   * @see #create()
   * @see ext.data.operation.Operation
   * @see ext.data.operation.CreateOperation
   * @see #clientIdProperty
   * @see ext.data.writer.DataWriter#clientIdProperty
   * @see #idProperty
   * @see #identifier
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-clientIdProperty Original Ext JS documentation of 'clientIdProperty'
   */
  public native function get clientIdProperty():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set clientIdProperty(value:String):void;

  [ExtConfig]
  /**
   * Set to <code>false</code> to prevent any converters from being called on fields specified in
   * a →<code>ext.data.Model.set()</code> operation.
   * <p><b>Note:</b> Setting the config to <code>false</code> will only prevent the convert / calculate
   * call when the set <code>fieldName</code> param matches the field's <code>→name</code>. In the
   * following example the calls to set <code>salary</code> will not execute the convert method
   * on <code>→set()</code> while the calls to set <code>vested</code> will execute the convert method on the
   * initial read as well as on <code>→set()</code>.</p>
   * <p>Example model definition:</p>
   * <pre>
   * Ext.define('MyApp.model.Employee', {
   *     extend: 'Ext.data.Model',
   *     fields: ['yearsOfService', {
   *         name: 'salary',
   *         convert: function (val) {
   *             var startingBonus = val &#42; .1;
   *             return val + startingBonus;
   *         }
   *     }, {
   *         name: 'vested',
   *         convert: function (val, record) {
   *             return record.get('yearsOfService') &gt;= 4;
   *         },
   *         depends: 'yearsOfService'
   *     }],
   *     convertOnSet: false
   * });
   *
   * var tina = Ext.create('MyApp.model.Employee', {
   *     salary: 50000,
   *     yearsOfService: 3
   * });
   *
   * console.log(tina.get('salary')); // logs 55000
   * console.log(tina.get('vested')); // logs false
   *
   * tina.set({
   *     salary: 60000,
   *     yearsOfService: 4
   * });
   * console.log(tina.get('salary')); // logs 60000
   * console.log(tina.get('vested')); // logs true
   * </pre>
   * @default true
   * @see ext.data.Model#set()
   * @see #name
   * @see #set()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-convertOnSet Original Ext JS documentation of 'convertOnSet'
   */
  public native function get convertOnSet():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set convertOnSet(value:Boolean):void;

  /**
   * This value is initially "R" or "C" indicating the initial CRUD state. As the
   * record changes and the various joined parties (stores, sessions, etc.) are notified
   * this value is updated prior to these calls. In other words, the joined parties
   * are notified after the <code>→crudState</code> is updated. This means that the <code>→crudState</code>
   * property may be briefly out of sync with underlying changes if this state is used
   * outside of these notifications.
   * <p>The possible states have these meanings:</p>
   * <ul>
   * <li>"R" - The record is in a cleanly retrieved (unmodified) state.</li>
   * <li>"C" - The record is in a newly created (<code>→phantom</code>) state.</li>
   * <li>"U" - The record is in an updated, <code>→modified</code> (<code>→dirty</code>) state.</li>
   * <li>"D" - The record is in a <code>→dropped</code> state.</li>
   * </ul>
   * @default 'R'
   * @since 6.2.0
   * @see #crudState
   * @see #phantom
   * @see #modified
   * @see #dirty
   * @see #dropped
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-crudState Original Ext JS documentation of 'crudState'
   */
  public native function get crudState():String;

  /**
   * This value is initially <code>null</code> indicating there is no previous CRUD state. As the
   * record changes and the various joined parties (stores, sessions, etc.) are notified
   * this value is updated for the <i>subsequent</i> calls. In other words, the joined parties
   * are notified and then <code>→crudStateWas</code> is modified for the next update.
   * <p>The value of this property has the same meaning as <code>→crudState</code>.</p>
   * @since 6.2.0
   * @see #crudStateWas
   * @see #crudState
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-crudStateWas Original Ext JS documentation of 'crudStateWas'
   */
  public native function get crudStateWas():String;

  /**
   * An object hash representing the data for this Record. Every field name in the Record definition is represented by a
   * property of that name in this object. Note that unless you specified a field with
   * name "id" in the Record definition, this will <b>not</b> contain an <tt>id</tt> property.
   */
  public native function get data():Object;

  /**
   * True if this record has been modified.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-dirty Original Ext JS documentation of 'dirty'
   */
  public native function get dirty():Boolean;

  /**
   * True if this record is pending delete on the server. This is set by the <code>→drop()</code>
   * method and transmitted to the server by the <code>→save()</code> method.
   * @default false
   * @see #drop()
   * @see #save()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-dropped Original Ext JS documentation of 'dropped'
   */
  public native function get dropped():Boolean;

  /**
   * Internal flag used to track whether or not the model instance is currently being edited.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-editing Original Ext JS documentation of 'editing'
   */
  public native function get editing():Boolean;

  /**
   * The short name of this entity class. This name is derived from the <code>namespace</code> of
   * the associated <code>schema</code> and this class name. By default, a class is not given a
   * shortened name.
   * <p>All entities in a given <code>schema</code> must have a unique <code>→entityName</code>.</p>
   * <p>For more details see "Relative Naming" in →<code>ext.data.schema.Schema</code>.</p>
   * @see #entityName
   * @see ext.data.schema.Schema
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-entityName Original Ext JS documentation of 'entityName'
   */
  public native function get entityName():String;

  /**
   * @private
   */
  public native function set entityName(value:String):void;

  /**
   * True if this record has been erased on the server. This flag is set of the <code>→erase()</code>
   * method.
   * @default false
   * @see #erase()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-erased Original Ext JS documentation of 'erased'
   */
  public native function get erased():Boolean;

  [ExtConfig]
  /**
   * An Array of <code>→ext.data.field.DataField</code> config objects, simply the field
   * →<code>ext.data.field.DataField.name</code>, or a mix of config objects and strings.
   * If just a name is given, the field type defaults to <code>auto</code>.
   * <p>In a →<code>ext.data.field.DataField</code> config object you may pass the alias of
   * the <code>Ext.data.field.&#42;</code> type using the <code>type</code> config option.</p>
   * <pre>
   * // two fields are set:
   * // - an 'auto' field with a name of 'firstName'
   * // - and an Ext.data.field.Integer field with a name of 'age'
   * fields: ['firstName', {
   *     type: 'int',
   *     name: 'age'
   * }]
   * </pre>
   * <p>Fields will automatically be created at read time for any for any keys in the
   * data passed to the Model's <i>proxy's</i> (→<code>proxy</code>)
   * <i>reader</i> (→<code>ext.data.reader.DataReader</code>) whose name is not explicitly configured in
   * the <code>fields</code> config.</p>
   * <p>Extending a Model class will inherit all the <code>fields</code> from the superclass /
   * ancestor classes.</p>
   * @see ext.data.field.DataField
   * @see ext.data.field.DataField#name
   * @see #proxy
   * @see ext.data.reader.DataReader
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-fields Original Ext JS documentation of 'fields'
   */
  public native function get fields():Array;

  [ExtConfig]
  [ArrayElementType("ext.data.field.DataField")]
  /**
   * @private
   */
  public native function set fields(value:Array):void;

  /**
   * This property is incremented on each modification of a record.
   * @default 1
   * @since 5.0.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-generation Original Ext JS documentation of 'generation'
   */
  public native function get generation():Number;

  [ExtConfig]
  /**
   * One or more <code>→ext.data.schema.HasMany</code> associations for this model.
   * @see ext.data.schema.HasMany
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-hasMany Original Ext JS documentation of 'hasMany'
   */
  public native function get hasMany():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set hasMany(value:*):void;

  [ExtConfig]
  /**
   * One or more <code>→ext.data.schema.HasOne</code> associations for this model.
   * @see ext.data.schema.HasOne
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-hasOne Original Ext JS documentation of 'hasOne'
   */
  public native function get hasOne():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set hasOne(value:*):void;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * The name of the field treated as this Model's unique id.
   * <p>If changing the idProperty in a subclass, the generated id field will replace the
   * one generated by the superclass, for example;</p>
   * <pre>
   *  Ext.define('Super', {
   *      extend: 'Ext.data.Model',
   *      fields: ['name']
   *  });
   *
   *  Ext.define('Sub', {
   *      extend: 'Super',
   *      idProperty: 'customId'
   *  });
   *
   *  var fields = Super.getFields();
   *  // Has 2 fields, "name" &amp; "id"
   *  console.log(fields[0].name, fields[1].name, fields.length);
   *
   *  fields = Sub.getFields();
   *  // Has 2 fields, "name" &amp; "customId", "id" is replaced
   *  console.log(fields[0].name, fields[1].name, fields.length);
   * </pre>
   * <p>The data values for this field must be unique or there will be id value collisions
   * in the →<code>ext.data.Store</code>.</p>
   * @default 'id'
   * @see ext.data.Store
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-idProperty Original Ext JS documentation of 'idProperty'
   * @see #getIdProperty()
   */
  public native function get idProperty():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set idProperty(value:String):void;

  [ExtConfig]
  /**
   * The id generator to use for this model. The <code>→identifier</code> generates values for the
   * →<code>idProperty</code> when no value is given. Records with client-side generated
   * values for →<code>idProperty</code> are called →<code>phantom</code> records since they are
   * not yet known to the server.
   * <p>This can be overridden at the model level to provide a custom generator for a
   * model. The simplest form of this would be:</p>
   * <pre>
   *  Ext.define('MyApp.data.MyModel', {
   *      extend: 'Ext.data.Model',
   *      requires: ['Ext.data.identifier.Sequential'],
   *      identifier: 'sequential',
   *      ...
   *  });
   * </pre>
   * <p>The above would generate <i>sequential</i> (→<code>ext.data.identifier.SequentialIdentifier</code>) id's
   * such as 1, 2, 3 etc..</p>
   * <p>Another useful id generator is →<code>ext.data.identifier.UuidIdentifier</code>:</p>
   * <pre>
   *  Ext.define('MyApp.data.MyModel', {
   *      extend: 'Ext.data.Model',
   *      requires: ['Ext.data.identifier.Uuid'],
   *      identifier: 'uuid',
   *      ...
   *  });
   * </pre>
   * <p>An id generator can also be further configured:</p>
   * <pre>
   *  Ext.define('MyApp.data.MyModel', {
   *      extend: 'Ext.data.Model',
   *      identifier: {
   *          type: 'sequential',
   *          seed: 1000,
   *          prefix: 'ID_'
   *      }
   *  });
   * </pre>
   * <p>The above would generate id's such as ID_1000, ID_1001, ID_1002 etc..</p>
   * <p>If multiple models share an id space, a single generator can be shared:</p>
   * <pre>
   *  Ext.define('MyApp.data.MyModelX', {
   *      extend: 'Ext.data.Model',
   *      identifier: {
   *          type: 'sequential',
   *          id: 'xy'
   *      }
   *  });
   *
   *  Ext.define('MyApp.data.MyModelY', {
   *      extend: 'Ext.data.Model',
   *      identifier: {
   *          type: 'sequential',
   *          id: 'xy'
   *      }
   *  });
   * </pre>
   * <p>For more complex, shared id generators, a custom generator is the best approach.
   * See →<code>ext.data.identifier.GeneratorIdentifier</code> for details on creating custom id
   * generators.</p>
   * @default null
   * @see #identifier
   * @see #idProperty
   * @see #phantom
   * @see ext.data.identifier.SequentialIdentifier
   * @see ext.data.identifier.UuidIdentifier
   * @see ext.data.identifier.GeneratorIdentifier
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-identifier Original Ext JS documentation of 'identifier'
   */
  public native function get identifier():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set identifier(value:*):void;

  /**
   * The value <code>true</code> to identify this class and its subclasses.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-isEntity Original Ext JS documentation of 'isEntity'
   */
  public native function get isEntity():Boolean;

  /**
   * The value <code>true</code> to identify this class and its subclasses.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-isModel Original Ext JS documentation of 'isModel'
   */
  public native function get isModel():Boolean;

  [ExtConfig]
  /**
   * A config object for a →<code>ext.data.schema.ManyToMany</code> association.
   * See the class description for →<code>ext.data.schema.ManyToMany</code> for
   * configuration examples.
   * @see ext.data.schema.ManyToMany
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-manyToMany Original Ext JS documentation of 'manyToMany'
   */
  public native function get manyToMany():Object;

  [ExtConfig]
  /**
   * @private
   */
  public native function set manyToMany(value:Object):void;

  /**
   * A hash of field values which holds the initial values of fields before a set of
   * edits are <i>committed</i> (→<code>commit()</code>).
   * @see #commit()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-modified Original Ext JS documentation of 'modified'
   */
  public native function get modified():Object;

  /**
   * @private
   */
  public native function set modified(value:Object):void;

  /**
   * True when the record does not yet exist in a server-side database. Any record which
   * has a real database identity set as its <code>→idProperty</code> is NOT a phantom -- it's real.
   * @default false
   * @see #idProperty
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-phantom Original Ext JS documentation of 'phantom'
   */
  public native function get phantom():Boolean;

  /**
   * @private
   */
  public native function set phantom(value:Boolean):void;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * The <i>proxy</i> (→<code>ext.data.proxy.DataProxy</code>) to use for this class.
   * <p>By default, the proxy is configured from the <i>schema</i> (→<code>ext.data.schema.Schema</code>).
   * You can ignore the schema defaults by setting <code>schema: false</code> on the <code>→proxy</code> config.</p>
   * <pre>
   *  Ext.define('MyApp.data.CustomProxy', {
   *      extend: 'Ext.data.proxy.Ajax',
   *      alias: 'proxy.customproxy',
   *
   *      url: 'users.json'
   *  });
   *
   *  Ext.define('MyApp.models.CustomModel', {
   *      extend: 'Ext.data.Model',
   *
   *      fields: ['name'],
   *      proxy: {
   *          type: 'customproxy,
   *          schema: false
   *      }
   *  });
   * </pre>
   * <p>With <code>schema: false</code>, the <code>url</code> of the proxy will be used instead of what has been defined
   * on the schema.</p>
   * @see ext.data.proxy.DataProxy
   * @see ext.data.schema.Schema
   * @see #proxy
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-proxy Original Ext JS documentation of 'proxy'
   * @see #getProxy()
   */
  public native function get proxy():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set proxy(value:*):void;

  [ExtConfig]
  /**
   * The <code>→ext.data.schema.Schema</code> to which this entity and its associations belong.
   * @default 'default'
   * @see ext.data.schema.Schema
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-schema Original Ext JS documentation of 'schema'
   */
  public native function get schema():Schema;

  [ExtConfig]
  /**
   * @private
   */
  public native function set schema(value:Schema):void;

  /**
   * The →<code>ext.data.Session</code> for this record.
   * @see ext.data.Session
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-session Original Ext JS documentation of 'session'
   */
  public native function get session():Session;

  /**
   * The →<code>ext.data.Store</code> to which this instance belongs.
   * <p><b>Note:</b> If this instance is bound to multiple stores, this property
   * will reference only the first.</p>
   * @default owner
   * @see ext.data.Store
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#property-store Original Ext JS documentation of 'store'
   */
  public native function get store():Store;

  /**
   * @private
   */
  public native function set store(value:Store):void;

  [ExtConfig]
  /**
   * Summary fields are a special kind of field that is used to assist in creating an
   * aggregation for this model. A new model type that extends this model will be
   * created, accessible via →<code>getSummaryModel()</code>. This summary model will
   * have these virtual aggregate fields in the fields collection like a normal model.
   * Each key in the object is the field name. The value for each field should mirror
   * the →<code>fields</code>, excluding the <code>name</code> option. The summary model generated
   * will have 2 fields, 'rate', which will aggregate using an average and maxRate,
   * which will aggregate using the maximum value.
   * <p>See →<code>ext.data.summary.SummaryBase</code> for more information.</p>
   * <pre>
   *  Ext.define('User', {
   *      extend: 'Ext.data.Model',
   *      fields: [{
   *          name: 'rate',
   *          summary: 'avg'
   *      }],
   *
   *      summary: {
   *          maxRate: {
   *              field: 'rate', // calculated from rate field
   *              summary: 'max'
   *          }
   *      }
   *  });
   * </pre>
   * @since 6.5.0
   * @see #getSummaryModel()
   * @see #fields
   * @see ext.data.summary.SummaryBase
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-summary Original Ext JS documentation of 'summary'
   */
  public native function get summary():Object;

  [ExtConfig]
  /**
   * @private
   */
  public native function set summary(value:Object):void;

  [ExtConfig]
  /**
   * If specified this property is used to concatenate multiple errors for each field
   * as reported by the <code>→validators</code>.
   * @see #validators
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-validationSeparator Original Ext JS documentation of 'validationSeparator'
   */
  public native function get validationSeparator():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set validationSeparator(value:String):void;

  [ExtConfig]
  /**
   * An array of <i>validators</i> (→<code>ext.data.validator.Validator</code>) for this model.
   * @see ext.data.validator.Validator
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-validators Original Ext JS documentation of 'validators'
   */
  public native function get validators():Array;

  [ExtConfig]
  [ArrayElementType("ext.data.validator.Validator")]
  /**
   * @private
   */
  public native function set validators(value:Array):void;

  [ExtConfig]
  /**
   * If specified, this is the name of the property that contains the entity "version".
   * The version property is used to manage a long-running transaction and allows the
   * detection of simultaneous modification.
   * <p>The way a version property is used is that the client receives the version as it
   * would any other entity property. When saving an entity, this property is always
   * included in the request and the server uses the value in a "conditional update".
   * If the current version of the entity on the server matches the version property
   * sent by the client, the update is allowed. Otherwise, the update fails.</p>
   * <p>On successful update, both the client and server increment the version. This is
   * done on the server in the conditional update and on the client when it receives a
   * success on its update request.</p>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#cfg-versionProperty Original Ext JS documentation of 'versionProperty'
   */
  public native function get versionProperty():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set versionProperty(value:String):void;

  /**
   * Aborts a pending →<code>load()</code> operation. If the record is not loading, this does
   * nothing.
   * @see #load()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-abort Original Ext JS documentation of 'abort'
   */
  public native function abort():void;

  /**
   * Begins an edit. While in edit mode, no events (e.g.. the <code>update</code> event) are
   * relayed to the containing store. When an edit has begun, it must be followed by
   * either <code>→endEdit()</code> or <code>→cancelEdit()</code>.
   * @see #endEdit()
   * @see #cancelEdit()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-beginEdit Original Ext JS documentation of 'beginEdit'
   */
  public native function beginEdit():void;

  /**
   * Calculate all summary fields on this record.
   * @since 6.5.0
   * @param records The records to use for calculation.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-calculateSummary Original Ext JS documentation of 'calculateSummary'
   */
  public native function calculateSummary(records:Array):void;

  /**
   * Cancels all changes made in the current edit operation.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-cancelEdit Original Ext JS documentation of 'cancelEdit'
   */
  public native function cancelEdit():void;

  /**
   * Creates a clone of this record. States like <code>→dropped</code>, <code>→phantom</code> and <code>→dirty</code> are
   * all preserved in the cloned record.
   * @param session The session to which the new record
   * belongs.
   * @return The cloned record.
   * @see #dropped
   * @see #phantom
   * @see #dirty
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-clone Original Ext JS documentation of 'clone'
   */
  public native function clone(session:Session = null):Model;

  /**
   * Usually called by the →<code>ext.data.Store</code> which owns the model instance. Commits all
   * changes made to the instance since either creation or the last commit operation.
   * <p>Developers should subscribe to the →<code>ext.data.Store.event:onUpdate</code> event to have their
   * code notified of commit operations.</p>
   * @param silent Pass <code>true</code> to skip notification of the owning store of the
   * change.
   * @default false
   * @param modifiedFieldNames Array of field names changed during sync with server
   * if known. Omit or pass <code>null</code> if unknown. An empty array means that it is known that
   * no fields were modified by the server's response.
   * Defaults to false.
   * @see ext.data.Store
   * @see ext.data.Store#event:onUpdate
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-commit Original Ext JS documentation of 'commit'
   */
  public native function commit(silent:Boolean = false, modifiedFieldNames:Array = null):void;

  /**
   * Creates a clean copy of this record. The returned record will not consider any its
   * fields as modified.
   * <p>To generate a phantom instance with a new id pass <code>null</code>:</p>
   * <pre>
   * var rec = record.copy(null); // clone the record but no id (one is generated)
   * </pre>
   * @param newId A new id, defaults to the id of the instance being copied.
   * See <code>→ext.data.Model.idProperty</code>.
   * @param session The session to which the new record
   * belongs.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-copy Original Ext JS documentation of 'copy'
   * @see ext.data.Model#idProperty
   */
  public native function copy(newId:String = null, session:Session = null):Model;

  /**
   * Marks this record as <code>→dropped</code> and waiting to be deleted on the server. When a
   * record is dropped, it is automatically removed from all association stores and
   * any child records associated to this record are also dropped (a "cascade delete")
   * depending on the <code>cascade</code> parameter.
   * @since 5.0.0
   * @param cascade Pass <code>false</code> to disable the cascade to drop child
   * records.
   * @default true
   * @see #dropped
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-drop Original Ext JS documentation of 'drop'
   */
  public native function drop(cascade:Boolean = true):void;

  /**
   * Ends an edit. If any data was modified, the containing store is notified
   * (ie, the store's <code>update</code> event will fire).
   * @param silent True to not notify any stores of the change.
   * @param modifiedFieldNames Array of field names changed during edit.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-endEdit Original Ext JS documentation of 'endEdit'
   */
  public native function endEdit(silent:Boolean = false, modifiedFieldNames:Array = null):void;

  /**
   * Destroys the model using the configured proxy. The erase action is
   * asynchronous. Any processing of the erased record should be done in a callback.
   * <pre>
   * Ext.define('MyApp.model.User', {
   *     extend: 'Ext.data.Model',
   *     fields: [
   *         {name: 'id', type: 'int'},
   *         {name: 'name', type: 'string'}
   *     ],
   *     proxy: {
   *         type: 'ajax',
   *         url: 'server.url'
   *     }
   * });
   *
   * var user = new MyApp.model.User({
   *     name: 'Foo'
   * });
   *
   * // pass the phantom record data to the server to be saved
   * user.save({
   *     success: function(record, operation) {
   *         // do something if the save succeeded
   *         // erase the created record
   *         record.erase({
   *             failure: function(record, operation) {
   *                 // do something if the erase failed
   *             },
   *             success: function(record, operation) {
   *                 // do something if the erase succeeded
   *             },
   *             callback: function(record, operation, success) {
   *                 // do something if the erase succeeded or failed
   *             }
   *         });
   *     }
   * });
   * </pre>
   * <p><b>NOTE:</b> If a →<code>phantom</code> record is erased it will not be processed via the
   * proxy. However, any passed <code>success</code> or <code>callback</code> functions will be called.</p>
   * <p>The options param is an →<code>ext.data.operation.DestroyOperation</code> config object
   * containing success, failure and callback functions, plus optional scope.</p>
   * @param options Options to pass to the proxy.
   * @return The destroy operation
   * @see #phantom
   * @see ext.data.operation.DestroyOperation
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-erase Original Ext JS documentation of 'erase'
   */
  public native function erase(options:Object = null):DestroyOperation;

  /**
   * Returns the value of the given field.
   * @param fieldName The name of the field.
   * @return The value of the specified field.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-get Original Ext JS documentation of 'get'
   */
  public native function get(fieldName:String):Object;

  /**
   * Gets all of the data from this Models <i>loaded</i> associations. It does this
   * recursively. For example if we have a User which hasMany Orders, and each Order
   * hasMany OrderItems, it will return an object like this:
   * <pre>
   * {
   *     orders: [
   *         {
   *             id: 123,
   *             status: 'shipped',
   *             orderItems: [
   *                 ...
   *             ]
   *         }
   *     ]
   * }
   * </pre>
   * @param result The object on to which the associations will be added. If
   * no object is passed one is created. This object is then returned.
   * @param options An object containing options describing the data
   * desired.
   * <ul>
   * <li><code>associated:Boolean</code> (default = <code>true</code>) —
   * Pass <code>true</code> to include associated data from
   * other associated records.
   * </li>
   * <li><code>changes:Boolean</code> (default = <code>false</code>) —
   * Pass <code>true</code> to only include fields that
   * have been modified. Note that field modifications are only tracked for fields that
   * are not declared with <code>persist</code> set to <code>false</code>. In other words, only persistent
   * fields have changes tracked so passing <code>true</code> for this means <code>options.persist</code> is
   * redundant.
   * </li>
   * <li><code>critical:Boolean</code> (optional) —
   * Pass <code>true</code> to include fields set as <code>critical</code>.
   * This is only meaningful when <code>options.changes</code> is <code>true</code> since critical fields may
   * not have been modified.
   * </li>
   * <li><code>persist:Boolean</code> (optional) —
   * Pass <code>true</code> to only return persistent fields.
   * This is implied when <code>options.changes</code> is set to <code>true</code>.
   * </li>
   * <li><code>serialize:Boolean</code> (default = <code>false</code>) —
   * Pass <code>true</code> to invoke the <code>serialize</code>
   * method on the returned fields.
   * </li>
   * </ul>
   * @return The nested data set for the Model's loaded associations.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-getAssociatedData Original Ext JS documentation of 'getAssociatedData'
   */
  public native function getAssociatedData(result:Object = null, options:* = undefined):Object;

  /**
   * Gets an object of only the fields that have been modified since this record was
   * created or committed. Only persistent fields are tracked in the <code>→modified</code> set so
   * this method will only return changes to persistent fields.
   * <p>For more control over the returned data, see <code>→getData()</code>.</p>
   * @see #modified
   * @see #getData()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-getChanges Original Ext JS documentation of 'getChanges'
   */
  public native function getChanges():Object;

  [ArrayElementType("ext.data.field.DataField")]
  /**
   * Returns the array of fields that are declared as critical (must always send).
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-getCriticalFields Original Ext JS documentation of 'getCriticalFields'
   */
  public native function getCriticalFields():Array;

  /**
   * Gets all values for each field in this model and returns an object containing the
   * current data. This can be tuned by passing an <code>options</code> object with various
   * properties describing the desired result. Passing <code>true</code> simply returns all fields
   * <i>and</i> all associated record data.
   * <p>To selectively gather some associated data, the <code>options</code> object can be used as
   * follows:</p>
   * <pre>
   *  var data = order.getData({
   *      associated: {
   *          orderItems: true
   *      }
   *  });
   * </pre>
   * <p>This will include all data fields as well as an "orderItems" array with the data
   * for each <code>OrderItem</code>. To include the associated <code>Item</code> for each <code>OrderItem</code>, the
   * call would look like:</p>
   * <pre>
   *  var data = order.getData({
   *      associated: {
   *          orderItems: {
   *              item: true
   *          }
   *      }
   *  });
   * </pre>
   * @param options An object containing options describing the data
   * desired. If <code>true</code> is passed it is treated as an object with <code>associated</code> set to
   * <code>true</code>.
   * <ul>
   * <li><code>associated</code> (default = <code>false</code>) —
   * Pass <code>true</code> to recursively
   * include all associated data. This is equivalent to pass <code>true</code> as the only argument.
   * See <code>→getAssociatedData()</code>. If <code>associated</code> is an object, it describes the specific
   * associations to gather.
   * </li>
   * <li><code>changes:Boolean</code> (default = <code>false</code>) —
   * Pass <code>true</code> to only include fields that
   * have been modified. Note that field modifications are only tracked for fields that
   * are not declared with <code>persist</code> set to <code>false</code>. In other words, only persistent
   * fields have changes tracked so passing <code>true</code> for this means <code>options.persist</code> is
   * redundant.
   * </li>
   * <li><code>critical:Boolean</code> (optional) —
   * Pass <code>true</code> to include fields set as <code>critical</code>.
   * This is only meaningful when <code>options.changes</code> is <code>true</code> since critical fields may
   * not have been modified.
   * </li>
   * <li><code>persist:Boolean</code> (optional) —
   * Pass <code>true</code> to only return persistent fields.
   * This is implied when <code>options.changes</code> is set to <code>true</code>.
   * </li>
   * <li><code>serialize:Boolean</code> (default = <code>false</code>) —
   * Pass <code>true</code> to invoke the <code>serialize</code>
   * method on the returned fields.
   * </li>
   * </ul>
   * @return An object containing all the values in this model.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-getData Original Ext JS documentation of 'getData'
   * @see #getAssociatedData()
   */
  public native function getData(options:* = undefined):Object;

  [ArrayElementType("ext.data.field.DataField")]
  /**
   * Get the fields array for this model.
   * @return The fields array
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-getFields Original Ext JS documentation of 'getFields'
   */
  public native function getFields():Array;

  /**
   * @private
   */
  public native function getFieldsMap():Object;

  /**
   * Returns the unique ID allocated to this model instance as defined by <code>→idProperty</code>.
   * @return The id
   * @see #idProperty
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-getId Original Ext JS documentation of 'getId'
   */
  public native function getId():*;

  /**
   * Get the idProperty for this model.
   * @return The idProperty
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-getIdProperty Original Ext JS documentation of 'getIdProperty'
   */
  public native function getIdProperty():String;

  /**
   * Returns the original value of a modified field. If there is no modified value,
   * <code>undefined</code> will be return. Also see →<code>isModified()</code>.
   * @param fieldName The name of the field for which to return the original value.
   * @return modified
   * @see #isModified()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-getModified Original Ext JS documentation of 'getModified'
   */
  public native function getModified(fieldName:String):Object;

  /**
   * Return a unique observable ID. Model is not observable but tree nodes
   * (<code>→ext.data.NodeInterface</code>) are, so they must be globally unique within the
   * →<code>observableType</code>.
   * @see ext.data.NodeInterface
   * @see #observableType
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-getObservableId Original Ext JS documentation of 'getObservableId'
   */
  protected native function getObservableId():String;

  /**
   * This method returns the value of a field given its name prior to its most recent
   * change.
   * @param fieldName The field's →<code>ext.data.field.DataField.name</code>.
   * @return The value of the given field prior to its current value. <code>undefined</code>
   * if there is no previous value;
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-getPrevious Original Ext JS documentation of 'getPrevious'
   * @see ext.data.field.DataField#name
   */
  public native function getPrevious(fieldName:String):Object;

  /**
   * Returns the configured Proxy for this Model.
   * @return The proxy
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-getProxy Original Ext JS documentation of 'getProxy'
   */
  public native function getProxy():DataProxy;

  [ArrayElementType("ext.data.field.DataField")]
  /**
   * Returns the array of fields that are declared as non-persist or "transient".
   * @since 5.0.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-getTransientFields Original Ext JS documentation of 'getTransientFields'
   */
  public native function getTransientFields():Array;

  /**
   * Returns the <code>→ext.data.Validation</code> record holding the results of this record's
   * <code>→validators</code>. This record is lazily created on first request and is then kept on
   * this record to be updated later.
   * <p>See the class description for more about <code>→validators</code>.</p>
   * @since 5.0.0
   * @param refresh Pass <code>false</code> to not call the <code>refresh</code> method on the
   * validation instance prior to returning it. Pass <code>true</code> to force a <code>refresh</code> of the
   * validation instance. By default the returned record is only refreshed if changes
   * have been made to this record.
   * @return The <code>Validation</code> record for this record.
   * @see ext.data.Validation
   * @see #validators
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-getValidation Original Ext JS documentation of 'getValidation'
   */
  public native function getValidation(refresh:Boolean = false):Validation;

  /**
   * Checks whether this model is loading data from the →<code>proxy</code>.
   * @return <code>true</code> if in a loading state.
   * @see #proxy
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-isLoading Original Ext JS documentation of 'isLoading'
   */
  public native function isLoading():Boolean;

  /**
   * Returns true if the passed field name has been <code>→modified</code> since the load
   * or last commit.
   * @param fieldName The field's →<code>ext.data.field.DataField.name</code>.
   * @see #modified
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-isModified Original Ext JS documentation of 'isModified'
   * @see ext.data.field.DataField#name
   */
  public native function isModified(fieldName:String):Boolean;

  /**
   * Checks if the model is valid. See →<code>getValidation()</code>.
   * @return True if the model is valid.
   * @see #getValidation()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-isValid Original Ext JS documentation of 'isValid'
   */
  public native function isValid():Boolean;

  /**
   * Tells this model instance that an observer is looking at it.
   * @param owner The store or other owner object to which this model
   * has been added.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-join Original Ext JS documentation of 'join'
   */
  protected native function join(owner:Store):void;

  /**
   * Loads the model instance using the configured proxy. The load action
   * is asynchronous. Any processing of the loaded record should be done in a
   * callback.
   * <pre>
   * Ext.define('MyApp.model.User', {
   *     extend: 'Ext.data.Model',
   *     fields: [
   *         {name: 'id', type: 'int'},
   *         {name: 'name', type: 'string'}
   *     ],
   *     proxy: {
   *         type: 'ajax',
   *         url: 'server.url'
   *     }
   * });
   *
   * var user = new MyApp.model.User();
   * user.load({
   *     scope: this,
   *     failure: function(record, operation) {
   *         // do something if the load failed
   *     },
   *     success: function(record, operation) {
   *         // do something if the load succeeded
   *     },
   *     callback: function(record, operation, success) {
   *         // do something whether the load succeeded or failed
   *     }
   * });
   * </pre>
   * <p>The options param is an →<code>ext.data.operation.ReadOperation</code> config object containing
   * success, failure and callback functions, plus optional scope.</p>
   * @param options Options to pass to the proxy.
   * <ul>
   * <li><code>success:Function</code> (optional) —
   * A function to be called when the
   * model is processed by the proxy successfully.
   * The callback is passed the following parameters:
   * <ul>
   * <li><code>record:ext.data.Model</code> (optional) —
   * The record.
   * </li>
   * <li><code>operation:ext.data.operation.Operation</code> (optional) —
   * The operation.
   * </li>
   * </ul>
   * </li>
   * <li><code>failure:Function</code> (optional) —
   * A function to be called when the
   * model is unable to be processed by the server.
   * The callback is passed the following parameters:
   * <ul>
   * <li><code>record:ext.data.Model</code> (optional) —
   * The record.
   * </li>
   * <li><code>operation:ext.data.operation.Operation</code> (optional) —
   * The operation.
   * </li>
   * </ul>
   * </li>
   * <li><code>callback:Function</code> (optional) —
   * A function to be called whether the proxy
   * transaction was successful or not.
   * The callback is passed the following parameters:
   * <ul>
   * <li><code>record:ext.data.Model</code> (optional) —
   * The record.
   * </li>
   * <li><code>operation:ext.data.operation.Operation</code> (optional) —
   * The operation.
   * </li>
   * <li><code>success:Boolean</code> (optional) —
   * <code>true</code> if the operation was successful.
   * </li>
   * </ul>
   * </li>
   * <li><code>scope:Object</code> (optional) —
   * The scope in which to execute the callback
   * functions. Defaults to the model instance.
   * </li>
   * </ul>
   * @return The read operation.
   * @see ext.data.operation.ReadOperation
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-load Original Ext JS documentation of 'load'
   */
  public native function load(options:Object = null):ReadOperation;

  /**
   * Merge incoming data from the server when this record exists
   * in an active session. This method is not called if this record is
   * loaded directly via →<code>load()</code>. The default behaviour is to use incoming
   * data if the record is not →<code>dirty</code>, otherwise the data is
   * discarded. This method should be overridden in subclasses to
   * provide a different behavior.
   * @since 6.5.0
   * @param data The model data retrieved from the server.
   * @see #load()
   * @see #dirty
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-mergeData Original Ext JS documentation of 'mergeData'
   */
  protected native function mergeData(data:Object):void;

  /**
   * Called when the model id is changed.
   * @param id The new id.
   * @param oldId The old id.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-onIdChanged Original Ext JS documentation of 'onIdChanged'
   */
  public native function onIdChanged(id:Object, oldId:Object):void;

  /**
   * This method is called by the →<code>ext.data.reader.DataReader</code> after loading a model from
   * the server. This is after processing any inline associations that are available.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @see ext.data.reader.DataReader
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-onLoad Original Ext JS documentation of 'onLoad'
   */
  protected native function onLoad():void;

  /**
   * Usually called by the →<code>ext.data.Store</code> to which this model instance has been
   * <i>joined</i> (→<code>join()</code>). Rejects all changes made to the model instance since either creation,
   * or the last commit operation. Modified fields are reverted to their original values.
   * <p>Developers should subscribe to the →<code>ext.data.Store.event:onUpdate</code> event to have their
   * code notified of reject operations.</p>
   * @param silent <code>true</code> to skip notification of the owning store of the
   * change.
   * @default false
   * @see ext.data.Store
   * @see #join()
   * @see ext.data.Store#event:onUpdate
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-reject Original Ext JS documentation of 'reject'
   */
  public native function reject(silent:Boolean = false):void;

  /**
   * Saves the model instance using the configured proxy. The save action
   * is asynchronous. Any processing of the saved record should be done in a callback.
   * <p>Create example:</p>
   * <pre>
   * Ext.define('MyApp.model.User', {
   *     extend: 'Ext.data.Model',
   *     fields: [
   *         {name: 'id', type: 'int'},
   *         {name: 'name', type: 'string'}
   *     ],
   *     proxy: {
   *         type: 'ajax',
   *         url: 'server.url'
   *     }
   * });
   *
   * var user = new MyApp.model.User({
   *     name: 'Foo'
   * });
   *
   * // pass the phantom record data to the server to be saved
   * user.save({
   *     failure: function(record, operation) {
   *         // do something if the save failed
   *     },
   *     success: function(record, operation) {
   *         // do something if the save succeeded
   *     },
   *     callback: function(record, operation, success) {
   *         // do something whether the save succeeded or failed
   *     }
   * });
   * </pre>
   * <p>The response from a create operation should include the ID for the newly created
   * record:</p>
   * <pre>
   * // sample response
   * {
   *     success: true,
   *     id: 1
   * }
   *
   * // the id may be nested if the proxy's reader has a rootProperty config
   * Ext.define('MyApp.model.User', {
   *     extend: 'Ext.data.Model',
   *     proxy: {
   *         type: 'ajax',
   *         url: 'server.url',
   *         reader: {
   *             type: 'ajax',
   *             rootProperty: 'data'
   *         }
   *     }
   * });
   *
   * // sample nested response
   * {
   *     success: true,
   *     data: {
   *         id: 1
   *     }
   * }
   * </pre>
   * <p>(Create + ) Update example:</p>
   * <pre>
   * Ext.define('MyApp.model.User', {
   *     extend: 'Ext.data.Model',
   *     fields: [
   *         {name: 'id', type: 'int'},
   *         {name: 'name', type: 'string'}
   *     ],
   *     proxy: {
   *         type: 'ajax',
   *         url: 'server.url'
   *     }
   * });
   *
   * var user = new MyApp.model.User({
   *     name: 'Foo'
   * });
   * user.save({
   *     success: function(record, operation) {
   *         record.set('name', 'Bar');
   *         // updates the remote record via the proxy
   *         record.save();
   *     }
   * });
   * </pre>
   * <p>(Create + ) Destroy example - see also →<code>erase()</code>:</p>
   * <pre>
   * Ext.define('MyApp.model.User', {
   *     extend: 'Ext.data.Model',
   *     fields: [
   *         {name: 'id', type: 'int'},
   *         {name: 'name', type: 'string'}
   *     ],
   *     proxy: {
   *         type: 'ajax',
   *         url: 'server.url'
   *     }
   * });
   *
   * var user = new MyApp.model.User({
   *     name: 'Foo'
   * });
   * user.save({
   *     success: function(record, operation) {
   *         record.drop();
   *         // destroys the remote record via the proxy
   *         record.save();
   *     }
   * });
   * </pre>
   * <p><b>NOTE:</b> If a →<code>phantom</code> record is <i>dropped</i> (→<code>drop()</code>) and subsequently
   * saved it will not be processed via the proxy. However, any passed <code>success</code>
   * or <code>callback</code> functions will be called.</p>
   * <p>The options param is an Operation config object containing success, failure and
   * callback functions, plus optional scope. The type of Operation depends on the
   * state of the model being saved.</p>
   * <ul>
   * <li>→<code>phantom</code> model - →<code>ext.data.operation.CreateOperation</code></li>
   * <li><i>modified</i> (→<code>isModified()</code>) model - →<code>ext.data.operation.UpdateOperation</code></li>
   * <li>→<code>dropped</code> model - →<code>ext.data.operation.DestroyOperation</code></li>
   * </ul>
   * @param options Options to pass to the proxy.
   * @return The operation instance for saving this model. The type of operation returned
   * depends on the model state at the time of the action.
   * <ul>
   * <li>→<code>phantom</code> model - →<code>ext.data.operation.CreateOperation</code></li>
   * <li><i>modified</i> (→<code>isModified()</code>) model - →<code>ext.data.operation.UpdateOperation</code></li>
   * <li>→<code>dropped</code> model - →<code>ext.data.operation.DestroyOperation</code></li>
   * </ul>
   * @see #erase()
   * @see #phantom
   * @see #drop()
   * @see ext.data.operation.CreateOperation
   * @see #isModified()
   * @see ext.data.operation.UpdateOperation
   * @see #dropped
   * @see ext.data.operation.DestroyOperation
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-save Original Ext JS documentation of 'save'
   */
  public native function save(options:Object):*;

  [ArrayElementType("String")]
  /**
   * Sets the given field to the given value. For example:
   * <pre>
   *  record.set('name', 'value');
   * </pre>
   * <p>This method can also be passed an object containing multiple values to set at once.
   * For example:</p>
   * <pre>
   *  record.set({
   *      name: 'value',
   *      age: 42
   *  });
   * </pre>
   * <p>The following store events are fired when the modified record belongs to a store:</p>
   * <ul>
   * <li>→<code>ext.data.Store.event:onBeginUpdate</code></li>
   * <li>→<code>ext.data.Store.event:onUpdate</code></li>
   * <li>→<code>ext.data.Store.event:onEndUpdate</code></li>
   * </ul>
   * @param fieldName The field to set, or an object containing key/value
   * pairs.
   * @param newValue The value for the field (if <code>fieldName</code> is a string).
   * @param options Options for governing this update.
   * <ul>
   * <li><code>convert:Boolean</code> (default = <code>true</code>) —
   * Set to <code>false</code> to prevent any converters from
   * being called during the set operation. This may be useful when setting a large bunch of
   * raw values.
   * </li>
   * <li><code>dirty:Boolean</code> (default = <code>true</code>) —
   * Pass <code>false</code> if the field values are to be
   * understood as non-dirty (fresh from the server). When <code>true</code>, this change will be
   * reflected in the <code>→modified</code> collection.
   * </li>
   * <li><code>commit:Boolean</code> (default = <code>false</code>) —
   * Pass <code>true</code> to call the →<code>commit()</code> method
   * after setting fields. If this option is passed, the usual after change processing will
   * be bypassed. <i>Commit</i> (→<code>commit()</code>) will be called even if there are no field changes.
   * </li>
   * <li><code>silent:Boolean</code> (default = <code>false</code>) —
   * Pass <code>true</code> to suppress notification of any
   * changes made by this call. Use with caution.
   * </li>
   * </ul>
   * @return The array of modified field names or null if nothing was modified.
   * @see ext.data.Store#event:onBeginUpdate
   * @see ext.data.Store#event:onUpdate
   * @see ext.data.Store#event:onEndUpdate
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-set Original Ext JS documentation of 'set'
   * @see #modified
   * @see #commit()
   */
  public native function set(fieldName:String, newValue:*, options:Object = null):Array;

  /**
   * Sets the model instance's id field to the given id.
   * @param id The new id.
   * @param options See →<code>set()</code>.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-setId Original Ext JS documentation of 'setId'
   * @see #set()
   */
  public native function setId(id:*, options:Object = null):void;

  /**
   * Returns a url-suitable string for this model instance. By default this just returns the
   * name of the Model class followed by the instance ID - for example an instance of
   * MyApp.model.User with ID 123 will return 'user/123'.
   * @return The url string for this model instance.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-toUrl Original Ext JS documentation of 'toUrl'
   */
  public native function toUrl():String;

  /**
   * Tells this model instance that it has been removed from the store.
   * @param owner The store or other owner object from which this
   * model has been removed.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-unjoin Original Ext JS documentation of 'unjoin'
   */
  public native function unjoin(owner:Store):void;

  [Deprecated(replacement="getValidation", since="5.0")]
  /**
   * Validates the current data against all of its configured →<code>validators</code>. The
   * returned collection holds an object for each reported problem from a <code>validator</code>.
   * @return The errors collection.
   * @see #validators
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Model.html#method-validate Original Ext JS documentation of 'validate'
   */
  public native function validate():DataErrorCollection;
}
}