package ext.data {
import ext.mixin.IFactoryable;
import ext.util.BasicFilter;

[Native("Ext.data.Query", require)]
/**
 * This class is a filter that compiles from an SQL-like expression. For example:
 * <pre>
 *  store.addFilter(new Ext.data.Query('name like "Bob" or age &lt; 20'));
 * </pre>
 * <p>Queries can also be assigned an <code>→id</code>:</p>
 * <pre>
 *  store.addFilter(new Ext.data.Query({
 *      id: 'myquery',
 *      source: 'name like "Bob" or age &lt; 20'
 *  ));
 * </pre>
 * <p><b><i>Query Syntax</i></b></p>
 * <p>The syntax for a query is SQL-like. The goal of the query syntax is to be as natural
 * to end-users as possible and therefore does not exactly match JavaScript.</p>
 * <p><i>Keyword Operators</i></p>
 * <ul>
 * <li><code>and</code> or <code>&amp;&amp;</code> - Logical AND</li>
 * <li><code>or</code> or <code>||</code> - Logical OR</li>
 * <li><code>like</code> - String containment or regex match</li>
 * <li><code>in</code> - Set membership</li>
 * <li><code>not</code> or <code>!</code> - Logical negation</li>
 * <li><code>between</code> - Bounds check a value (<code>age between 18 and 99</code>)</li>
 * </ul>
 * <p><i>The <code>like</code> Operator</i></p>
 * <p>There are several forms of <code>like</code>. The first uses a simple string on the right-side:</p>
 * <pre>
 *  name like "Bob"
 * </pre>
 * <p>This expression evaluates as <code>true</code> if the <code>name</code> contains the substring <code>'Bob'</code>
 * (ignoring case).</p>
 * <p>The second form will be more typical of those familiar with SQL. It is when the
 * right-side uses the SQL <code>%</code> or <code>_</code> wildcards (or the shell <code>&#42;</code> or <code>?</code> wildcards) and/or
 * character sets (such as <code>'[a-f]'</code> and <code>'[^abc]'</code>):</p>
 * <pre>
 *  name like "[BR]ob%"
 * </pre>
 * <p>If any wildcards are used, the typical SQL meaning is assumed (strict match, including
 * case).</p>
 * <p>The right-side can also use shell wildcards <code>'&#42;'</code> or <code>'?'</code> instead of SQL wildcards.</p>
 * <p>These wildcards can be escaped with a backslash (<code>\</code>) character (the <code>escape</code> keyword
 * is not supported).</p>
 * <pre>
 *  text like 'To be or not to be\?'
 * </pre>
 * <p>The final form of <code>like</code> is when the right-side is a regular expression:</p>
 * <pre>
 *  name like /^Bob/i
 * </pre>
 * <p>This form uses the <code>test()</code> method of the <code>→RegExp</code> to match the value of <code>name</code>.</p>
 * <p><i>The <code>in</code> Operator</i></p>
 * <p>This operator accepts a parenthesized list of values and evaluates to <code>true</code> if the
 * left-side value matches an item in the right-side list:</p>
 * <pre>
 *  name in ("Bob", 'Robert')
 * </pre>
 * <p><i>Relational Operators</i></p>
 * <ul>
 * <li><code>&lt;</code></li>
 * <li><code>&lt;=</code></li>
 * <li><code>&gt;</code></li>
 * <li><code>&gt;=</code></li>
 * </ul>
 * <p><i>Equality and Inequality</i></p>
 * <ul>
 * <li><code>=</code> - Equality after conversion (like <code>==</code> in JavaScript)</li>
 * <li><code>==</code> or <code>===</code> - Strict equality (like <code>===</code> in JavaScript)</li>
 * <li><code>!=</code> or <code>&lt;&gt;</code> - Inequality after conversion (like <code>!=</code> in JavaScript)</li>
 * <li><code>!==</code> - Strict inequality (like <code>!==</code> in JavaScript)</li>
 * </ul>
 * <p><i>Helper Functions</i></p>
 * <p>The following functions can be used in a query:</p>
 * <ul>
 * <li><code>abs(x)</code> - Absolute value of <code>x</code></li>
 * <li><code>avg(...)</code> - The average of all parameters.</li>
 * <li><code>date(d)</code> - Converts the argument into a date.</li>
 * <li><code>lower(s)</code> - The lower-case conversion of the given string.</li>
 * <li><code>max(...)</code> - The maximum value of all parameters.</li>
 * <li><code>min(...)</code> - The minimum value of all parameters.</li>
 * <li><code>sum(...)</code> - The sum of all parameters.</li>
 * <li><code>upper(s)</code> - The upper-case conversion of the given string.</li>
 * </ul>
 * <p>These functions are used as needed in queries, such as:</p>
 * <pre>
 *  upper(name) = 'BOB'
 * </pre>
 * @see #id
 * @see RegExp
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Query.html Original Ext JS documentation of 'Ext.data.Query'
 */
public class Query extends BasicFilter implements IFactoryable {
  /**
   * @inheritDoc
   */
  public function Query(config:Object = null) {
    super(null);
  }

  /**
   * @inheritDoc
   */
  public native function get factoryConfig():Object;

  /**
   * @inheritDoc
   */
  public native function set factoryConfig(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * @default 'ast'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Query.html#cfg-format Original Ext JS documentation of 'format'
   * @see #getFormat()
   * @see #setFormat()
   */
  public native function get format():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set format(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * This config contains the methods that will be made available to queries. To
   * add a custom function:
   * <pre>
   *  Ext.define('MyQuery', {
   *      extend: 'Ext.data.Query',
   *
   *      functions: {
   *          round: function (x) {
   *              return Math.round(x);
   *          },
   *
   *          // When a function name ends with "..." it is called
   *          // with the arguments as an array.
   *          //
   *          'concat...': function (args) {
   *              return args.join('');
   *          }
   *      }
   *  });
   * </pre>
   * @default {
   *     abs: function(arg) {
   *         return Math.abs(arg);
   *     },
   *     'avg...': function(args) {
   *         var count = 0,
   *             sum = 0,
   *             i = args.length,
   *             v;
   *         for (; i-- > 0; /&#42; empty &#42;/
   *         ) {
   *             v = args[i];
   *             if (v != null) {
   *                 sum += v;
   *                 ++count;
   *             }
   *         }
   *         return count ? sum / count : 0;
   *     },
   *     date: function(arg) {
   *         return (arg instanceof Date) ? arg : Ext.Date.parse(arg);
   *     },
   *     lower: function(arg) {
   *         return (arg == null) ? '' : String(arg).toLowerCase();
   *     },
   *     'max...': function(args) {
   *         var ret = null,
   *             i = args.length,
   *             v;
   *         for (; i-- > 0; /&#42; empty &#42;/
   *         ) {
   *             v = args[i];
   *             if (v != null) {
   *                 ret = (ret === null) ? v : (ret < v ? v : ret);
   *             }
   *         }
   *         return ret;
   *     },
   *     'min...': function(args) {
   *         var ret = null,
   *             i = args.length,
   *             v;
   *         for (; i-- > 0; /&#42; empty &#42;/
   *         ) {
   *             v = args[i];
   *             if (v != null) {
   *                 ret = (ret === null) ? v : (ret < v ? ret : v);
   *             }
   *         }
   *         return ret;
   *     },
   *     'sum...': function(args) {
   *         var ret = null,
   *             i = args.length,
   *             v;
   *         for (; i-- > 0; /&#42; empty &#42;/
   *         ) {
   *             v = args[i];
   *             if (v != null) {
   *                 ret = (ret === null) ? v : (ret + v);
   *             }
   *         }
   *         return ret === null ? 0 : ret;
   *     },
   *     upper: function(arg) {
   *         return (arg == null) ? '' : String(arg).toUpperCase();
   *     }
   * }
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Query.html#cfg-functions Original Ext JS documentation of 'functions'
   * @see #getFunctions()
   * @see #setFunctions()
   */
  public native function get functions():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set functions(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * The source text of this query. See <i>class documentation</i> (→<code>ext.data.Query</code>)
   * for syntax details.
   * @default ''
   * @see ext.data.Query
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Query.html#cfg-source Original Ext JS documentation of 'source'
   * @see #getSource()
   * @see #setSource()
   */
  public native function get source():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set source(value:String):void;

  /**
   * Returns the array of serialized <i>filter</i> (→<code>ext.util.Filter</code>) objects equivalent
   * to this query if possible. If this query is empty, this method returns <code>null</code>. If
   * the query cannot be converted without loss into a filters array, this method will
   * return <code>undefined</code>.
   * @see ext.util.Filter
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.query.Converter.html#method-getFilters Original Ext JS documentation of 'getFilters'
   */
  public native function getFilters():Array;

  /**
   * Returns the value of <code>format</code>.
   * @see #format
   */
  public native function getFormat():String;

  /**
   * Returns the value of <code>functions</code>.
   * @see #functions
   */
  public native function getFunctions():Object;

  /**
   * Returns the value of <code>source</code>.
   * @see #source
   */
  public native function getSource():String;

  /**
   * This method should be called if the <code>ast</code> has been manipulated directly.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Query.html#method-refresh Original Ext JS documentation of 'refresh'
   */
  public native function refresh():void;

  /**
   * Sets the value of <code>format</code>.
   * @param format The new value.
   * @see #format
   */
  public native function setFormat(format:String):void;

  /**
   * Sets the value of <code>functions</code>.
   * @param functions The new value.
   * @see #functions
   */
  public native function setFunctions(functions:Object):void;

  /**
   * Sets the value of <code>source</code>.
   * @param source The new value.
   * @see #source
   */
  public native function setSource(source:String):void;
}
}