package ext.data {
import ext.util.Collection;

[Native("Ext.data.Store", require)]
/**
 * @see ext.data.events.Store_operationEvent.BEFORE_PREFETCH
 * @eventType ext.data.events.Store_operationEvent.BEFORE_PREFETCH
 */
[Event(name='onBeforePrefetch', type='ext.data.events.Store_operationEvent')]
/**
 * @see ext.data.events.Store_filtersEvent.FILTER_CHANGE
 * @eventType ext.data.events.Store_filtersEvent.FILTER_CHANGE
 */
[Event(name='onFilterChange', type='ext.data.events.Store_filtersEvent')]
/**
 * @see ext.data.events.Store_grouperEvent.GROUP_CHANGE
 * @eventType ext.data.events.Store_grouperEvent.GROUP_CHANGE
 */
[Event(name='onGroupChange', type='ext.data.events.Store_grouperEvent')]
/**
 * @see ext.data.events.Store_records_successful_operationEvent.PREFETCH
 * @eventType ext.data.events.Store_records_successful_operationEvent.PREFETCH
 */
[Event(name='onPrefetch', type='ext.data.events.Store_records_successful_operationEvent')]
/**
 * The Store class encapsulates a client side cache of →<code>ext.data.Model</code> objects.
 * Stores load data via a →<code>ext.data.proxy.DataProxy</code>, and also provide functions
 * for <i>sorting</i> (→<code>sort()</code>), <i>filtering</i> (→<code>filter()</code>) and querying the
 * <i>model</i> (→<code>ext.data.Model</code>) instances contained within it.
 * <p>Creating a Store is easy - we just tell it the Model and the Proxy to use for loading and saving
 * its data:</p>
 * <pre>
 *  // Set up a model to use in our Store
 *  Ext.define('User', {
 *      extend: 'Ext.data.Model',
 *      fields: [
 *          {name: 'firstName', type: 'string'},
 *          {name: 'lastName',  type: 'string'},
 *          {name: 'age',       type: 'int'},
 *          {name: 'eyeColor',  type: 'string'}
 *      ]
 *  });
 *
 *  var myStore = Ext.create('Ext.data.Store', {
 *      model: 'User',
 *      proxy: {
 *          type: 'ajax',
 *          url: '/users.json',
 *          reader: {
 *              type: 'json',
 *              rootProperty: 'users'
 *          }
 *      },
 *      autoLoad: true
 *  });
 * </pre>
 * <p>In the example above we configured an AJAX proxy to load data from the url '/users.json'.
 * We told our Proxy to use a →<code>ext.data.reader.JsonReader</code> to parse the response
 * from the server into Model object - <i>see the docs on JsonReader</i> (→<code>ext.data.reader.JsonReader</code>)
 * for details.</p>
 * <p><b><i>Inline data</i></b></p>
 * <p>Stores can also load data inline. Internally, Store converts each of the objects we pass in
 * as →<code>data</code> into Model instances:</p>
 * <pre>
 *  Ext.create('Ext.data.Store', {
 *      model: 'User',
 *      data : [
 *          {firstName: 'Peter',   lastName: 'Venkman'},
 *          {firstName: 'Egon',    lastName: 'Spengler'},
 *          {firstName: 'Ray',     lastName: 'Stantz'},
 *          {firstName: 'Winston', lastName: 'Zeddemore'}
 *      ]
 *  });
 * </pre>
 * <p>Loading inline data using the method above is great if the data is in the correct format already
 * (e.g. it doesn't need to be processed by a <i>reader</i> (→<code>ext.data.reader.DataReader</code>)). If your inline
 * data requires processing to decode the data structure, use a
 * →<code>ext.data.proxy.MemoryProxy</code> instead (see the
 * →<code>ext.data.proxy.MemoryProxy</code> docs for an example).</p>
 * <p>Additional data can also be loaded locally using →<code>add()</code>.</p>
 * <p><b><i>Dynamic Loading</i></b></p>
 * <p>Stores can be dynamically updated by calling the →<code>load()</code> method:</p>
 * <pre>
 * store.load({
 *     params: {
 *         group: 3,
 *         type: 'user'
 *     },
 *     callback: function(records, operation, success) {
 *         // do something after the load finishes
 *     },
 *     scope: this
 * });
 * </pre>
 * <p>Here a bunch of arbitrary parameters is passed along with the load request and a callback
 * function is set up to do something after the loading is over.</p>
 * <p><b><i>Loading Nested Data</i></b></p>
 * <p>Applications often need to load sets of associated data - for example a CRM system might load
 * a User and her Orders. Instead of issuing an AJAX request for the User and a series of additional
 * AJAX requests for each Order, we can load a nested dataset and allow the Reader to automatically
 * populate the associated models. Below is a brief example, see the →<code>ext.data.reader.DataReader</code>
 * intro docs for a full explanation:</p>
 * <pre>
 *  var store = Ext.create('Ext.data.Store', {
 *      autoLoad: true,
 *      model: "User",
 *      proxy: {
 *          type: 'ajax',
 *          url: 'users.json',
 *          reader: {
 *              type: 'json',
 *              rootProperty: 'users'
 *          }
 *      }
 *  });
 * </pre>
 * <p>Which would consume a response like this:</p>
 * <pre>
 *  {
 *      "users": [{
 *          "id": 1,
 *          "name": "Peter",
 *          "orders": [{
 *              "id": 10,
 *              "total": 10.76,
 *              "status": "invoiced"
 *         },{
 *              "id": 11,
 *              "total": 13.45,
 *              "status": "shipped"
 *         }]
 *      }]
 *  }
 * </pre>
 * <p>See the →<code>ext.data.reader.DataReader</code> intro docs for a full explanation.</p>
 * <p><b><i>Filtering and Sorting</i></b></p>
 * <p>Stores can be sorted and filtered - in both cases either remotely or locally. The
 * →<code>sorters</code> and →<code>filters</code> are held inside
 * →<code>ext.util.Collection</code> instances to make them easy to manage. Usually it is
 * sufficient to either just specify sorters and filters in the Store configuration or call
 * →<code>sort()</code> or →<code>filter()</code>:</p>
 * <pre>
 *  var store = Ext.create('Ext.data.Store', {
 *      model: 'User',
 *      sorters: [{
 *          property: 'age',
 *          direction: 'DESC'
 *      }, {
 *          property: 'firstName',
 *          direction: 'ASC'
 *      }],
 *
 *      filters: [{
 *          property: 'firstName',
 *          value: /Peter/
 *      }]
 *  });
 * </pre>
 * <p>The new Store will keep the configured sorters and filters in the Collection instances mentioned
 * above. By default, sorting and filtering are both performed locally by the Store - see
 * →<code>remoteSort</code> and →<code>remoteFilter</code> to allow the server to perform these operations
 * instead.</p>
 * <p>Filtering and sorting after the Store has been instantiated is also easy. Calling →<code>filter()</code>
 * adds another filter to the Store and automatically filters the dataset (calling →<code>filter()</code>
 * with no arguments simply re-applies all existing filters).</p>
 * <pre>
 * store.filter('eyeColor', 'Brown');
 * </pre>
 * <p>Change the sorting at any time by calling →<code>sort()</code>:</p>
 * <pre>
 * store.sort('height', 'ASC');
 * </pre>
 * <p>Note that all existing sorters will be removed in favor of the new sorter data (if
 * →<code>sort()</code> is called with no arguments, the existing sorters are just reapplied
 * instead of being removed). To keep existing sorters and add new ones, just add them to the
 * Collection:</p>
 * <pre>
 * store.sorters.add(new Ext.util.Sorter({
 *     property : 'shoeSize',
 *     direction: 'ASC'
 * }));
 *
 * store.sort();
 * </pre>
 * <p><b><i>Registering with StoreManager</i></b></p>
 * <p>Any Store that is instantiated with a →<code>storeId</code> will automatically be registered with
 * the →<code>ext.data.StoreManager</code>. This makes it easy to reuse the same store
 * in multiple views:</p>
 * <pre>
 * //this store can be used several times
 * Ext.create('Ext.data.Store', {
 *     model: 'User',
 *     storeId: 'usersStore'
 * });
 *
 * new Ext.List({
 *     store: 'usersStore',
 *     //other config goes here
 * });
 *
 * new Ext.view.View({
 *     store: 'usersStore',
 *     //other config goes here
 * });
 * </pre>
 * <p><b><i>Further Reading</i></b></p>
 * <p>Stores are backed up by an ecosystem of classes that enables their operation. To gain a full
 * understanding of these pieces and how they fit together, see:</p>
 * <ul>
 * <li>→<code>ext.data.proxy.DataProxy</code> - overview of what Proxies are and how they are used</li>
 * <li>→<code>ext.data.Model</code> - the core class in the data package</li>
 * <li>→<code>ext.data.reader.DataReader</code> - used by any subclass of
 * →<code>ext.data.proxy.ServerProxy</code> to read a response</li>
 * </ul>
 * @see ext.data.Model
 * @see ext.data.proxy.DataProxy
 * @see #sort()
 * @see #filter()
 * @see ext.data.reader.JsonReader
 * @see #data
 * @see ext.data.reader.DataReader
 * @see ext.data.proxy.MemoryProxy
 * @see #add()
 * @see #load()
 * @see #sorters
 * @see #filters
 * @see ext.util.Collection
 * @see #remoteSort
 * @see #remoteFilter
 * @see #storeId
 * @see ext.data.#StoreManager ext.data.StoreManager
 * @see ext.data.proxy.ServerProxy
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html Original Ext JS documentation of 'Ext.data.Store'
 */
public class Store extends ProxyStore implements ILocalStore {
  /**
   * Creates the store.
   * @param config Config object.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-constructor Original Ext JS documentation of 'constructor'
   */
  public function Store(config:Store = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * True to empty the store when loading another page via →<code>loadPage()</code>,
   * →<code>nextPage()</code> or →<code>previousPage()</code>. Setting to false keeps existing records,
   * allowing large data sets to be loaded one page at a time but rendered all together.
   * @default true
   * @see #loadPage()
   * @see #nextPage()
   * @see #previousPage()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#cfg-clearOnPageLoad Original Ext JS documentation of 'clearOnPageLoad'
   * @see #getClearOnPageLoad()
   * @see #setClearOnPageLoad()
   */
  public native function get clearOnPageLoad():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set clearOnPageLoad(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * <code>true</code> to clear anything in the →<code>removed</code> record collection when the store loads.
   * @default true
   * @see #removed
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#cfg-clearRemovedOnLoad Original Ext JS documentation of 'clearRemovedOnLoad'
   * @see #getClearRemovedOnLoad()
   * @see #setClearRemovedOnLoad()
   */
  public native function get clearRemovedOnLoad():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set clearRemovedOnLoad(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * Array of Model instances or data objects to load locally. See "Inline data"
   * above for details.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#cfg-data Original Ext JS documentation of 'data'
   * @see #getData()
   * @see #setData()
   */
  public native function get data():Array;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set data(value:*):void;

  /**
   * The number of times records have been loaded into the store. This includes loads via
   * →<code>loadData()</code> &amp; →<code>loadRecords()</code>.
   * @default 0
   * @see #loadData()
   * @see #loadRecords()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#property-loadCount Original Ext JS documentation of 'loadCount'
   */
  public native function get loadCount():Number;

  [ExtConfig]
  [Bindable]
  /**
   * The session for this store. By specifying a session, it ensures any records that are
   * added to this store are also included in the session. This store does not become a member
   * of the session itself.
   * @since 5.0.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#cfg-session Original Ext JS documentation of 'session'
   * @see #getSession()
   * @see #setSession()
   */
  public native function get session():Session;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set session(value:Session):void;

  [ArrayElementType("ext.data.Model")]
  /**
   * Adds Model instance to the Store. This method accepts either:
   * <ul>
   * <li>An array of Model instances or Model configuration objects.</li>
   * <li>Any number of Model instance or Model configuration object arguments.</li>
   * </ul>
   * <p>The new Model instances will be added at the end of the existing collection.</p>
   * <p>Sample usage:</p>
   * <pre>
   * myStore.add({some: 'data'}, {some: 'other data'});
   * </pre>
   * <p>Note that if this Store is sorted, the new Model instances will be inserted
   * at the correct point in the Store to maintain the sort order.</p>
   * @param record An array of
   * records or configuration objects, or variable number of record or config arguments.
   * @return The record instances that were added.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-add Original Ext JS documentation of 'add'
   */
  public native function add(...record):Array;

  /**
   * (Local sort only) Inserts the passed Record into the Store at the index where it
   * should go based on the current sort information.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-addSorted Original Ext JS documentation of 'addSorted'
   */
  public native function addSorted(record:Model):void;

  /**
   * Runs the aggregate function for all the records in the store.
   * <p>When store is filtered, only items within the filter are aggregated.</p>
   * @param fn The function to execute. The function is called with a single parameter,
   * an array of records for that group.
   * @param scope The scope to execute the function in. Defaults to the store.
   * @param grouped True to perform the operation for each group
   * in the store. The value returned will be an object literal with the key being the group
   * name and the group average being the value. The grouped parameter is only honored if
   * the store has a groupField.
   * @param field The field to get the value from
   * @return An object literal with the group names and their appropriate values.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-aggregate Original Ext JS documentation of 'aggregate'
   */
  public native function aggregate(fn:Function, scope:Object = null, grouped:Boolean = false, field:String = null):Object;

  /**
   * Gets the average value in the store.
   * <p>When store is filtered, only items within the filter are aggregated.</p>
   * @param field The field in each record
   * @param grouped True to perform the operation for each group
   * in the store. The value returned will be an object literal with the key being the group
   * name and the group average being the value. The grouped parameter is only honored if
   * the store has a groupField.
   * @return The average value, if no items exist, 0.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-average Original Ext JS documentation of 'average'
   */
  public native function average(field:String, grouped:Boolean = false):Object;

  /**
   * Collects unique values for a particular dataIndex from this store.
   * <p>Note that the <code>filtered</code> option can also be passed as a separate parameter for
   * compatibility with previous versions.</p>
   * <pre>
   * var store = Ext.create('Ext.data.Store', {
   *     fields: ['name'],
   *     data: [{
   *         name: 'Larry'
   *     }, {
   *         name: 'Darryl'
   *     }, {
   *         name: 'Darryl'
   *     }]
   * });
   *
   * store.collect('name');
   * // returns ["Larry", "Darryl"]
   * </pre>
   * @param property The property to collect
   * @param includeOptions An object which contains options which modify how
   * the store is traversed. For compatibility, this argument may be the <code>allowNull</code>
   * value itself. If so, the next argument is the <code>filtered</code> value.
   * <ul>
   * <li><code>allowNull:Boolean</code> (optional) —
   * Pass true to allow null, undefined or
   * empty string values.
   * </li>
   * <li><code>filtered:Boolean</code> (optional) —
   * Pass <code>true</code> to collect from all records,
   * even ones which are filtered.
   * </li>
   * </ul>
   * @param filtered This argument only applies when the legacy call form
   * is used and <code>includeOptions</code> is actually the <code>allowNull</code> value.
   * @return An array of the unique values
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-collect Original Ext JS documentation of 'collect'
   */
  public native function collect(property:String, includeOptions:Object = null, filtered:Boolean = false):Array;

  /**
   * Commits all Records with <i>outstanding changes</i> (→<code>getModifiedRecords()</code>). To handle updates
   * for changes, subscribe to the Store's <i>update event</i> (→<code>event:onUpdate</code>), and perform
   * updating when the third parameter is Ext.data.Record.COMMIT.
   * @see #getModifiedRecords()
   * @see #event:onUpdate
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-commitChanges Original Ext JS documentation of 'commitChanges'
   */
  public native function commitChanges():void;

  /**
   * Gets the count of items in the store.
   * <p>When store is filtered, only items within the filter are counted.</p>
   * @param grouped True to perform the operation for each group
   * in the store. The value returned will be an object literal with the key being the group
   * name and the count for each group being the value. The grouped parameter is only honored if
   * the store has a groupField.
   * @return the count
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-count Original Ext JS documentation of 'count'
   */
  public native function count(grouped:Boolean = false):Number;

  /**
   * Calls the specified function for each <i>record</i> (→<code>ext.data.Model</code>) in the store.
   * <p>When store is filtered, only loops over the filtered records.</p>
   * @param fn The function to call. The <i>Record</i> (→<code>ext.data.Model</code>) is passed
   * as the first parameter. Returning <code>false</code> aborts and exits the iteration.
   * @param scope The scope (<code>this</code> reference) in which the function is executed.
   * Defaults to the current <i>record</i> (→<code>ext.data.Model</code>) in the iteration.
   * @param includeOptions An object which contains options which
   * modify how the store is traversed. Or simply the <code>filtered</code> option.
   * <ul>
   * <li><code>filtered:Boolean</code> (optional) —
   * Pass <code>true</code> to include filtered out
   * nodes in the iteration.
   * </li>
   * </ul>
   * @see ext.data.Model
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-each Original Ext JS documentation of 'each'
   */
  public native function each(fn:Function, scope:Object = null, includeOptions:* = undefined):void;

  /**
   * Convenience function for getting the first model instance in the store.
   * <p>When store is filtered, will return first item within the filter.</p>
   * @param grouped True to perform the operation for each group
   * in the store. The value returned will be an object literal with the key being the group
   * name and the first record being the value. The grouped parameter is only honored if
   * the store has a groupField.
   * @return The first model instance in the store, or undefined
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-first Original Ext JS documentation of 'first'
   */
  public native function first(grouped:Boolean = false):*;

  /**
   * Get the Record with the specified id.
   * <p>This method is not affected by filtering, lookup will be performed from all records
   * inside the store, filtered or not.</p>
   * @param id The id of the Record to find.
   * @return The Record with the passed id. Returns null if not found.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-getById Original Ext JS documentation of 'getById'
   */
  override public native function getById(id:*):Model;

  /**
   * Get the Record with the specified internalId.
   * <p>This method is not affected by filtering, lookup will be performed from all records
   * inside the store, filtered or not.</p>
   * @param internalId The id of the Record to find.
   * @return The Record with the passed internalId. Returns null if not found.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-getByInternalId Original Ext JS documentation of 'getByInternalId'
   */
  public native function getByInternalId(internalId:*):Model;

  /**
   * Returns the value of <code>clearOnPageLoad</code>.
   * @see #clearOnPageLoad
   */
  public native function getClearOnPageLoad():Boolean;

  /**
   * Returns the value of <code>clearRemovedOnLoad</code>.
   * @see #clearRemovedOnLoad
   */
  public native function getClearRemovedOnLoad():Boolean;

  /**
   * Returns the value of <code>data</code>.
   * @see #data
   */
  public native function getData():*;

  /**
   * Returns the value of <code>session</code>.
   * @see #session
   */
  public native function getSession():Session;

  /**
   * @inheritDoc
   */
  public native function getSummaryRecord():Model;

  /**
   * Get the index of the record within the store.
   * <p>When store is filtered, records outside of filter will not be found.</p>
   * @param record The Ext.data.Model object to find.
   * @return The index of the passed Record. Returns -1 if not found.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-indexOf Original Ext JS documentation of 'indexOf'
   */
  public native function indexOf(record:Model):Number;

  /**
   * Get the index within the store of the Record with the passed id.
   * <p>Like #indexOf, this method is affected by filtering.</p>
   * @param id The id of the Record to find.
   * @return The index of the Record. Returns -1 if not found.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-indexOfId Original Ext JS documentation of 'indexOfId'
   */
  public native function indexOfId(id:String):Number;

  [ArrayElementType("ext.data.Model")]
  /**
   * Inserts Model instances into the Store at the given index and fires the add event.
   * See also →<code>add()</code>.
   * @param index The start index at which to insert the passed Records.
   * @param records An <code>→ext.data.Model</code>
   * instance, the data needed to populate an instance or an array of either of these.
   * @return records The added records
   * @see #add()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-insert Original Ext JS documentation of 'insert'
   * @see ext.data.Model
   */
  public native function insert(index:Number, records:*):Array;

  /**
   * Convenience function for getting the last model instance in the store.
   * <p>When store is filtered, will return last item within the filter.</p>
   * @param grouped True to perform the operation for each group
   * in the store. The value returned will be an object literal with the key being the group
   * name and the last record being the value. The grouped parameter is only honored if
   * the store has a groupField.
   * @return The last model instance in the store, or undefined
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-last Original Ext JS documentation of 'last'
   */
  public native function last(grouped:Boolean = false):*;

  /**
   * Loads an array of data straight into the Store.
   * <p>Using this method is great if the data is in the correct format already (e.g. it doesn't
   * need to be processed by a reader). If your data requires processing to decode the data
   * structure, use a →<code>ext.data.proxy.MemoryProxy</code> or →<code>loadRawData()</code>.</p>
   * @param data Array of data to load. Any non-model instances will
   * be cast into model instances first.
   * @param append <code>true</code> to add the records to the existing records in the
   * store, <code>false</code> to remove the old ones first.
   * @default false
   * @see ext.data.proxy.MemoryProxy
   * @see #loadRawData()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-loadData Original Ext JS documentation of 'loadData'
   */
  public native function loadData(data:Object, append:Boolean = false):void;

  /**
   * Loads a given 'page' of data by setting the start and limit values appropriately. Internally
   * this just causes a normal load operation, passing in calculated 'start' and 'limit' params.
   * @param page The number of the page to load.
   * @param options See options for →<code>load()</code>.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-loadPage Original Ext JS documentation of 'loadPage'
   * @see #load()
   */
  public native function loadPage(page:Number, options:Object = null):void;

  /**
   * Loads data via the bound Proxy's reader.
   * <p>Use this method if you are attempting to load data and want to utilize the configured data
   * reader.</p>
   * <p>As of 4.2, this method will no longer fire the →<code>event:onLoad</code> event.</p>
   * @param data The full JSON object you'd like to load into the Data store.
   * @param append <code>true</code> to add the records to the existing records in the
   * store, <code>false</code> to remove the old ones first.
   * @default false
   * @return <code>true</code> if the reader processed the records correctly. See
   * →<code>ext.data.reader.DataReader.successProperty</code>. If the reader did not process the records,
   * nothing will be added.
   * @see #event:onLoad
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-loadRawData Original Ext JS documentation of 'loadRawData'
   * @see ext.data.reader.DataReader#successProperty
   */
  public native function loadRawData(data:Array, append:Boolean = false):Boolean;

  /**
   * Loads an array of <i>model</i> (→<code>ext.data.Model</code>) instances into the store, fires the
   * datachanged event. This should only usually be called internally when loading from the
   * →<code>ext.data.proxy.DataProxy</code>, when adding records manually use →<code>add()</code>
   * instead
   * @param records The array of records to load
   * @param options
   * <ul>
   * <li><code>addRecords:Boolean</code> (default = <code>false</code>) —
   * Pass <code>true</code> to add these records to the existing
   * records, <code>false</code> to remove the Store's existing records first.
   * </li>
   * </ul>
   * @see ext.data.Model
   * @see ext.data.proxy.DataProxy
   * @see #add()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-loadRecords Original Ext JS documentation of 'loadRecords'
   */
  public native function loadRecords(records:Array, options:Object):void;

  /**
   * Gets the maximum value in the store.
   * <p>When store is filtered, only items within the filter are aggregated.</p>
   * @param field The field in each record
   * @param grouped True to perform the operation for each group
   * in the store. The value returned will be an object literal with the key being the group
   * name and the maximum in the group being the value. The grouped parameter is only honored if
   * the store has a groupField.
   * @return The maximum value, if no items exist, undefined.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-max Original Ext JS documentation of 'max'
   */
  public native function max(field:String, grouped:Boolean = false):Object;

  /**
   * Gets the minimum value in the store.
   * <p>When store is filtered, only items within the filter are aggregated.</p>
   * @param field The field in each record
   * @param grouped True to perform the operation for each group
   * in the store. The value returned will be an object literal with the key being the group
   * name and the minimum in the group being the value. The grouped parameter is only honored if
   * the store has a groupField.
   * @return The minimum value, if no items exist, undefined.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-min Original Ext JS documentation of 'min'
   */
  public native function min(field:String, grouped:Boolean = false):Object;

  /**
   * Loads the next 'page' in the current data set
   * @param options See options for →<code>load()</code>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-nextPage Original Ext JS documentation of 'nextPage'
   * @see #load()
   */
  public native function nextPage(options:Object):void;

  /**
   * Called internally when a Proxy has completed a load request
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-onProxyLoad Original Ext JS documentation of 'onProxyLoad'
   */
  protected native function onProxyLoad(operation:*):void;

  /**
   * Loads the previous 'page' in the current data set
   * @param options See options for →<code>load()</code>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-previousPage Original Ext JS documentation of 'previousPage'
   * @see #load()
   */
  public native function previousPage(options:Object):void;

  /**
   * Query all the cached records in this Store by name/value pair.
   * The parameters will be used to generated a filter function that is given
   * to the queryBy method.
   * <p>This method complements queryBy by generating the query function automatically.</p>
   * <p>This method is not affected by filtering, it will always search <i>all</i> records in the store
   * regardless of filtering.</p>
   * @param property The property to create the filter function for
   * @param value The string/regex to compare the property value to
   * @param anyMatch True to match any part of the string, not just the
   * beginning.
   * @default false
   * @param caseSensitive <code>true</code> to create a case-sensitive regex.
   * @default false
   * @param exactMatch True to force exact match (^ and $ characters
   * added to the regex). Ignored if <code>anyMatch</code> is <code>true</code>.
   * @default false
   * @return The matched records
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-query Original Ext JS documentation of 'query'
   */
  public native function query(property:String, value:*, anyMatch:Boolean = false, caseSensitive:Boolean = false, exactMatch:Boolean = false):Collection;

  /**
   * Query all the cached records in this Store using a filtering function. The specified function
   * will be called with each record in this Store. If the function returns <code>true</code> the record is
   * included in the results.
   * <p>This method is not affected by filtering, it will always search <i>all</i> records in the store
   * regardless of filtering.</p>
   * @param fn The function to be called. It will be passed the following parameters:
   * <ul>
   * <li><code>record:ext.data.Model</code> (optional) —
   * The record to test for filtering. Access field values
   * using →<code>ext.data.Model.get()</code>.
   * </li>
   * <li><code>id:Object</code> (optional) —
   * The ID of the Record passed.
   * </li>
   * </ul>
   * @param scope The scope (this reference) in which the function is executed
   * Defaults to this Store.
   * @return The matched records
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-queryBy Original Ext JS documentation of 'queryBy'
   * @see ext.data.Model#get()
   */
  public native function queryBy(fn:Function, scope:Object = null):Collection;

  /**
   * <i>Rejects</i> (→<code>ext.data.Model.reject()</code>) outstanding changes on all <i>modified records</i> (→<code>getModifiedRecords()</code>) and re-insert any records that were removed locally. Any phantom records
   * will be removed.
   * @see ext.data.Model#reject()
   * @see #getModifiedRecords()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-rejectChanges Original Ext JS documentation of 'rejectChanges'
   */
  public native function rejectChanges():void;

  /**
   * Removes the specified record(s) from the Store, firing the →<code>event:onRemove</code>
   * event for the removed records.
   * <p>After all records have been removed a single <code>→event:onDataChanged</code> is fired.</p>
   * @param records Model instance or
   * array of instances to remove or an array of indices from which to remove records.
   * @param isMove (private)
   * @param silent (private)
   * @see #event:onRemove
   * @see #event:onDataChanged
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-remove Original Ext JS documentation of 'remove'
   */
  public native function remove(records:*, isMove:Boolean = false, silent:Boolean = false):void;

  [ArrayElementType("ext.data.Model")]
  /**
   * Removes all unfiltered items from the store. Filtered records will not be removed.
   * Individual record <code>→event:onRemove</code> events are not fired by this method.
   * @see #event:onRemove
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-removeAll Original Ext JS documentation of 'removeAll'
   */
  override public native function removeAll():void;

  /**
   * Removes the model instance(s) at the given index
   * @param index The record index
   * @param count The number of records to delete
   * @default 1
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-removeAt Original Ext JS documentation of 'removeAt'
   */
  public native function removeAt(index:Number, count:Number = 1):void;

  /**
   * Sets the value of <code>clearOnPageLoad</code>.
   * @param clearOnPageLoad The new value.
   * @see #clearOnPageLoad
   */
  public native function setClearOnPageLoad(clearOnPageLoad:Boolean):void;

  /**
   * Sets the value of <code>clearRemovedOnLoad</code>.
   * @param clearRemovedOnLoad The new value.
   * @see #clearRemovedOnLoad
   */
  public native function setClearRemovedOnLoad(clearRemovedOnLoad:Boolean):void;

  /**
   * Loads an array of data directly into the Store.
   * <p>setData() is ideal if your data's format is already in its appropriate format (e.g. it
   * doesn't need to be processed by a reader). If your data's structure requires processing,
   * use a →<code>ext.data.proxy.MemoryProxy</code> or →<code>loadRawData()</code>.</p>
   * <p>Use →<code>loadData()</code>, →<code>add()</code>, or →<code>insert()</code> if records need to be
   * appended to the current recordset.</p>
   * @param data Array of data to load. Any non-model instances
   * will be cast into model instances first.
   * @see ext.data.proxy.MemoryProxy
   * @see #loadRawData()
   * @see #loadData()
   * @see #add()
   * @see #insert()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-setData Original Ext JS documentation of 'setData'
   */
  public native function setData(data:*):void;

  /**
   * Sets the value of <code>session</code>.
   * @param session The new value.
   * @see #session
   */
  public native function setSession(session:Session):void;

  /**
   * Sums the value of <code>field</code> for each <i>record</i> (→<code>ext.data.Model</code>) in store
   * and returns the result.
   * <p>When store is filtered, only sums items within the filter.</p>
   * @param field A field in each record
   * @param grouped True to perform the operation for each group
   * in the store. The value returned will be an object literal with the key being the group
   * name and the sum for that group being the value. The grouped parameter is only honored if
   * the store has a groupField.
   * @return The sum
   * @see ext.data.Model
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.Store.html#method-sum Original Ext JS documentation of 'sum'
   */
  public native function sum(field:String, grouped:Boolean = false):Number;
}
}