package ext.data.field {
import ext.Base;
import ext.ExtClass;
import ext.data.Model;
import ext.data.summary.SummaryBase;
import ext.mixin.IFactoryable;

[Native("Ext.data.field.Field", require)]
[ExtConfig]
/**
 * Fields are used to define the members of a Model. They aren't instantiated directly;
 * instead, when we create a class that extends →<code>ext.data.Model</code>, it automatically
 * creates Field instances for each field configured in a →<code>ext.data.Model</code>.
 * For example, we might set up a model like this:
 * <pre>
 * Ext.define('User', {
 *     extend: 'Ext.data.Model',
 *     fields: [
 *         'name', 'email',
 *         { name: 'age', type: 'int' },
 *         { name: 'gender', type: 'string', defaultValue: 'Unknown' }
 *     ]
 * });
 * </pre>
 * <p>Four fields will have been created for the User Model - name, email, age and gender.
 * Note that we specified a couple of different formats here; if we only pass in the string
 * name of the field (as with name and email), the field is set up with the 'auto' type.
 * It's as if we'd done this instead:</p>
 * <pre>
 * Ext.define('User', {
 *     extend: 'Ext.data.Model',
 *     fields: [
 *         { name: 'name', type: 'auto' },
 *         { name: 'email', type: 'auto' },
 *         { name: 'age', type: 'int' },
 *         { name: 'gender', type: 'string', defaultValue: 'Unknown' }
 *     ]
 * });
 * </pre>
 * <p><b>Field Types</b></p>
 * <p>Fields come in various types. When declaring a field, the <code>type</code> property is used to
 * specify the type of <code>Field</code> derived class used to manage values.</p>
 * <p>The predefined set of types are:</p>
 * <ul>
 * <li><i>auto</i> (→<code>ext.data.field.DataField</code>) (Default, implies no conversion)</li>
 * <li><i>string</i> (→<code>ext.data.field.StringDataField</code>)</li>
 * <li><i>int</i> (→<code>ext.data.field.IntegerDataField</code>)</li>
 * <li><i>number</i> (→<code>ext.data.field.NumberDataField</code>)</li>
 * <li><i>boolean</i> (→<code>ext.data.field.BooleanField</code>)</li>
 * <li><i>date</i> (→<code>ext.data.field.DateDataField</code>)</li>
 * </ul>
 * <p><b>Conversion</b></p>
 * <p>When reading fields it is often necessary to convert the values received before using
 * them or storing them in records. To handle these cases there is the
 * <code>→convert()</code> method. This method is passed the received value (as
 * well as the current record instance, but see below) and it returns the value to carry
 * forward.</p>
 * <p>For <code>auto</code> fields there is no <code>→convert()</code> method. This is for
 * efficiency. For other field types, there are often <code>convert</code> methods. You can provide
 * a <code>→convert</code> config when the field is defined like this:</p>
 * <pre>
 *  {
 *      name: 'timestamp',
 *
 *      convert: function (value) {
 *          return new Date(value);
 *      }
 *  }
 * </pre>
 * <p>While this can be convenient, see below for details on defining Custom Types as that is
 * often a better practice and avoids repeating these functions.</p>
 * <p>Note that when a <code>→defaultValue</code> is specified, it will also be passed through to
 * <code>convert</code> (either to the <code>→convert()</code> method or to the
 * <code>→convert config)</code>.</p>
 * <p><b><i>Calculated Values</i></b></p>
 * <p>In some cases fields are the result of a calculation from other fields. Historically
 * this was a second role for <code>→convert()</code> but that has some short
 * comings. The simpler solution is the <code>→calculate</code> config.</p>
 * <p>Values produced by <code>→calculate</code> and <code>→convert()</code>
 * are stored in the record as with any other field. In fact, if we define a calculated
 * "firstName" field and log out all of the data, we'll see this:</p>
 * <pre>
 * var ed = Ext.create('User', { name: 'Ed Spencer' });
 *
 * console.log(ed.data);
 *
 * //outputs this:
 * {
 *     age: 0,
 *     email: "",
 *     firstName: "Ed",  // calculated field
 *     gender: "Unknown",
 *     name: "Ed Spencer"
 * }
 * </pre>
 * <p><i>Using <code>calculate (→calculate)</code></i></p>
 * <pre>
 *  {
 *      name: 'firstName',
 *
 *      calculate: function (data) {
 *          return data.name.split(' ')[0];
 *      }
 *  }
 * </pre>
 * <p>Using <code>→calculate</code> is the simplest and safest way to define a
 * calculated field. The most important part of this is that, internally, the code of the
 * supplied function is parsed to extract its dependencies. In this case, the "name" field
 * is the only dependency. This means that "firstName" will only need to be recalculated
 * when "name" is modified.</p>
 * <p><b>Note:</b> Fields used by the calculate method must be explicitly defined in the
 * <i>fields</i> (→<code>ext.data.Model.fields</code>) of the model.</p>
 * <p><i>Using <code>convert</code></i></p>
 * <p>Following is the equivalent technique using <code>→convert</code></p>
 * <pre>
 *  {
 *      name: 'firstName',
 *
 *      convert: function (value, record) {
 *          return record.get('name').split(' ')[0];
 *      },
 *
 *      depends: [ 'name' ]
 *  }
 * </pre>
 * <p>When a <code>→convert()</code> function accepts a 2nd argument (a reference to
 * the record), it is considered a calculated field. If a <code>→depends</code>
 * config is not provided then this field's dependencies are unknown. In this case, the
 * <code>→depends</code> are provided as would be automatically determined with
 * the <code>→calculate</code> config.</p>
 * <p><i>Updating</i></p>
 * <p>Fields modified with the →<code>ext.data.Model.set()</code> method will have their stored
 * value set using the convert / calculate method when present.</p>
 * <p>For example:</p>
 * <pre>
 * Ext.define('MyApp.model.Employee', {
 *     extend: 'Ext.data.Model',
 *     fields: [{
 *         name: 'salary',
 *         convert: function (val) {
 *             var startingBonus = val &#42; .1;
 *             return val + startingBonus;
 *         }
 *     }],
 *     convertOnSet: false
 * });
 *
 * var tina = Ext.create('MyApp.model.Employee', {
 *     salary: 50000
 * });
 *
 * console.log(tina.get('salary')); // logs 55000
 *
 * tina.set('salary', 60000);
 * console.log(tina.get('salary')); // logs 60000
 * </pre>
 * <p>This default behavior can be disabled by setting the Model's
 * <code>→ext.data.Model.convertOnSet</code> config to <code>false</code>.</p>
 * <p><b>Note:</b> convertOnSet <code>false</code> only prevents the convert / calculate call when the
 * set <code>fieldName</code> param matches the field's <code>→name</code>. See
 * →<code>ext.data.Model.convertOnSet</code> for additional details.</p>
 * <p><i>Dependencies</i></p>
 * <p>When a field's <code>→convert()</code> method processes values from the record
 * (vs. just the field's value), it is best to also provide a <code>→depends</code> config as shown
 * above. Fields that provide a <code>→calculate</code> method must follow the
 * proper form for using fields so that dependencies can be extracted.</p>
 * <p>Calculated fields are processed after other fields based on their dependencies. Fields
 * with <code>→convert()</code> methods that use the provided record that do <i>not</i>
 * specify a <code>→depends</code> config are processed as a group after all other
 * fields since such converters can rely on anything in the record. The order of processing
 * these fields with respect to each other is unspecified and should not be relied upon.</p>
 * <p><b>Serialization</b></p>
 * <p>To handle the inverse scenario of <code>convert</code> there is the <code>serialize</code> method. This
 * method is called to produce the value to send to a server based on the internal value
 * as would be returned from <code>convert</code>. In most cases, these methods should "round trip"
 * a value:</p>
 * <pre>
 *  assertEqual(value, field.serialize(field.convert(value)));
 * </pre>
 * <p>By default, only <code>date (→ext.data.field.DateDataField)</code> fields have a <code>serialize</code> method.
 * Other types simply send their value unmodified.</p>
 * <p><b>Custom Types</b></p>
 * <p>Developers may create their own application-specific data types by deriving from this
 * class. This is typically much better than applying multiple configuration values on
 * field instances as these often become repetitive.</p>
 * <p>To illustrate, we define a "time" field type that stores a time-of-day represented as a
 * number of minutes since Midnight.</p>
 * <pre>
 *  Ext.define('App.field.Time', {
 *      extend: 'Ext.data.field.Field',
 *
 *      alias: 'data.field.time',
 *
 *      timeFormat: 'g:i',
 *
 *      convert: function (value) {
 *          if (value &amp;&amp; Ext.isString(value)) {
 *              var date = Ext.Date.parse(value, this.timeFormat);
 *              if (!date) {
 *                  return null;
 *              }
 *              return (date.getHours() - 1) &#42; 60 + date.getMinutes();
 *          }
 *          return value;
 *      }
 *  });
 * </pre>
 * <p><b><i>Validation</i></b></p>
 * <p>Custom field types can override the <code>→validate()</code> method or
 * provide a set of <code>→validators</code>.</p>
 * <pre>
 *  Ext.define('App.field.PhoneNumber', {
 *      extend: 'Ext.data.field.Field',
 *
 *      alias: 'data.field.phonenumber',
 *
 *      // Match U.S. phone numbers for example purposes
 *      validators: {
 *          type: 'format',
 *          matcher: /\d{3}\-\d{3}\-\d{4}/
 *      }
 *  });
 * </pre>
 * <p>Once the class is defined, fields can be declared using the new type (based on its
 * <code>alias</code>) like so:</p>
 * <pre>
 *  Ext.define('App.model.PhoneCall', {
 *      fields: [
 *          { name: 'startTime', type: 'time' },
 *          { name: 'phoneNumber', type: 'phonenumber' }
 *      ]
 *  });
 * </pre>
 * @see ext.data.Model
 * @see ext.data.field.DataField
 * @see ext.data.field.StringDataField
 * @see ext.data.field.IntegerDataField
 * @see ext.data.field.NumberDataField
 * @see ext.data.field.BooleanField
 * @see ext.data.field.DateDataField
 * @see #convert()
 * @see #convert
 * @see #defaultValue
 * @see #calculate
 * @see ext.data.Model#fields
 * @see #depends
 * @see ext.data.Model#set()
 * @see ext.data.Model#convertOnSet
 * @see #name
 * @see #validate()
 * @see #validators
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html Original Ext JS documentation of 'Ext.data.field.Field'
 */
public class DataField extends Base implements IFactoryable {
  /**
   * @param config @inheritDoc
   */
  public function DataField(config:DataField = null) {
    super();
  }

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * Use when converting received data into a <i><code>int</code></i> (→<code>ext.data.field.IntegerDataField</code>),
   * <i><code>float</code></i> (→<code>ext.data.field.NumberDataField</code>), <i><code>bool</code></i> (→<code>ext.data.field.BooleanField</code>)
   * or <i><code>string</code></i> (→<code>ext.data.field.StringDataField</code>) type. If the value cannot be
   * parsed, <code>null</code> will be used if allowNull is true, otherwise a default value for
   * that type will be used:
   * <ul>
   * <li>for <code>int</code> and <code>float</code> - <code>0</code>.</li>
   * <li>for <code>string</code> - <code>""</code>.</li>
   * <li>for <code>bool</code> - <code>false</code>.</li>
   * </ul>
   * <p>Note that when parsing of <i><code>date</code></i> (→<code>ext.data.field.DateDataField</code>) type fails, the value
   * will be <code>null</code> regardless of this setting.</p>
   * @default false
   * @see ext.data.field.IntegerDataField
   * @see ext.data.field.NumberDataField
   * @see ext.data.field.BooleanField
   * @see ext.data.field.StringDataField
   * @see ext.data.field.DateDataField
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-allowNull Original Ext JS documentation of 'allowNull'
   * @see #getAllowNull()
   */
  public native function get allowNull():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set allowNull(value:Boolean):void;

  [ExtConfig]
  /**
   * This config defines a simple field calculation function. A calculate method only
   * has access to the record data and should return the value of the calculated field.
   * When provided in this way, the <code>→depends</code> config is automatically determined by
   * parsing the <code>→calculate</code> function. For example:
   * <pre>
   *  fields: [{
   *      name: 'firstName',
   *      type: 'string'
   *  },{
   *      name: 'lastName',
   *      type: 'string'
   *  },{
   *      name: 'fullName',
   *      calculate: function (data) {
   *          return data.firstName + ' ' + data.lastName;
   *      }
   *  }]
   * </pre>
   * <p>The above 'fullName' field is equivalent to:</p>
   * <pre>
   *  {
   *      name: 'fullName',
   *      convert: function (v, rec) {
   *          return rec.get('firstName') + ' ' + rec.get('lastName');
   *      },
   *      depends: ['firstName', 'lastName']
   *  }
   * </pre>
   * <p>The restrictions on form for a <code>→calculate</code> method are that the accesses to field
   * values must match the following regular expression (case insensitive):</p>
   * <pre>
   *  data.([a-z_][a-z0-9_]&#42;)
   *  // where 'data' is the param passed to the calculate method
   * </pre>
   * <p>The only advantage of a <code>→calculate</code> method over a <code>convert</code> method is automatic
   * determination of <code>→depends</code>.</p>
   * <p><b>Note:</b> The use of calculate and →<code>convert()</code> are exclusive. The
   * calculate method will override the convert method if both are configured.</p>
   * <p><b>Note:</b> Fields used by the calculate method must be explicitly defined in the
   * <i>fields</i> (→<code>ext.data.Model.fields</code>) of the model.</p>
   * <ul>
   * <li><code>data:Object</code> —
   * An object with all values for each field in the parent
   * model. See →<code>ext.data.Model.getData()</code>.
   * </li>
   * <li><code>null</code> —
   * value The value of the calculated field
   * </li>
   * </ul>
   * @see #depends
   * @see #calculate
   * @see #convert()
   * @see ext.data.Model#fields
   * @see ext.data.Model#getData()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-calculate Original Ext JS documentation of 'calculate'
   */
  public native function get calculate():Function;

  [ExtConfig]
  /**
   * @private
   */
  public native function set calculate(value:Function):void;

  /**
   * This property is <code>true</code> if this field has a <code>→calculate</code>
   * method or a <code>→convert()</code> method that operates on the entire
   * record as opposed to just the data value. This property is determined from the
   * <code>length</code> of the <code>→convert()</code> function which means this is
   * <i>not</i> calculated:
   * <pre>
   *  convert: function (value) {
   *      return ...
   *  }
   * </pre>
   * <p>While this <i>is</i> calculated:</p>
   * <pre>
   *  convert: function (value, record) {
   *      return ...
   *  }
   * </pre>
   * <p><b>NOTE:</b> It is recommended for such fields to use
   * <code>→calculate</code> or explicitly specify the fields used by
   * <code>→convert()</code> using <code>→depends</code>.</p>
   * @default false
   * @see #calculate
   * @see #convert()
   * @see #depends
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#property-calculated Original Ext JS documentation of 'calculated'
   */
  public native function get calculated():Boolean;

  /**
   * A function which converts the value provided by the Reader into the value that will
   * be stored in the record. This method can be overridden by a derived class or set as
   * a <code>→convert</code> config.
   * <p>If configured as <code>null</code>, then no conversion will be applied to the raw data property
   * when this Field is read. This will increase performance. but you must ensure that
   * the data is of the correct type and does not <i>need</i> converting.</p>
   * <p>Example of convert functions:</p>
   * <pre>
   * function fullName(v, record){
   *     return record.data.last + ', ' + record.data.first;
   * }
   *
   * function location(v, record){
   *     return !record.data.city ? '' : (record.data.city + ', ' + record.data.state);
   * }
   *
   * Ext.define('Dude', {
   *     extend: 'Ext.data.Model',
   *     fields: [
   *         {name: 'fullname',  convert: fullName},
   *         {name: 'firstname', mapping: 'name.first'},
   *         {name: 'lastname',  mapping: 'name.last'},
   *         {name: 'city', defaultValue: 'unknown'},
   *         'state',
   *         {name: 'location',  convert: location}
   *     ]
   * });
   *
   * // create the data store
   * var store = Ext.create('Ext.data.Store', {
   *     model: 'Dude',
   *     proxy: {
   *         type: 'memory',
   *         reader: {
   *             type: 'json',
   *             rootProperty: 'daRoot',
   *             totalProperty: 'total'
   *         }
   *     }
   * });
   *
   * var myData = [
   *     { key: 1,
   *       name: { first: 'Fat',    last:  'Albert' }
   *       // notice no city, state provided in data object
   *     },
   *     { key: 2,
   *       name: { first: 'Barney', last:  'Rubble' },
   *       city: 'Bedrock', state: 'Stoneridge'
   *     },
   *     { key: 3,
   *       name: { first: 'Cliff',  last:  'Claven' },
   *       city: 'Boston',  state: 'MA'
   *     }
   * ];
   * </pre>
   * @param value The data value as read by the Reader, if undefined will use
   * the configured <code>→defaultValue</code>.
   * @param record The data object containing the Model as read so far
   * by the Reader. Note that the Model may not be fully populated at this point as the
   * fields are read in the order that they are defined.
   * →<code>ext.data.Model.fields</code> array.
   * @return The converted value for storage in the record.
   * @see #convert
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-convert Original Ext JS documentation of 'convert'
   * @see #defaultValue
   * @see ext.data.Model#fields
   */
  public native function get convert():Function;

  [ExtConfig]
  /**
   * @private
   */
  public native function set convert(value:Function):void;

  [ExtConfig("convert")]
  [Bindable(style="methods")]
  /**
   * If specified this config overrides the <code>→convert()</code> method.
   * See also <code>→calculate</code> for simple field calculations.
   * <p><b>Note:</b> The use of →<code>calculate</code> and convert are exclusive. The calculate
   * method will override the convert method if both are configured.</p>
   * @see #convert()
   * @see #calculate
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-convert Original Ext JS documentation of 'convert'
   * @see #getConvert()
   */
  public native function get convert_():Function;

  [ExtConfig("convert")]
  /**
   * @private
   */
  public native function set convert_(value:Function):void;

  [ExtConfig]
  /**
   * A critical field is a field that must always be sent to the server even if it has
   * not changed. The most common example of such a field is the "id" of a record (see
   * <code>→ext.data.Model.idProperty</code> but the
   * <code>→ext.data.Model.versionProperty</code> is similarly a <code>→critical</code> field.
   * @default false
   * @see ext.data.Model#idProperty
   * @see ext.data.Model#versionProperty
   * @see #critical
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-critical Original Ext JS documentation of 'critical'
   */
  public native function get critical():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set critical(value:Boolean):void;

  /**
   * The default message to present for an invalid field.
   * @default 'This field is invalid'
   * @since 5.0.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#property-defaultInvalidMessage Original Ext JS documentation of 'defaultInvalidMessage'
   */
  public native function get defaultInvalidMessage():String;

  /**
   * @private
   */
  public native function set defaultInvalidMessage(value:String):void;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * The default value used when the creating an instance from a raw data object,
   * and the property referenced by the <code>→ext.data.field.DataField.mapping</code>
   * does not exist in that data object.
   * <p>The value <code>undefined</code> prevents defaulting in a value.</p>
   * @default undefined
   * @see ext.data.field.DataField#mapping
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-defaultValue Original Ext JS documentation of 'defaultValue'
   * @see #getDefaultValue()
   */
  public native function get defaultValue():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set defaultValue(value:*):void;

  /**
   * The class (derived from →<code>ext.data.Model</code>) that defined this field.
   * <pre>
   *  Ext.define('MyApp.models.Foo', {
   *      extend: 'Ext.data.Model',
   *
   *      fields: [
   *          { name: 'bar' }
   *      ],
   *      ...
   *  });
   *
   *  var barField = MyApp.models.Foo.getField('bar');
   *
   *  alert(barField.definedBy === MyApp.models.Foo); // alerts 'true'
   * </pre>
   * <p>When a field is inherited, this value will reference the class that originally
   * defined the field.</p>
   * <pre>
   *  Ext.define('MyApp.models.Base', {
   *      extend: 'Ext.data.Model',
   *
   *      fields: [
   *          { name: 'foo' }
   *      ],
   *      ...
   *  });
   *
   *  Ext.define('MyApp.models.Derived', {
   *      extend: 'MyApp.models.Base',
   *
   *      fields: [
   *          { name: 'bar' }
   *      ],
   *      ...
   *  });
   *
   *  var fooField = MyApp.models.Derived.getField('foo');
   *
   *  alert(fooField.definedBy === MyApp.models.Base); // alerts 'true'
   * </pre>
   * @see ext.data.Model
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#property-definedBy Original Ext JS documentation of 'definedBy'
   */
  public native function get definedBy():ExtClass;

  /**
   * @private
   */
  public native function set definedBy(value:ExtClass):void;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * The field name or names within the →<code>ext.data.Model</code> on which the value
   * of this field depends, and from which a new value may be calculated. These values
   * are the values used by the <code>convert</code> method. If you do not have a <code>convert</code> method
   * then this config should not be specified.
   * <p>Before using this config you should consider if using a <code>→calculate</code> method instead
   * of a <code>convert</code> method would be simpler.</p>
   * <p>Whenever any of the named fields are set using the →<code>ext.data.Model.set()</code>
   * method, this fields will have its <code>convert</code> method called passing the
   * <i>record</i> (→<code>ext.data.Model</code>) so that the dependent value can be calculated from
   * all fields which it needs.</p>
   * <p>For example, to display a person's full name, using two separate <code>firstName</code> and
   * <code>lastName</code> fields, configure the name field like this:</p>
   * <pre>
   * {
   *     name: 'name',
   *
   *     // Will be called whenever forename or surname fields are set
   *     convert: function (v, rec) {
   *         return rec.get('firstName') + ' ' + rec.get('lastName');
   *     },
   *
   *     depends: [ 'firstName', 'lastName' ],
   *
   *     // It should not be returned to the server - it's not a database field
   *     persist: false
   * }
   * </pre>
   * <p>Note that if you do not want the calculated field to be part of the field set sent
   * back to the server when the store is synchronized, you should configure the field
   * with <code>→persist</code> set to <code>false</code>.</p>
   * @default null
   * @see ext.data.Model
   * @see #calculate
   * @see ext.data.Model#set()
   * @see #persist
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-depends Original Ext JS documentation of 'depends'
   * @see #getDepends()
   */
  public native function get depends():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set depends(value:*):void;

  /**
   * If this property is specified by the target class of this mixin its properties are
   * used to configure the created <code>→ext.Factory</code>.
   * @default {
   *     defaultProperty: 'name'
   * }
   * @see ext.Factory
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#property-factoryConfig Original Ext JS documentation of 'factoryConfig'
   */
  public native function get factoryConfig():Object;

  /**
   * @private
   */
  public native function set factoryConfig(value:Object):void;

  /**
   * This property is set to <code>true</code> if this is an <i>id</i> (→<code>ext.data.Model.idProperty</code>)
   * field.
   * @default false
   * @see ext.data.Model#idProperty
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#property-identifier Original Ext JS documentation of 'identifier'
   */
  public native function get identifier():Boolean;

  [ExtConfig]
  /**
   * (Optional) A path expression for use by the →<code>ext.data.reader.DataReader</code>
   * implementation that is creating the →<code>ext.data.Model</code> to extract the
   * Field value from the data object. If the path expression is the same as the field
   * name, the mapping may be omitted. A function may be passed to do complex data
   * extraction. The examples below are simple just to demonstrate the capability,
   * typically, a function would not be used to extract such simple data.
   * <p>The form of the mapping expression depends on the Reader being used.</p>
   * <ul>
   * <li>
   * <p>→<code>ext.data.reader.JsonReader</code></p>
   * <p>The mapping is a string containing the javascript expression to reference the
   * data from an element of the data item's
   * →<code>ext.data.reader.JsonReader.rootProperty</code> Array. Defaults to
   * the field name. If a function is passed, a single argument is received which
   * contains the raw json object:</p>
   * <pre>
   * // Server returns [{"name": "Foo", "age": 1}, {"name": "Bar", "age": 2}]
   * mapping: function(data) {
   *     return data.name;
   * }
   * </pre>
   * </li>
   * <li>
   * <p>→<code>ext.data.reader.XmlReader</code></p>
   * <p>The mapping is an →<code>ext.dom.DomQuery</code> path to the data item relative to the DOM
   * element that represents the →<code>ext.data.reader.XmlReader.record</code>. Defaults
   * to the field name. If a function is passed, a single argument is received which
   * contains the record node:</p>
   * <pre>
   * // Server returns:
   * // &lt;Root&gt;
   * //     &lt;Person&gt;
   * //         &lt;Name&gt;Foo&lt;/Name&gt;
   * //         &lt;Age&gt;1&lt;/Age&gt;
   * //     &lt;/Person&gt;
   * //     &lt;Person&gt;
   * //         &lt;Name&gt;Bar&lt;/Name&gt;
   * //         &lt;Age&gt;2&lt;/Age&gt;
   * //     &lt;/Person&gt;
   * // &lt;/Root&gt;
   * mapping: function(data) {
   *     return data.firstChild.textContent;
   * }
   * </pre>
   * </li>
   * <li>
   * <p>→<code>ext.data.reader.ArrayReader</code></p>
   * <p>The mapping is a number indicating the Array index of the field's value.
   * Defaults to the field specification's Array position. If a function is passed, a
   * single argument is received which contains the child array.</p>
   * <pre>
   * // Server returns [["Foo", 1], ["Bar", 2]]
   * mapping: function(data) {
   *     return data[0];
   * }
   * </pre>
   * </li>
   * </ul>
   * <p>If a more complex value extraction strategy is required, then configure the Field
   * with a →<code>convert</code> function. This is passed the whole row object, and may
   * interrogate it in whatever way is necessary in order to return the desired data.</p>
   * @default null
   * @see ext.data.reader.DataReader
   * @see ext.data.Model
   * @see ext.data.reader.JsonReader
   * @see ext.data.reader.JsonReader#rootProperty
   * @see ext.data.reader.XmlReader
   * @see ext.dom.#DomQuery ext.dom.DomQuery
   * @see ext.data.reader.XmlReader#record
   * @see ext.data.reader.ArrayReader
   * @see #convert
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-mapping Original Ext JS documentation of 'mapping'
   * @see #getMapping()
   */
  public native function get mapping():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set mapping(value:*):void;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * The name by which the field is referenced within the Model. This is referenced by,
   * for example, the <code>dataIndex</code> property in column definition objects passed to
   * →<code>ext.grid.property.PropertyGridHeaderContainer</code>.
   * <p>Note: In the simplest case, if no properties other than <code>→name</code> are required, a
   * field definition may consist of just a String for the field name.</p>
   * @see ext.grid.property.PropertyGridHeaderContainer
   * @see #name
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-name Original Ext JS documentation of 'name'
   * @see #getName()
   */
  public native function get name():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set name(value:String):void;

  /**
   * The position of this field in the →<code>ext.data.Model</code> in which it was defined.
   * @default undefined
   * @see ext.data.Model
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#property-ordinal Original Ext JS documentation of 'ordinal'
   */
  public native function get ordinal():Number;

  /**
   * @private
   */
  public native function set ordinal(value:Number):void;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * False to exclude this field from the →<code>ext.data.Model.modified</code> fields in a
   * record. This will also exclude the field from being written using a
   * →<code>ext.data.writer.DataWriter</code>. This option is useful when fields are used to keep
   * state on the client but do not need to be persisted to the server.
   * <p>Defaults to <code>false</code> for <code>→calculated</code> fields and <code>true</code> otherwise.</p>
   * @default null
   * @see ext.data.Model#modified
   * @see ext.data.writer.DataWriter
   * @see #calculated
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-persist Original Ext JS documentation of 'persist'
   * @see #getPersist()
   */
  public native function get persist():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set persist(value:Boolean):void;

  [ExtConfig]
  /**
   * The <i>name</i> (→<code>ext.data.Model.entityName</code>) of the entity referenced by this field.
   * In most databases, this relationship is represented by a "foreign key". That is, a
   * value for such a field matches the value of the
   * <i>id</i> (→<code>ext.data.Model.idProperty</code>) for an entity of this type.
   * <p>For further documentation, see →<code>ext.data.schema.Reference</code>.</p>
   * @default null
   * @see ext.data.Model#entityName
   * @see ext.data.Model#idProperty
   * @see ext.data.schema.Reference
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-reference Original Ext JS documentation of 'reference'
   */
  public native function get reference():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set reference(value:*):void;

  [ExtConfig("serialize")]
  /**
   * A function which converts the Model's value for this Field into a form which can be used
   * by whatever →<code>ext.data.writer.DataWriter</code> is being used to sync data with the server.
   * <ul>
   * <li><code>value</code> —
   * The Field's value - the value to be serialized.
   * </li>
   * <li><code>record:ext.data.Model</code> —
   * The record being serialized.
   * </li>
   * <li><code>null:String</code> —
   * The string that represents the Field's value.
   * </li>
   * </ul>
   * @see ext.data.writer.DataWriter
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-serialize Original Ext JS documentation of 'serialize'
   */
  public native function get serialize_():*;

  [ExtConfig("serialize")]
  /**
   * @private
   */
  public native function set serialize_(value:*):void;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * A function which converts a Field's value to a comparable value in order to ensure
   * correct sort ordering.
   * <p>Predefined functions are provided in →<code>ext.data.SortTypes</code>. A custom sort
   * example:</p>
   * <pre>
   * // current sort     after sort we want
   * // +-+------+          +-+------+
   * // |1|First |          |1|First |
   * // |2|Last  |          |3|Second|
   * // |3|Second|          |2|Last  |
   * // +-+------+          +-+------+
   *
   * sortType: function(value) {
   *    switch (value.toLowerCase()) // native toLowerCase():
   *    {
   *       case 'first': return 1;
   *       case 'second': return 2;
   *       default: return 3;
   *    }
   * }
   * </pre>
   * <p>May also be set to a String value, corresponding to one of the named sort types in
   * →<code>ext.data.SortTypes</code>.</p>
   * @see ext.data.#SortTypes ext.data.SortTypes
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-sortType Original Ext JS documentation of 'sortType'
   * @see #getSortType()
   */
  public native function get sortType():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set sortType(value:*):void;

  /**
   * A regular expression for stripping non-numeric characters from a numeric value.
   * This should be overridden for localization.
   * @default /[$,%]/g
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#property-stripRe Original Ext JS documentation of 'stripRe'
   */
  public native function get stripRe():RegExp;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * The summary type for this field. This is used to calculate a
   * summary value by the →<code>ext.data.Model</code>.
   * <ul>
   * <li>If a string, it should be an alias for one of the Ext.data.summary types.</li>
   * <li>If an object, a config for one of the Ext.data.summary types.</li>
   * <li>If a function, it should match the signature for
   * →<code>ext.data.summary.SummaryBase.calculate()</code>.</li>
   * </ul>
   * @default null
   * @since 6.5.0
   * @see ext.data.Model
   * @see ext.data.summary.SummaryBase#calculate()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-summary Original Ext JS documentation of 'summary'
   * @see #getSummary()
   */
  public native function get summary():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set summary(value:*):void;

  /**
   * The type of field.
   */
  public native function get type():String;

  /**
   * @private
   */
  public native function set type(value:String):void;

  [ExtConfig]
  /**
   * <code>true</code> if the value of this field is unique amongst all instances. When used with a
   * <code>→reference</code> this describes a "one-to-one" relationship. It is almost always the
   * case that a <code>→unique</code> field cannot also be <i>nullable</i> (→<code>allowBlank</code>).
   * @default false
   * @see #reference
   * @see #unique
   * @see #allowBlank
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-unique Original Ext JS documentation of 'unique'
   */
  public native function get unique():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set unique(value:Boolean):void;

  [ExtConfig]
  /**
   * An array of <i>validators</i> (→<code>ext.data.validator.Validator</code>) for this field. These
   * <code>→validators</code> will only be passed a field value to validate.
   * @see ext.data.validator.Validator
   * @see #validators
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#cfg-validators Original Ext JS documentation of 'validators'
   */
  public native function get validators():Array;

  [ExtConfig]
  /**
   * @private
   */
  public native function set validators(value:Array):void;

  /**
   * Compares two values to retrieve their relative position in sort order, taking into account
   * any →<code>sortType</code>. Also see →<code>compare()</code>.
   * @param value1 The first value.
   * @param value2 The second value.
   * @return <code>-1</code> if <code>value1</code> is less than <code>value2</code>; <code>1</code> if <code>value1</code> is greater than
   * <code>value2</code>; <code>0</code> otherwise.
   * @see #sortType
   * @see #compare()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-collate Original Ext JS documentation of 'collate'
   */
  public native function collate(value1:Object, value2:Object):Number;

  /**
   * Compares two values to retrieve their relative position in sort order. Also see
   * →<code>collate()</code>.
   * @param value1 The first value.
   * @param value2 The second value.
   * @return <code>-1</code> if <code>value1</code> is less than <code>value2</code>; <code>1</code> if <code>value1</code> is greater than
   * <code>value2</code>; <code>0</code> otherwise.
   * @see #collate()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-compare Original Ext JS documentation of 'compare'
   */
  public native function compare(value1:Object, value2:Object):Number;

  /**
   * Gets allowBlank for this field. See →<code>allowBlank</code>.
   * @return allowBlank
   * @see #allowBlank
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-getAllowBlank Original Ext JS documentation of 'getAllowBlank'
   */
  public native function getAllowBlank():Boolean;

  /**
   * Gets allowNull for this field. See →<code>allowNull</code>.
   * @return allowNull
   * @see #allowNull
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-getAllowNull Original Ext JS documentation of 'getAllowNull'
   */
  public native function getAllowNull():Boolean;

  /**
   * Gets converter for this field. See →<code>convert()</code>.
   * @return convert
   * @see #convert()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-getConvert Original Ext JS documentation of 'getConvert'
   */
  public native function getConvert():Function;

  /**
   * Gets the defaultValue for this field. See →<code>defaultValue</code>.
   * @return defaultValue
   * @see #defaultValue
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-getDefaultValue Original Ext JS documentation of 'getDefaultValue'
   */
  public native function getDefaultValue():*;

  [ArrayElementType("String")]
  /**
   * Gets the depends for this field. See →<code>depends</code>.
   * @return depends
   * @see #depends
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-getDepends Original Ext JS documentation of 'getDepends'
   */
  public native function getDepends():Array;

  /**
   * Get the mapping for this field. See →<code>mapping</code>.
   * @return mapping
   * @see #mapping
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-getMapping Original Ext JS documentation of 'getMapping'
   */
  public native function getMapping():Object;

  /**
   * Gets the name for this field. See →<code>name</code>.
   * @return name
   * @see #name
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-getName Original Ext JS documentation of 'getName'
   */
  public native function getName():String;

  /**
   * Gets the persist for this field. See →<code>persist</code>.
   * @return persist
   * @see #persist
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-getPersist Original Ext JS documentation of 'getPersist'
   */
  public native function getPersist():Boolean;

  [Deprecated(message="Setting sortDir and calling getSortDir were never applied by the\nthe Sorter.  This functionality does not natively exist on field instances.\n", since="5.1")]
  /**
   * Gets the sortDir for this field.
   * @return sortDir
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-getSortDir Original Ext JS documentation of 'getSortDir'
   */
  public native function getSortDir():String;

  /**
   * Gets the sortType for this field. See →<code>sortType</code>.
   * @return sortType
   * @see #sortType
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-getSortType Original Ext JS documentation of 'getSortType'
   */
  public native function getSortType():Function;

  /**
   * Gets the summary for this field. See →<code>summary</code>.
   * @return The summary.
   * @see #summary
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-getSummary Original Ext JS documentation of 'getSummary'
   */
  public native function getSummary():SummaryBase;

  /**
   * Gets a string representation of the type of this field.
   * @return type
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-getType Original Ext JS documentation of 'getType'
   */
  public native function getType():String;

  /**
   * Checks if this field has a mapping applied.
   * @return <code>true</code> if this field has a mapping.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-hasMapping Original Ext JS documentation of 'hasMapping'
   */
  public native function hasMapping():Boolean;

  /**
   * Tests whether two values are equal based on this field type.
   * This uses the →<code>compare()</code> method to determine equality, so
   * this method should generally not be overridden.
   * @param value1 The first value.
   * @param value2 The second value.
   * @return <code>true</code> if the values are equal.
   * @see #compare()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-isEqual Original Ext JS documentation of 'isEqual'
   */
  public native function isEqual(value1:Object, value2:Object):Boolean;

  /**
   * A function which converts the Model's value for this Field into a form which can be used
   * by whatever →<code>ext.data.writer.DataWriter</code> is being used to sync data with the server.
   * @param value The Field's value - the value to be serialized.
   * @param record The record being serialized.
   * @return The string that represents the Field's value.
   * @see ext.data.writer.DataWriter
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-serialize Original Ext JS documentation of 'serialize'
   */
  public native function serialize(value:*, record:Model):String;

  /**
   * Validates the passed value for this field.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @since 5.0.0
   * @param value The value to validate.
   * @param separator This string is passed if the caller wants all validation
   * messages concatenated with this string between each. This can be handled as a
   * "falsy" value because concatenating with no separator is seldom desirable.
   * @param errors This parameter is
   * passed if the caller wants all validation results individually added to the collection.
   * @param record The record being validated
   * @return <code>true</code> if the value is valid. A string may be returned if
   * the value is not valid, to indicate an error message. Any other non <code>true</code> value
   * indicates the value is not valid. This method is not implemented by default,
   * subclasses may override it to provide an implementation.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.field.Field.html#method-validate Original Ext JS documentation of 'validate'
   */
  protected native function validate(value:Object, separator:String = null, errors:* = undefined, record:Model = null):*;
}
}