package ext.data.identifier {
import ext.Base;
import ext.mixin.IFactoryable;

[Native("Ext.data.identifier.Generator", require)]
/**
 * This class is a base for all id generators. It also provides lookup of id generators by
 * their id.
 * <p>Generally, id generators are used to generate a primary key for new model instances. There
 * are different approaches to solving this problem, so this mechanism has both simple use
 * cases and is open to custom implementations. A →<code>ext.data.Model</code> requests id generation
 * using the →<code>ext.data.Model.identifier</code> property.</p>
 * <p>The following types of <code>identifiers</code> are provided:</p>
 * <ul>
 * <li><code>sequential (→ext.data.identifier.SequentialIdentifier)</code></li>
 * <li><code>negative (→ext.data.identifier.NegativeIdentifier)</code></li>
 * <li><code>uuid (→ext.data.identifier.UuidIdentifier)</code></li>
 * </ul>
 * <p>In most cases (other than <code>uuid</code>), the server is the only party that can generate
 * authoritative id values. This means that any id generated by an <code>identifier</code> should be
 * consider "provisional" and must eventually be reconciled with the server. This makes a
 * <code>uuid</code> very attractive as an <code>identifier</code> because they are designed to be generated in
 * a distributed manner and therefore never require reconciliation.</p>
 * <p>It is common for id values to be generated as increasing integer values (1, 2, etc.) by
 * the server when records are inserted. A <code>negative (→ext.data.identifier.NegativeIdentifier)</code>
 * <code>identifier</code> may be useful as it generates client-side values of -1, -2, etc.. These
 * values are of the same data type (integer) and so can typically be read by servers
 * using typed languages (such as Java or C#) and easily recognized as provisional.</p>
 * <p>In the end, the choice of <code>identifier</code> strategy requires agreement between client and
 * server.</p>
 * <p><b>Identity, Type and Shared Generators</b></p>
 * <p>It is often desirable to share Generators to ensure uniqueness or common configuration.
 * This is done by giving Generator instances an id property by which they can be looked
 * up using the →<code>ext.Factory.dataIdentifier()</code> method. To configure two
 * →<code>ext.data.Model</code> classes to share one
 * <i>sequential</i> (→<code>ext.data.identifier.SequentialIdentifier</code>) id generator, you simply assign them
 * the same id:</p>
 * <pre>
 * Ext.define('MyApp.data.MyModelA', {
 *     extend: 'Ext.data.Model',
 *     identifier: {
 *         type: 'sequential',
 *         id: 'foo'
 *     }
 * });
 *
 * Ext.define('MyApp.data.MyModelB', {
 *     extend: 'Ext.data.Model',
 *     identifier: {
 *         type: 'sequential',
 *         id: 'foo'
 *     }
 * });
 * </pre>
 * <p>To make this as simple as possible for generator types that are shared by many (or all)
 * Models, the Generator types (such as 'sequential' or 'uuid') are also reserved as
 * generator ids. This is used by the →<code>ext.data.identifier.UuidIdentifier</code> which has an id equal
 * to its type ('uuid'). In other words, the following Models share the same generator:</p>
 * <pre>
 * Ext.define('MyApp.data.MyModelX', {
 *     extend: 'Ext.data.Model',
 *     identifier: 'uuid'
 * });
 *
 * Ext.define('MyApp.data.MyModelY', {
 *     extend: 'Ext.data.Model',
 *     identifier: 'uuid'
 * });
 * </pre>
 * <p>This can be overridden (by specifying the id explicitly), but there is no particularly
 * good reason to do so for this generator type.</p>
 * <p><b>Creating Custom Generators</b></p>
 * <p>An id generator should derive from this class and implement the →<code>generate()</code> method.</p>
 * <p>To register an id generator type, a derived class should provide an <code>alias</code> like so:</p>
 * <pre>
 * Ext.define('MyApp.data.identifier.Custom', {
 *     extend: 'Ext.data.identifier.Generator',
 *     alias: 'data.identifier.custom',
 *     config: {
 *         configProp: 42 // some config property w/default value
 *     }
 *
 *     generate: function () {
 *         return ... // a new id
 *     }
 * });
 * </pre>
 * <p>Using the custom id generator is then straightforward:</p>
 * <pre>
 * Ext.define('MyApp.data.MyModel', {
 *     extend: 'Ext.data.Model',
 *     identifier: 'custom'
 * });
 * // or...
 *
 * Ext.define('MyApp.data.MyModel', {
 *     extend: 'Ext.data.Model',
 *     identifier: {
 *         type: 'custom',
 *         configProp: value
 *     }
 * });
 * </pre>
 * <p>It is not recommended to mix shared generators with generator configuration. This leads
 * to unpredictable results unless all configurations match (which is also redundant). In
 * such cases, a custom generator with a default id is the best approach.</p>
 * <pre>
 * Ext.define('MyApp.data.identifier.Custom', {
 *     extend: 'Ext.data.identifier.Sequential',
 *     alias: 'data.identifier.custom',
 *
 *     config: {
 *         id: 'custom',
 *         prefix: 'ID_',
 *         seed: 1000
 *     }
 * });
 *
 * Ext.define('MyApp.data.MyModelX', {
 *     extend: 'Ext.data.Model',
 *     identifier: 'custom'
 * });
 *
 * Ext.define('MyApp.data.MyModelY', {
 *     extend: 'Ext.data.Model',
 *     identifier: 'custom'
 * });
 *
 * // the above models share a generator that produces ID_1000, ID_1001, etc..
 * </pre>
 * @see ext.data.Model
 * @see ext.data.Model#identifier
 * @see ext.data.identifier.SequentialIdentifier
 * @see ext.data.identifier.NegativeIdentifier
 * @see ext.data.identifier.UuidIdentifier
 * @see ext.Factory#dataIdentifier()
 * @see #generate()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.identifier.Generator.html Original Ext JS documentation of 'Ext.data.identifier.Generator'
 */
public class GeneratorIdentifier extends Base implements IFactoryable {
  /**
   * Initializes a new instance.
   * @param config Configuration object to be applied to the new instance.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.identifier.Generator.html#method-constructor Original Ext JS documentation of 'constructor'
   */
  public function GeneratorIdentifier(config:GeneratorIdentifier = null) {
    super();
  }

  /**
   * @inheritDoc
   */
  public native function get factoryConfig():Object;

  /**
   * @inheritDoc
   */
  public native function set factoryConfig(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * The id for this generator.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.identifier.Generator.html#cfg-id Original Ext JS documentation of 'id'
   * @see #getId()
   * @see #setId()
   */
  public native function get id():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set id(value:String):void;

  /**
   * <code>true</code> in this class to identify an object as an instantiated IdGenerator, or subclass
   * thereof.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.identifier.Generator.html#property-isGenerator Original Ext JS documentation of 'isGenerator'
   */
  public native function get isGenerator():Boolean;

  /**
   * @private
   */
  public native function set isGenerator(value:Boolean):void;

  /**
   * Generates and returns the next id. This method must be implemented by the derived
   * class.
   * @return The next id.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.identifier.Generator.html#method-generate Original Ext JS documentation of 'generate'
   */
  public native function generate():*;

  /**
   * Returns the value of <code>id</code>.
   * @see #id
   */
  public native function getId():String;

  /**
   * Sets the value of <code>id</code>.
   * @param id The new value.
   * @see #id
   */
  public native function setId(id:String):void;
}
}