package ext.data.proxy {
import ext.data.Request;

[Native("Ext.data.proxy.Ajax", require)]
/**
 * AjaxProxy is one of the most widely-used ways of getting data into your application. It uses AJAX
 * requests to load data from the server, usually to be placed into a →<code>ext.data.Store</code>.
 * Let's take a look at a typical setup. Here we're going to set up a Store that has an Ajax Proxy.
 * To prepare, we'll also set up a →<code>ext.data.Model</code>:
 * <pre>
 * Ext.define('User', {
 *     extend: 'Ext.data.Model',
 *     fields: ['id', 'name', 'email']
 * });
 *
 * // The Store contains the AjaxProxy as an inline configuration
 * var store = Ext.create('Ext.data.Store', {
 *     model: 'User',
 *     proxy: {
 *         type: 'ajax',
 *         url: 'users.json'
 *     }
 * });
 *
 * store.load();
 * </pre>
 * <p>Our example is going to load user data into a Store, so we start off by defining a
 * →<code>ext.data.Model</code> with the fields that we expect the server to return. Next we set up
 * the Store itself, along with a →<code>ext.data.Store.proxy</code> configuration.
 * This configuration was automatically turned into an Ext.data.proxy.Ajax instance, with the url
 * we specified being passed into AjaxProxy's constructor. It's as if we'd done this:</p>
 * <pre>
 * new Ext.data.proxy.Ajax({
 *     url: 'users.json',
 *     model: 'User',
 *     reader: 'json'
 * });
 * </pre>
 * <p>A couple of extra configurations appeared here - →<code>model</code> and →<code>reader</code>. These are
 * set by default when we create the proxy via the Store - the Store already knows about the Model,
 * and Proxy's default →<code>ext.data.reader.DataReader</code> is
 * →<code>ext.data.reader.JsonReader</code>.</p>
 * <p>Now when we call store.load(), the AjaxProxy springs into action, making a request to the url
 * we configured ('users.json' in this case). As we're performing a read, it sends a GET request
 * to that url (see →<code>actionMethods</code> to customize this - by default any kind of read will be
 * sent as a GET request and any kind of write will be sent as a POST request).</p>
 * <p><b>Limitations</b></p>
 * <p>AjaxProxy cannot be used to retrieve data from other domains. If your application is running
 * on http://domainA.com it cannot load data from http://domainB.com because browsers have
 * a built-in security policy that prohibits domains talking to each other via AJAX.</p>
 * <p>If you need to read data from another domain and can't set up a proxy server (some software
 * that runs on your own domain's web server and transparently forwards requests to
 * http://domainB.com, making it look like they actually came from http://domainA.com), you can use
 * →<code>ext.data.proxy.JsonPProxy</code> and a technique known as JSON-P (JSON with Padding), which can help
 * you get around the problem so long as the server on http://domainB.com is set up to support
 * JSON-P responses. See →<code>ext.data.proxy.JsonPProxy</code>'s introduction docs for more
 * details.</p>
 * <p><b>Readers and Writers</b></p>
 * <p>AjaxProxy can be configured to use any type of →<code>ext.data.reader.DataReader</code> to decode
 * the server's response. If no Reader is supplied, AjaxProxy will default to using a
 * →<code>ext.data.reader.JsonReader</code>. Reader configuration can be passed in as a simple
 * object, which the Proxy automatically turns into a →<code>ext.data.reader.DataReader</code>
 * instance:</p>
 * <pre>
 * var proxy = new Ext.data.proxy.Ajax({
 *     model: 'User',
 *     reader: {
 *         type: 'xml',
 *         rootProperty: 'users'
 *     }
 * });
 *
 * proxy.getReader(); // returns an XmlReader instance based on the config we supplied
 * </pre>
 * <p><b>Url generation</b></p>
 * <p>AjaxProxy automatically inserts any sorting, filtering, paging and grouping options into the url
 * it generates for each request. These are controlled with the following configuration options:</p>
 * <ul>
 * <li>→<code>pageParam</code> - controls how the page number is sent to the server (see also
 * →<code>startParam</code> and →<code>limitParam</code>)</li>
 * <li>→<code>sortParam</code> - controls how sort information is sent to the server</li>
 * <li>→<code>groupParam</code> - controls how grouping information is sent to the server</li>
 * <li>→<code>filterParam</code> - controls how filter information is sent to the server</li>
 * </ul>
 * <p>Each request sent by AjaxProxy is described by an →<code>ext.data.operation.Operation</code>.
 * To see how we can customize the generated urls, let's say we're loading the Proxy
 * with the following Operation:</p>
 * <pre>
 * var proxy = new Ext.data.proxy.Ajax({
 *     url: '/users'
 * });
 *
 * var operation = proxy.createOperation('read', {
 *     page: 2
 * });
 * </pre>
 * <p>Now we'll issue the request for this Operation by calling →<code>read()</code>:</p>
 * <pre>
 * proxy.read(operation); // GET /users?page=2
 * </pre>
 * <p>Easy enough - the Proxy just copied the page property from the Operation. We can customize
 * how this page data is sent to the server:</p>
 * <pre>
 * var proxy = new Ext.data.proxy.Ajax({
 *     url: '/users',
 *     pageParam: 'pageNumber'
 * });
 *
 * proxy.read(operation); // GET /users?pageNumber=2
 * </pre>
 * <p>Alternatively, our Operation could have been configured to send start and limit parameters
 * instead of page:</p>
 * <pre>
 * var proxy = new Ext.data.proxy.Ajax({
 *     url: '/users'
 * });
 *
 * var operation = proxy.createOperation('read', {
 *     start: 50,
 *     limit: 25
 * });
 *
 * proxy.read(operation); // GET /users?start=50&amp;limit;=25
 * </pre>
 * <p>Again we can customize this url:</p>
 * <pre>
 * var proxy = new Ext.data.proxy.Ajax({
 *     url: '/users',
 *     startParam: 'startIndex',
 *     limitParam: 'limitIndex'
 * });
 *
 * proxy.read(operation); // GET /users?startIndex=50&amp;limitIndex;=25
 * </pre>
 * <p>AjaxProxy will also send sort and filter information to the server. Let's take a look at how
 * this looks with a more expressive Operation object:</p>
 * <pre>
 * var operation = proxy.createOperation('read', {
 *     sorters: [
 *         new Ext.util.Sorter({
 *             property: 'name',
 *             direction: 'ASC'
 *         }),
 *         new Ext.util.Sorter({
 *             property: 'age',
 *             direction: 'DESC'
 *         })
 *     ],
 *     filters: [
 *         new Ext.util.Filter({
 *             property: 'eyeColor',
 *             value: 'brown'
 *         })
 *     ]
 * });
 * </pre>
 * <p>This is the type of object that is generated internally when loading a
 * →<code>ext.data.Store</code> with sorters and filters defined. By default the AjaxProxy will
 * JSON encode the sorters and filters, resulting in something like this (note that the url
 * is escaped before sending the request, but is left unescaped here for clarity):</p>
 * <pre>
 * var proxy = new Ext.data.proxy.Ajax({
 *     url: '/users'
 * });
 *
 * // GET /users?sort=[{"property":"name","direction":"ASC"},
 *                     {"property":"age","direction":"DESC"}]
 *                     &amp;filter;=[{"property":"eyeColor","value":"brown"}]
 * proxy.read(operation);
 * </pre>
 * <p>We can again customize how this is created by supplying a few configuration options. Let's say
 * our server is set up to receive sorting information is a format like "sortBy=name#ASC,age#DESC".
 * We can configure AjaxProxy to provide that format like this:</p>
 * <pre>
 *  var proxy = new Ext.data.proxy.Ajax({
 *      url: '/users',
 *      sortParam: 'sortBy',
 *      filterParam: 'filterBy',
 *
 *      // our custom implementation of sorter encoding -
 *      // turns our sorters into "name#ASC,age#DESC"
 *      encodeSorters: function(sorters) {
 *          var length = sorters.length,
 *              sortStrs = [],
 *              sorter, i;
 *
 *          for (i = 0; i &lt; length; i++) {
 *              sorter = sorters[i];
 *
 *              sortStrs[i] = sorter.property + '#' + sorter.direction
 *          }
 *
 *          return sortStrs.join(",");
 *      }
 *  });
 *
 *  // GET /users?sortBy=name#ASC,age#DESC&amp;filterBy;=[{"property":"eyeColor","value":"brown"}]
 *  proxy.read(operation);
 * </pre>
 * <p>We can also provide a custom →<code>encodeFilters()</code> function to encode our filters.</p>
 * <p><b>Debugging your Ajax Proxy</b></p>
 * <p>If the data is not being loaded into the store as expected, it could be due to a mismatch
 * between the the way that the →<code>reader</code> is configured, and the shape of the incoming data.</p>
 * <p>To debug from the point that your data arrives back from the network, set a breakpoint inside
 * the callback function created in the <code>→createRequestCallback()</code> method of the Ajax Proxy class,
 * and follow the data to where the →<code>reader</code> attempts to consume it.</p>
 * @see ext.data.Store
 * @see ext.data.Model
 * @see ext.data.Store#proxy
 * @see #model
 * @see #reader
 * @see ext.data.reader.DataReader
 * @see ext.data.reader.JsonReader
 * @see #actionMethods
 * @see ext.data.proxy.JsonPProxy
 * @see #pageParam
 * @see #startParam
 * @see #limitParam
 * @see #sortParam
 * @see #groupParam
 * @see #filterParam
 * @see ext.data.operation.Operation
 * @see #read()
 * @see #encodeFilters()
 * @see #createRequestCallback()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Ajax.html Original Ext JS documentation of 'Ext.data.proxy.Ajax'
 */
public class AjaxProxy extends ServerProxy {
  /**
   * Note that if this HttpProxy is being used by a →<code>ext.data.Store</code>, then the Store's
   * call to →<code>ext.data.Store.load()</code> will override any specified callback and params
   * options. In this case, use the →<code>ext.data.Store</code>'s events to modify parameters,
   * or react to loading events.
   * @param config Config object.
   * If an options parameter is passed, the singleton →<code>ext.Ajax</code> object will be used to make
   * the request.
   * @see ext.data.Store
   * @see ext.data.Store#load()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Ajax.html#method-constructor Original Ext JS documentation of 'constructor'
   * @see ext.#Ajax ext.Ajax
   */
  public function AjaxProxy(config:AjaxProxy = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * Mapping of action name to HTTP request method. In the basic AjaxProxy these are set to
   * 'GET' for 'read' actions and 'POST' for 'create', 'update' and 'destroy' actions. The
   * →<code>ext.data.proxy.RestProxy</code> maps these to the correct RESTful methods.
   * @default {
   *     create: 'POST',
   *     read: 'GET',
   *     update: 'POST',
   *     destroy: 'POST'
   * }
   * @see ext.data.proxy.RestProxy
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Ajax.html#cfg-actionMethods Original Ext JS documentation of 'actionMethods'
   * @see #getActionMethods()
   * @see #setActionMethods()
   */
  public native function get actionMethods():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set actionMethods(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * True to request binary data from the server. This feature requires
   * the use of a binary reader such as <i>AMF Reader</i> (→<code>null</code>)
   * @default false
   * @see null
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Ajax.html#cfg-binary Original Ext JS documentation of 'binary'
   * @see #getBinary()
   * @see #setBinary()
   */
  public native function get binary():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set binary(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * Any headers to add to the Ajax request.
   * <p>example:</p>
   * <pre>
   * proxy: {
   *     headers: {'Content-Type': "text/plain" }
   *     ...
   * }
   * </pre>
   * @default undefined
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Ajax.html#cfg-headers Original Ext JS documentation of 'headers'
   * @see #getHeaders()
   * @see #setHeaders()
   */
  public native function get headers():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set headers(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * Set to <code>true</code> to have any request parameters sent as
   * <i>jsonData</i> (→<code>ext.data.Connection.request()</code>) where they can be parsed from the
   * raw request. By default, parameters are sent via the
   * <i>params</i> (→<code>ext.data.Connection.request()</code>) property.
   * <b>Note</b>: This setting does not apply when the request is sent as a 'GET' request.
   * See →<code>actionMethods</code> for controlling the HTTP verb that is used when sending
   * requests.
   * @default false
   * @see ext.data.Connection#request()
   * @see #actionMethods
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Ajax.html#cfg-paramsAsJson Original Ext JS documentation of 'paramsAsJson'
   * @see #getParamsAsJson()
   * @see #setParamsAsJson()
   */
  public native function get paramsAsJson():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set paramsAsJson(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * Most oData feeds require basic HTTP authentication. This configuration allows
   * you to specify the password.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Ajax.html#cfg-password Original Ext JS documentation of 'password'
   * @see #getPassword()
   * @see #setPassword()
   */
  public native function get password():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set password(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * Set this to false to not send the default Xhr header (X-Requested-With) with every
   * request. This should be set to false when making CORS (cross-domain) requests.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Ajax.html#cfg-useDefaultXhrHeader Original Ext JS documentation of 'useDefaultXhrHeader'
   * @see #getUseDefaultXhrHeader()
   * @see #setUseDefaultXhrHeader()
   */
  public native function get useDefaultXhrHeader():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set useDefaultXhrHeader(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * Most oData feeds require basic HTTP authentication. This configuration allows
   * you to specify the username.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Ajax.html#cfg-username Original Ext JS documentation of 'username'
   * @see #getUsername()
   * @see #setUsername()
   */
  public native function get username():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set username(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * This configuration is sometimes necessary when using cross-origin resource sharing.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Ajax.html#cfg-withCredentials Original Ext JS documentation of 'withCredentials'
   * @see #getWithCredentials()
   * @see #setWithCredentials()
   */
  public native function get withCredentials():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set withCredentials(value:Boolean):void;

  /**
   * Aborts a running request.
   * @param request The request to abort. If not passed, the most recent
   * active request will be aborted.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Ajax.html#method-abort Original Ext JS documentation of 'abort'
   */
  public native function abort(request:Request = null):void;

  /**
   * Returns the value of <code>actionMethods</code>.
   * @see #actionMethods
   */
  public native function getActionMethods():Object;

  /**
   * Returns the value of <code>binary</code>.
   * @see #binary
   */
  public native function getBinary():Boolean;

  /**
   * Returns the value of <code>headers</code>.
   * @see #headers
   */
  public native function getHeaders():Object;

  /**
   * Returns the HTTP method name for a given request. By default this returns based on a lookup
   * on →<code>actionMethods</code>.
   * @param request The request object
   * @return The HTTP method to use (should be one of 'GET', 'POST', 'PUT' or 'DELETE')
   * @see #actionMethods
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Ajax.html#method-getMethod Original Ext JS documentation of 'getMethod'
   */
  public native function getMethod(request:Request):String;

  /**
   * Returns the value of <code>paramsAsJson</code>.
   * @see #paramsAsJson
   */
  public native function getParamsAsJson():Boolean;

  /**
   * Returns the value of <code>password</code>.
   * @see #password
   */
  public native function getPassword():String;

  /**
   * Returns the value of <code>useDefaultXhrHeader</code>.
   * @see #useDefaultXhrHeader
   */
  public native function getUseDefaultXhrHeader():Boolean;

  /**
   * Returns the value of <code>username</code>.
   * @see #username
   */
  public native function getUsername():String;

  /**
   * Returns the value of <code>withCredentials</code>.
   * @see #withCredentials
   */
  public native function getWithCredentials():Boolean;

  /**
   * Sets the value of <code>actionMethods</code>.
   * @param actionMethods The new value.
   * @see #actionMethods
   */
  public native function setActionMethods(actionMethods:Object):void;

  /**
   * Sets the value of <code>binary</code>.
   * @param binary The new value.
   * @see #binary
   */
  public native function setBinary(binary:Boolean):void;

  /**
   * Sets the value of <code>headers</code>.
   * @param headers The new value.
   * @see #headers
   */
  public native function setHeaders(headers:Object):void;

  /**
   * Sets the value of <code>paramsAsJson</code>.
   * @param paramsAsJson The new value.
   * @see #paramsAsJson
   */
  public native function setParamsAsJson(paramsAsJson:Boolean):void;

  /**
   * Sets the value of <code>password</code>.
   * @param password The new value.
   * @see #password
   */
  public native function setPassword(password:String):void;

  /**
   * Sets the value of <code>useDefaultXhrHeader</code>.
   * @param useDefaultXhrHeader The new value.
   * @see #useDefaultXhrHeader
   */
  public native function setUseDefaultXhrHeader(useDefaultXhrHeader:Boolean):void;

  /**
   * Sets the value of <code>username</code>.
   * @param username The new value.
   * @see #username
   */
  public native function setUsername(username:String):void;

  /**
   * Sets the value of <code>withCredentials</code>.
   * @param withCredentials The new value.
   * @see #withCredentials
   */
  public native function setWithCredentials(withCredentials:Boolean):void;
}
}