package ext.data.proxy {
import ext.data.Request;
import ext.data.operation.Operation;

[Native("Ext.data.proxy.Direct", require)]
/**
 * This class is used to send requests to the server using <i>Ext Direct</i> (→<code>ext.direct.DirectManager</code>).
 * When a request is made, the transport mechanism is handed off to the appropriate
 * <i>Provider</i> (→<code>ext.direct.RemotingProvider</code>) to complete the call.
 * <p><b>Specifying the functions</b></p>
 * <p>This proxy expects Direct remoting method to be passed in order to be able to complete requests,
 * one Direct function per CRUD method. This is done via →<code>api</code> configuration:</p>
 * <pre>
 *  api: {
 *      read: 'MyApp.readRecords',
 *      create: 'MyApp.createRecords',
 *      update: 'MyApp.updateRecords',
 *      destroy: 'MyApp.destroyRecords'
 *  }
 * </pre>
 * <p>You can also use a <code>prefix</code> config to avoid duplicating full namespaces for Direct functions:</p>
 * <pre>
 *  api: {
 *      prefix: 'MyApp',
 *      read: 'readRecords',
 *      create: 'createRecords',
 *      update: 'updateRecords',
 *      destroy: 'destroyRecords'
 *  }
 * </pre>
 * <p>The preferred way is to specify function names to allow late resolution, however you can
 * pass function references instead if desired:</p>
 * <pre>
 *  api: {
 *      read: MyApp.readRecords,
 *      create: MyApp.createRecords,
 *      update: MyApp.updateRecords,
 *      destroy: MyApp.destroyRecords
 *  }
 * </pre>
 * <p>This method of configuring API is not recommended because this way the Direct functions
 * need to be created very early in the application lifecycle, long before
 * →<code>ext.app.Application</code> instance is initialized.</p>
 * <p>You can also use the →<code>directFn</code> configuration instead of →<code>api</code>. This will use
 * the same Direct function for all types of requests.</p>
 * <p><b>Server API</b></p>
 * <p>The server side methods are expected to conform to the following calling conventions:</p>
 * <p><b><i><code>read (→read())</code></i></b></p>
 * <p>Accept one argument which is either named arguments in an object (default), or an array
 * of values depending on the →<code>paramsAsHash</code> configuration. Return an array of records
 * or an object with format recognizable by the configured →<code>ext.data.reader.DataReader</code>
 * instance.</p>
 * <p>Example <i>Direct API declaration</i> (→<code>ext.direct.RemotingProvider.actions</code>):</p>
 * <pre>
 *  actions: {
 *      MyApp: [{
 *          name: 'readRecords',
 *          params: [],
 *          strict: false
 *      }]
 *  }
 * </pre>
 * <p>Example function invocation:</p>
 * <pre>
 *  MyApp.readRecords(
 *      {
 *          start: 0,
 *          limit: 10
 *      },
 *      // Results are passed to the callback function
 *      function(records) {
 *          console.log(records);
 *          // Logs:  [{ id: 'r0', text: 'foo' }, { id: 'r1', text: 'bar' }]
 *      }
 *  );
 * </pre>
 * <p><b><i><code>create</code></i></b></p>
 * <p>Accept one ordered argument which is either an object with data for the new record,
 * or an array of objects for multiple records. Return an array of identifiers for actually
 * created records. See →<code>ext.data.Model.clientIdProperty</code> for more information.</p>
 * <p>Example <i>Direct API declaration</i> (→<code>ext.direct.RemotingProvider.actions</code>):</p>
 * <pre>
 *  actions: [
 *      MyApp: [{
 *          name: 'createRecords',
 *          len: 1
 *      }]
 *  }
 * </pre>
 * <p>Example function invocation:</p>
 * <pre>
 *  MyApp.createRecords(
 *      [
 *          { id: 0, text: 'foo' },
 *          { id: 1, text: 'bar' }
 *      ],
 *      // Results are passed to the callback function
 *      function(records) {
 *          console.log(records);
 *          // Logs: [{ clientId: 0, id: 'r0' }, { clientId: 1, id: 'r1' }]
 *      }
 *  );
 * </pre>
 * <p><b><i><code>update (→update())</code></i></b></p>
 * <p>Accept one ordered argument which is either an object with updated data and valid
 * record identifier, or an array of objects for multiple records. Return an array of
 * objects with updated record data.</p>
 * <p>Example <i>Direct API declaration</i> (→<code>ext.direct.RemotingProvider.actions</code>):</p>
 * <pre>
 *  actions: [
 *      MyApp: [{
 *          name: 'updateRecords',
 *          len: 1
 *      }]
 *  }
 * </pre>
 * <p>Example function invocation:</p>
 * <pre>
 *  MyApp.updateRecords(
 *      [
 *          { id: 'r0', text: 'blerg' },
 *          { id: 'r1', text: 'throbbe' }
 *      ],
 *      // Results are passed to the callback function
 *      function(records) {
 *          console.log(records);
 *          // Logs: [{ id: 'r0', text: 'blerg' }, { id: 'r1', text: 'throbbe }]
 *      }
 *  );
 * </pre>
 * <p><b><i><code>destroy (→destroy())</code></i></b></p>
 * <p>Accept one ordered argument which is an array of record identifiers to be deleted.
 * Return an object with at least one →<code>ext.data.reader.JsonReader.successProperty</code>
 * property set to <code>true</code> or <code>false</code>, with more optional properties recognizable by configured
 * →<code>ext.data.reader.DataReader</code> instance.</p>
 * <p>Example <i>Direct API declaration</i> (→<code>ext.direct.RemotingProvider.actions</code>):</p>
 * <pre>
 *  actions: [
 *      MyApp: [{
 *          name: 'destroyRecords',
 *          len: 1
 *      }]
 *  }
 * </pre>
 * <p>Example function invocation:</p>
 * <pre>
 *  MyApp.destroyRecords(
 *      [
 *          { id: 'r0' },
 *          { id: 'r1' }
 *      ],
 *      // Results are passed to the callback function
 *      function(result) {
 *          // Default successProperty is `success`
 *          if (!result.success) {
 *              // Handle the error
 *          }
 *      }
 *  );
 * </pre>
 * <p><b><i>Read method parameters</i></b></p>
 * <p>Direct proxy provides options to help configure which parameters will be sent to the server
 * for Read operations. By setting the →<code>paramsAsHash</code> option to <code>true</code>, the proxy will
 * send an object literal containing each of the passed parameters. This is the default. When
 * →<code>paramsAsHash</code> is set to <code>false</code>, Proxy will pass the Read function an array of values
 * instead of an object, with the order determined by →<code>paramOrder</code> value.</p>
 * <p>Setting →<code>paramOrder</code> to any value other than <code>undefined</code> will automatically reset
 * →<code>paramsAsHash</code> to <code>false</code>.</p>
 * <p><b>Example Usage</b></p>
 * <pre>
 *  Ext.define('User', {
 *      extend: 'Ext.data.Model',
 *      fields: ['firstName', 'lastName']
 *  });
 *
 *  Ext.define('Users', {
 *      extend: 'Ext.data.Store',
 *      model: 'User',
 *      proxy: {
 *          type: 'direct',
 *          directFn: 'MyApp.getUsers',
 *          // Tells the proxy to pass `start` and `limit` as two by-position arguments:
 *          paramOrder: 'start,limit'
 *      }
 *  });
 *
 *  var store = new Users();
 *  store.load();
 * </pre>
 * @see ext.direct.#DirectManager ext.direct.DirectManager
 * @see ext.direct.RemotingProvider
 * @see #api
 * @see ext.app.Application
 * @see #directFn
 * @see #read()
 * @see #paramsAsHash
 * @see ext.data.reader.DataReader
 * @see ext.direct.RemotingProvider#actions
 * @see ext.data.Model#clientIdProperty
 * @see #update()
 * @see #destroy()
 * @see ext.data.reader.JsonReader#successProperty
 * @see #paramOrder
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Direct.html Original Ext JS documentation of 'Ext.data.proxy.Direct'
 */
public class DirectProxy extends ServerProxy {
  /**
   * @param config @inheritDoc
   */
  public function DirectProxy(config:DirectProxy = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * The same as →<code>ext.data.proxy.ServerProxy.api</code>, however instead of providing
   * urls you should provide a Direct function name for each CRUD method.
   * <p>Instead of providing fully qualified names for each function, you can use
   * <code>prefix</code> property to provide a common prefix for all functions:</p>
   * <p>api: {
   * prefix: 'MyApp',
   * read: 'readRecords',
   * create: 'createRecords',
   * update: 'updateRecords',
   * destroy: 'destroyRecords'
   * }</p>
   * <p>This way function names will be resolved to <code>'MyApp.readRecords'</code>,
   * <code>'MyApp.createRecords'</code>, etc. Note that using <code>prefix</code> and fully qualified
   * function names is <b>not</b> supported, and prefix will be used for every
   * function name when configured.</p>
   * <p>See also →<code>directFn</code>.</p>
   * @default undefined
   * @see ext.data.proxy.ServerProxy#api
   * @see #directFn
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Direct.html#cfg-api Original Ext JS documentation of 'api'
   * @see #getApi()
   * @see #setApi()
   */
  override public native function get api():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  override public native function set api(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * Function to call when executing a request. <code>→directFn</code> is a simple alternative
   * to defining the api configuration parameter for Stores which will not
   * implement a full CRUD api. The <code>→directFn</code> may also be a string reference to
   * the fully qualified name of the function, for example:
   * <code>'MyApp.company.GetProfile'</code>. This can be useful when using dynamic loading.
   * The string will be resolved before calling the function for the first time.
   * @see #directFn
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Direct.html#cfg-directFn Original Ext JS documentation of 'directFn'
   * @see #getDirectFn()
   * @see #setDirectFn()
   */
  public native function get directFn():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set directFn(value:*):void;

  [ExtConfig]
  [Bindable]
  /**
   * Optional set of fixed parameters to send with every Proxy request, similar to
   * →<code>extraParams</code> but available with all CRUD requests. Also unlike
   * →<code>extraParams</code>, metadata is not mixed with the ordinary data but sent
   * separately in the data packet.
   * You may need to update your server side Ext Direct stack to use this feature.
   * @see #extraParams
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Direct.html#cfg-metadata Original Ext JS documentation of 'metadata'
   * @see #getMetadata()
   * @see #setMetadata()
   */
  public native function get metadata():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set metadata(value:*):void;

  [ExtConfig]
  [Bindable]
  /**
   * A list of params to be passed to server side Read function. Specify the params
   * in the order in which they must be executed on the server-side as either (1)
   * an Array of String values, or (2) a String of params delimited by either
   * whitespace, comma, or pipe. For example, any of the following would be
   * acceptable:
   * <pre>
   * paramOrder: ['param1','param2','param3']
   * paramOrder: 'param1 param2 param3'
   * paramOrder: 'param1,param2,param3'
   * paramOrder: 'param1|param2|param'
   * </pre>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Direct.html#cfg-paramOrder Original Ext JS documentation of 'paramOrder'
   * @see #getParamOrder()
   * @see #setParamOrder()
   */
  public native function get paramOrder():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set paramOrder(value:*):void;

  [ExtConfig]
  [Bindable]
  /**
   * Send Read function parameters as a collection of named arguments. Providing a
   * →<code>paramOrder</code> nullifies this configuration.
   * @default true
   * @see #paramOrder
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Direct.html#cfg-paramsAsHash Original Ext JS documentation of 'paramsAsHash'
   * @see #getParamsAsHash()
   * @see #setParamsAsHash()
   */
  public native function get paramsAsHash():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set paramsAsHash(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Direct.html#cfg-url Original Ext JS documentation of 'url'
   * @see #getUrl()
   * @see #setUrl()
   */
  override public native function get url():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  override public native function set url(value:String):void;

  /**
   * Aborts a running request by operation.
   * @param operation The operation to abort. This parameter
   * is mandatory.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Direct.html#method-abort Original Ext JS documentation of 'abort'
   */
  public native function abort(operation:Operation):void;

  /**
   * Encode any values being sent to the server. Can be overridden in subclasses.
   * @param value An array of sorters/filters.
   * @return The encoded value
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Direct.html#method-applyEncoding Original Ext JS documentation of 'applyEncoding'
   */
  override protected native function applyEncoding(value:Array):Object;

  /**
   * Generates a url based on a given Ext.data.Request object. By default, ServerProxy's buildUrl
   * will add the cache-buster param to the end of the url. Subclasses may need to perform
   * additional modifications to the url.
   * @param request The request object
   * @return The url
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Direct.html#method-buildUrl Original Ext JS documentation of 'buildUrl'
   */
  override public native function buildUrl(request:Request):String;

  /**
   * Template method to allow subclasses to specify how to get the response for the reader.
   * @param response The server response
   * @return The response data to be used by the reader
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Direct.html#method-extractResponseData Original Ext JS documentation of 'extractResponseData'
   */
  public native function extractResponseData(response:Object):Object;

  /**
   * Returns the value of <code>api</code>.
   * @see #api
   */
  override public native function getApi():Object;

  /**
   * Returns the value of <code>directFn</code>.
   * @see #directFn
   */
  public native function getDirectFn():*;

  /**
   * Returns the value of <code>metadata</code>.
   * @see #metadata
   */
  public native function getMetadata():*;

  /**
   * Returns the value of <code>paramOrder</code>.
   * @see #paramOrder
   */
  public native function getParamOrder():*;

  /**
   * Returns the value of <code>paramsAsHash</code>.
   * @see #paramsAsHash
   */
  public native function getParamsAsHash():Boolean;

  /**
   * Returns the value of <code>url</code>.
   * @see #url
   */
  override public native function getUrl():String;

  /**
   * Sets the value of <code>api</code>.
   * @param api The new value.
   * @see #api
   */
  override public native function setApi(api:Object):void;

  /**
   * Sets the value of <code>directFn</code>.
   * @param directFn The new value.
   * @see #directFn
   */
  public native function setDirectFn(directFn:*):void;

  /**
   * Sets up an exception on the operation
   * @param operation The operation
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Direct.html#method-setException Original Ext JS documentation of 'setException'
   */
  public native function setException(operation:Operation, response:* = undefined):void;

  /**
   * Sets the value of <code>metadata</code>.
   * @param metadata The new value.
   * @see #metadata
   */
  public native function setMetadata(metadata:*):void;

  /**
   * Sets the value of <code>paramOrder</code>.
   * @param paramOrder The new value.
   * @see #paramOrder
   */
  public native function setParamOrder(paramOrder:*):void;

  /**
   * Sets the value of <code>paramsAsHash</code>.
   * @param paramsAsHash The new value.
   * @see #paramsAsHash
   */
  public native function setParamsAsHash(paramsAsHash:Boolean):void;

  /**
   * Sets the value of <code>url</code>.
   * @param url The new value.
   * @see #url
   */
  override public native function setUrl(url:String):void;
}
}