package ext.data.proxy {
import ext.data.Request;

[Native("Ext.data.proxy.JsonP", require)]
/**
 * The JsonP proxy is useful when you need to load data from a domain other than the one your
 * application is running on. If your application is running on http://domainA.com it cannot use
 * →<code>ext.data.proxy.AjaxProxy</code> to load its data from http://domainB.com because cross-domain
 * Ajax requests are prohibited by the browser.
 * <p>We can get around this using a JsonP proxy. JsonP proxy injects a <code>&lt;script&gt;</code> tag into the DOM
 * whenever an AJAX request would usually be made. Let's say we want to load data from
 * http://domainB.com/users - the script tag that would be injected might look like this:</p>
 * <pre>
 * &lt;script src="http://domainB.com/users?callback=someCallback"&gt;&lt;/script&gt;
 * </pre>
 * <p>When we inject the tag above, the browser makes a request to that url and includes the response
 * as if it was any other type of JavaScript include. By passing a callback in the url above,
 * we're telling domainB's server that we want to be notified when the result comes in and that it
 * should call our callback function with the data it sends back. So long as the server formats
 * the response to look like this, everything will work:</p>
 * <pre>
 * someCallback({
 *     users: [
 *         {
 *             id: 1,
 *             name: "Ed Spencer",
 *             email: "ed&#64;sencha.com"
 *         }
 *     ]
 * });
 * </pre>
 * <p>As soon as the script finishes loading, the 'someCallback' function that we passed in the url
 * is called with the JSON object that the server returned.</p>
 * <p>JsonP proxy takes care of all of this automatically. It formats the url you pass, adding
 * the callback parameter automatically. It even creates a temporary callback function, waits for it
 * to be called and then puts the data into the Proxy making it look just like you loaded it
 * through a normal →<code>ext.data.proxy.AjaxProxy</code>. Here's how we might set that up:</p>
 * <pre>
 * Ext.define('User', {
 *     extend: 'Ext.data.Model',
 *     fields: ['id', 'name', 'email']
 * });
 *
 * var store = Ext.create('Ext.data.Store', {
 *     model: 'User',
 *     proxy: {
 *         type: 'jsonp',
 *         url: 'http://domainB.com/users'
 *     }
 * });
 *
 * store.load();
 * </pre>
 * <p>That's all we need to do - JsonP proxy takes care of the rest. In this case the Proxy will have
 * injected a script tag like this:</p>
 * <pre>
 * &lt;script src="http://domainB.com/users?callback=callback1"&gt;&lt;/script&gt;
 * </pre>
 * <p><b>Customization</b></p>
 * <p>This script tag can be customized using the →<code>callbackKey</code> configuration. For example:</p>
 * <pre>
 * var store = Ext.create('Ext.data.Store', {
 *     model: 'User',
 *     proxy: {
 *         type: 'jsonp',
 *         url : 'http://domainB.com/users',
 *         callbackKey: 'theCallbackFunction'
 *     }
 * });
 *
 * store.load();
 * </pre>
 * <p>Would inject a script tag like this:</p>
 * <pre>
 * &lt;script src="http://domainB.com/users?theCallbackFunction=callback1"&gt;&lt;/script&gt;
 * </pre>
 * <p><b>Implementing on the server side</b></p>
 * <p>The remote server side needs to be configured to return data in this format. Here are suggestions
 * for how you might achieve this using Java, PHP and ASP.net:</p>
 * <p>Java:</p>
 * <pre>
 * boolean jsonP = false;
 * String cb = request.getParameter("callback");
 * if (cb != null) {
 *     jsonP = true;
 *     response.setContentType("text/javascript");
 * } else {
 *     response.setContentType("application/x-json");
 * }
 * Writer out = response.getWriter();
 * if (jsonP) {
 *     out.write(cb + "(");
 * }
 * out.print(dataBlock.toJsonString());
 * if (jsonP) {
 *     out.write(");");
 * }
 * </pre>
 * <p>PHP:</p>
 * <pre>
 * $callback = $_REQUEST['callback'];
 *
 * // Create the output object.
 * $output = array('a' =&gt; 'Apple', 'b' =&gt; 'Banana');
 *
 * //start output
 * if ($callback) {
 *     header('Content-Type: text/javascript');
 *     echo $callback . '(' . json_encode($output) . ');';
 * } else {
 *     header('Content-Type: application/x-json');
 *     echo json_encode($output);
 * }
 * </pre>
 * <p>ASP.net:</p>
 * <pre>
 * String jsonString = "{"success": true}";
 * String cb = Request.Params.Get("callback");
 * String responseString = "";
 * if (!String.IsNullOrEmpty(cb)) {
 *     responseString = cb + "(" + jsonString + ")";
 * } else {
 *     responseString = jsonString;
 * }
 * Response.Write(responseString);
 * </pre>
 * @see ext.data.proxy.AjaxProxy
 * @see #callbackKey
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.JsonP.html Original Ext JS documentation of 'Ext.data.proxy.JsonP'
 */
public class JsonPProxy extends ServerProxy {
  /**
   * @param config @inheritDoc
   */
  public function JsonPProxy(config:JsonPProxy = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * True to automatically append the request's params to the generated url. Defaults to true
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.JsonP.html#cfg-autoAppendParams Original Ext JS documentation of 'autoAppendParams'
   * @see #getAutoAppendParams()
   * @see #setAutoAppendParams()
   */
  public native function get autoAppendParams():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set autoAppendParams(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * See →<code>ext.data.JsonP.callbackKey</code>.
   * @default 'callback'
   * @see ext.data.SJsonP#callbackKey ext.data.JsonP.callbackKey
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.JsonP.html#cfg-callbackKey Original Ext JS documentation of 'callbackKey'
   * @see #getCallbackKey()
   * @see #setCallbackKey()
   */
  public native function get callbackKey():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set callbackKey(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * The HTTP parameter name to use when passing records to the server and the
   * <i>Json writer</i> (→<code>writer</code>) is not configured to
   * →<code>ext.data.writer.JsonWriter.encode</code> records into a parameter.
   * <p>The →<code>encodeRecords()</code> method is used to encode the records to create this parameter's
   * value.</p>
   * @default 'records'
   * @see #writer
   * @see ext.data.writer.JsonWriter#encode
   * @see #encodeRecords()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.JsonP.html#cfg-recordParam Original Ext JS documentation of 'recordParam'
   * @see #getRecordParam()
   * @see #setRecordParam()
   */
  public native function get recordParam():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set recordParam(value:String):void;

  /**
   * Aborts a server request. If no request is passed, the most recent request
   * will be aborted.
   * @param request The request to abort.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.JsonP.html#method-abort Original Ext JS documentation of 'abort'
   */
  public native function abort(request:Request = null):void;

  /**
   * Generates a url based on a given Ext.data.Request object. Adds the params and callback
   * function name to the url
   * @param request The request object
   * @return The url
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.JsonP.html#method-buildUrl Original Ext JS documentation of 'buildUrl'
   */
  override public native function buildUrl(request:Request):String;

  /**
   * Encodes an array of records into a value suitable to be added to the request <code>params</code>
   * as the →<code>recordParam</code> parameter. This is broken out into its own function so that
   * it can be easily overridden.
   * <p>The default implementation</p>
   * @param records The records array
   * @return An array of record data objects
   * @see #recordParam
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.JsonP.html#method-encodeRecords Original Ext JS documentation of 'encodeRecords'
   */
  public native function encodeRecords(records:Array):Array;

  /**
   * Returns the value of <code>autoAppendParams</code>.
   * @see #autoAppendParams
   */
  public native function getAutoAppendParams():Boolean;

  /**
   * Returns the value of <code>callbackKey</code>.
   * @see #callbackKey
   */
  public native function getCallbackKey():String;

  /**
   * Returns the value of <code>recordParam</code>.
   * @see #recordParam
   */
  public native function getRecordParam():String;

  /**
   * Sets the value of <code>autoAppendParams</code>.
   * @param autoAppendParams The new value.
   * @see #autoAppendParams
   */
  public native function setAutoAppendParams(autoAppendParams:Boolean):void;

  /**
   * Sets the value of <code>callbackKey</code>.
   * @param callbackKey The new value.
   * @see #callbackKey
   */
  public native function setCallbackKey(callbackKey:String):void;

  /**
   * Sets the value of <code>recordParam</code>.
   * @param recordParam The new value.
   * @see #recordParam
   */
  public native function setRecordParam(recordParam:String):void;
}
}