package ext.data.proxy {
import ext.data.operation.Operation;

[Native("Ext.data.proxy.Memory", require)]
/**
 * In-memory proxy. This proxy simply uses a local variable for data storage/retrieval, so its
 * contents are lost on every page refresh.
 * <p>Usually this Proxy isn't used directly, serving instead as a helper to a
 * →<code>ext.data.Store</code> where a reader is required to load data. For example, say we have
 * a Store for a User model and have some inline data we want to load, but this data isn't in quite
 * the right format: we can use a MemoryProxy with a JsonReader to read it into our Store:</p>
 * <pre>
 * // this is the model we will be using in the store
 * Ext.define('User', {
 *     extend: 'Ext.data.Model',
 *     fields: [
 *         { name: 'id',    type: 'int' },
 *         { name: 'name',  type: 'string' },
 *         { name: 'phone', type: 'string', mapping: 'phoneNumber' }
 *     ]
 * });
 *
 * // this data does not line up to our model fields - the phone field is called phoneNumber
 * var data = {
 *     users: [
 *         {
 *             id: 1,
 *             name: 'Ed Spencer',
 *             phoneNumber: '555 1234'
 *         },
 *         {
 *             id: 2,
 *             name: 'Abe Elias',
 *             phoneNumber: '666 1234'
 *         }
 *     ]
 * };
 *
 * // note how we set the 'root' in the reader to match the data structure above
 * var store = Ext.create('Ext.data.Store', {
 *     autoLoad: true,
 *     model: 'User',
 *     data: data,
 *     proxy: {
 *         type: 'memory',
 *         reader: {
 *             type: 'json',
 *             rootProperty: 'users'
 *         }
 *     }
 * });
 * </pre>
 * @see ext.data.Store
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Memory.html Original Ext JS documentation of 'Ext.data.proxy.Memory'
 */
public class MemoryProxy extends ClientProxy {
  /**
   * @param config @inheritDoc
   */
  public function MemoryProxy(config:MemoryProxy = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * By default MemoryProxy data is persistent, and subsequent reads will read the
   * same data. If this is not required, configure the proxy using <code>clearOnRead: true</code>.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Memory.html#cfg-clearOnRead Original Ext JS documentation of 'clearOnRead'
   * @see #getClearOnRead()
   * @see #setClearOnRead()
   */
  public native function get clearOnRead():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set clearOnRead(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * Optional data to pass to configured Reader.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Memory.html#cfg-data Original Ext JS documentation of 'data'
   * @see #getData()
   * @see #setData()
   */
  public native function get data():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set data(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * Configure as <code>true</code> to enable this MemoryProxy to honour a read operation's <code>start</code>
   * and <code>limit</code> options.
   * <p>When <code>true</code>, read operations will be able to read <i>pages</i> of records from the data object.</p>
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Memory.html#cfg-enablePaging Original Ext JS documentation of 'enablePaging'
   * @see #getEnablePaging()
   * @see #setEnablePaging()
   */
  public native function get enablePaging():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set enablePaging(value:Boolean):void;

  /**
   * Currently this is a hard-coded method that simply commits any records and sets the operation
   * to successful, then calls the callback function, if provided. It is essentially mocking
   * a server call in memory, but since there is no real back end in this case there's not much
   * else to do. This method can be easily overridden to implement more complex logic if needed.
   * @param operation The Operation to perform
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Memory.html#method-create Original Ext JS documentation of 'create'
   */
  override public native function create(operation:Operation):void;

  /**
   * Currently this is a hard-coded method that simply commits any records and sets the operation
   * to successful, then calls the callback function, if provided. It is essentially mocking
   * a server call in memory, but since there is no real back end in this case there's not much
   * else to do. This method can be easily overridden to implement more complex logic if needed.
   * @param operation The Operation to perform
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Memory.html#method-erase Original Ext JS documentation of 'erase'
   */
  override public native function erase(operation:Operation):void;

  /**
   * Returns the value of <code>clearOnRead</code>.
   * @see #clearOnRead
   */
  public native function getClearOnRead():Boolean;

  /**
   * Returns the value of <code>data</code>.
   * @see #data
   */
  public native function getData():Object;

  /**
   * Returns the value of <code>enablePaging</code>.
   * @see #enablePaging
   */
  public native function getEnablePaging():Boolean;

  /**
   * Reads data from the configured →<code>data</code> object. Uses the Proxy's →<code>reader</code>,
   * if present.
   * @param operation The read Operation
   * @see #data
   * @see #reader
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Memory.html#method-read Original Ext JS documentation of 'read'
   */
  override public native function read(operation:Operation):void;

  /**
   * Sets the value of <code>clearOnRead</code>.
   * @param clearOnRead The new value.
   * @see #clearOnRead
   */
  public native function setClearOnRead(clearOnRead:Boolean):void;

  /**
   * Sets the value of <code>data</code>.
   * @param data The new value.
   * @see #data
   */
  public native function setData(data:Object):void;

  /**
   * Sets the value of <code>enablePaging</code>.
   * @param enablePaging The new value.
   * @see #enablePaging
   */
  public native function setEnablePaging(enablePaging:Boolean):void;

  /**
   * Currently this is a hard-coded method that simply commits any records and sets the operation
   * to successful, then calls the callback function, if provided. It is essentially mocking
   * a server call in memory, but since there is no real back end in this case there's not much
   * else to do. This method can be easily overridden to implement more complex logic if needed.
   * @param operation The Operation to perform
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Memory.html#method-update Original Ext JS documentation of 'update'
   */
  override public native function update(operation:Operation):void;
}
}