package ext.data.proxy {
import ext.data.Request;

[Native("Ext.data.proxy.Rest", require)]
/**
 * The Rest proxy is a specialization of the →<code>ext.data.proxy.AjaxProxy</code> which simply maps
 * the four actions (create, read, update and destroy) to RESTful HTTP verbs. For example, let's
 * set up a →<code>ext.data.Model</code> with an inline Rest proxy
 * <pre>
 * Ext.define('User', {
 *     extend: 'Ext.data.Model',
 *     fields: ['id', 'name', 'email'],
 *
 *     proxy: {
 *         type: 'rest',
 *         url: '/users'
 *     }
 * });
 * </pre>
 * <p>Now we can create a new User instance and save it via the Rest proxy. Doing this will cause
 * the Proxy to send a POST request to '/users':</p>
 * <pre>
 * var user = Ext.create('User', { name: 'Ed Spencer', email: 'ed&#64;sencha.com' });
 *
 * user.save(); // POST /users
 * </pre>
 * <p>Let's expand this a little and provide a callback for the →<code>ext.data.Model.save()</code> call
 * to update the Model once it has been created. We'll assume the creation went successfully
 * and that the server gave this user an ID of 123:</p>
 * <pre>
 * user.save({
 *     success: function(user) {
 *         user.set('name', 'Khan Noonien Singh');
 *
 *         user.save(); // PUT /users/123
 *     }
 * });
 * </pre>
 * <p>Now that we're no longer creating a new Model instance, the request method is changed to an
 * HTTP PUT, targeting the relevant url for that user. Now let's delete this user, which will use
 * the DELETE method:</p>
 * <pre>
 *     user.erase(); // DELETE /users/123
 * </pre>
 * <p>Finally, when we perform a load of a Model or Store, Rest proxy will use the GET method:</p>
 * <pre>
 * // 1. Load via Store
 *
 * // the Store automatically picks up the Proxy from the User model
 * var store = Ext.create('Ext.data.Store', {
 *     model: 'User'
 * });
 *
 * store.load(); // GET /users
 *
 * // 2. Load directly from the Model
 *
 * // GET /users/123
 * User.load(123, {
 *     success: function(user) {
 *         console.log(user.getId()); // outputs 123
 *     }
 * });
 * </pre>
 * <p><b>Url generation</b></p>
 * <p>The Rest proxy is able to automatically generate the urls above based on two configuration
 * options - →<code>appendId</code> and →<code>format</code>. If appendId is true (it is by default) then
 * Rest proxy will automatically append the ID of the Model instance in question to the configured
 * url, resulting in the '/users/123' that we saw above.</p>
 * <p>If the request is not for a specific Model instance (e.g. loading a Store), the url is not
 * appended with an id. The Rest proxy will automatically insert a '/' before the ID if one is not
 * already present.</p>
 * <pre>
 * new Ext.data.proxy.Rest({
 *     url: '/users',
 *     appendId: true // default
 * });
 *
 * // Collection url: /users
 * // Instance url: /users/123
 * </pre>
 * <p>The Rest proxy can also optionally append a format string to the end of any generated url:</p>
 * <pre>
 * new Ext.data.proxy.Rest({
 *     url: '/users',
 *     format: 'json'
 * });
 *
 * // Collection url: /users.json
 * // Instance url: /users/123.json
 * </pre>
 * <p>If further customization is needed, simply implement the →<code>buildUrl()</code> method and add your
 * custom generated url onto the →<code>ext.data.Request</code> object that is passed to buildUrl.
 * See <a href="https://docs.sencha.com/extjs/7.2.0/classic/source/Rest.html#Ext-data-proxy-Rest-method-buildUrl">Rest proxy's implementation</a> for an example of how to achieve this.</p>
 * <p>Note that Rest proxy inherits from →<code>ext.data.proxy.AjaxProxy</code>, which already injects
 * all of the sorter, filter, group and paging options into the generated url. See the
 * <i>AjaxProxy docs</i> (→<code>ext.data.proxy.AjaxProxy</code>) for more details.</p>
 * @see ext.data.proxy.AjaxProxy
 * @see ext.data.Model
 * @see ext.data.Model#save()
 * @see #appendId
 * @see #format
 * @see #buildUrl()
 * @see ext.data.Request
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Rest.html Original Ext JS documentation of 'Ext.data.proxy.Rest'
 */
public class RestProxy extends AjaxProxy {
  /**
   * @param config @inheritDoc
   */
  public function RestProxy(config:RestProxy = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * Mapping of action name to HTTP request method. These default to RESTful conventions for the
   * 'create', 'read', 'update' and 'destroy' actions (which map to 'POST', 'GET', 'PUT', and
   * 'DELETE' respectively). This object should not be changed except globally via
   * →<code>ext.Ext.override_()</code> - the →<code>getMethod()</code> function can be overridden
   * instead.
   * @see ext.SExt#override_() ext.Ext.override_()
   * @see #getMethod()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Rest.html#property-actionMethods Original Ext JS documentation of 'actionMethods'
   * @see #getActionMethods()
   * @see #setActionMethods()
   */
  override public native function get actionMethods():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  override public native function set actionMethods(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * True to automatically append the ID of a Model instance when performing a
   * request based on that single instance. See Rest proxy intro docs for more
   * details.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Rest.html#cfg-appendId Original Ext JS documentation of 'appendId'
   * @see #getAppendId()
   * @see #setAppendId()
   */
  public native function get appendId():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set appendId(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * True to batch actions of a particular type when synchronizing the store.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Rest.html#cfg-batchActions Original Ext JS documentation of 'batchActions'
   * @see #getBatchActions()
   * @see #setBatchActions()
   */
  override public native function get batchActions():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  override public native function set batchActions(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * Optional data format to send to the server when making any request (e.g.
   * 'json'). See the Rest proxy intro docs for full details.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Rest.html#cfg-format Original Ext JS documentation of 'format'
   * @see #getFormat()
   * @see #setFormat()
   */
  public native function get format():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set format(value:String):void;

  /**
   * Specialized version of buildUrl that incorporates the →<code>appendId</code> and
   * →<code>format</code> options into the generated url. Override this to provide further
   * customizations, but remember to call the superclass buildUrl so that additional
   * parameters like the cache buster string are appended.
   * @param request The request object
   * @see #appendId
   * @see #format
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.proxy.Rest.html#method-buildUrl Original Ext JS documentation of 'buildUrl'
   */
  override public native function buildUrl(request:Request):String;

  /**
   * Returns the value of <code>actionMethods</code>.
   * @see #actionMethods
   */
  override public native function getActionMethods():Object;

  /**
   * Returns the value of <code>appendId</code>.
   * @see #appendId
   */
  public native function getAppendId():Boolean;

  /**
   * Returns the value of <code>batchActions</code>.
   * @see #batchActions
   */
  override public native function getBatchActions():Boolean;

  /**
   * Returns the value of <code>format</code>.
   * @see #format
   */
  public native function getFormat():String;

  /**
   * Sets the value of <code>actionMethods</code>.
   * @param actionMethods The new value.
   * @see #actionMethods
   */
  override public native function setActionMethods(actionMethods:Object):void;

  /**
   * Sets the value of <code>appendId</code>.
   * @param appendId The new value.
   * @see #appendId
   */
  public native function setAppendId(appendId:Boolean):void;

  /**
   * Sets the value of <code>batchActions</code>.
   * @param batchActions The new value.
   * @see #batchActions
   */
  override public native function setBatchActions(batchActions:Boolean):void;

  /**
   * Sets the value of <code>format</code>.
   * @param format The new value.
   * @see #format
   */
  public native function setFormat(format:String):void;
}
}