package ext.data.reader {
import ext.data.ResultSet;
import ext.mixin.IFactoryable;
import ext.mixin.Observable;

[Native("Ext.data.reader.Reader", require)]
/**
 * @see ext.data.reader.events.DataReader_response_errorEvent.EXCEPTION
 * @eventType ext.data.reader.events.DataReader_response_errorEvent.EXCEPTION
 */
[Event(name='onException', type='ext.data.reader.events.DataReader_response_errorEvent')]
/**
 * Readers are used to interpret data to be loaded into a →<code>ext.data.Model</code> instance
 * or a →<code>ext.data.Store</code> - often in response to an AJAX request. In general there is
 * usually no need to create a Reader instance directly, since a Reader is almost always used
 * together with a →<code>ext.data.proxy.DataProxy</code>, and is configured using the Proxy's
 * →<code>ext.data.proxy.DataProxy.reader</code> configuration property:
 * <pre>
 * Ext.create('Ext.data.Store', {
 *     model: 'User',
 *     proxy: {
 *         type: 'ajax',
 *         url: 'users.json',
 *         reader: {
 *             type: 'json',
 *             rootProperty: 'users'
 *         }
 *     },
 * });
 * </pre>
 * <p>The above reader is configured to consume a JSON string that looks something like this:</p>
 * <pre>
 * {
 *     "success": true,
 *     "users": [
 *         { "name": "User 1" },
 *         { "name": "User 2" }
 *     ]
 * }
 * </pre>
 * <p><b>Loading Nested Data</b></p>
 * <p>Readers have the ability to automatically load deeply-nested data objects based on the
 * <i>associations</i> (→<code>ext.data.schema.Association</code>) configured on each Model. Below is an example
 * demonstrating the flexibility of these associations in a fictional CRM system which manages
 * a User, their Orders, OrderItems and Products. First we'll define the models:</p>
 * <pre>
 * Ext.define("User", {
 *     extend: 'Ext.data.Model',
 *     fields: [
 *         'id', 'name'
 *     ],
 *
 *     hasMany: {model: 'Order', name: 'orders'},
 *
 *     proxy: {
 *         type: 'rest',
 *         url : 'users.json',
 *         reader: {
 *             type: 'json',
 *             rootProperty: 'users'
 *         }
 *     }
 * });
 *
 * Ext.define("Order", {
 *     extend: 'Ext.data.Model',
 *     fields: [
 *         'id', 'total'
 *     ],
 *
 *     hasMany  : {model: 'OrderItem', name: 'orderItems', associationKey: 'order_items'},
 *     belongsTo: 'User'
 * });
 *
 * Ext.define("OrderItem", {
 *     extend: 'Ext.data.Model',
 *     fields: [
 *         'id', 'price', 'quantity', 'order_id', 'product_id'
 *     ],
 *
 *     belongsTo: ['Order', {model: 'Product', associationKey: 'product'}]
 * });
 *
 * Ext.define("Product", {
 *     extend: 'Ext.data.Model',
 *     fields: [
 *         'id', 'name'
 *     ],
 *
 *     hasMany: 'OrderItem'
 * });
 * </pre>
 * <p>This may be a lot to take in - basically a User has many Orders, each of which is composed of
 * several OrderItems. Finally, each OrderItem has a single Product. This allows us to consume data
 * like this:</p>
 * <pre>
 * {
 *     "users": [
 *         {
 *             "id": 123,
 *             "name": "Ed",
 *             "orders": [
 *                 {
 *                     "id": 50,
 *                     "total": 100,
 *                     "order_items": [
 *                         {
 *                             "id": 20,
 *                             "price": 40,
 *                             "quantity": 2,
 *                             "product": {
 *                                 "id": 1000,
 *                                 "name": "MacBook Pro"
 *                             }
 *                         },
 *                         {
 *                             "id": 21,
 *                             "price": 20,
 *                             "quantity": 3,
 *                             "product": {
 *                                 "id": 1001,
 *                                 "name": "iPhone"
 *                             }
 *                         }
 *                     ]
 *                 }
 *             ]
 *         }
 *     ]
 * }
 * </pre>
 * <p>The JSON response is deeply nested - it returns all Users (in this case just 1 for simplicity's
 * sake), all of the Orders for each User (again just 1 in this case), all of the OrderItems
 * for each Order (2 order items in this case), and finally the Product associated with each
 * OrderItem. Now we can read the data and use it as follows:</p>
 * <pre>
 * var store = Ext.create('Ext.data.Store', {
 *     model: "User"
 * });
 *
 * store.load({
 *     callback: function() {
 *         // the user that was loaded
 *         var user = store.first();
 *
 *         console.log("Orders for " + user.get('name') + ":")
 *
 *         // iterate over the Orders for each User
 *         user.orders().each(function(order) {
 *             console.log("Order ID: " + order.getId() + ", which contains items:");
 *
 *             // iterate over the OrderItems for each Order
 *             order.orderItems().each(function(orderItem) {
 *                 // we know that the Product data is already loaded,
 *                 // so we can use the synchronous getProduct
 *                 // usually, we would use the asynchronous version (see #belongsTo)
 *                 var product = orderItem.getProduct();
 *
 *                 console.log(orderItem.get('quantity') + ' orders of ' + product.get('name'));
 *             });
 *         });
 *     }
 * });
 * </pre>
 * <p>Running the code above results in the following:</p>
 * <pre>
 * Orders for Ed:
 * Order ID: 50, which contains items:
 * 2 orders of MacBook Pro
 * 3 orders of iPhone
 * </pre>
 * @see ext.data.Model
 * @see ext.data.Store
 * @see ext.data.proxy.DataProxy
 * @see ext.data.proxy.DataProxy#reader
 * @see ext.data.schema.Association
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html Original Ext JS documentation of 'Ext.data.reader.Reader'
 */
public class DataReader extends Observable implements IFactoryable {
  /**
   * Creates new Reader.
   * @param config Config object.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#method-constructor Original Ext JS documentation of 'constructor'
   */
  public function DataReader(config:DataReader = null) {
    super();
  }

  /**
   * @inheritDoc
   */
  public native function get factoryConfig():Object;

  /**
   * @inheritDoc
   */
  public native function set factoryConfig(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * Name of the property from which to retrieve remote grouping summary information.
   * There should be an item for each group.
   * <p>The remote summary data should be parseable as a →<code>model</code> used by this reader.</p>
   * @default ''
   * @see #model
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#cfg-groupRootProperty Original Ext JS documentation of 'groupRootProperty'
   * @see #getGroupRootProperty()
   * @see #setGroupRootProperty()
   */
  public native function get groupRootProperty():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set groupRootProperty(value:*):void;

  [ExtConfig]
  [Bindable]
  /**
   * True to automatically parse models nested within other models in a response object.
   * See the Ext.data.reader.Reader intro docs for full explanation.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#cfg-implicitIncludes Original Ext JS documentation of 'implicitIncludes'
   * @see #getImplicitIncludes()
   * @see #setImplicitIncludes()
   */
  public native function get implicitIncludes():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set implicitIncludes(value:Boolean):void;

  /**
   * <code>true</code> in this class to identify an object as an instantiated Reader, or subclass thereof.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#property-isReader Original Ext JS documentation of 'isReader'
   */
  public native function get isReader():Boolean;

  /**
   * @private
   */
  public native function set isReader(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * Determines if the Reader will keep raw data
   * received from the server in the →<code>rawData</code> property.
   * <p>While this might seem useful to do additional data processing, keeping raw data
   * might cause adverse effects such as memory leaks. It is recommended to set
   * <code>→keepRawData</code> to <code>false</code> if you do not need the raw data.</p>
   * <p>If you need to process data packet to extract additional data such as row summaries,
   * it is recommended to use →<code>transform</code> function for that purpose.</p>
   * <p>Note that starting with Ext JS 6.0 the default behavior has been changed to
   * <b>not</b> keep the raw data because of the high potential for memory leaks.</p>
   * @default null
   * @since 5.1.1
   * @see #rawData
   * @see #keepRawData
   * @see #transform
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#cfg-keepRawData Original Ext JS documentation of 'keepRawData'
   * @see #getKeepRawData()
   * @see #setKeepRawData()
   */
  public native function get keepRawData():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set keepRawData(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * The name of the property which contains a response message for exception handling. If you
   * want to return a false success response from the server, maybe due to some server-side
   * validation, the messageProperty can hold the error message. For example:
   * <pre>
   * {
   *     "success": false,
   *     "error": "There was an error with your request"
   * }
   * </pre>
   * <p>You can retrieve this error message in a callback when loading a
   * →<code>ext.data.Store</code> or →<code>ext.data.Model</code> like:</p>
   * <pre>
   * var store = new Ext.data.Store({
   *     fields : ['foo'],
   *     proxy  : {
   *         type   : 'ajax',
   *         url    : 'data.json',
   *         reader : {
   *             type            : 'json',
   *             rootProperty    : 'data',
   *             messageProperty : 'error'
   *         }
   *     }
   * });
   *
   * store.load({
   *     callback: function(records, operation, success) {
   *         if (success) {
   *             // ...
   *         } else {
   *             var error = operation.getError();
   *
   *             Ext.Msg.alert('Error', error);
   *         }
   *     }
   * });
   * </pre>
   * <p>In this example, the callback will execute with <code>success</code> being <code>false</code> and will
   * therefore show the <i>Ext.Msg.alert</i> (→<code>ext.MessageBox.alert()</code>) with the error string
   * returned in the response.</p>
   * @default ''
   * @see ext.data.Store
   * @see ext.data.Model
   * @see ext.SMessageBox#alert() ext.MessageBox.alert()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#cfg-messageProperty Original Ext JS documentation of 'messageProperty'
   * @see #getMessageProperty()
   * @see #setMessageProperty()
   */
  public native function get messageProperty():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set messageProperty(value:*):void;

  /**
   * The raw meta data that was most recently read, if any. Meta data can include existing
   * Reader config options like →<code>totalProperty</code>, etc. that get
   * automatically applied to the Reader, and those can still be accessed directly from the Reader
   * if needed. However, meta data is also often used to pass other custom data to be processed
   * by application code. For example, it is common when reconfiguring the data model of a grid to
   * also pass a corresponding column model config to be applied to the grid. Any such data will
   * not get applied to the Reader directly (it just gets passed through and is ignored by Ext).
   * This metaData property gives you access to all meta data that was passed, including any such
   * custom data ignored by the reader.
   * <p>This is a read-only property, and it will get replaced each time a new meta data object is
   * passed to the reader. Note that typically you would handle proxy's
   * →<code>ext.data.proxy.DataProxy.event:onMetaChange</code> event which passes this exact same meta
   * object to listeners. However this property is available if it's more convenient to access it
   * via the reader directly in certain cases.</p>
   * @see #totalProperty
   * @see ext.data.proxy.DataProxy#event:onMetaChange
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#property-metaData Original Ext JS documentation of 'metaData'
   */
  public native function get metaData():Object;

  [ExtConfig]
  [Bindable]
  /**
   * The model to use for this reader. This config is only required if the reader is being
   * used without a proxy, otherwise the proxy will automatically set the model.
   * @default null
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#cfg-model Original Ext JS documentation of 'model'
   * @see #getModel()
   * @see #setModel()
   */
  public native function get model():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set model(value:*):void;

  /**
   * The raw data object that was last passed to →<code>readRecords()</code>. rawData is populated
   * based on the results of →<code>ext.data.proxy.ServerProxy.processResponse()</code>. rawData will
   * maintain a cached copy of the last successfully returned records. In other words,
   * if processResponse is unsuccessful, the records from the last successful response
   * will remain cached in rawData.
   * <p>Since Ext JS 5.1.1 you can use the →<code>keepRawData</code> config option to
   * control this behavior.</p>
   * @see #readRecords()
   * @see ext.data.proxy.ServerProxy#processResponse()
   * @see #keepRawData
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#property-rawData Original Ext JS documentation of 'rawData'
   */
  public native function get rawData():Object;

  /**
   * @private
   */
  public native function set rawData(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * True to extract the records from a data packet even if the →<code>successProperty</code>
   * returns false.
   * @default true
   * @see #successProperty
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#cfg-readRecordsOnFailure Original Ext JS documentation of 'readRecordsOnFailure'
   * @see #getReadRecordsOnFailure()
   * @see #setReadRecordsOnFailure()
   */
  public native function get readRecordsOnFailure():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set readRecordsOnFailure(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * The property that contains data items corresponding to the
   * Model(s) of the configured Reader. <code>→rootProperty</code> varies by Reader type.
   * <p><b><i>JSON Reader</i></b></p>
   * <p><code>→rootProperty</code> is a property name. It may also be a dot-separated
   * list of property names if the root is nested. The root JSON array will be
   * used by default.</p>
   * <pre>
   * // rootPropety config
   * rootProperty: 'embedded.myresults'
   *
   * // server response
   * {
   *     embedded: {
   *         myresults: [{
   *             name: 'Scott',
   *             age: 22
   *         }, {
   *             name: 'Ramona',
   *             age: 24
   *         }]
   *     },
   *     success: true
   * }
   * </pre>
   * <p><b><i>XML Reader</i></b></p>
   * <p><code>→rootProperty</code> is a CSS selector. The root XML element will be used
   * by default.</p>
   * <pre>
   * // rootProperty config (plus record config)
   * rootProperty: 'myresults',
   * record: 'user'
   *
   * // server response
   * &lt;?xml version="1.0" encoding="UTF-8"?&gt;
   * &lt;embedded&gt;
   *     &lt;myresults&gt;
   *         &lt;user&gt;
   *             &lt;name&gt;Scott&lt;/name&gt;
   *             &lt;age&gt;22&lt;/age&gt;
   *         &lt;/user&gt;
   *         &lt;user&gt;
   *             &lt;name&gt;Ramona&lt;/name&gt;
   *             &lt;age&gt;24&lt;/age&gt;
   *         &lt;/user&gt;
   *     &lt;/myresults&gt;
   * &lt;/embedded&gt;
   * </pre>
   * <p><b><i>Array Reader</i></b></p>
   * <p><code>→rootProperty</code> is not typically applicable since the data is assumed to be a
   * single-level array of arrays. However, if the array of records is returned
   * within a JSON response a <code>→rootProperty</code> config may be used:</p>
   * <pre>
   * // rootProperty config
   * rootProperty: 'embedded.myresults'
   *
   * // server response
   * {
   *     embedded: {
   *         myresults: [['Scott', 22], ['Ramona', 24]]
   *     },
   *     success: true
   * }
   * </pre>
   * <p><b><i>rootProperty as a function</i></b></p>
   * <p>The <code>→rootProperty</code> may also be a function that returns the root node from
   * the dataset. For example:</p>
   * <pre>
   * var store = Ext.create('Ext.data.TreeStore', {
   *     proxy: {
   *         type: 'memory',
   *         reader: {
   *             type: 'json',
   *             rootProperty: function(data){
   *                 // Extract child nodes from the items or children property
   *                 // in the dataset
   *                 return data.items || data.children;
   *             }
   *         }
   *     },
   *     data: {
   *         items: [{
   *             text: 'item 1',
   *             children: [{
   *                 text: 'child A',
   *                 leaf: true
   *             }]
   *         }]
   *     }
   * });
   *
   * Ext.create('Ext.tree.Panel', {
   *     title: 'rootProperty as a function',
   *     width: 200,
   *     height:150,
   *     store: store,
   *     rootVisible: false,
   *     renderTo: Ext.getBody()
   * });
   * </pre>
   * @default ''
   * @see #rootProperty
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#cfg-rootProperty Original Ext JS documentation of 'rootProperty'
   * @see #getRootProperty()
   * @see #setRootProperty()
   */
  public native function get rootProperty():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set rootProperty(value:*):void;

  [ExtConfig]
  [Bindable]
  /**
   * Name of the property from which to retrieve the <code>success</code> attribute, the value of which
   * indicates whether a given request succeeded or failed (typically a boolean or
   * 'true'|'false'). See
   * →<code>ext.data.proxy.ServerProxy</code>.→<code>ext.data.proxy.ServerProxy.event:onException</code> for
   * additional information.
   * @default 'success'
   * @see ext.data.proxy.ServerProxy
   * @see ext.data.proxy.ServerProxy#event:onException
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#cfg-successProperty Original Ext JS documentation of 'successProperty'
   * @see #getSuccessProperty()
   * @see #setSuccessProperty()
   */
  public native function get successProperty():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set successProperty(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * Name of the property from which to retrieve remote summary information.
   * <p>The remote summary data should be parseable as a →<code>model</code> used by this reader.</p>
   * @default ''
   * @see #model
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#cfg-summaryRootProperty Original Ext JS documentation of 'summaryRootProperty'
   * @see #getSummaryRootProperty()
   * @see #setSummaryRootProperty()
   */
  public native function get summaryRootProperty():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set summaryRootProperty(value:*):void;

  [ExtConfig]
  [Bindable]
  /**
   * Name of the property from which to retrieve the total number of records in the dataset.
   * This is only needed if the whole dataset is not passed in one go, but is being paged from
   * the remote server.
   * @default 'total'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#cfg-totalProperty Original Ext JS documentation of 'totalProperty'
   * @see #getTotalProperty()
   * @see #setTotalProperty()
   */
  public native function get totalProperty():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set totalProperty(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * If a transform function is set, it will be invoked just before →<code>readRecords()</code>
   * executes. It is passed the raw (deserialized) data object. The transform function
   * returns a data object, which can be a modified version of the original data object,
   * or a completely new data object. The transform can be a function, or a method name
   * on the Reader instance, or an object with a 'fn' key and an optional 'scope' key.
   * <p>Example usage:</p>
   * <pre>
   * Ext.create('Ext.data.Store', {
   *     model: 'User',
   *     proxy: {
   *         type: 'ajax',
   *         url : 'users.json',
   *         reader: {
   *             type: 'json',
   *             transform: {
   *                 fn: function(data) {
   *                     // do some manipulation of the raw data object
   *                     return data;
   *                 },
   *                 scope: this
   *             }
   *         }
   *     },
   * });
   * </pre>
   * @default null
   * @see #readRecords()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#cfg-transform Original Ext JS documentation of 'transform'
   * @see #getTransform()
   * @see #setTransform()
   */
  public native function get transform():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set transform(value:*):void;

  [ExtConfig]
  [Bindable]
  /**
   * The name of the property in a node raw data block which indicates the type of the model
   * to be created from that raw data. Useful for heterogeneous trees.
   * <p>For example, hierarchical geographical data may look like this:</p>
   * <pre>
   * {
   *     nodeType: 'Territory',
   *     name: 'EMEA',
   *     children: [{
   *         nodeType: 'Country',
   *         name: 'United Kingdon',
   *         children: [{
   *             nodeType: 'City',
   *             name: 'London'
   *         }]
   *     }]
   * }
   * </pre>
   * <p>You would configure the typeProperty in this case to be <code>"nodeType"</code> which would cause
   * the models named "Territory", "Country" and "City" to be used.</p>
   * @default ''
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#cfg-typeProperty Original Ext JS documentation of 'typeProperty'
   * @see #getTypeProperty()
   * @see #setTypeProperty()
   */
  public native function get typeProperty():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set typeProperty(value:String):void;

  /**
   * This method provides a hook to do any data transformation before the reading process
   * begins. By default this function just returns what is passed to it. It can be
   * overridden in a subclass to return something else.
   * See →<code>ext.data.reader.XmlReader</code> for an example.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @param data The data object
   * @return The normalized data object
   * @see ext.data.reader.XmlReader
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#method-getData Original Ext JS documentation of 'getData'
   */
  protected native function getData(data:Object):Object;

  /**
   * Returns the value of <code>groupRootProperty</code>.
   * @see #groupRootProperty
   */
  public native function getGroupRootProperty():*;

  /**
   * Returns the value of <code>implicitIncludes</code>.
   * @see #implicitIncludes
   */
  public native function getImplicitIncludes():Boolean;

  /**
   * Returns the value of <code>keepRawData</code>.
   * @see #keepRawData
   */
  public native function getKeepRawData():Boolean;

  /**
   * Returns the value of <code>messageProperty</code>.
   * @see #messageProperty
   */
  public native function getMessageProperty():*;

  /**
   * Returns the value of <code>model</code>.
   * @see #model
   */
  public native function getModel():*;

  /**
   * Returns the value of <code>readRecordsOnFailure</code>.
   * @see #readRecordsOnFailure
   */
  public native function getReadRecordsOnFailure():Boolean;

  /**
   * Takes a raw response object (as passed to the →<code>read()</code> method) and returns the useful
   * data segment from it. This must be implemented by each subclass.
   * @param response The response object
   * @return The extracted data from the response. For example, a JSON object or an XML
   * document.
   * @see #read()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#method-getResponseData Original Ext JS documentation of 'getResponseData'
   */
  public native function getResponseData(response:Object):Object;

  /**
   * Returns the value of <code>rootProperty</code>.
   * @see #rootProperty
   */
  public native function getRootProperty():*;

  /**
   * Returns the value of <code>successProperty</code>.
   * @see #successProperty
   */
  public native function getSuccessProperty():String;

  /**
   * Returns the value of <code>summaryRootProperty</code>.
   * @see #summaryRootProperty
   */
  public native function getSummaryRootProperty():*;

  /**
   * Returns the value of <code>totalProperty</code>.
   * @see #totalProperty
   */
  public native function getTotalProperty():String;

  /**
   * Returns the value of <code>transform</code>.
   * @see #transform
   */
  public native function getTransform():*;

  /**
   * Returns the value of <code>typeProperty</code>.
   * @see #typeProperty
   */
  public native function getTypeProperty():String;

  /**
   * Reads the given response object. This method normalizes the different types of response
   * object that may be passed to it. If it's an XMLHttpRequest object, hand off to the subclass'
   * →<code>getResponseData()</code> method. Else, hand off the reading of records to the
   * →<code>readRecords()</code> method.
   * @param response The response object. This may be either an XMLHttpRequest object or
   * a plain JS object
   * @param readOptions Various options that instruct the reader on how to read the
   * data
   * <ul>
   * <li><code>recordCreator:Function</code> (optional) —
   * A function to construct the model based on the
   * processed data. By default, this just calls the model constructor and passes the raw data.
   * </li>
   * </ul>
   * @return The parsed or default ResultSet object
   * @see #getResponseData()
   * @see #readRecords()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#method-read Original Ext JS documentation of 'read'
   */
  public native function read(response:Object, readOptions:Object = null):ResultSet;

  /**
   * Abstracts common functionality used by all Reader subclasses. Each subclass is expected to
   * call this function before running its own logic and returning the Ext.data.ResultSet
   * instance. For most Readers additional processing should not be needed.
   * @param data The raw data object
   * @param readOptions See →<code>read()</code> for details.
   * @return A ResultSet object
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Reader.html#method-readRecords Original Ext JS documentation of 'readRecords'
   * @see #read()
   */
  public native function readRecords(data:Object, readOptions:Object = null):ResultSet;

  /**
   * Sets the value of <code>groupRootProperty</code>.
   * @param groupRootProperty The new value.
   * @see #groupRootProperty
   */
  public native function setGroupRootProperty(groupRootProperty:*):void;

  /**
   * Sets the value of <code>implicitIncludes</code>.
   * @param implicitIncludes The new value.
   * @see #implicitIncludes
   */
  public native function setImplicitIncludes(implicitIncludes:Boolean):void;

  /**
   * Sets the value of <code>keepRawData</code>.
   * @param keepRawData The new value.
   * @see #keepRawData
   */
  public native function setKeepRawData(keepRawData:Boolean):void;

  /**
   * Sets the value of <code>messageProperty</code>.
   * @param messageProperty The new value.
   * @see #messageProperty
   */
  public native function setMessageProperty(messageProperty:*):void;

  /**
   * Sets the value of <code>model</code>.
   * @param model The new value.
   * @see #model
   */
  public native function setModel(model:*):void;

  /**
   * Sets the value of <code>readRecordsOnFailure</code>.
   * @param readRecordsOnFailure The new value.
   * @see #readRecordsOnFailure
   */
  public native function setReadRecordsOnFailure(readRecordsOnFailure:Boolean):void;

  /**
   * Sets the value of <code>rootProperty</code>.
   * @param rootProperty The new value.
   * @see #rootProperty
   */
  public native function setRootProperty(rootProperty:*):void;

  /**
   * Sets the value of <code>successProperty</code>.
   * @param successProperty The new value.
   * @see #successProperty
   */
  public native function setSuccessProperty(successProperty:String):void;

  /**
   * Sets the value of <code>summaryRootProperty</code>.
   * @param summaryRootProperty The new value.
   * @see #summaryRootProperty
   */
  public native function setSummaryRootProperty(summaryRootProperty:*):void;

  /**
   * Sets the value of <code>totalProperty</code>.
   * @param totalProperty The new value.
   * @see #totalProperty
   */
  public native function setTotalProperty(totalProperty:String):void;

  /**
   * Sets the value of <code>transform</code>.
   * @param transform The new value.
   * @see #transform
   */
  public native function setTransform(transform:*):void;

  /**
   * Sets the value of <code>typeProperty</code>.
   * @param typeProperty The new value.
   * @see #typeProperty
   */
  public native function setTypeProperty(typeProperty:String):void;
}
}