package ext.data.reader {
import ext.data.ResultSet;

[Native("Ext.data.reader.Json", require)]
/**
 * The JSON Reader is used by a Proxy to read a server response that is sent back in JSON format.
 * This usually happens as a result of loading a Store - for example we might create something
 * like this:
 * <pre>
 * Ext.define('User', {
 *     extend: 'Ext.data.Model',
 *     fields: ['id', 'name', 'email']
 * });
 *
 * var store = Ext.create('Ext.data.Store', {
 *     model: 'User',
 *     proxy: {
 *         type: 'ajax',
 *         url: 'users.json',
 *         reader: {
 *             type: 'json'
 *         }
 *     }
 * });
 * </pre>
 * <p>The example above creates a 'User' model. Models are explained in the
 * →<code>ext.data.Model</code> docs if you're not already familiar with them.</p>
 * <p>We created the simplest type of JSON Reader possible by simply telling our
 * →<code>ext.data.Store</code>'s →<code>ext.data.proxy.DataProxy</code> that we want a JSON Reader.
 * The Store automatically passes the configured model to the Store, so it is as if we passed
 * this instead:</p>
 * <pre>
 * reader: {
 *     type: 'json',
 *     model: 'User'
 * }
 * </pre>
 * <p>The reader we set up is ready to read data from our server - at the moment it will accept
 * a response like this:</p>
 * <pre>
 * [
 *     {
 *         "id": 1,
 *         "name": "Ed Spencer",
 *         "email": "ed&#64;sencha.com"
 *     },
 *     {
 *         "id": 2,
 *         "name": "Abe Elias",
 *         "email": "abe&#64;sencha.com"
 *     }
 * ]
 * </pre>
 * <p><b><i>Reading other JSON formats</i></b></p>
 * <p>If you already have your JSON format defined and it doesn't look quite like what we have above,
 * you can usually pass JsonReader a couple of configuration options to make it parse your format.
 * For example, we can use the →<code>rootProperty</code> configuration to parse data that comes back
 * like this:</p>
 * <pre>
 * {
 *     "users": [
 *        {
 *            "id": 1,
 *            "name": "Ed Spencer",
 *            "email": "ed&#64;sencha.com"
 *        },
 *        {
 *            "id": 2,
 *            "name": "Abe Elias",
 *            "email": "abe&#64;sencha.com"
 *        }
 *     ]
 * }
 * </pre>
 * <p>To parse this we just pass in a →<code>rootProperty</code> configuration that matches the 'users'
 * above:</p>
 * <pre>
 * reader: {
 *     type: 'json',
 *     rootProperty: 'users'
 * }
 * </pre>
 * <p>Sometimes the JSON structure is even more complicated. Document databases like CouchDB often
 * provide metadata around each record inside a nested structure like this:</p>
 * <pre>
 * {
 *     "total": 122,
 *     "offset": 0,
 *     "users": [
 *         {
 *             "id": "ed-spencer-1",
 *             "value": 1,
 *             "user": {
 *                 "id": 1,
 *                 "name": "Ed Spencer",
 *                 "email": "ed&#64;sencha.com"
 *             }
 *         }
 *     ]
 * }
 * </pre>
 * <p>In the case above the record data is nested an additional level inside the "users" array as each
 * "user" item has additional metadata surrounding it ('id' and 'value' in this case). To parse
 * data out of each "user" item in the JSON above we need to specify the →<code>record</code>
 * configuration like this:</p>
 * <pre>
 * reader: {
 *     type: 'json',
 *     rootProperty: 'users',
 *     record: 'user'
 * }
 * </pre>
 * <p><b><i>Response MetaData</i></b></p>
 * <p>The server can return metadata in its response, in addition to the record data, that describe
 * attributes of the data set itself or are used to reconfigure the Reader. To pass metadata
 * in the response you simply add a <code>→metaData</code> attribute to the root of the response data.
 * The metaData attribute can contain anything, but supports a specific set of properties
 * that are handled by the Reader if they are present:</p>
 * <ul>
 * <li>→<code>rootProperty</code>: the property name of the root response node containing the record data</li>
 * <li>→<code>totalProperty</code>: property name for the total number of records in the data</li>
 * <li>→<code>successProperty</code>: property name for the success status of the response</li>
 * <li>→<code>messageProperty</code>: property name for an optional response message</li>
 * <li>→<code>ext.data.Model.fields</code>: Config used to reconfigure the Model's fields
 * before converting the response data into records</li>
 * </ul>
 * <p>An initial Reader configuration containing all of these properties might look like this
 * ("fields" would be included in the Model definition, not shown):</p>
 * <pre>
 * reader: {
 *     type: 'json',
 *     rootProperty: 'root',
 *     totalProperty: 'total',
 *     successProperty: 'success',
 *     messageProperty: 'message'
 * }
 * </pre>
 * <p>If you were to pass a response object containing attributes different from those initially
 * defined above, you could use the <code>→metaData</code> attribute to reconfigure the Reader on the fly.
 * For example:</p>
 * <pre>
 * {
 *     "count": 1,
 *     "ok": true,
 *     "msg": "Users found",
 *     "users": [{
 *         "userId": 123,
 *         "name": "Ed Spencer",
 *         "email": "ed&#64;sencha.com"
 *     }],
 *     "metaData": {
 *         "rootProperty": "users",
 *         "totalProperty": 'count',
 *         "successProperty": 'ok',
 *         "messageProperty": 'msg'
 *     }
 * }
 * </pre>
 * <p>You can also place any other arbitrary data you need into the <code>→metaData</code> attribute which will be
 * ignored by the Reader, but will be accessible via the Reader's →<code>metaData</code> property
 * (which is also passed to listeners via the Proxy's
 * →<code>ext.data.proxy.DataProxy.event:onMetaChange</code> event (also relayed by the store).
 * Application code can then process the passed metadata in any way it chooses.</p>
 * <p>A simple example for how this can be used would be customizing the fields for a Model that is
 * bound to a grid. By passing the <code>fields</code> property the Model will be automatically updated by the
 * Reader internally, but that change will not be reflected automatically in the grid unless you
 * also update the column configuration. You could do this manually, or you could simply pass
 * a standard grid <i>column</i> (→<code>ext.panel.TablePanel.columns</code>) config object as part of the <code>→metaData</code>
 * attribute and then pass that along to the grid. Here's a very simple example for how that
 * could be accomplished:</p>
 * <pre>
 * // response format:
 * {
 *     ...
 *     "metaData": {
 *         "fields": [
 *             { "name": "userId", "type": "int" },
 *             { "name": "name", "type": "string" },
 *             { "name": "birthday", "type": "date", "dateFormat": "Y-j-m" },
 *         ],
 *         "columns": [
 *             { "text": "User ID", "dataIndex": "userId", "width": 40 },
 *             { "text": "User Name", "dataIndex": "name", "flex": 1 },
 *             { "text": "Birthday", "dataIndex": "birthday", "flex": 1,
 *               "format": 'Y-j-m', "xtype": "datecolumn" }
 *         ]
 *     }
 * }
 * </pre>
 * <p>The Reader will automatically read the meta fields config and rebuild the Model based on the new
 * fields, but to handle the new column configuration you would need to handle the metadata within
 * the application code. This is done simply enough by handling the metachange event on either
 * the store or the proxy, e.g.:</p>
 * <pre>
 * var store = Ext.create('Ext.data.Store', {
 *     ...
 *     listeners: {
 *         'metachange': function(store, meta) {
 *             myGrid.reconfigure(store, meta.columns);
 *         }
 *     }
 * });
 * </pre>
 * @see ext.data.Model
 * @see ext.data.Store
 * @see ext.data.proxy.DataProxy
 * @see #rootProperty
 * @see #record
 * @see #metaData
 * @see #totalProperty
 * @see #successProperty
 * @see #messageProperty
 * @see ext.data.Model#fields
 * @see ext.data.proxy.DataProxy#event:onMetaChange
 * @see ext.panel.TablePanel#columns
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Json.html Original Ext JS documentation of 'Ext.data.reader.Json'
 */
public class JsonReader extends DataReader {
  /**
   * @param config @inheritDoc
   */
  public function JsonReader(config:JsonReader = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * Name of the property from which to retrieve the <code>→metaData</code> attribute. See
   * →<code>metaData</code>.
   * @default 'metaData'
   * @see #metaData
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Json.html#cfg-metaProperty Original Ext JS documentation of 'metaProperty'
   * @see #getMetaProperty()
   * @see #setMetaProperty()
   */
  public native function get metaProperty():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set metaProperty(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * The reader will keep a copy of the most recent request in the →<code>rawData</code> property.
   * For performance reasons, the data object for each record is used directly as the model
   * data. This means that these objects may be modified and thus modify the raw data.
   * To ensure the objects are copied, set this option to <code>true</code>.
   * NB: This only applies to items that are read as part of the data array, any other
   * metadata will not be modified:
   * <pre>
   * {
   *     "someOtherData": 1, // Won't be modified
   *     "root": [{}, {}, {}] // The objects here will be modified
   * }
   * </pre>
   * @default false
   * @see #rawData
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Json.html#cfg-preserveRawData Original Ext JS documentation of 'preserveRawData'
   * @see #getPreserveRawData()
   * @see #setPreserveRawData()
   */
  public native function get preserveRawData():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set preserveRawData(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * The optional location within the JSON response that the record data
   * itself can be found at. See the JsonReader intro docs for more details. This is not often
   * needed.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Json.html#cfg-record Original Ext JS documentation of 'record'
   * @see #getRecord()
   * @see #setRecord()
   */
  public native function get record():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set record(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * True to ensure that field names/mappings are treated
   * as literals when reading values.
   * <p>For example, by default, using the mapping "foo.bar.baz" will try and read a property foo
   * from the root, then a property bar from foo, then a property baz from bar. Setting the
   * simple accessors to true will read the property with the name "foo.bar.baz" direct from
   * the root object.</p>
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Json.html#cfg-useSimpleAccessors Original Ext JS documentation of 'useSimpleAccessors'
   * @see #getUseSimpleAccessors()
   * @see #setUseSimpleAccessors()
   */
  public native function get useSimpleAccessors():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set useSimpleAccessors(value:Boolean):void;

  /**
   * Returns the value of <code>metaProperty</code>.
   * @see #metaProperty
   */
  public native function getMetaProperty():String;

  /**
   * Returns the value of <code>preserveRawData</code>.
   * @see #preserveRawData
   */
  public native function getPreserveRawData():Boolean;

  /**
   * Returns the value of <code>record</code>.
   * @see #record
   */
  public native function getRecord():String;

  /**
   * Returns the value of <code>useSimpleAccessors</code>.
   * @see #useSimpleAccessors
   */
  public native function getUseSimpleAccessors():Boolean;

  /**
   * Reads a JSON object and returns a ResultSet. Uses the internal getTotal and getSuccess
   * extractors to retrieve meta data from the response, and extractData to turn the JSON data
   * into model instances.
   * @param data The raw JSON data
   * @param readOptions See →<code>read()</code> for details.
   * @return A ResultSet containing model instances and meta data about
   * the results
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Json.html#method-readRecords Original Ext JS documentation of 'readRecords'
   * @see #read()
   */
  override public native function readRecords(data:Object, readOptions:Object = null):ResultSet;

  /**
   * Sets the value of <code>metaProperty</code>.
   * @param metaProperty The new value.
   * @see #metaProperty
   */
  public native function setMetaProperty(metaProperty:String):void;

  /**
   * Sets the value of <code>preserveRawData</code>.
   * @param preserveRawData The new value.
   * @see #preserveRawData
   */
  public native function setPreserveRawData(preserveRawData:Boolean):void;

  /**
   * Sets the value of <code>record</code>.
   * @param record The new value.
   * @see #record
   */
  public native function setRecord(record:String):void;

  /**
   * Sets the value of <code>useSimpleAccessors</code>.
   * @param useSimpleAccessors The new value.
   * @see #useSimpleAccessors
   */
  public native function setUseSimpleAccessors(useSimpleAccessors:Boolean):void;
}
}