package ext.data.reader {
import ext.data.ResultSet;

[Native("Ext.data.reader.Xml", require)]
/**
 * The XML Reader is used by a Proxy to read a server response that is sent back in XML format.
 * This usually happens as a result of loading a Store - for example we might create something
 * like this:
 * <pre>
 * Ext.define('User', {
 *     extend: 'Ext.data.Model',
 *     fields: ['id', 'name', 'email']
 * });
 *
 * var store = Ext.create('Ext.data.Store', {
 *     model: 'User',
 *     proxy: {
 *         type: 'ajax',
 *         url: 'users.xml',
 *         reader: {
 *             type: 'xml',
 *             record: 'user',
 *             rootProperty: 'users'
 *         }
 *     }
 * });
 * </pre>
 * <p>The example above creates a 'User' model. Models are explained in the
 * →<code>ext.data.Model</code> docs if you're not already familiar with them.</p>
 * <p>We created the simplest type of XML Reader possible by simply telling our
 * →<code>ext.data.Store</code>'s →<code>ext.data.proxy.DataProxy</code> that we want a XML Reader.
 * The Store automatically passes the configured model to the Store, so it is as if we passed
 * this instead:</p>
 * <pre>
 * reader: {
 *     type: 'xml',
 *     model: 'User',
 *     record: 'user',
 *     rootProperty: 'users'
 * }
 * </pre>
 * <p>The reader we set up is ready to read data from our server - at the moment it will accept
 * a response like this:</p>
 * <pre>
 * &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 * &lt;users&gt;
 *     &lt;user&gt;
 *         &lt;id&gt;1&lt;/id&gt;
 *         &lt;name&gt;Ed Spencer&lt;/name&gt;
 *         &lt;email&gt;ed&#64;sencha.com&lt;/email&gt;
 *     &lt;/user&gt;
 *     &lt;user&gt;
 *         &lt;id&gt;2&lt;/id&gt;
 *         &lt;name&gt;Abe Elias&lt;/name&gt;
 *         &lt;email&gt;abe&#64;sencha.com&lt;/email&gt;
 *     &lt;/user&gt;
 * &lt;/users&gt;
 * </pre>
 * <p>First off there's →<code>rootProperty</code> option to define the root node <code>&lt;users&gt;</code> (there should be
 * only one in a well-formed XML document). Then the XML Reader uses the configured →<code>record</code>
 * option to pull out the data for each record - in this case we set record to 'user', so each
 * <code>&lt;user&gt;</code> above will be converted into a User model.</p>
 * <p>Note that XmlReader doesn't care whether your →<code>rootProperty</code> and →<code>record</code> elements
 * are nested deep inside a larger structure, so a response like this will still work:</p>
 * <pre>
 * &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 * &lt;deeply&gt;
 *     &lt;nested&gt;
 *         &lt;xml&gt;
 *             &lt;users&gt;
 *                 &lt;user&gt;
 *                     &lt;id&gt;1&lt;/id&gt;
 *                     &lt;name&gt;Ed Spencer&lt;/name&gt;
 *                     &lt;email&gt;ed&#64;sencha.com&lt;/email&gt;
 *                 &lt;/user&gt;
 *                 &lt;user&gt;
 *                     &lt;id&gt;2&lt;/id&gt;
 *                     &lt;name&gt;Abe Elias&lt;/name&gt;
 *                     &lt;email&gt;abe&#64;sencha.com&lt;/email&gt;
 *                 &lt;/user&gt;
 *             &lt;/users&gt;
 *         &lt;/xml&gt;
 *     &lt;/nested&gt;
 * &lt;/deeply&gt;
 * </pre>
 * <p>If this Reader is being used by a →<code>ext.data.TreeStore</code> to read tree-structured
 * data in which records are nested as descendant nodes of other records, then this lenient
 * behaviour must be overridden by using a more specific child node selector as your →<code>record</code>
 * selector which will not select all descendants, such as:</p>
 * <pre>
 * record: '&gt;user'
 * </pre>
 * <p><b>Response metadata</b></p>
 * <p>The server can return additional data in its response, such as the <i>total
 * number of records</i> (→<code>totalProperty</code>) and the <i>success status of the response</i> (→<code>successProperty</code>). These are
 * typically included in the XML response like this:</p>
 * <pre>
 * &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 * &lt;users&gt;
 *     &lt;total&gt;100&lt;/total&gt;
 *     &lt;success&gt;true&lt;/success&gt;
 *     &lt;user&gt;
 *         &lt;id&gt;1&lt;/id&gt;
 *         &lt;name&gt;Ed Spencer&lt;/name&gt;
 *         &lt;email&gt;ed&#64;sencha.com&lt;/email&gt;
 *     &lt;/user&gt;
 *     &lt;user&gt;
 *         &lt;id&gt;2&lt;/id&gt;
 *         &lt;name&gt;Abe Elias&lt;/name&gt;
 *         &lt;email&gt;abe&#64;sencha.com&lt;/email&gt;
 *     &lt;/user&gt;
 * &lt;/users&gt;
 * </pre>
 * <p>If these properties are present in the XML response they can be parsed out by the XmlReader
 * and used by the Store that loaded it. We can set up the names of these properties by specifying
 * a final pair of configuration options:</p>
 * <pre>
 * reader: {
 *     type: 'xml',
 *     rootProperty: 'users',
 *     totalProperty: 'total',
 *     successProperty: 'success'
 * }
 * </pre>
 * <p>These final options are not necessary to make the Reader work, but can be useful when the server
 * needs to report an error or if it needs to indicate that there is a lot of data available
 * of which only a subset is currently being returned.</p>
 * <p><b>Response format</b></p>
 * <p><b>Note:</b> in order for the browser to parse a returned XML document, the Content-Type header
 * in the HTTP response must be set to "text/xml" or "application/xml". This is very important -
 * the XmlReader will not work correctly otherwise.</p>
 * @see ext.data.Model
 * @see ext.data.Store
 * @see ext.data.proxy.DataProxy
 * @see #rootProperty
 * @see #record
 * @see ext.data.TreeStore
 * @see #totalProperty
 * @see #successProperty
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Xml.html Original Ext JS documentation of 'Ext.data.reader.Xml'
 */
public class XmlReader extends DataReader {
  /**
   * @param config @inheritDoc
   */
  public function XmlReader(config:XmlReader = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * A namespace prefix that will be prepended to the field name when reading a
   * field from an XML node. Take, for example, the following Model:
   * <pre>
   * Ext.define('Foo', {
   *     extend: 'Ext.data.Model',
   *     fields: ['bar', 'baz']
   * });
   * </pre>
   * <p>The reader would need to be configured with a namespace of 'n' in order to read XML
   * data in the following format:</p>
   * <pre>
   * &lt;foo&gt;
   *     &lt;n:bar&gt;bar&lt;/n:bar&gt;
   *     &lt;n:baz&gt;baz&lt;/n:baz&gt;
   * &lt;/foo&gt;
   * </pre>
   * @default ''
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Xml.html#cfg-namespace Original Ext JS documentation of 'namespace'
   * @see #getNamespace()
   * @see #setNamespace()
   */
  public native function get namespace():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set namespace(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * <b>This is a required setting</b>. The DomQuery path to the repeated element which contains record information.
   * <p>By default, the elements which match the selector may be nested at any level
   * below the →<code>rootProperty</code></p>
   * <p>If this Reader is being used by a →<code>ext.data.TreeStore</code> to read
   * tree-structured data, then only first generation child nodes of the root element must be
   * selected, so the record selector must be specified with a more specific selector which
   * will not select all descendants. For example:</p>
   * <pre>
   * record: '&gt;node'
   * </pre>
   * @default ''
   * @see #rootProperty
   * @see ext.data.TreeStore
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Xml.html#cfg-record Original Ext JS documentation of 'record'
   * @see #getRecord()
   * @see #setRecord()
   */
  public native function get record():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set record(value:String):void;

  [Deprecated(replacement="rawData", since="5.1.1")]
  /**
   * Copy of →<code>rawData</code>.
   * @see #rawData
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Xml.html#property-xmlData Original Ext JS documentation of 'xmlData'
   */
  public native function get xmlData():Object;

  /**
   * @private
   */
  public native function set xmlData(value:Object):void;

  /**
   * Normalizes the data object.
   * @param data The data object
   * @return The documentElement property of the data object if present, or the same
   * object if not.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Xml.html#method-getData Original Ext JS documentation of 'getData'
   */
  override protected native function getData(data:Object):Object;

  /**
   * Returns the value of <code>namespace</code>.
   * @see #namespace
   */
  public native function getNamespace():String;

  /**
   * Returns the value of <code>record</code>.
   * @see #record
   */
  public native function getRecord():String;

  /**
   * Parses an XML document and returns a ResultSet containing the model instances.
   * @param doc Parsed XML document
   * @param readOptions See →<code>read()</code> for details.
   * @return The parsed result set
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.data.reader.Xml.html#method-readRecords Original Ext JS documentation of 'readRecords'
   * @see #read()
   */
  override public native function readRecords(doc:Object, readOptions:Object = null):ResultSet;

  /**
   * Sets the value of <code>namespace</code>.
   * @param namespace The new value.
   * @see #namespace
   */
  public native function setNamespace(namespace:String):void;

  /**
   * Sets the value of <code>record</code>.
   * @param record The new value.
   * @see #record
   */
  public native function setRecord(record:String):void;
}
}