package ext.dd {
import ext.event.Event;

[Native("Ext.dd.DropZone", require)]
/**
 * This class provides a container DD instance that allows dropping on multiple child target nodes.
 * <p>By default, this class requires that child nodes accepting drop are registered with
 * →<code>ext.dd.Registry</code>. However a simpler way to allow a DropZone to manage any number of target
 * elements is to configure the DropZone with an implementation of →<code>getTargetFromEvent()</code>
 * which interrogates the passed mouse event to see if it has taken place within an element,
 * or class of elements. This is easily done by using the event's
 * →<code>ext.event.Event.getTarget()</code> method to identify a node based on a CSS selector.</p>
 * <p>Once the DropZone has detected through calling getTargetFromEvent, that the mouse is over
 * a drop target, that target is passed as the first parameter to →<code>onNodeEnter()</code>,
 * →<code>onNodeOver()</code>, →<code>onNodeOut()</code>, →<code>onNodeDrop()</code>. You may configure the instance
 * of DropZone with implementations of these methods to provide application-specific behaviour
 * for these events to update both application state, and UI state.</p>
 * <p>For example to make a GridPanel a cooperating target with the example illustrated in
 * →<code>ext.dd.DragZone</code>, the following technique might be used:</p>
 * <pre>
 * myGridPanel.on('render', function() {
 *     myGridPanel.dropZone = new Ext.dd.DropZone(myGridPanel.getView().scroller, {
 *
 *         // If the mouse is over a grid row, return that node. This is
 *         // provided as the "target" parameter in all "onNodeXXXX"
 *         // node event handling functions
 *         getTargetFromEvent: function(e) {
 *             return e.getTarget(myGridPanel.getView().rowSelector);
 *         },
 *
 *         // On entry into a target node, highlight that node.
 *         onNodeEnter: function(target, dd, e, data) {
 *             Ext.fly(target).addCls('my-row-highlight-class');
 *         },
 *
 *         // On exit from a target node, unhighlight that node.
 *         onNodeOut: function(target, dd, e, data) {
 *             Ext.fly(target).removeCls('my-row-highlight-class');
 *         },
 *
 *         // While over a target node, return the default drop allowed class which
 *         // places a "tick" icon into the drag proxy.
 *         onNodeOver: function(target, dd, e, data) {
 *             return Ext.dd.DropZone.prototype.dropAllowed;
 *         },
 *
 *         // On node drop we can interrogate the target to find the underlying
 *         // application object that is the real target of the dragged data.
 *         // In this case, it is a Record in the GridPanel's Store.
 *         // We can use the data set up by the DragZone's getDragData method to read
 *         // any data we decided to attach in the DragZone's getDragData method.
 *         onNodeDrop: function(target, dd, e, data) {
 *             var rowIndex = myGridPanel.getView().findRowIndex(target);
 *             var r = myGridPanel.getStore().getAt(rowIndex);
 *             Ext.Msg.alert('Drop gesture', 'Dropped Record id ' + data.draggedRecord.id +
 *                 ' on Record id ' + r.id);
 *             return true;
 *         }
 *     });
 * }
 * </pre>
 * <p>See the →<code>ext.dd.DragZone</code> documentation for details about building
 * a DragZone which cooperates with this DropZone.</p>
 * @see ext.dd.#Registry ext.dd.Registry
 * @see #getTargetFromEvent()
 * @see ext.event.Event#getTarget()
 * @see #onNodeEnter()
 * @see #onNodeOver()
 * @see #onNodeOut()
 * @see #onNodeDrop()
 * @see ext.dd.DragZone
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dd.DropZone.html Original Ext JS documentation of 'Ext.dd.DropZone'
 */
public class DropZone extends DropTarget {
  /**
   * @param config @inheritDoc
   */
  public function DropZone(el:* = undefined, config:DropZone = null) {
    super(undefined);
  }

  /**
   * Returns a custom data object associated with the DOM node that is the target of the event.
   * By default this looks up the event target in the →<code>ext.dd.Registry</code>, although you can
   * override this method to provide your own custom lookup.
   * @param e The event
   * @return data The custom data
   * @see ext.dd.#Registry ext.dd.Registry
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dd.DropZone.html#method-getTargetFromEvent Original Ext JS documentation of 'getTargetFromEvent'
   */
  public native function getTargetFromEvent(e:Event):Object;

  /**
   * The function a →<code>ext.dd.DragSource</code> calls once to notify this drop zone that the dragged
   * item has been dropped on it. The drag zone will look up the target node based on the event
   * passed in, and if there is a node registered for that event, it will delegate to
   * →<code>onNodeDrop()</code> for node-specific handling, otherwise it will call
   * →<code>onContainerDrop()</code>.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @param source The drag source that was dragged over this drop zone
   * @param e The event
   * @param data An object containing arbitrary data supplied by the drag source
   * @return False if the drop was invalid.
   * @see ext.dd.DragSource
   * @see #onNodeDrop()
   * @see #onContainerDrop()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dd.DropZone.html#method-notifyDrop Original Ext JS documentation of 'notifyDrop'
   */
  override public native function notifyDrop(source:DragSource, e:Event, data:Object):Boolean;

  /**
   * The function a →<code>ext.dd.DragSource</code> calls once to notify this drop zone that the source
   * is now over the zone. The default implementation returns this.dropNotAllowed and expects
   * that only registered drop nodes can process drag drop operations, so if you need the drop
   * zone itself to be able to process drops you should override this method and provide a custom
   * implementation.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @param source The drag source that was dragged over this drop zone
   * @param e The event
   * @param data An object containing arbitrary data supplied by the drag source
   * @return status The CSS class that communicates the drop status back to the source
   * so that the underlying →<code>ext.dd.StatusProxy</code> can be updated
   * @see ext.dd.DragSource
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dd.DropZone.html#method-notifyEnter Original Ext JS documentation of 'notifyEnter'
   * @see ext.dd.StatusProxy
   */
  override public native function notifyEnter(source:DragSource, e:Event, data:Object):String;

  /**
   * The function a →<code>ext.dd.DragSource</code> calls once to notify this drop zone that the source
   * has been dragged out of the zone without dropping. If the drag source is currently over
   * a registered node, the notification will be delegated to →<code>onNodeOut()</code> for node-specific
   * handling, otherwise it will be ignored.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @param source The drag source that was dragged over this drop target
   * @param e The event
   * @param data An object containing arbitrary data supplied by the drag zone
   * @see ext.dd.DragSource
   * @see #onNodeOut()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dd.DropZone.html#method-notifyOut Original Ext JS documentation of 'notifyOut'
   */
  override public native function notifyOut(source:DragSource, e:Event, data:Object):void;

  /**
   * The function a →<code>ext.dd.DragSource</code> calls continuously while it is being dragged over
   * the drop zone. This method will be called on every mouse movement while the drag source
   * is over the drop zone. It will call →<code>onNodeOver()</code> while the drag source is over
   * a registered node, and will also automatically delegate to the appropriate node-specific
   * methods as necessary when the drag source enters and exits registered nodes
   * (→<code>onNodeEnter()</code>, →<code>onNodeOut()</code>). If the drag source is not currently over a
   * registered node, it will call →<code>onContainerOver()</code>.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @param source The drag source that was dragged over this drop zone
   * @param e The event
   * @param data An object containing arbitrary data supplied by the drag source
   * @return status The CSS class that communicates the drop status back to the source
   * so that the
   * underlying →<code>ext.dd.StatusProxy</code> can be updated
   * @see ext.dd.DragSource
   * @see #onNodeOver()
   * @see #onNodeEnter()
   * @see #onNodeOut()
   * @see #onContainerOver()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dd.DropZone.html#method-notifyOver Original Ext JS documentation of 'notifyOver'
   * @see ext.dd.StatusProxy
   */
  override public native function notifyOver(source:DragSource, e:Event, data:Object):String;

  /**
   * Called when the DropZone determines that a →<code>ext.dd.DragSource</code> has been dropped on it,
   * but not on any of its registered drop nodes. The default implementation returns false, so it
   * should be overridden to provide the appropriate processing of the drop event if you need
   * the drop zone itself to be able to accept drops. It should return true when valid so that
   * the drag source's repair action does not run.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @param source The drag source that was dragged over this drop zone
   * @param e The event
   * @param data An object containing arbitrary data supplied by the drag source
   * @return True if the drop was valid, else false
   * @see ext.dd.DragSource
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dd.DropZone.html#method-onContainerDrop Original Ext JS documentation of 'onContainerDrop'
   */
  public native function onContainerDrop(source:DragSource, e:Event, data:Object):Boolean;

  /**
   * Called while the DropZone determines that a →<code>ext.dd.DragSource</code> is being dragged
   * over it, but not over any of its registered drop nodes. The default implementation returns
   * this.dropNotAllowed, so it should be overridden to provide the proper feedback if necessary.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @param source The drag source that was dragged over this drop zone
   * @param e The event
   * @param data An object containing arbitrary data supplied by the drag source
   * @return status The CSS class that communicates the drop status back to the source
   * so that the underlying →<code>ext.dd.StatusProxy</code> can be updated
   * @see ext.dd.DragSource
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dd.DropZone.html#method-onContainerOver Original Ext JS documentation of 'onContainerOver'
   * @see ext.dd.StatusProxy
   */
  public native function onContainerOver(source:DragSource, e:Event, data:Object):String;

  /**
   * Called when the DropZone determines that a →<code>ext.dd.DragSource</code> has been dropped onto
   * the drop node. The default implementation returns false, so it should be overridden
   * to provide the appropriate processing of the drop event and return true so that the drag
   * source's repair action does not run.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @param nodeData The custom data associated with the drop node (this is the same
   * value returned from
   * →<code>getTargetFromEvent()</code> for this node)
   * @param source The drag source that was dragged over this drop zone
   * @param e The event
   * @param data An object containing arbitrary data supplied by the drag source
   * @return True if the drop was valid, else false
   * @see ext.dd.DragSource
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dd.DropZone.html#method-onNodeDrop Original Ext JS documentation of 'onNodeDrop'
   * @see #getTargetFromEvent()
   */
  public native function onNodeDrop(nodeData:Object, source:DragSource, e:Event, data:Object):Boolean;

  /**
   * Called when the DropZone determines that a →<code>ext.dd.DragSource</code> has entered a drop node
   * that has either been registered or detected by a configured implementation of
   * →<code>getTargetFromEvent()</code>. This method has no default implementation and should be
   * overridden to provide node-specific processing if necessary.
   * @param nodeData The custom data associated with the drop node (this is the same
   * value returned from →<code>getTargetFromEvent()</code> for this node)
   * @param source The drag source that was dragged over this drop zone
   * @param e The event
   * @param data An object containing arbitrary data supplied by the drag source
   * @see ext.dd.DragSource
   * @see #getTargetFromEvent()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dd.DropZone.html#method-onNodeEnter Original Ext JS documentation of 'onNodeEnter'
   */
  public native function onNodeEnter(nodeData:Object, source:DragSource, e:Event, data:Object):void;

  /**
   * Called when the DropZone determines that a →<code>ext.dd.DragSource</code> has been dragged out of
   * the drop node without dropping. This method has no default implementation and should be
   * overridden to provide node-specific processing if necessary.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @param nodeData The custom data associated with the drop node (this is the same
   * value returned from →<code>getTargetFromEvent()</code> for this node)
   * @param source The drag source that was dragged over this drop zone
   * @param e The event
   * @param data An object containing arbitrary data supplied by the drag source
   * @see ext.dd.DragSource
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dd.DropZone.html#method-onNodeOut Original Ext JS documentation of 'onNodeOut'
   * @see #getTargetFromEvent()
   */
  public native function onNodeOut(nodeData:Object, source:DragSource, e:Event, data:Object):void;

  /**
   * Called while the DropZone determines that a →<code>ext.dd.DragSource</code> is over a drop node
   * that has either been registered or detected by a configured implementation of
   * →<code>getTargetFromEvent()</code>. The default implementation returns this.dropAllowed,
   * so it should be overridden to provide the proper feedback.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @param nodeData The custom data associated with the drop node (this is the same
   * value returned from →<code>getTargetFromEvent()</code> for this node)
   * @param source The drag source that was dragged over this drop zone
   * @param e The event
   * @param data An object containing arbitrary data supplied by the drag source
   * @return status The CSS class that communicates the drop status back to the source
   * so that the underlying →<code>ext.dd.StatusProxy</code> can be updated
   * @see ext.dd.DragSource
   * @see #getTargetFromEvent()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dd.DropZone.html#method-onNodeOver Original Ext JS documentation of 'onNodeOver'
   * @see ext.dd.StatusProxy
   */
  public native function onNodeOver(nodeData:Object, source:DragSource, e:Event, data:Object):String;
}
}