package ext.dom {

[Native("Ext.dom.Helper", require)]
/**
 * The DomHelper class provides a layer of abstraction from DOM and transparently supports creating
 * elements via DOM or using HTML fragments. It also has the ability to create HTML fragment
 * templates from your DOM building code.
 * <p><b><i>DomHelper element specification object</i></b></p>
 * <p>A specification object is used when creating elements. Attributes of this object are assumed
 * to be element attributes, except for 4 special attributes:</p>
 * <ul>
 * <li><b>tag</b>: The tag name of the element</li>
 * <li><b>children (or cn)</b>: An array of the same kind of element definition objects to be created
 * and appended. These can be nested as deep as you want.</li>
 * <li><b>cls</b>: The class attribute of the element. This will end up being either the "class"
 * attribute on a HTML fragment or className for a DOM node, depending on whether DomHelper is using
 * fragments or DOM.</li>
 * <li><b>html</b>: The innerHTML for the element</li>
 * </ul>
 * <p><b><i>Insertion methods</i></b></p>
 * <p>Commonly used insertion methods:</p>
 * <ul>
 * <li>→<code>append()</code></li>
 * <li>→<code>insertBefore()</code></li>
 * <li>→<code>insertAfter()</code></li>
 * <li>→<code>overwrite()</code></li>
 * <li>→<code>insertHtml()</code></li>
 * </ul>
 * <p><b><i>Example</i></b></p>
 * <p>This is an example, where an unordered list with 3 children items is appended to an existing
 * element with id 'my-div':</p>
 * <pre>
 * var dh = Ext.DomHelper; // create shorthand alias
 * // specification object
 * var spec = {
 *     id: 'my-ul',
 *     tag: 'ul',
 *     cls: 'my-list',
 *     // append children after creating
 *     children: [     // may also specify 'cn' instead of 'children'
 *         {tag: 'li', id: 'item0', html: 'List Item 0'},
 *         {tag: 'li', id: 'item1', html: 'List Item 1'},
 *         {tag: 'li', id: 'item2', html: 'List Item 2'}
 *     ]
 * };
 * var list = dh.append(
 *     'my-div', // the context element 'my-div' can either be the id or the actual node
 *     spec      // the specification object
 * );
 * </pre>
 * <p>Element creation specification parameters in this class may also be passed as an Array
 * of specification objects. This can be used to insert multiple sibling nodes into an existing
 * container very efficiently. For example, to add more list items to the example above:</p>
 * <pre>
 * dh.append('my-ul', [
 *     {tag: 'li', id: 'item3', html: 'List Item 3'},
 *     {tag: 'li', id: 'item4', html: 'List Item 4'}
 * ]);
 * </pre>
 * <p><b><i>Templating</i></b></p>
 * <p>The real power is in the built-in templating. Instead of creating or appending any elements,
 * createTemplate returns a Template object which can be used over and over to insert new elements.
 * Revisiting the example above, we could utilize templating this time:</p>
 * <pre>
 * // create the node
 * var list = dh.append('my-div', {tag: 'ul', cls: 'my-list'});
 * // get template
 * var tpl = dh.createTemplate({tag: 'li', id: 'item{0}', html: 'List Item {0}'});
 *
 * for(var i = 0; i &lt; 5; i++){
 *     tpl.append(list, i); // use template to append to the actual node
 * }
 * </pre>
 * <p>An example using a template:</p>
 * <pre>
 * var html = '"{0}" href="{1}" class="nav"&gt;{2}';
 *
 * var tpl = new Ext.DomHelper.createTemplate(html);
 * tpl.append('blog-roll', ['link1', 'http://www.foxmulder.com/', "Fox's Site"]);
 * tpl.append('blog-roll', ['link2', 'http://www.danascully.org/', "Scully's Site"]);
 * </pre>
 * <p>The same example using named parameters:</p>
 * <pre>
 * var html = '"{id}" href="{url}" class="nav"&gt;{text}';
 *
 * var tpl = new Ext.DomHelper.createTemplate(html);
 * tpl.append('blog-roll', {
 *     id: 'link1',
 *     url: 'http://www.danascully.org/',
 *     text: "Scully's Site"
 * });
 * tpl.append('blog-roll', {
 *     id: 'link2',
 *     url: 'http://www.foxmulder.com/',
 *     text: "Fox's Site"
 * });
 * </pre>
 * <p><b><i>Compiling Templates</i></b></p>
 * <p>Templates are applied using regular expressions. The performance is great, but if you are adding
 * a bunch of DOM elements using the same template, you can increase performance even further by
 * "compiling" the template. The way "compile()" works is the template is parsed and broken up
 * at the different variable points and a dynamic function is created and eval'ed. The generated
 * function performs string concatenation of these parts and the passed variables instead of using
 * regular expressions.</p>
 * <pre>
 * var html = '"{id}" href="{url}" class="nav"&gt;{text}';
 *
 * var tpl = new Ext.DomHelper.createTemplate(html);
 * tpl.compile();
 *
 * // ... use template like normal
 * </pre>
 * <p><b><i>Performance Boost</i></b></p>
 * <p>DomHelper will transparently create HTML fragments when it can. Using HTML fragments instead of
 * DOM can significantly boost performance.</p>
 * <p>Element creation specification parameters may also be strings which are used as innerHTML.</p>
 * @see SDomHelper#append()
 * @see SDomHelper#insertBefore()
 * @see SDomHelper#insertAfter()
 * @see SDomHelper#overwrite()
 * @see SDomHelper#insertHtml()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html Original Ext JS documentation of 'Ext.dom.Helper'
 */
public const DomHelper:SDomHelper = new SDomHelper;
}