package ext.dom {
import ext.Base;
import ext.Template;

import js.HTMLElement;

[Native]
/**
 * The DomHelper class provides a layer of abstraction from DOM and transparently supports creating
 * elements via DOM or using HTML fragments. It also has the ability to create HTML fragment
 * templates from your DOM building code.
 * <p><b><i>DomHelper element specification object</i></b></p>
 * <p>A specification object is used when creating elements. Attributes of this object are assumed
 * to be element attributes, except for 4 special attributes:</p>
 * <ul>
 * <li><b>tag</b>: The tag name of the element</li>
 * <li><b>children (or cn)</b>: An array of the same kind of element definition objects to be created
 * and appended. These can be nested as deep as you want.</li>
 * <li><b>cls</b>: The class attribute of the element. This will end up being either the "class"
 * attribute on a HTML fragment or className for a DOM node, depending on whether DomHelper is using
 * fragments or DOM.</li>
 * <li><b>html</b>: The innerHTML for the element</li>
 * </ul>
 * <p><b><i>Insertion methods</i></b></p>
 * <p>Commonly used insertion methods:</p>
 * <ul>
 * <li>→<code>append()</code></li>
 * <li>→<code>insertBefore()</code></li>
 * <li>→<code>insertAfter()</code></li>
 * <li>→<code>overwrite()</code></li>
 * <li>→<code>insertHtml()</code></li>
 * </ul>
 * <p><b><i>Example</i></b></p>
 * <p>This is an example, where an unordered list with 3 children items is appended to an existing
 * element with id 'my-div':</p>
 * <pre>
 * var dh = Ext.DomHelper; // create shorthand alias
 * // specification object
 * var spec = {
 *     id: 'my-ul',
 *     tag: 'ul',
 *     cls: 'my-list',
 *     // append children after creating
 *     children: [     // may also specify 'cn' instead of 'children'
 *         {tag: 'li', id: 'item0', html: 'List Item 0'},
 *         {tag: 'li', id: 'item1', html: 'List Item 1'},
 *         {tag: 'li', id: 'item2', html: 'List Item 2'}
 *     ]
 * };
 * var list = dh.append(
 *     'my-div', // the context element 'my-div' can either be the id or the actual node
 *     spec      // the specification object
 * );
 * </pre>
 * <p>Element creation specification parameters in this class may also be passed as an Array
 * of specification objects. This can be used to insert multiple sibling nodes into an existing
 * container very efficiently. For example, to add more list items to the example above:</p>
 * <pre>
 * dh.append('my-ul', [
 *     {tag: 'li', id: 'item3', html: 'List Item 3'},
 *     {tag: 'li', id: 'item4', html: 'List Item 4'}
 * ]);
 * </pre>
 * <p><b><i>Templating</i></b></p>
 * <p>The real power is in the built-in templating. Instead of creating or appending any elements,
 * createTemplate returns a Template object which can be used over and over to insert new elements.
 * Revisiting the example above, we could utilize templating this time:</p>
 * <pre>
 * // create the node
 * var list = dh.append('my-div', {tag: 'ul', cls: 'my-list'});
 * // get template
 * var tpl = dh.createTemplate({tag: 'li', id: 'item{0}', html: 'List Item {0}'});
 *
 * for(var i = 0; i &lt; 5; i++){
 *     tpl.append(list, i); // use template to append to the actual node
 * }
 * </pre>
 * <p>An example using a template:</p>
 * <pre>
 * var html = '"{0}" href="{1}" class="nav"&gt;{2}';
 *
 * var tpl = new Ext.DomHelper.createTemplate(html);
 * tpl.append('blog-roll', ['link1', 'http://www.foxmulder.com/', "Fox's Site"]);
 * tpl.append('blog-roll', ['link2', 'http://www.danascully.org/', "Scully's Site"]);
 * </pre>
 * <p>The same example using named parameters:</p>
 * <pre>
 * var html = '"{id}" href="{url}" class="nav"&gt;{text}';
 *
 * var tpl = new Ext.DomHelper.createTemplate(html);
 * tpl.append('blog-roll', {
 *     id: 'link1',
 *     url: 'http://www.danascully.org/',
 *     text: "Scully's Site"
 * });
 * tpl.append('blog-roll', {
 *     id: 'link2',
 *     url: 'http://www.foxmulder.com/',
 *     text: "Fox's Site"
 * });
 * </pre>
 * <p><b><i>Compiling Templates</i></b></p>
 * <p>Templates are applied using regular expressions. The performance is great, but if you are adding
 * a bunch of DOM elements using the same template, you can increase performance even further by
 * "compiling" the template. The way "compile()" works is the template is parsed and broken up
 * at the different variable points and a dynamic function is created and eval'ed. The generated
 * function performs string concatenation of these parts and the passed variables instead of using
 * regular expressions.</p>
 * <pre>
 * var html = '"{id}" href="{url}" class="nav"&gt;{text}';
 *
 * var tpl = new Ext.DomHelper.createTemplate(html);
 * tpl.compile();
 *
 * // ... use template like normal
 * </pre>
 * <p><b><i>Performance Boost</i></b></p>
 * <p>DomHelper will transparently create HTML fragments when it can. Using HTML fragments instead of
 * DOM can significantly boost performance.</p>
 * <p>Element creation specification parameters may also be strings which are used as innerHTML.</p>
 * <p>Type of singleton DomHelper.</p>
 * @see SDomHelper#append()
 * @see SDomHelper#insertBefore()
 * @see SDomHelper#insertAfter()
 * @see SDomHelper#overwrite()
 * @see SDomHelper#insertHtml()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html Original Ext JS documentation of 'Ext.dom.Helper'
 * @see ext.dom.#DomHelper ext.dom.DomHelper
 */
public class SDomHelper extends Base {
  /**
   * Creates new DOM element(s) and appends them to el.
   * @param el The context element
   * @param o The DOM object spec (and children) or raw HTML blob
   * @param returnElement true to return a Ext.Element
   * @return The new node
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html#method-append Original Ext JS documentation of 'append'
   */
  public native function append(el:*, o:*, returnElement:Boolean = false):*;

  /**
   * Applies a style specification to an element.
   * <p>Styles in object form should be a valid DOM element style property.
   * <a href="http://www.w3schools.com/jsref/dom_obj_style.asp">Valid style property names</a>
   * (<i>along with the supported CSS version for each</i>)</p>
   * <pre>
   * // &lt;div id="my-el"&gt;Phineas Flynn&lt;/div&gt;
   *
   * var el = Ext.get('my-el'),
   *     dh = Ext.dom.Helper;
   *
   * dh.applyStyles(el, 'color: white;');
   *
   * dh.applyStyles(el, {
   *     fontWeight: 'bold',
   *     backgroundColor: 'gray',
   *     padding: '10px'
   * });
   *
   * dh.applyStyles(el, function () {
   *     if (name.initialConfig.html === 'Phineas Flynn') {
   *         return 'font-style: italic;';
   *         // OR return { fontStyle: 'italic' };
   *     }
   * });
   * </pre>
   * @param el The element to apply styles to
   * @param styles A style specification string e.g. 'width:100px',
   * or object in the form {width:'100px'}, or a function which returns such a specification.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html#method-applyStyles Original Ext JS documentation of 'applyStyles'
   */
  public native function applyStyles(el:*, styles:*):void;

  /**
   * Creates new DOM element(s) without inserting them to the document.
   * @param o The DOM object spec (and children) or raw HTML blob
   * @return The new un-inserted node
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html#method-createDom Original Ext JS documentation of 'createDom'
   */
  public native function createDom(o:*):HTMLElement;

  [Deprecated(replacement="markup", since="5.0.0")]
  /**
   * Alias for →<code>markup()</code>.
   * @see SDomHelper#markup()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html#method-createHtml Original Ext JS documentation of 'createHtml'
   */
  public native function createHtml(spec:*):void;

  /**
   * Creates a new Ext.Template from the DOM object spec.
   * @param o The DOM object spec (and children)
   * @return The new template
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html#method-createTemplate Original Ext JS documentation of 'createTemplate'
   */
  public native function createTemplate(o:Object):Template;

  /**
   * Converts the styles from the given object to text. The styles are CSS style names
   * with their associated value.
   * <p>The basic form of this method returns a string:</p>
   * <pre>
   *  var s = Ext.DomHelper.generateStyles({
   *      backgroundColor: 'red'
   *  });
   *
   *  // s = 'background-color:red;'
   * </pre>
   * <p>Alternatively, this method can append to an output array.</p>
   * <pre>
   *  var buf = [];
   *
   *  ...
   *
   *  Ext.DomHelper.generateStyles({
   *      backgroundColor: 'red'
   *  }, buf);
   * </pre>
   * <p>In this case, the style text is pushed on to the array and the array is returned.</p>
   * @param styles The object describing the styles.
   * @param buffer The output buffer.
   * @param encode <code>true</code> to →<code>ext.StringUtil.htmlEncode()</code> property values if they
   * are going to be inserted as HTML attributes.
   * @return If buffer is passed, it is returned. Otherwise the style
   * string is returned.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html#method-generateStyles Original Ext JS documentation of 'generateStyles'
   * @see ext.SStringUtil#htmlEncode() ext.StringUtil.htmlEncode()
   */
  public native function generateStyles(styles:Object, buffer:Array = null, encode:Boolean = false):*;

  /**
   * Creates new DOM element(s) and inserts them after el.
   * @param el The context element
   * @param o The DOM object spec (and children)
   * @param returnElement true to return a Ext.Element
   * @return The new node
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html#method-insertAfter Original Ext JS documentation of 'insertAfter'
   */
  public native function insertAfter(el:*, o:Object, returnElement:Boolean = false):*;

  /**
   * Creates new DOM element(s) and inserts them before el.
   * @param el The context element
   * @param o The DOM object spec (and children) or raw HTML blob
   * @param returnElement true to return a Ext.Element
   * @return The new node
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html#method-insertBefore Original Ext JS documentation of 'insertBefore'
   */
  public native function insertBefore(el:*, o:*, returnElement:Boolean = false):*;

  /**
   * Creates new DOM element(s) and inserts them as the first child of el.
   * @param el The context element
   * @param o The DOM object spec (and children) or raw HTML blob
   * @param returnElement true to return a Ext.Element
   * @return The new node
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html#method-insertFirst Original Ext JS documentation of 'insertFirst'
   */
  public native function insertFirst(el:*, o:*, returnElement:Boolean = false):*;

  /**
   * Inserts an HTML fragment into the DOM.
   * @param where Where to insert the html in relation to el - beforeBegin,
   * afterBegin, beforeEnd, afterEnd.
   * <p>For example take the following HTML: <code>&lt;div&gt;Contents&lt;/div&gt;</code></p>
   * <p>Using different <code>where</code> values inserts element to the following places:</p>
   * <ul>
   * <li>beforeBegin: <code>&lt;HERE&gt;&lt;div&gt;Contents&lt;/div&gt;</code></li>
   * <li>afterBegin: <code>&lt;div&gt;&lt;HERE&gt;Contents&lt;/div&gt;</code></li>
   * <li>beforeEnd: <code>&lt;div&gt;Contents&lt;HERE&gt;&lt;/div&gt;</code></li>
   * <li>afterEnd: <code>&lt;div&gt;Contents&lt;/div&gt;&lt;HERE&gt;</code></li>
   * </ul>
   * @param el The context element
   * @param html The HTML fragment
   * @return The new node
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html#method-insertHtml Original Ext JS documentation of 'insertHtml'
   */
  public native function insertHtml(where:String, el:*, html:String):HTMLElement;

  /**
   * Returns the markup for the passed Element(s) config.
   * @param spec The DOM object spec (and children).
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html#method-markup Original Ext JS documentation of 'markup'
   */
  public native function markup(spec:Object):String;

  /**
   * Creates new DOM element(s) and overwrites the contents of el with them.
   * @param el The context element
   * @param html The DOM object spec (and children) or raw HTML blob
   * @param returnElement true to return an Ext.Element
   * @default false
   * @return The new node
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Helper.html#method-overwrite Original Ext JS documentation of 'overwrite'
   */
  public native function overwrite(el:*, html:*, returnElement:Boolean = false):*;
}
}