package ext.dom {
import ext.Base;

import js.HTMLElement;

[Native]
/**
 * Provides high performance selector/xpath processing by compiling queries
 * into reusable functions. New pseudo classes and matchers can be plugged. It works
 * on HTML and XML documents (if a content node is passed in).
 * <p>DomQuery supports most of the <a href="http://www.w3.org/TR/2005/WD-css3-selectors-20051215/#selectors">CSS3 selectors spec</a>, along with some custom selectors
 * and basic XPath.</p>
 * <p>All selectors, attribute filters and pseudos below can be combined infinitely in any order.
 * For example <code>div.foo:nth-child(odd)[&#64;foo=bar].bar:first</code> would be a perfectly valid selector.
 * Node filters are processed in the order in which they appear, which allows you to optimize
 * your queries for your document structure.</p>
 * <p><b><i>Simple Selectors</i></b></p>
 * <p>For performance reasons, some query methods accept selectors that are termed as
 * <b>simple selectors</b>. A simple selector is a selector that does not include contextual
 * information about any parent/sibling elements.</p>
 * <p>Some examples of valid simple selectors:</p>
 * <pre>
 * var simple = '.foo'; // Only asking for the class name on the element
 * var simple = 'div.bar'; // Only asking for the tag/class name on the element
 * var simple = '[href];' // Asking for an attribute on the element.
 * var simple = ':not(.foo)'; // Only asking for the non-matches against the class name
 * var simple = 'span:first-child'; // Doesn't require any contextual information about
 *                                  // the parent node
 * </pre>
 * <p>Simple examples of invalid simple selectors:</p>
 * <pre>
 * var notSimple = 'div.foo div.bar'; // Requires matching a parent node by class name
 * var notSimple = 'span + div'; //  Requires matching a sibling by tag name
 * </pre>
 * <p><b><i>Element Selectors:</i></b></p>
 * <ul>
 * <li><b><code>&#42;</code></b> any element</li>
 * <li><b><code>E</code></b> an element with the tag E</li>
 * <li><b><code>E F</code></b> All descendent elements of E that have the tag F</li>
 * <li><b><code>E &gt; F</code></b> or <b>E/F</b> all direct children elements of E that have the tag F</li>
 * <li><b><code>E + F</code></b> all elements with the tag F that are immediately preceded by an element
 * with the tag E</li>
 * <li><b><code>E ~ F</code></b> all elements with the tag F that are preceded by a sibling element with the tag E</li>
 * </ul>
 * <p><b><i>Attribute Selectors:</i></b></p>
 * <p>The use of <code>&#64;</code> and quotes are optional. For example, <code>div[&#64;foo='bar']</code> is also a valid attribute
 * selector.</p>
 * <ul>
 * <li><b><code>E[foo]</code></b> has an attribute "foo"</li>
 * <li><b><code>E[foo=bar]</code></b> has an attribute "foo" that equals "bar"</li>
 * <li><b><code>E[foo^=bar]</code></b> has an attribute "foo" that starts with "bar"</li>
 * <li><b><code>E[foo$=bar]</code></b> has an attribute "foo" that ends with "bar"</li>
 * <li><b><code>E[foo&#42;=bar]</code></b> has an attribute "foo" that contains the substring "bar"</li>
 * <li><b><code>E[foo%=2]</code></b> has an attribute "foo" that is evenly divisible by 2</li>
 * <li><b><code>E[foo!=bar]</code></b> attribute "foo" does not equal "bar"</li>
 * </ul>
 * <p><b><i>Pseudo Classes:</i></b></p>
 * <ul>
 * <li><b><code>E:first-child</code></b> E is the first child of its parent</li>
 * <li><b><code>E:last-child</code></b> E is the last child of its parent</li>
 * <li><b><code>E:nth-child(_n_)</code></b> E is the _n_th child of its parent (1 based as per the spec)</li>
 * <li><b><code>E:nth-child(odd)</code></b> E is an odd child of its parent</li>
 * <li><b><code>E:nth-child(even)</code></b> E is an even child of its parent</li>
 * <li><b><code>E:only-child</code></b> E is the only child of its parent</li>
 * <li><b><code>E:checked</code></b> E is an element that is has a checked attribute that is true
 * (e.g. a radio or checkbox)</li>
 * <li><b><code>E:first</code></b> the first E in the resultset</li>
 * <li><b><code>E:last</code></b> the last E in the resultset</li>
 * <li><b><code>E:nth(_n_)</code></b> the _n_th E in the resultset (1 based)</li>
 * <li><b><code>E:odd</code></b> shortcut for :nth-child(odd)</li>
 * <li><b><code>E:even</code></b> shortcut for :nth-child(even)</li>
 * <li><b><code>E:contains(foo)</code></b> E's innerHTML contains the substring "foo"</li>
 * <li><b><code>E:nodeValue(foo)</code></b> E contains a textNode with a nodeValue that equals "foo"</li>
 * <li><b><code>E:not(S)</code></b> an E element that does not match simple selector S</li>
 * <li><b><code>E:has(S)</code></b> an E element that has a descendent that matches simple selector S</li>
 * <li><b><code>E:next(S)</code></b> an E element whose next sibling matches simple selector S</li>
 * <li><b><code>E:prev(S)</code></b> an E element whose previous sibling matches simple selector S</li>
 * <li><b><code>E:any(S1|S2|S2)</code></b> an E element which matches any of the simple selectors S1, S2 or S3</li>
 * <li><b><code>E:visible(true)</code></b> an E element which is deeply visible according to
 * →<code>ext.dom.Element.isVisible()</code></li>
 * </ul>
 * <p><b><i>CSS Value Selectors:</i></b></p>
 * <ul>
 * <li><b><code>E{display=none}</code></b> css value "display" that equals "none"</li>
 * <li><b><code>E{display^=none}</code></b> css value "display" that starts with "none"</li>
 * <li><b><code>E{display$=none}</code></b> css value "display" that ends with "none"</li>
 * <li><b><code>E{display&#42;=none}</code></b> css value "display" that contains the substring "none"</li>
 * <li><b><code>E{display%=2}</code></b> css value "display" that is evenly divisible by 2</li>
 * <li><b><code>E{display!=none}</code></b> css value "display" that does not equal "none"</li>
 * </ul>
 * <p><b><i>XML Namespaces:</i></b></p>
 * <ul>
 * <li><b><code>ns|E</code></b> an element with tag E and namespace prefix ns</li>
 * </ul>
 * <p>Type of singleton DomQuery.</p>
 * @see ext.dom.Element#isVisible()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Query.html Original Ext JS documentation of 'Ext.dom.Query'
 * @see ext.dom.#DomQuery ext.dom.DomQuery
 */
public class SDomQuery extends Base {
  /**
   * Collection of matching regular expressions and code snippets.
   * Each capture group within <code>()</code> will be replace the <code>{}</code> in the select
   * statement as specified by their index.
   * @default [
   *     {
   *         re: /^\.([\w\-\\]+)/,
   *         select: useClassList ? 'n = byClassName(n, "{1}");' : 'n = byClassName(n, " {1} ");'
   *     },
   *     {
   *         re: /^\:([\w\-]+)(?:\(((?:[^\s>\/]&#42;|.&#42;?))\))?/,
   *         select: 'n = byPseudo(n, "{1}", "{2}");'
   *     },
   *     {
   *         re: /^(?:([\[\{])(?:&#64;)?([\w\-]+)\s?(?:(=|.=)\s?['"]?(.&#42;?)["']?)?[\]\}])/,
   *         select: 'n = byAttribute(n, "{2}", "{4}", "{3}", "{1}");'
   *     },
   *     {
   *         re: /^#([\w\-\\]+)/,
   *         select: 'n = byId(n, "{1}");'
   *     },
   *     {
   *         re: /^&#64;([\w\-\.]+)/,
   *         select: 'return {firstChild:{nodeValue:attrValue(n, "{1}")}};'
   *     }
   * ]
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Query.html#property-matchers Original Ext JS documentation of 'matchers'
   */
  public native function get matchers():*;

  /**
   * @private
   */
  public native function set matchers(value:*):void;

  /**
   * Collection of operator comparison functions.
   * The default operators are <code>=</code>, <code>!=</code>, <code>^=</code>, <code>$=</code>, <code>&#42;=</code>, <code>%=</code>, <code>|=</code> and <code>~=</code>.
   * <p>New operators can be added as long as the match the format <i>c</i><code>=</code> where <i>c</i>
   * is any character other than space, <code>&gt;</code>, or <code>&lt;</code>.</p>
   * <p>Operator functions are passed the following parameters:</p>
   * <ul>
   * <li><code>propValue</code> : The property value to test.</li>
   * <li><code>compareTo</code> : The value to compare to.</li>
   * </ul>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Query.html#property-operators Original Ext JS documentation of 'operators'
   */
  public native function get operators():Object;

  /**
   * @private
   */
  public native function set operators(value:Object):void;

  /**
   * Object hash of "pseudo class" filter functions which are used when filtering selections.
   * Each function is passed two parameters:
   * <ul>
   * <li>
   * <p><b>c</b> : Array
   * An Array of DOM elements to filter.</p>
   * </li>
   * <li>
   * <p><b>v</b> : String
   * The argument (if any) supplied in the selector.</p>
   * </li>
   * </ul>
   * <p>A filter function returns an Array of DOM elements which conform to the pseudo class.
   * In addition to the provided pseudo classes listed above such as <code>first-child</code> and
   * <code>nth-child</code>, developers may add additional, custom psuedo class filters to select
   * elements according to application-specific requirements.</p>
   * <p>For example, to filter <code>a</code> elements to only return links to <b>external</b> resources:</p>
   * <pre>
   * Ext.DomQuery.pseudos.external = function(c, v) {
   *     var r = [], ri = -1;
   *     for(var i = 0, ci; ci = c[i]; i++) {
   *         // Include in result set only if it's a link to an external resource
   *         if (ci.hostname != location.hostname) {
   *             r[++ri] = ci;
   *         }
   *     }
   *     return r;
   * };
   * </pre>
   * <p>Then external links could be gathered with the following statement:</p>
   * <pre>
   * var externalLinks = Ext.select("a:external");
   * </pre>
   * @default {
   *     "first-child": function(c) {
   *         var r = [],
   *             ri = -1,
   *             n, i, ci;
   *         for (i = 0; (ci = n = c[i]); i++) {
   *             while ((n = n.previousSibling) && n.nodeType != 1) {}
   *             // This block is intentionally left blank
   *             if (!n) {
   *                 r[++ri] = ci;
   *             }
   *         }
   *         return r;
   *     },
   *     "last-child": function(c) {
   *         var r = [],
   *             ri = -1,
   *             n, i, ci;
   *         for (i = 0; (ci = n = c[i]); i++) {
   *             while ((n = n.nextSibling) && n.nodeType != 1) {}
   *             // This block is intentionally left blank
   *             if (!n) {
   *                 r[++ri] = ci;
   *             }
   *         }
   *         return r;
   *     },
   *     "nth-child": function(c, a) {
   *         var r = [],
   *             ri = -1,
   *             m = nthRe.exec(a == "even" && "2n" || a == "odd" && "2n+1" || !nthRe2.test(a) && "n+" + a || a),
   *             // eslint-disable-line max-len
   *             f = (m[1] || 1) - 0,
   *             l = m[2] - 0,
   *             i, n, j, cn, pn;
   *         for (i = 0; n = c[i]; i++) {
   *             pn = n.parentNode;
   *             if (batch != pn._batch) {
   *                 j = 0;
   *                 for (cn = pn.firstChild; cn; cn = cn.nextSibling) {
   *                     if (cn.nodeType == 1) {
   *                         cn.nodeIndex = ++j;
   *                     }
   *                 }
   *                 pn._batch = batch;
   *             }
   *             if (f == 1) {
   *                 if (l === 0 || n.nodeIndex == l) {
   *                     r[++ri] = n;
   *                 }
   *             } else if ((n.nodeIndex + l) % f === 0) {
   *                 r[++ri] = n;
   *             }
   *         }
   *         return r;
   *     },
   *     "only-child": function(c) {
   *         var r = [],
   *             ri = -1,
   *             i, ci;
   *         for (i = 0; ci = c[i]; i++) {
   *             if (!prev(ci) && !next(ci)) {
   *                 r[++ri] = ci;
   *             }
   *         }
   *         return r;
   *     },
   *     "empty": function(c) {
   *         var r = [],
   *             ri = -1,
   *             i, ci, cns, j, cn, empty;
   *         for (i = 0; ci = c[i]; i++) {
   *             cns = ci.childNodes;
   *             j = 0;
   *             empty = true;
   *             while (cn = cns[j]) {
   *                 ++j;
   *                 if (cn.nodeType == 1 || cn.nodeType == 3) {
   *                     empty = false;
   *                     break;
   *                 }
   *             }
   *             if (empty) {
   *                 r[++ri] = ci;
   *             }
   *         }
   *         return r;
   *     },
   *     "contains": function(c, v) {
   *         var r = [],
   *             ri = -1,
   *             i, ci;
   *         for (i = 0; ci = c[i]; i++) {
   *             if ((ci.textContent || ci.innerText || ci.text || '').indexOf(v) != -1) {
   *                 r[++ri] = ci;
   *             }
   *         }
   *         return r;
   *     },
   *     "nodeValue": function(c, v) {
   *         var r = [],
   *             ri = -1,
   *             i, ci;
   *         for (i = 0; ci = c[i]; i++) {
   *             if (ci.firstChild && ci.firstChild.nodeValue == v) {
   *                 r[++ri] = ci;
   *             }
   *         }
   *         return r;
   *     },
   *     "checked": function(c) {
   *         var r = [],
   *             ri = -1,
   *             i, ci;
   *         for (i = 0; ci = c[i]; i++) {
   *             if (ci.checked === true) {
   *                 r[++ri] = ci;
   *             }
   *         }
   *         return r;
   *     },
   *     "not": function(c, ss) {
   *         return DQ.filter(c, ss, true);
   *     },
   *     "any": function(c, selectors) {
   *         var ss = selectors.split('|'),
   *             r = [],
   *             ri = -1,
   *             s, i, ci, j;
   *         for (i = 0; ci = c[i]; i++) {
   *             for (j = 0; s = ss[j]; j++) {
   *                 if (DQ.is(ci, s)) {
   *                     r[++ri] = ci;
   *                     break;
   *                 }
   *             }
   *         }
   *         return r;
   *     },
   *     "odd": function(c) {
   *         return this["nth-child"](c, "odd");
   *     },
   *     "even": function(c) {
   *         return this["nth-child"](c, "even");
   *     },
   *     "nth": function(c, a) {
   *         return c[a - 1] || [];
   *     },
   *     "first": function(c) {
   *         return c[0] || [];
   *     },
   *     "last": function(c) {
   *         return c[c.length - 1] || [];
   *     },
   *     "has": function(c, ss) {
   *         var s = DQ.select,
   *             r = [],
   *             ri = -1,
   *             i, ci;
   *         for (i = 0; ci = c[i]; i++) {
   *             if (s(ss, ci).length > 0) {
   *                 r[++ri] = ci;
   *             }
   *         }
   *         return r;
   *     },
   *     "next": function(c, ss) {
   *         var is = DQ.is,
   *             r = [],
   *             ri = -1,
   *             i, ci, n;
   *         for (i = 0; ci = c[i]; i++) {
   *             n = next(ci);
   *             if (n && is(n, ss)) {
   *                 r[++ri] = ci;
   *             }
   *         }
   *         return r;
   *     },
   *     "prev": function(c, ss) {
   *         var is = DQ.is,
   *             r = [],
   *             ri = -1,
   *             i, ci, n;
   *         for (i = 0; ci = c[i]; i++) {
   *             n = prev(ci);
   *             if (n && is(n, ss)) {
   *                 r[++ri] = ci;
   *             }
   *         }
   *         return r;
   *     },
   *     focusable: function(candidates) {
   *         var len = candidates.length,
   *             results = [],
   *             i = 0,
   *             c;
   *         for (; i < len; i++) {
   *             c = candidates[i];
   *             if (Ext.fly(c, '_DomQuery').isFocusable()) {
   *                 results.push(c);
   *             }
   *         }
   *         return results;
   *     },
   *     visible: function(candidates, deep) {
   *         var len = candidates.length,
   *             results = [],
   *             i = 0,
   *             c;
   *         for (; i < len; i++) {
   *             c = candidates[i];
   *             if (Ext.fly(c, '_DomQuery').isVisible(deep)) {
   *                 results.push(c);
   *             }
   *         }
   *         return results;
   *     },
   *     isScrolled: function(c) {
   *         var r = [],
   *             ri = -1,
   *             i, ci, s;
   *         for (i = 0; ci = c[i]; i++) {
   *             s = Ext.fly(ci, '_DomQuery').getScroll();
   *             if (s.top > 0 || s.left > 0) {
   *                 r[++ri] = ci;
   *             }
   *         }
   *         return r;
   *     }
   * }
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Query.html#property-pseudos Original Ext JS documentation of 'pseudos'
   */
  public native function get pseudos():*;

  /**
   * @private
   */
  public native function set pseudos(value:*):void;

  /**
   * Compiles a selector/xpath query into a reusable function. The returned function
   * takes one parameter "root" (optional), which is the context node from where the query
   * should start.
   * @param path The selector/xpath query
   * @param type Either "select" or "simple" for a simple selector match
   * @default "select"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Query.html#method-compile Original Ext JS documentation of 'compile'
   */
  public native function compile(path:String, type:String = "select"):Function;

  [ArrayElementType("js.HTMLElement")]
  /**
   * Filters an array of elements to only include matches of a simple selector
   * @param els An array of elements to filter
   * @param ss The simple selector to test
   * @param nonMatches If true, it returns the elements that DON'T match
   * the selector instead of the ones that match
   * @return An Array of DOM elements which match the selector. If there are
   * no matches, and empty Array is returned.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Query.html#method-filter Original Ext JS documentation of 'filter'
   */
  public native function filter(els:Array, ss:String, nonMatches:Boolean):Array;

  /**
   * Get the text value for a node, optionally replacing null with the defaultValue.
   * @param node The node
   * @param defaultValue When specified, this is return as empty value.
   * @return The value
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Query.html#method-getNodeValue Original Ext JS documentation of 'getNodeValue'
   */
  public native function getNodeValue(node:Object, defaultValue:String = null):String;

  [ArrayElementType("js.HTMLElement")]
  /**
   * Selects an array of DOM nodes using JavaScript-only implementation.
   * <p>Use →<code>select()</code> to take advantage of browsers built-in support for CSS selectors.</p>
   * @param path The selector/xpath query (can be a comma separated list
   * of selectors)
   * @param root (Default document) The start of the query.
   * @default document
   * @return An Array of DOM elements which match the selector. If there are
   * no matches, and empty Array is returned.
   * @see SDomQuery#select()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Query.html#method-jsSelect Original Ext JS documentation of 'jsSelect'
   */
  public native function jsSelect(path:String, root:* = undefined, type:* = undefined):Array;

  /**
   * Returns true if the passed element(s) match the passed simple selector
   * @param el An element id, element or array of elements
   * @param ss The simple selector to test
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Query.html#method-is Original Ext JS documentation of 'is'
   */
  public native function matches(el:*, ss:String):Boolean;

  [ArrayElementType("js.HTMLElement")]
  /**
   * Selects an array of DOM nodes by CSS/XPath selector.
   * <p>Uses <a href="https://developer.mozilla.org/en/DOM/document.querySelectorAll">document.querySelectorAll</a> if browser supports that, otherwise falls back to
   * →<code>ext.dom.DomQuery.jsSelect()</code> to do the work.</p>
   * @param path The selector/xpath query
   * @param root (Default document) The start of the query.
   * @default document
   * @param type Either "select" or "simple" for a simple selector match
   * (only valid when used when the call is deferred to the jsSelect method)
   * @default "select"
   * @param single Pass <code>true</code> to select only the first matching node using
   * <code>document.querySelector</code> (where available)
   * @return An array of DOM elements (not a NodeList as returned by
   * <code>querySelectorAll</code>).
   * @see ext.dom.SDomQuery#jsSelect() ext.dom.DomQuery.jsSelect()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Query.html#method-select Original Ext JS documentation of 'select'
   */
  public native function select(path:String, root:HTMLElement = null, type:String = "select", single:Boolean = false):Array;

  /**
   * Selects a single element.
   * @param path The selector/xpath query
   * @param root (Default document) The start of the query.
   * @default document
   * @return The DOM element which matched the selector.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Query.html#method-selectNode Original Ext JS documentation of 'selectNode'
   */
  public native function selectNode(path:String, root:HTMLElement = null):HTMLElement;

  /**
   * Selects the value of a node, parsing integers and floats.
   * Returns the defaultValue, or 0 if none is specified.
   * @param path The selector/xpath query
   * @param root (Default document) The start of the query.
   * @default document
   * @param defaultValue When specified, this is return as empty value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Query.html#method-selectNumber Original Ext JS documentation of 'selectNumber'
   */
  public native function selectNumber(path:String, root:HTMLElement = null, defaultValue:Number = NaN):Number;

  /**
   * Selects the value of a node, optionally replacing null with the defaultValue.
   * @param path The selector/xpath query
   * @param root (Default document) The start of the query.
   * @default document
   * @param defaultValue When specified, this is return as empty value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.dom.Query.html#method-selectValue Original Ext JS documentation of 'selectValue'
   */
  public native function selectValue(path:String, root:HTMLElement = null, defaultValue:String = null):String;
}
}