package ext.draw {
import ext.Base;

[Native("Ext.draw.Matrix", require)]
/**
 * Ext.draw.Matix is a utility class used to calculate
 * <a href="http://en.wikipedia.org/wiki/Affine_transformation">affine transformation</a> matrix.
 * The matrix class is used to apply transformations to existing
 * <i>sprites</i> (→<code>ext.draw.sprite.Sprite</code>) using a number of convenience transform
 * methods.
 * <p>Transformations configured directly on a sprite are processed in the following order:
 * scaling, rotation, and translation. The matrix class offers additional flexibility.
 * Once a sprite is created, you can use the matrix class's transform methods as many
 * times as needed and in any order you choose.</p>
 * <p>To demonstrate, we'll start with a simple <i>rect</i> (→<code>ext.draw.sprite.RectSprite</code>) sprite
 * with the intent of rotating it 180 degrees with the bottom right corner being the
 * center of rotation. To begin, let's look at the initial, untransformed sprite:</p>
 * <p><b>Example:</b></p>
 * <pre>
 * var drawContainer = new Ext.draw.Container({
 *     renderTo: Ext.getBody(),
 *     width: 380,
 *     height: 380,
 *     sprites: [{
 *         type: 'rect',
 *         width: 100,
 *         height: 100,
 *         fillStyle: 'red'
 *     }]
 * });
 * </pre>
 * <p>Next, we'll use the →<code>rotate()</code> and →<code>translate()</code> methods from our matrix
 * class to position the rect sprite.</p>
 * <p><b>Example:</b></p>
 * <pre>
 * var drawContainer = new Ext.draw.Container({
 *     renderTo: Ext.getBody(),
 *     width: 380,
 *     height: 380,
 *     sprites: [{
 *         type: 'rect',
 *         width: 100,
 *         height: 100,
 *         fillStyle: 'red'
 *     }]
 * });
 *
 * var main = drawContainer.getSurface();
 * var rect = main.getItems()[0];
 *
 * var m = new Ext.draw.Matrix().translate(100, 100).
 * rotate(Math.PI).
 * translate(-100, - 100);
 *
 * rect.setTransform(m);
 * main.renderFrame();
 * </pre>
 * <p>In the previous example we perform the following steps in order to achieve our
 * desired rotated output:</p>
 * <ul>
 * <li>translate the rect to the right and down by 100</li>
 * <li>rotate by 180 degrees</li>
 * <li>translate the rect to the right and down by 100</li>
 * </ul>
 * <p><b>Note:</b> A couple of things to note at this stage; 1) the rotation center point is
 * the upper left corner of the sprite by default and 2) with transformations, the
 * sprite itself isn't transformed, but rather the entire coordinate plane of the sprite
 * is transformed. The coordinate plane itself is translated by 100 and then rotated
 * 180 degrees. And that is why in the third step we translate the sprite using
 * negative values. Translating by -100 in the third step results in the sprite
 * visually moving to the right and down within the draw container.</p>
 * <p>Fortunately there is a shortcut we can apply using two optional params of the rotate
 * method allowing us to specify the center point of rotation:</p>
 * <p><b>Example:</b></p>
 * <pre>
 * var drawContainer = new Ext.draw.Container({
 *     renderTo: Ext.getBody(),
 *     width: 380,
 *     height: 380,
 *     sprites: [{
 *         type: 'rect',
 *         width: 100,
 *         height: 100,
 *         fillStyle: 'red'
 *     }]
 * });
 *
 * var main = drawContainer.getSurface();
 * var rect = main.getItems()[0];
 *
 * var m = new Ext.draw.Matrix().rotate(Math.PI, 100, 100);
 *
 * rect.setTransform(m);
 * main.renderFrame();
 * </pre>
 * <p>This class is compatible with
 * <a href="http://www.w3.org/TR/SVG11/coords.html#InterfaceSVGMatrix">SVGMatrix</a> except:</p>
 * <ol>
 * <li>Ext.draw.Matrix is not read only</li>
 * <li>Using Number as its values rather than floats</li>
 * </ol>
 * <p>Using this class helps to reduce the severe numeric
 * <a href="http://stackoverflow.com/questions/8784405/large-numbers-in-html-canvas-translate-result-in-strange-behavior">problem with HTML Canvas and SVG transformation</a></p>
 * <p>Additionally, there's no way to get the current transformation matrix
 * <a href="http://stackoverflow.com/questions/7395813/html5-canvas-get-transform-matrix">in Canvas</a>.</p>
 * @see ext.draw.sprite.Sprite
 * @see ext.draw.sprite.RectSprite
 * @see #rotate()
 * @see #translate()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html Original Ext JS documentation of 'Ext.draw.Matrix'
 */
public class Matrix extends Base {
  /**
   * Create a matrix from <code>mat</code>. If <code>mat</code> is already a matrix, returns it.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#static-method-create Original Ext JS documentation of 'create'
   */
  public static native function create(mat:*):Matrix;

  /**
   * Return the affine matrix that transform two points (x0, y0) and (x1, y1) to (x0p, y0p)
   * and (x1p, y1p)
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#static-method-createAffineMatrixFromTwoPair Original Ext JS documentation of 'createAffineMatrixFromTwoPair'
   */
  public static native function createAffineMatrixFromTwoPair(x0:Number, y0:Number, x1:Number, y1:Number, x0p:Number, y0p:Number, x1p:Number, y1p:Number):void;

  /**
   * Return the affine matrix that transform two points (x0, y0) and (x1, y1) to (x0p, y0p)
   * and (x1p, y1p)
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#static-method-createPanZoomFromTwoPair Original Ext JS documentation of 'createPanZoomFromTwoPair'
   */
  public static native function createPanZoomFromTwoPair(x0:Number, y0:Number, x1:Number, y1:Number, x0p:Number, y0p:Number, x1p:Number, y1p:Number):void;

  /**
   * Create a flyweight to wrap the given array.
   * The flyweight will directly refer the object and the elements can be changed
   * by other methods.
   * <p>Do not hold the instance of flyweight matrix.</p>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#static-method-fly Original Ext JS documentation of 'fly'
   */
  public static native function fly(elements:Array):Matrix;

  /**
   * Create an affine transform matrix.
   * @param xx Coefficient from x to x
   * @param xy Coefficient from x to y
   * @param yx Coefficient from y to x
   * @param yy Coefficient from y to y
   * @param dx Offset of x
   * @param dy Offset of y
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-constructor Original Ext JS documentation of 'constructor'
   */
  public function Matrix(xx:Number, xy:Number, yx:Number, yy:Number, dx:Number, dy:Number) {
    super();
  }

  /**
   * Get x-to-x component of the matrix. Avoid using it for performance
   * consideration.
   * Use →<code>getXX()</code> instead.
   * @see #getXX()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#property-a Original Ext JS documentation of 'a'
   */
  public native function get a():Number;

  /**
   * @private
   */
  public native function set a(value:Number):void;

  /**
   * Postpend a matrix onto the current.
   * <p><b>Note:</b> The given transform will come before the current one.</p>
   * @param xx Coefficient from x to x.
   * @param xy Coefficient from x to y.
   * @param yx Coefficient from y to x.
   * @param yy Coefficient from y to y.
   * @param dx Offset of x.
   * @param dy Offset of y.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-append Original Ext JS documentation of 'append'
   */
  public native function append(xx:Number, xy:Number, yx:Number, yy:Number, dx:Number, dy:Number):Matrix;

  /**
   * Postpend a matrix onto the current.
   * <p><b>Note:</b> The given transform will come before the current one.</p>
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-appendMatrix Original Ext JS documentation of 'appendMatrix'
   */
  public native function appendMatrix(matrix:Matrix):Matrix;

  /**
   * Clone this matrix.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-clone Original Ext JS documentation of 'clone'
   */
  public native function clone():Matrix;

  [Deprecated(message="This method is deprecated.\nDetermines if this matrix has the same values as another matrix.\n", since="6.0.1")]
  /**
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-equals Original Ext JS documentation of 'equals'
   */
  public native function equals(matrix:Matrix):Boolean;

  /**
   * Horizontally flip the matrix
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-flipX Original Ext JS documentation of 'flipX'
   */
  public native function flipX():Matrix;

  /**
   * Vertically flip the matrix
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-flipY Original Ext JS documentation of 'flipY'
   */
  public native function flipY():Matrix;

  /**
   * Get offset x component of the matrix.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-getDX Original Ext JS documentation of 'getDX'
   */
  public native function getDX():Number;

  /**
   * Get offset y component of the matrix.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-getDY Original Ext JS documentation of 'getDY'
   */
  public native function getDY():Number;

  /**
   * Get the x scale of the matrix.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-getScaleX Original Ext JS documentation of 'getScaleX'
   */
  public native function getScaleX():Number;

  /**
   * Get the y scale of the matrix.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-getScaleY Original Ext JS documentation of 'getScaleY'
   */
  public native function getScaleY():Number;

  /**
   * Get x-to-x component of the matrix
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-getXX Original Ext JS documentation of 'getXX'
   */
  public native function getXX():Number;

  /**
   * Get x-to-y component of the matrix.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-getXY Original Ext JS documentation of 'getXY'
   */
  public native function getXY():Number;

  /**
   * Get y-to-x component of the matrix.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-getYX Original Ext JS documentation of 'getYX'
   */
  public native function getYX():Number;

  /**
   * Get y-to-y component of the matrix.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-getYY Original Ext JS documentation of 'getYY'
   */
  public native function getYY():Number;

  /**
   * Return a new matrix represents the opposite transformation of the current one.
   * @param target A target matrix. If present, it will receive
   * the result of inversion to avoid creating a new object.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-inverse Original Ext JS documentation of 'inverse'
   */
  public native function inverse(target:Matrix = null):Matrix;

  /**
   * Determines if this matrix has the same values as another matrix.
   * @param matrix A maxtrix or array of its elements.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-isEqual Original Ext JS documentation of 'isEqual'
   */
  public native function isEqual(matrix:Matrix):Boolean;

  /**
   * Determines whether this matrix is an identity matrix (no transform).
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-isIdentity Original Ext JS documentation of 'isIdentity'
   */
  public native function isIdentity():Boolean;

  /**
   * Performs matrix multiplication. This matrix is post-multiplied by another matrix.
   * <p><b>Note:</b> The given transform will come before the current one.</p>
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-multiply Original Ext JS documentation of 'multiply'
   */
  public native function multiply(matrix:Matrix):Matrix;

  /**
   * Prepend a matrix onto the current.
   * <p><b>Note:</b> The given transform will come after the current one.</p>
   * @param xx Coefficient from x to x.
   * @param xy Coefficient from x to y.
   * @param yx Coefficient from y to x.
   * @param yy Coefficient from y to y.
   * @param dx Offset of x.
   * @param dy Offset of y.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-prepend Original Ext JS documentation of 'prepend'
   */
  public native function prepend(xx:Number, xy:Number, yx:Number, yy:Number, dx:Number, dy:Number):Matrix;

  /**
   * Prepend a matrix onto the current.
   * <p><b>Note:</b> The given transform will come after the current one.</p>
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-prependMatrix Original Ext JS documentation of 'prependMatrix'
   */
  public native function prependMatrix(matrix:Matrix):Matrix;

  /**
   * Reset the matrix to identical.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-reset Original Ext JS documentation of 'reset'
   */
  public native function reset():Matrix;

  /**
   * Rotate the matrix.
   * @param angle Radians to rotate
   * @param rcx Center of rotation.
   * @param rcy Center of rotation.
   * @param prepend If <code>true</code>, this will transformation be prepended to the matrix.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-rotate Original Ext JS documentation of 'rotate'
   */
  public native function rotate(angle:Number, rcx:*, rcy:*, prepend:Boolean = false):Matrix;

  /**
   * Rotate the matrix by the angle of a vector.
   * @param x
   * @param y
   * @param prepend If <code>true</code>, this will transformation be prepended to the matrix.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-rotateFromVector Original Ext JS documentation of 'rotateFromVector'
   */
  public native function rotateFromVector(x:Number, y:Number, prepend:Boolean = false):Matrix;

  /**
   * Scale the matrix.
   * @param sx
   * @param sy
   * @param scx
   * @param scy
   * @param prepend If <code>true</code>, this will transformation be prepended to the matrix.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-scale Original Ext JS documentation of 'scale'
   */
  public native function scale(sx:Number, sy:Number, scx:Number, scy:Number, prepend:Boolean = false):Matrix;

  /**
   * Set the elements of a Matrix
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-set Original Ext JS documentation of 'set'
   */
  public native function set(xx:Number, xy:Number, yx:Number, yy:Number, dx:Number, dy:Number):Matrix;

  /**
   * Shear the matrix along the x-axis.
   * @param factor The horizontal shear factor.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-shearX Original Ext JS documentation of 'shearX'
   */
  public native function shearX(factor:*):Matrix;

  /**
   * Shear the matrix along the y-axis.
   * @param factor The vertical shear factor.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-shearY Original Ext JS documentation of 'shearY'
   */
  public native function shearY(factor:*):Matrix;

  /**
   * Skew the matrix
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-skewX Original Ext JS documentation of 'skewX'
   */
  public native function skewX(angle:Number):Matrix;

  /**
   * Skew the matrix
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-skewY Original Ext JS documentation of 'skewY'
   */
  public native function skewY(angle:Number):Matrix;

  /**
   * Splits this transformation matrix into Scale, Rotate, Translate components,
   * assuming it was produced by applying transformations in that order.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-split Original Ext JS documentation of 'split'
   */
  public native function split():Object;

  /**
   * Create an array of elements by horizontal order (xx,yx,dx,yx,yy,dy).
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-toArray Original Ext JS documentation of 'toArray'
   */
  public native function toArray():Array;

  /**
   * Apply the matrix to a drawing context.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-toContext Original Ext JS documentation of 'toContext'
   */
  public native function toContext(ctx:Object):Matrix;

  /**
   * Get an array of elements.
   * The numbers are rounded to keep only 4 decimals.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-toString Original Ext JS documentation of 'toString'
   */
  public native function toString():Array;

  /**
   * Return a string that can be used as transform attribute in SVG.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-toSvg Original Ext JS documentation of 'toSvg'
   */
  public native function toSvg():String;

  /**
   * Create an array of elements by vertical order (xx,xy,yx,yy,dx,dy).
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-toVerticalArray Original Ext JS documentation of 'toVerticalArray'
   */
  public native function toVerticalArray():*;

  /**
   * @param bbox Given as <code>{x: Number, y: Number, width: Number, height: Number}</code>.
   * @param radius
   * @param target Optional target object to recieve the result.
   * Recommended to use it for better gc.
   * @return Object with x, y, width and height.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-transformBBox Original Ext JS documentation of 'transformBBox'
   */
  public native function transformBBox(bbox:Object, radius:Number = NaN, target:Object = null):Object;

  /**
   * Transform a list for points.
   * <p><b>Note:</b> will change the original list but not points inside it.</p>
   * @return list
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-transformList Original Ext JS documentation of 'transformList'
   */
  public native function transformList(list:Array):Array;

  /**
   * Transform a point to a new array.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-transformPoint Original Ext JS documentation of 'transformPoint'
   */
  public native function transformPoint(point:Array):Array;

  /**
   * Translate the matrix.
   * @param x
   * @param y
   * @param prepend If <code>true</code>, this will transformation be prepended to the matrix.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-translate Original Ext JS documentation of 'translate'
   */
  public native function translate(x:Number, y:Number, prepend:Boolean = false):Matrix;

  /**
   * Transform point returning the x component of the result.
   * @return x component of the result.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-x Original Ext JS documentation of 'x'
   */
  public native function x(x:Number, y:Number):Number;

  /**
   * Transform point returning the y component of the result.
   * @return y component of the result.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.Matrix.html#method-y Original Ext JS documentation of 'y'
   */
  public native function y(x:Number, y:Number):Number;
}
}