package ext.draw.sprite {
import ext.draw.Path;

[Native("Ext.draw.sprite.Path", require)]
/**
 * A sprite that represents a path.
 * <p><b>Example:</b></p>
 * <pre>
 * Ext.create({
 *    xtype: 'draw',
 *    renderTo: document.body,
 *    width: 600,
 *    height: 400,
 *    sprites: [{
 *        type: 'path',
 *        path: 'M20,30 c0,-50 75,50 75,0 c0,-50 -75,50 -75,0',
 *        fillStyle: '#1F6D91'
 *    }]
 * });
 * </pre>
 * <p><i>Drawing with SVG Paths</i></p>
 * <p>You may use special SVG Path syntax to "describe" the drawing path.
 * Here are the SVG path commands:</p>
 * <ul>
 * <li>M = moveto</li>
 * <li>L = lineto</li>
 * <li>H = horizontal lineto</li>
 * <li>V = vertical lineto</li>
 * <li>C = curveto</li>
 * <li>S = smooth curveto</li>
 * <li>Q = quadratic Bézier curve</li>
 * <li>T = smooth quadratic Bézier curveto</li>
 * <li>A = elliptical Arc</li>
 * <li>Z = closepath</li>
 * </ul>
 * <p><b>Note:</b> Capital letters indicate that the item should be absolutely positioned.
 * Use lower case letters for relative positioning.</p>
 * <p><b>From override Ext.draw.overrides.hittest.sprite.Path:</b></p>
 * Adds hit testing methods to the Ext.draw.sprite.Path sprite.
 * Included by the Ext.draw.PathUtil.
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Path.html Original Ext JS documentation of 'Ext.draw.sprite.Path'
 */
public class PathSprite extends Sprite {
  /**
   * @param config @inheritDoc
   */
  public function PathSprite(config:PathSprite = null) {
    super(config);
  }

  [ExtConfig]
  /**
   * The SVG based path string used by the sprite.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Path.html#cfg-path Original Ext JS documentation of 'path'
   */
  public native function get path():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set path(value:String):void;

  /**
   * Returns all points where this sprite intersects the given sprite.
   * The given sprite must be an instance of the →<code>ext.draw.sprite.PathSprite</code> class
   * or its subclass.
   * @see ext.draw.sprite.PathSprite
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Path.html#method-getIntersections Original Ext JS documentation of 'getIntersections'
   */
  public native function getIntersections(path:*):Array;

  /**
   * Performs a hit test on all sprites in the surface, returning the first matching one.
   * @param point A two-item array containing x and y coordinates of the point.
   * @param options Hit testing options.
   * @return A hit result object that contains more information about what
   * exactly was hit or null if nothing was hit.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Path.html#method-hitTest Original Ext JS documentation of 'hitTest'
   */
  override public native function hitTest(point:Array, options:Object):Object;

  /**
   * Tests whether the given point is inside the path.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Path.html#method-isPointInPath Original Ext JS documentation of 'isPointInPath'
   */
  public native function isPointInPath(x:*, y:*):Boolean;

  /**
   * Tests whether the given point is on the path.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Path.html#method-isPointOnPath Original Ext JS documentation of 'isPointOnPath'
   */
  public native function isPointOnPath(x:*, y:*):Boolean;

  /**
   * Update the path.
   * @param path An empty path to draw on using path API.
   * @param attr The attribute object. Note: DO NOT use the <code>sprite.attr</code> instead of this
   * if you want to work with instancing.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Path.html#method-updatePath Original Ext JS documentation of 'updatePath'
   */
  public native function updatePath(path:Path, attr:Object):void;
}
}