package ext.draw.sprite {
import ext.draw.Surface;
import ext.draw.modifier.AnimationModifier;
import ext.mixin.Observable;

[ExtConfig]
[Native("Ext.draw.sprite.Sprite", require)]
/**
 * A sprite is a basic primitive from the charts package which represents a graphical
 * object that can be drawn. Sprites are used extensively in the charts package to
 * create the visual elements of each chart. You can also create a desired image by
 * adding one or more sprites to a <i>draw container</i> (→<code>ext.draw.DrawContainer</code>).
 * <p>The Sprite class itself is an abstract class and is not meant to be used directly.
 * There are many different kinds of sprites available in the charts package that extend
 * Ext.draw.sprite.Sprite. Each sprite type has various attributes that define how that
 * sprite should look. For example, this is a <i>rect</i> (→<code>ext.draw.sprite.RectSprite</code>) sprite:</p>
 * <p><b>Example:</b></p>
 * <pre>
 * Ext.create({
 *     xtype: 'draw',
 *     renderTo: document.body,
 *     width: 400,
 *     height: 400,
 *     sprites: [{
 *         type: 'rect',
 *         x: 50,
 *         y: 50,
 *         width: 100,
 *         height: 100,
 *         fillStyle: '#1F6D91'
 *     }]
 * });
 * </pre>
 * <p>By default, sprites are added to the default 'main' <i>surface</i> (→<code>ext.draw.Surface</code>)
 * of the draw container. However, sprites may also be configured with a reference to a
 * specific Ext.draw.Surface when set in the draw container's
 * →<code>ext.draw.DrawContainer.sprites</code> config. Specifying a surface
 * other than 'main' will create a surface by that name if it does not already exist.</p>
 * <p><b>Example:</b></p>
 * <pre>
 * Ext.create({
 *     xtype: 'draw',
 *     renderTo: document.body,
 *     width: 400,
 *     height: 400,
 *     sprites: [{
 *         type: 'rect',
 *         surface: 'anim',  // a surface with id "anim" will be created automatically
 *         x: 50,
 *         y: 50,
 *         width: 100,
 *         height: 100,
 *         fillStyle: '#1F6D91'
 *     }]
 * });
 * </pre>
 * <p>The ability to have multiple surfaces is useful for performance (and battery life)
 * reasons. Because changes to sprite attributes cause the whole surface (and all
 * sprites in it) to re-render, it makes sense to group sprites by surface, so changes
 * to one group of sprites will only trigger the surface they are in to re-render.</p>
 * <p>You can add a sprite to an existing drawing by adding the sprite to a draw surface.</p>
 * <p><b>Example:</b></p>
 * <pre>
 * var drawCt = Ext.create({
 *     xtype: 'draw',
 *     renderTo: document.body,
 *     width: 400,
 *     height: 400
 * });
 *
 * // If the surface name is not specified then 'main' will be used
 * var surface = drawCt.getSurface();
 *
 * surface.add({
 *     type: 'rect',
 *     x: 50,
 *     y: 50,
 *     width: 100,
 *     height: 100,
 *     fillStyle: '#1F6D91'
 * });
 *
 * surface.renderFrame();
 * </pre>
 * <p><b>Note:</b> Changes to the sprites on a surface will be not be reflected in the DOM
 * until you call the surface's →<code>ext.draw.Surface.renderFrame()</code>
 * method. This must be done after adding, removing, or modifying sprites in order to
 * see the changes on-screen.</p>
 * <p>For information on configuring a sprite with an initial transformation see
 * →<code>scaling</code>, →<code>rotation</code>, and →<code>translation</code>.</p>
 * <p>For information on applying a transformation to an existing sprite see the
 * Ext.draw.Matrix class.</p>
 * @see ext.draw.DrawContainer
 * @see ext.draw.sprite.RectSprite
 * @see ext.draw.Surface
 * @see ext.draw.DrawContainer#sprites
 * @see ext.draw.Surface#renderFrame()
 * @see #scaling
 * @see #rotation
 * @see #translation
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html Original Ext JS documentation of 'Ext.draw.sprite.Sprite'
 */
public class Sprite extends Observable {
  /**
   * Debug rendering options...
   * <p>debug: {
   * bbox: true, // renders the bounding box of the sprite
   * xray: true // renders control points of the path (for Ext.draw.sprite.Path and descendants only)
   * }</p>
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#static-property-debug Original Ext JS documentation of 'debug'
   */
  public static native function get debug():*;

  /**
   * @private
   */
  public static native function set debug(value:*):void;

  /**
   * @inheritDoc
   */
  public function Sprite(config:Sprite = null) {
    super(config);
  }

  [ExtConfig]
  [Bindable]
  /**
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-animation Original Ext JS documentation of 'animation'
   * @see #getAnimation()
   * @see #setAnimation()
   */
  public native function get animation():AnimationModifier;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set animation(value:AnimationModifier):void;

  /**
   * The visual attributes of the sprite, e.g. strokeStyle, fillStyle, lineWidth...
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#property-attr Original Ext JS documentation of 'attr'
   */
  public native function get attr():Object;

  /**
   * @private
   */
  public native function set attr(value:Object):void;

  [ExtConfig]
  /**
   * The opacity of the fill. Limited from 0 to 1.
   * @default 1
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-fillOpacity Original Ext JS documentation of 'fillOpacity'
   */
  public native function get fillOpacity():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set fillOpacity(value:Number):void;

  [ExtConfig]
  /**
   * The color of the shape (a CSS color value).
   * @default "none"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-fillStyle Original Ext JS documentation of 'fillStyle'
   */
  public native function get fillStyle():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set fillStyle(value:String):void;

  [ExtConfig]
  /**
   * The opacity of the sprite. Limited from 0 to 1.
   * @default 1
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-globalAlpha Original Ext JS documentation of 'globalAlpha'
   */
  public native function get globalAlpha():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set globalAlpha(value:Number):void;

  [ExtConfig]
  /**
   * Indicates how source images are drawn onto a destination image.
   * globalCompositeOperation attribute is not supported by the SVG and VML
   * (excanvas) engines.
   * @default source-over
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-globalCompositeOperation Original Ext JS documentation of 'globalCompositeOperation'
   */
  public native function get globalCompositeOperation():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set globalCompositeOperation(value:String):void;

  [ExtConfig]
  /**
   * Determines whether or not the sprite is hidden.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-hidden Original Ext JS documentation of 'hidden'
   */
  public native function get hidden():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set hidden(value:Boolean):void;

  [ExtConfig]
  /**
   * The style of the line caps.
   * @default "butt"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-lineCap Original Ext JS documentation of 'lineCap'
   */
  public native function get lineCap():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set lineCap(value:String):void;

  [ExtConfig]
  /**
   * An even number of non-negative numbers specifying a dash/space sequence.
   * Note that while this is supported in IE8 (VML engine), the behavior is
   * different from Canvas and SVG. Please refer to this document for details:
   * http://msdn.microsoft.com/en-us/library/bb264085(v=vs.85).aspx
   * Although IE9 and IE10 have Canvas support, the 'lineDash'
   * attribute is not supported in those browsers.
   * @default []
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-lineDash Original Ext JS documentation of 'lineDash'
   */
  public native function get lineDash():Array;

  [ExtConfig]
  /**
   * @private
   */
  public native function set lineDash(value:Array):void;

  [ExtConfig]
  /**
   * A number specifying how far into the line dash sequence drawing commences.
   * @default 0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-lineDashOffset Original Ext JS documentation of 'lineDashOffset'
   */
  public native function get lineDashOffset():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set lineDashOffset(value:Number):void;

  [ExtConfig]
  /**
   * The style of the line join.
   * @default "miter"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-lineJoin Original Ext JS documentation of 'lineJoin'
   */
  public native function get lineJoin():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set lineJoin(value:String):void;

  [ExtConfig]
  /**
   * The width of the line stroke.
   * @default 1
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-lineWidth Original Ext JS documentation of 'lineWidth'
   */
  public native function get lineWidth():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set lineWidth(value:Number):void;

  [ExtConfig]
  /**
   * Sets the distance between the inner corner and the outer corner
   * where two lines meet.
   * @default 10
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-miterLimit Original Ext JS documentation of 'miterLimit'
   */
  public native function get miterLimit():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set miterLimit(value:Number):void;

  [ExtConfig]
  /**
   * Applies an initial angle of rotation to the sprite. May be a number
   * specifying the rotation in degrees. Or may be a config object using
   * the below config options.
   * <p><b>Note:</b> Rotation config options will be overridden by values set on
   * the →<code>rotationRads</code>, →<code>rotationCenterX</code>, and
   * →<code>rotationCenterY</code> configs.</p>
   * <pre>
   * Ext.create({
   *     xtype: 'draw',
   *     renderTo: Ext.getBody(),
   *     width: 600,
   *     height: 400,
   *     sprites: [{
   *         type: 'rect',
   *         x: 50,
   *         y: 50,
   *         width: 100,
   *         height: 100,
   *         fillStyle: '#1F6D91',
   *         //rotation: 45
   *         rotation: {
   *             degrees: 45,
   *             //rads: Math.PI / 4,
   *             //centerX: 50,
   *             //centerY: 50
   *         }
   *     }]
   * });
   * </pre>
   * <p><b>Note:</b> Transform configs are <i>always</i> performed in the following
   * order:</p>
   * <ol>
   * <li>Scaling</li>
   * <li>Rotation</li>
   * <li>Translation</li>
   * </ol>
   * <ul>
   * <li><code>rads:Number</code> (optional) —
   * The angle in radians to rotate the sprite
   * </li>
   * <li><code>degrees:Number</code> (optional) —
   * The angle in degrees to rotate the sprite (is ignored if rads or
   * →<code>rotationRads</code> is set
   * </li>
   * <li><code>centerX:Number</code> (optional) —
   * The central coordinate of the sprite's rotation on the x-axis.
   * Unless explicitly set, will default to the calculated center of the
   * sprite along the x-axis.
   * </li>
   * <li><code>centerY:Number</code> (optional) —
   * The central coordinate of the sprite's rotation on the y-axis.
   * Unless explicitly set, will default to the calculated center of the
   * sprite along the y-axis.
   * </li>
   * </ul>
   * @see #rotationRads
   * @see #rotationCenterX
   * @see #rotationCenterY
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-rotation Original Ext JS documentation of 'rotation'
   */
  public native function get rotation():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set rotation(value:*):void;

  [ExtConfig]
  /**
   * The central coordinate of the sprite's scale operation on the x-axis.
   * Unless explicitly set, will default to the calculated center of the
   * sprite along the x-axis.
   * <p><b>Note:</b> Transform configs are <i>always</i> performed in the following
   * order:</p>
   * <ol>
   * <li>Scaling</li>
   * <li>Rotation</li>
   * <li>Translation</li>
   * </ol>
   * <p>See also: →<code>rotation</code>, →<code>rotationRads</code>, and
   * →<code>rotationCenterY</code></p>
   * @default null
   * @see #rotation
   * @see #rotationRads
   * @see #rotationCenterY
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-rotationCenterX Original Ext JS documentation of 'rotationCenterX'
   */
  public native function get rotationCenterX():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set rotationCenterX(value:Number):void;

  [ExtConfig]
  /**
   * The central coordinate of the sprite's rotate operation on the y-axis.
   * Unless explicitly set, will default to the calculated center of the
   * sprite along the y-axis.
   * <p><b>Note:</b> Transform configs are <i>always</i> performed in the following
   * order:</p>
   * <ol>
   * <li>Scaling</li>
   * <li>Rotation</li>
   * <li>Translation</li>
   * </ol>
   * <p>See also: →<code>rotation</code>, →<code>rotationRads</code>, and
   * →<code>rotationCenterX</code></p>
   * @default null
   * @see #rotation
   * @see #rotationRads
   * @see #rotationCenterX
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-rotationCenterY Original Ext JS documentation of 'rotationCenterY'
   */
  public native function get rotationCenterY():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set rotationCenterY(value:Number):void;

  [ExtConfig]
  /**
   * The angle of rotation of the sprite in radians.
   * <p><b>Note:</b> Transform configs are <i>always</i> performed in the following
   * order:</p>
   * <ol>
   * <li>Scaling</li>
   * <li>Rotation</li>
   * <li>Translation</li>
   * </ol>
   * <p>See also: →<code>rotation</code>, →<code>rotationCenterX</code>, and
   * →<code>rotationCenterY</code></p>
   * @default 0
   * @see #rotation
   * @see #rotationCenterX
   * @see #rotationCenterY
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-rotationRads Original Ext JS documentation of 'rotationRads'
   */
  public native function get rotationRads():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set rotationRads(value:Number):void;

  [ExtConfig]
  /**
   * Applies initial scaling to the sprite. May be a number specifying
   * the amount to scale both the x and y-axis. The number value
   * represents a percentage by which to scale the sprite. <b>1</b> is equal
   * to 100%, <b>2</b> would be 200%, etc. Or may be a config object using
   * the below config options.
   * <p><b>Note:</b> Scaling config options will be overridden by values set on
   * the →<code>scalingX</code>, →<code>scalingY</code>, →<code>scalingCenterX</code>,
   * and →<code>scalingCenterY</code> configs.</p>
   * <pre>
   * Ext.create({
   *     xtype: 'draw',
   *     renderTo: Ext.getBody(),
   *     width: 600,
   *     height: 400,
   *     sprites: [{
   *         type: 'rect',
   *         x: 50,
   *         y: 50,
   *         width: 100,
   *         height: 100,
   *         fillStyle: '#1F6D91',
   *         //scaling: 2,
   *         scaling: {
   *             x: 2,
   *             y: 2
   *             //centerX: 100,
   *             //centerY: 100
   *         }
   *     }]
   * });
   * </pre>
   * <p><b>Note:</b> Transform configs are <i>always</i> performed in the following
   * order:</p>
   * <ol>
   * <li>Scaling</li>
   * <li>Rotation</li>
   * <li>Translation</li>
   * </ol>
   * <ul>
   * <li><code>x:Number</code> (optional) —
   * The amount by which to scale the sprite along the x-axis. The number
   * value represents a percentage by which to scale the sprite. <b>1</b> is
   * equal to 100%, <b>2</b> would be 200%, etc.
   * </li>
   * <li><code>y:Number</code> (optional) —
   * The amount by which to scale the sprite along the y-axis. The number
   * value represents a percentage by which to scale the sprite. <b>1</b> is
   * equal to 100%, <b>2</b> would be 200%, etc.
   * </li>
   * <li><code>centerX</code> (optional) —
   * The central coordinate of the sprite's scaling on the x-axis. Unless
   * explicitly set, will default to the calculated center of the sprite
   * along the x-axis.
   * </li>
   * <li><code>centerY:Number</code> (optional) —
   * The central coordinate of the sprite's scaling on the y-axis. Unless
   * explicitly set, will default to the calculated center of the sprite
   * along the y-axis.
   * </li>
   * </ul>
   * @see #scalingX
   * @see #scalingY
   * @see #scalingCenterX
   * @see #scalingCenterY
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-scaling Original Ext JS documentation of 'scaling'
   */
  public native function get scaling():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set scaling(value:*):void;

  [ExtConfig]
  /**
   * The central coordinate of the sprite's scale operation on the x-axis.
   * <p><b>Note:</b> Transform configs are <i>always</i> performed in the following
   * order:</p>
   * <ol>
   * <li>Scaling</li>
   * <li>Rotation</li>
   * <li>Translation</li>
   * </ol>
   * <p>See also: →<code>scaling</code>, →<code>scalingX</code>,
   * →<code>scalingY</code>, and →<code>scalingCenterY</code></p>
   * @default null
   * @see #scaling
   * @see #scalingX
   * @see #scalingY
   * @see #scalingCenterY
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-scalingCenterX Original Ext JS documentation of 'scalingCenterX'
   */
  public native function get scalingCenterX():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set scalingCenterX(value:Number):void;

  [ExtConfig]
  /**
   * The central coordinate of the sprite's scale operation on the y-axis.
   * <p><b>Note:</b> Transform configs are <i>always</i> performed in the following
   * order:</p>
   * <ol>
   * <li>Scaling</li>
   * <li>Rotation</li>
   * <li>Translation</li>
   * </ol>
   * <p>See also: →<code>scaling</code>, →<code>scalingX</code>,
   * →<code>scalingY</code>, and →<code>scalingCenterX</code></p>
   * @default null
   * @see #scaling
   * @see #scalingX
   * @see #scalingY
   * @see #scalingCenterX
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-scalingCenterY Original Ext JS documentation of 'scalingCenterY'
   */
  public native function get scalingCenterY():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set scalingCenterY(value:Number):void;

  [ExtConfig]
  /**
   * The scaling of the sprite on the x-axis.
   * The number value represents a percentage by which to scale the
   * sprite. <b>1</b> is equal to 100%, <b>2</b> would be 200%, etc.
   * <p><b>Note:</b> Transform configs are <i>always</i> performed in the following
   * order:</p>
   * <ol>
   * <li>Scaling</li>
   * <li>Rotation</li>
   * <li>Translation</li>
   * </ol>
   * <p>See also: →<code>scaling</code>, →<code>scalingY</code>,
   * →<code>scalingCenterX</code>, and →<code>scalingCenterY</code></p>
   * @default 1
   * @see #scaling
   * @see #scalingY
   * @see #scalingCenterX
   * @see #scalingCenterY
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-scalingX Original Ext JS documentation of 'scalingX'
   */
  public native function get scalingX():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set scalingX(value:Number):void;

  [ExtConfig]
  /**
   * The scaling of the sprite on the y-axis.
   * The number value represents a percentage by which to scale the
   * sprite. <b>1</b> is equal to 100%, <b>2</b> would be 200%, etc.
   * <p><b>Note:</b> Transform configs are <i>always</i> performed in the following
   * order:</p>
   * <ol>
   * <li>Scaling</li>
   * <li>Rotation</li>
   * <li>Translation</li>
   * </ol>
   * <p>See also: →<code>scaling</code>, →<code>scalingX</code>,
   * →<code>scalingCenterX</code>, and →<code>scalingCenterY</code></p>
   * @default 1
   * @see #scaling
   * @see #scalingX
   * @see #scalingCenterX
   * @see #scalingCenterY
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-scalingY Original Ext JS documentation of 'scalingY'
   */
  public native function get scalingY():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set scalingY(value:Number):void;

  [ExtConfig]
  /**
   * The amount blur used on the shadow.
   * @default 0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-shadowBlur Original Ext JS documentation of 'shadowBlur'
   */
  public native function get shadowBlur():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set shadowBlur(value:Number):void;

  [ExtConfig]
  /**
   * The color of the shadow (a CSS color value).
   * @default "none"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-shadowColor Original Ext JS documentation of 'shadowColor'
   */
  public native function get shadowColor():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set shadowColor(value:String):void;

  [ExtConfig]
  /**
   * The offset of the sprite's shadow on the x-axis.
   * @default 0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-shadowOffsetX Original Ext JS documentation of 'shadowOffsetX'
   */
  public native function get shadowOffsetX():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set shadowOffsetX(value:Number):void;

  [ExtConfig]
  /**
   * The offset of the sprite's shadow on the y-axis.
   * @default 0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-shadowOffsetY Original Ext JS documentation of 'shadowOffsetY'
   */
  public native function get shadowOffsetY():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set shadowOffsetY(value:Number):void;

  [ExtConfig]
  /**
   * The opacity of the stroke. Limited from 0 to 1.
   * @default 1
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-strokeOpacity Original Ext JS documentation of 'strokeOpacity'
   */
  public native function get strokeOpacity():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set strokeOpacity(value:Number):void;

  [ExtConfig]
  /**
   * The color of the stroke (a CSS color value).
   * @default "none"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-strokeStyle Original Ext JS documentation of 'strokeStyle'
   */
  public native function get strokeStyle():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set strokeStyle(value:String):void;

  [ExtConfig]
  /**
   * Determines whether the fill and stroke are affected by sprite transformations.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-transformFillStroke Original Ext JS documentation of 'transformFillStroke'
   */
  public native function get transformFillStroke():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set transformFillStroke(value:Boolean):void;

  [ExtConfig]
  /**
   * Applies an initial translation, adjustment in x/y positioning, to the
   * sprite.
   * <p><b>Note:</b> Translation config options will be overridden by values set
   * on the →<code>translationX</code> and →<code>translationY</code> configs.</p>
   * <pre>
   * Ext.create({
   *     xtype: 'draw',
   *     renderTo: Ext.getBody(),
   *     width: 600,
   *     height: 400,
   *         sprites: [{
   *         type: 'rect',
   *         x: 50,
   *         y: 50,
   *         width: 100,
   *         height: 100,
   *         fillStyle: '#1F6D91',
   *         translation: {
   *             x: 50,
   *             y: 50
   *         }
   *     }]
   * });
   * </pre>
   * <p><b>Note:</b> Transform configs are <i>always</i> performed in the following
   * order:</p>
   * <ol>
   * <li>Scaling</li>
   * <li>Rotation</li>
   * <li>Translation</li>
   * </ol>
   * <ul>
   * <li><code>x:Number</code> (optional) —
   * The amount to translate the sprite along the x-axis.
   * </li>
   * <li><code>y:Number</code> (optional) —
   * The amount to translate the sprite along the y-axis.
   * </li>
   * </ul>
   * @see #translationX
   * @see #translationY
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-translation Original Ext JS documentation of 'translation'
   */
  public native function get translation():Object;

  [ExtConfig]
  /**
   * @private
   */
  public native function set translation(value:Object):void;

  [ExtConfig]
  /**
   * The translation, position offset, of the sprite on the x-axis.
   * <p><b>Note:</b> Transform configs are <i>always</i> performed in the following
   * order:</p>
   * <ol>
   * <li>Scaling</li>
   * <li>Rotation</li>
   * <li>Translation</li>
   * </ol>
   * <p>See also: →<code>translation</code> and →<code>translationY</code></p>
   * @default 0
   * @see #translation
   * @see #translationY
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-translationX Original Ext JS documentation of 'translationX'
   */
  public native function get translationX():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set translationX(value:Number):void;

  [ExtConfig]
  /**
   * The translation, position offset, of the sprite on the y-axis.
   * <p><b>Note:</b> Transform configs are <i>always</i> performed in the following
   * order:</p>
   * <ol>
   * <li>Scaling</li>
   * <li>Rotation</li>
   * <li>Translation</li>
   * </ol>
   * <p>See also: →<code>translation</code> and →<code>translationX</code></p>
   * @default 0
   * @see #translation
   * @see #translationX
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-translationY Original Ext JS documentation of 'translationY'
   */
  public native function get translationY():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set translationY(value:Number):void;

  [ExtConfig]
  /**
   * The stacking order of the sprite.
   * @default 0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#cfg-zIndex Original Ext JS documentation of 'zIndex'
   */
  public native function get zIndex():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set zIndex(value:Number):void;

  /**
   * Removes the sprite and clears all listeners.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-destroy Original Ext JS documentation of 'destroy'
   */
  override public native function destroy(...params):void;

  /**
   * Returns the value of <code>animation</code>.
   * @see #animation
   */
  public native function getAnimation():AnimationModifier;

  /**
   * Returns the bounding box for the given Sprite as calculated with the Canvas engine.
   * @param isWithoutTransform Whether to calculate the bounding box
   * with the current transforms or not.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-getBBox Original Ext JS documentation of 'getBBox'
   */
  public native function getBBox(isWithoutTransform:Boolean = false):void;

  /**
   * Subclass can rewrite this function to gain better performance.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-getBBoxCenter Original Ext JS documentation of 'getBBoxCenter'
   */
  public native function getBBoxCenter(isWithoutTransform:Boolean):Array;

  /**
   * Hide the sprite.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-hide Original Ext JS documentation of 'hide'
   */
  public native function hide():Sprite;

  /**
   * Performs a hit test on the sprite.
   * @param point A two-item array containing x and y coordinates of the point.
   * @param options Hit testing options.
   * @return A hit result object that contains more information about what
   * exactly was hit or null if nothing was hit.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-hitTest Original Ext JS documentation of 'hitTest'
   */
  public native function hitTest(point:Array, options:Object):Object;

  /**
   * Called before rendering.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-preRender Original Ext JS documentation of 'preRender'
   */
  public native function preRender():void;

  /**
   * Removes this sprite from its surface.
   * The sprite itself is not destroyed.
   * @return Returns the removed sprite or <code>null</code> otherwise.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-remove Original Ext JS documentation of 'remove'
   */
  public native function remove():Sprite;

  /**
   * This is where the actual sprite rendering happens by calling <code>ctx</code> methods.
   * @param surface A draw container surface.
   * @param ctx A context object that is API compatible with the native
   * <a href="https://developer.mozilla.org/en/docs/Web/API/CanvasRenderingContext2D">CanvasRenderingContext2D</a>.
   * @param surfaceClipRect The clip rect: [left, top, width, height].
   * Not to be confused with the <code>surface.getRect()</code>, which represents the location
   * and size of the surface in a draw container, in draw container coordinates.
   * The clip rect on the other hand represents the portion of the surface that is being
   * rendered, in surface coordinates.
   * @return returns <code>false</code> to stop rendering in this frame.
   * All the sprites that haven't been rendered will have their dirty flag untouched.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-render Original Ext JS documentation of 'render'
   */
  public native function render(surface:Surface, ctx:Object, surfaceClipRect:Array):*;

  /**
   * Resets current transformation matrix of a sprite to the identify matrix.
   * @param isSplit If 'true', transformation attributes are updated.
   * @default false
   * @return This sprite.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-resetTransform Original Ext JS documentation of 'resetTransform'
   */
  public native function resetTransform(isSplit:Boolean = false):Sprite;

  /**
   * Sets the value of <code>animation</code>.
   * @param animation The new value.
   * @see #animation
   */
  public native function setAnimation(animation:AnimationModifier):void;

  /**
   * Set attributes of the sprite.
   * By default only the attributes that have processors will be set
   * and all other attributes will be filtered out as a result of the
   * normalization process.
   * The normalization process can be skipped. In that case all the given
   * attributes will be set unprocessed. This will result in better
   * performance, but might also pollute the sprite's attributes with
   * unwanted attributes or attributes with invalid values, if one is not
   * careful. See also →<code>setAttributesBypassingNormalization()</code>.
   * If normalization is skipped, one may also chose to avoid copying
   * the given object. This may result in even better performance, but
   * only in cases where most of the attributes have values that are
   * different from the old values, because copying additionally checks
   * if the value has changed.
   * @param changes The content of the change.
   * @param bypassNormalization <code>true</code> to avoid normalization of the given changes.
   * @param avoidCopy <code>true</code> to avoid copying the <code>changes</code> object.
   * <code>bypassNormalization</code> should also be <code>true</code>. The content of object may be destroyed.
   * @see #setAttributesBypassingNormalization()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-setAttributes Original Ext JS documentation of 'setAttributes'
   */
  public native function setAttributes(changes:Object, bypassNormalization:Boolean = false, avoidCopy:Boolean = false):void;

  [Deprecated(message="Use setAttributes directly with bypassNormalization argument being `true`.\n", since="6.5.0")]
  /**
   * Set attributes of the sprite, assuming the names and values have already been
   * normalized.
   * @param changes The content of the change.
   * @param avoidCopy <code>true</code> to avoid copying the <code>changes</code> object.
   * The content of object may be destroyed.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-setAttributesBypassingNormalization Original Ext JS documentation of 'setAttributesBypassingNormalization'
   */
  public native function setAttributesBypassingNormalization(changes:Object, avoidCopy:Boolean = false):void;

  /**
   * Resets current transformation matrix of a sprite to the identify matrix
   * and pre-multiplies it with the given matrix.
   * This is effectively the same as calling →<code>resetTransform()</code>,
   * followed by →<code>transform()</code> with the same arguments.
   * <p>See also: →<code>transform()</code></p>
   * <pre>
   * var drawContainer = new Ext.draw.Container({
   *     renderTo: Ext.getBody(),
   *     width: 380,
   *     height: 380,
   *     sprites: [{
   *         type: 'rect',
   *         width: 100,
   *         height: 100,
   *         fillStyle: 'red'
   *     }]
   * });
   *
   * var main = drawContainer.getSurface();
   * var rect = main.getItems()[0];
   *
   * var m = new Ext.draw.Matrix().rotate(Math.PI, 100, 100);
   *
   * rect.setTransform(m);
   * main.renderFrame();
   * </pre>
   * <p>There may be times where the transformation you need to apply cannot easily be
   * accomplished using the sprite’s convenience transform methods. Or, you may want
   * to pass a matrix directly to the sprite in order to set a transformation. The
   * <code>→setTransform()</code> method allows for this sort of advanced usage as well. The
   * following tables show each transformation matrix used when applying
   * transformations to a sprite.</p>
   * <p><i>Translate</i></p>
   * <table style="text-align: center;">
   *     <tr>
   *         <td style="font-weight: normal;">1</td>
   *         <td style="font-weight: normal;">0</td>
   *         <td style="font-weight: normal;">tx</td>
   *     </tr>
   *     <tr>
   *         <td>0</td>
   *         <td>1</td>
   *         <td>ty</td>
   *     </tr>
   *     <tr>
   *         <td>0</td>
   *         <td>0</td>
   *         <td>1</td>
   *     </tr>
   * </table>
   * <p><i>Rotate (θ is the angle of rotation)</i></p>
   * <table style="text-align: center;">
   *     <tr>
   *         <td style="font-weight: normal;">cos(θ)</td>
   *         <td style="font-weight: normal;">-sin(θ)</td>
   *         <td style="font-weight: normal;">0</td>
   *     </tr>
   *     <tr>
   *         <td>0</td>
   *         <td>cos(θ)</td>
   *         <td>0</td>
   *     </tr>
   *     <tr>
   *         <td>0</td>
   *         <td>0</td>
   *         <td>1</td>
   *     </tr>
   * </table>
   * <p><i>Scale</i></p>
   * <table style="text-align: center;">
   *     <tr>
   *         <td style="font-weight: normal;">sx</td>
   *         <td style="font-weight: normal;">0</td>
   *         <td style="font-weight: normal;">0</td>
   *     </tr>
   *     <tr>
   *         <td>0</td>
   *         <td>cos(θ)</td>
   *         <td>0</td>
   *     </tr>
   *     <tr>
   *         <td>0</td>
   *         <td>0</td>
   *         <td>1</td>
   *     </tr>
   * </table>
   * <p><i>Shear X <i>(λ is the distance on the x axis to shear by)</i></i></p>
   * <table style="text-align: center;">
   *     <tr>
   *         <td style="font-weight: normal;">1</td>
   *         <td style="font-weight: normal;">λx</td>
   *         <td style="font-weight: normal;">0</td>
   *     </tr>
   *     <tr>
   *         <td>0</td>
   *         <td>1</td>
   *         <td>0</td>
   *     </tr>
   *     <tr>
   *         <td>0</td>
   *         <td>0</td>
   *         <td>1</td>
   *     </tr>
   * </table>
   * <p><i>Shear Y (λ is the distance on the y axis to shear by)</i></p>
   * <table style="text-align: center;">
   *     <tr>
   *         <td style="font-weight: normal;">1</td>
   *         <td style="font-weight: normal;">0</td>
   *         <td style="font-weight: normal;">0</td>
   *     </tr>
   *     <tr>
   *         <td>λy</td>
   *         <td>1</td>
   *         <td>0</td>
   *     </tr>
   *     <tr>
   *         <td>0</td>
   *         <td>0</td>
   *         <td>1</td>
   *     </tr>
   * </table>
   * <p><i>Skew X (θ is the angle to skew by)</i></p>
   * <table style="text-align: center;">
   *     <tr>
   *         <td style="font-weight: normal;">1</td>
   *         <td style="font-weight: normal;">tan(θ)</td>
   *         <td style="font-weight: normal;">0</td>
   *     </tr>
   *     <tr>
   *         <td>0</td>
   *         <td>1</td>
   *         <td>0</td>
   *     </tr>
   *     <tr>
   *         <td>0</td>
   *         <td>0</td>
   *         <td>1</td>
   *     </tr>
   * </table>
   * <p><i>Skew Y (θ is the angle to skew by)</i></p>
   * <table style="text-align: center;">
   *     <tr>
   *         <td style="font-weight: normal;">1</td>
   *         <td style="font-weight: normal;">0</td>
   *         <td style="font-weight: normal;">0</td>
   *     </tr>
   *     <tr>
   *         <td>tan(θ)</td>
   *         <td>1</td>
   *         <td>0</td>
   *     </tr>
   *     <tr>
   *         <td>0</td>
   *         <td>0</td>
   *         <td>1</td>
   *     </tr>
   * </table>
   * <p>Multiplying matrices for translation, rotation, scaling, and shearing / skewing
   * any number of times in the desired order produces a single matrix for a composite
   * transformation. You can use the product as a value for the <code>→setTransform()</code>method
   * of a sprite:</p>
   * <pre>
   * mySprite.setTransform([a, b, c, d, e, f]);
   * </pre>
   * <p>Where <code>a</code>, <code>b</code>, <code>c</code>, <code>d</code>, <code>e</code>, <code>f</code> are numeric values that correspond to the
   * following transformation matrix components:</p>
   * <table style="text-align: center;">
   *     <tr>
   *         <td style="font-weight: normal;">a</td>
   *         <td style="font-weight: normal;">c</td>
   *         <td style="font-weight: normal;">e</td>
   *     </tr>
   *     <tr>
   *         <td>b</td>
   *         <td>d</td>
   *         <td>f</td>
   *     </tr>
   *     <tr>
   *         <td>0</td>
   *         <td>0</td>
   *         <td>1</td>
   *     </tr>
   * </table>
   * @param matrix The transformation matrix to apply or its
   * raw elements as an array.
   * @param isSplit If <code>true</code>, transformation attributes are updated.
   * @default false
   * @return This sprite.
   * @see #resetTransform()
   * @see #transform()
   * @see #setTransform()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-setTransform Original Ext JS documentation of 'setTransform'
   */
  public native function setTransform(matrix:*, isSplit:Boolean = false):Sprite;

  /**
   * Show the sprite.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-show Original Ext JS documentation of 'show'
   */
  public native function show():Sprite;

  /**
   * Pre-multiplies the current transformation matrix of a sprite with the given matrix.
   * If <code>isSplit</code> parameter is <code>true</code>, the resulting matrix is also split into
   * individual components (scaling, rotation, translation) and corresponding sprite
   * attributes are updated. The shearing component is not extracted.
   * Note, that transformation attributes work as if transformations are applied to the
   * local coordinate system of a sprite, while matrix transformations transform
   * the global coordinate space or the surface grid.
   * Since the <code>→transform()</code> method returns the sprite itself, calls to the method
   * can be chained. And if updating sprite transformation attributes is desired,
   * it can be achieved by setting the <code>isSplit</code> parameter of the last call to <code>true</code>.
   * For example:
   * <pre>
   * sprite.transform(matrixA).transform(matrixB).transform(matrixC, true);
   * </pre>
   * <p>See also: →<code>setTransform()</code></p>
   * @param matrix A transformation matrix or array of its elements.
   * @param isSplit If 'true', transformation attributes are updated.
   * @default false
   * @return This sprite.
   * @see #transform()
   * @see #setTransform()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-transform Original Ext JS documentation of 'transform'
   */
  public native function transform(matrix:*, isSplit:Boolean = false):Sprite;

  /**
   * Subclass will fill the plain object with <code>x</code>, <code>y</code>, <code>width</code>, <code>height</code> information
   * of the plain bounding box of this sprite.
   * @param plain Target object.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-updatePlainBBox Original Ext JS documentation of 'updatePlainBBox'
   */
  protected native function updatePlainBBox(plain:Object):void;

  /**
   * Subclass will fill the plain object with <code>x</code>, <code>y</code>, <code>width</code>, <code>height</code> information
   * of the transformed bounding box of this sprite.
   * @param transform Target object (transformed bounding box) to populate.
   * @param plain Untransformed bounding box.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-updateTransformedBBox Original Ext JS documentation of 'updateTransformedBBox'
   */
  protected native function updateTransformedBBox(transform:Object, plain:Object):void;

  /**
   * Applies sprite's attributes to the given context.
   * @param ctx Context to apply sprite's attributes to.
   * @param rect The rect of the context to be affected by gradients.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.draw.sprite.Sprite.html#method-useAttributes Original Ext JS documentation of 'useAttributes'
   */
  public native function useAttributes(ctx:Object, rect:Array):void;
}
}