package ext.form.field {

[Native("Ext.form.field.Number", require)]
/**
 * A numeric text field that provides automatic keystroke filtering to disallow non-numeric
 * characters, and numeric validation to limit the value to a range of valid numbers. The range
 * of acceptable number values can be controlled by setting the →<code>minValue</code> and
 * →<code>maxValue</code> configs, and fractional decimals can be disallowed by setting
 * →<code>allowDecimals</code> to <code>false</code>.
 * <p>By default, the number field is also rendered with a set of up/down spinner buttons and has
 * up/down arrow key and mouse wheel event listeners attached for incrementing/decrementing
 * the value by the →<code>step</code> value. To hide the spinner buttons set
 * <code>→hideTrigger:true</code>; to disable the arrow key and mouse wheel handlers set
 * <code>→keyNavEnabled:false</code> and
 * <code>→mouseWheelEnabled:false</code>. See the example below.</p>
 * <p><b>Example usage</b></p>
 * <p><b>Example:</b></p>
 * <pre>
 * Ext.create('Ext.form.Panel', {
 *     title: 'On The Wall',
 *     width: 300,
 *     bodyPadding: 10,
 *     renderTo: Ext.getBody(),
 *     items: [{
 *         xtype: 'numberfield',
 *         anchor: '100%',
 *         name: 'bottles',
 *         fieldLabel: 'Bottles of Beer',
 *         value: 99,
 *         maxValue: 99,
 *         minValue: 0
 *     }],
 *     buttons: [{
 *         text: 'Take one down, pass it around',
 *         handler: function() {
 *             this.up('form').down('[name=bottles]').spinDown();
 *         }
 *     }]
 * });
 * </pre>
 * <p><b>Removing UI Enhancements</b></p>
 * <p><b>Example:</b></p>
 * <pre>
 * Ext.create('Ext.form.Panel', {
 *     title: 'Personal Info',
 *     width: 300,
 *     bodyPadding: 10,
 *     renderTo: Ext.getBody(),
 *     items: [{
 *         xtype: 'numberfield',
 *         anchor: '100%',
 *         name: 'age',
 *         fieldLabel: 'Age',
 *         minValue: 0, //prevents negative numbers
 *
 *         // Remove spinner buttons, and arrow key and mouse wheel listeners
 *         hideTrigger: true,
 *         keyNavEnabled: false,
 *         mouseWheelEnabled: false
 *     }]
 * });
 * </pre>
 * <p><b>Using Step</b></p>
 * <p><b>Example:</b></p>
 * <pre>
 * Ext.create('Ext.form.Panel', {
 *     renderTo: Ext.getBody(),
 *     title: 'Step',
 *     width: 300,
 *     bodyPadding: 10,
 *     items: [{
 *         xtype: 'numberfield',
 *         anchor: '100%',
 *         name: 'evens',
 *         fieldLabel: 'Even Numbers',
 *
 *         // Set step so it skips every other number
 *         step: 2,
 *         value: 0,
 *
 *         // Add change handler to force user-entered numbers to evens
 *         listeners: {
 *             change: function(field, value) {
 *                 value = parseInt(value, 10);
 *                 field.setValue(value + value % 2);
 *             }
 *         }
 *     }]
 * });
 * </pre>
 * @see #minValue
 * @see #maxValue
 * @see #allowDecimals
 * @see #step
 * @see #hideTrigger
 * @see #keyNavEnabled
 * @see #mouseWheelEnabled
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html Original Ext JS documentation of 'Ext.form.field.Number'
 */
public class NumberField extends SpinnerField {
  /**
   * @param config @inheritDoc
   */
  public function NumberField(config:NumberField = null) {
    super();
  }

  [ExtConfig]
  /**
   * False to disallow decimal values
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-allowDecimals Original Ext JS documentation of 'allowDecimals'
   */
  public native function get allowDecimals():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set allowDecimals(value:Boolean):void;

  [ExtConfig]
  /**
   * Set to <code>false</code> to disallow Exponential number notation
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-allowExponential Original Ext JS documentation of 'allowExponential'
   */
  public native function get allowExponential():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set allowExponential(value:Boolean):void;

  [ExtConfig]
  /**
   * True to automatically strip not allowed characters from the field.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-autoStripChars Original Ext JS documentation of 'autoStripChars'
   */
  public native function get autoStripChars():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set autoStripChars(value:Boolean):void;

  [ExtConfig]
  /**
   * The base set of characters to evaluate as valid numbers.
   * @default '0123456789'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-baseChars Original Ext JS documentation of 'baseChars'
   */
  public native function get baseChars():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set baseChars(value:String):void;

  [ExtConfig]
  /**
   * The maximum precision to display after the decimal separator
   * @default 2
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-decimalPrecision Original Ext JS documentation of 'decimalPrecision'
   */
  public native function get decimalPrecision():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set decimalPrecision(value:Number):void;

  [ExtConfig]
  /**
   * Character(s) to allow as the decimal separator.
   * Defaults to →<code>ext.util.Format.decimalSeparator</code>.
   * @see ext.util.SFormat#decimalSeparator ext.util.Format.decimalSeparator
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-decimalSeparator Original Ext JS documentation of 'decimalSeparator'
   */
  public native function get decimalSeparator():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set decimalSeparator(value:String):void;

  [ExtConfig]
  /**
   * Error text to display if the maximum value validation fails.
   * @default 'The maximum value for this field is {0}'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-maxText Original Ext JS documentation of 'maxText'
   */
  public native function get maxText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set maxText(value:String):void;

  [ExtConfig]
  /**
   * The maximum allowed value. Will be used by the field's validation logic, and for
   * <i>enabling/disabling the up spinner button</i> (→<code>ext.form.field.SpinnerField.setSpinUpEnabled()</code>).
   * <p>Defaults to Number.MAX_VALUE.</p>
   * @default Number.MAX_VALUE
   * @see ext.form.field.SpinnerField#setSpinUpEnabled()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-maxValue Original Ext JS documentation of 'maxValue'
   * @see #setMaxValue()
   */
  public native function get maxValue():Number;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * @private
   */
  public native function set maxValue(value:Number):void;

  [ExtConfig]
  /**
   * Error text to display if the minimum value validation fails.
   * @default 'The minimum value for this field is {0}'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-minText Original Ext JS documentation of 'minText'
   */
  public native function get minText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set minText(value:String):void;

  [ExtConfig]
  /**
   * The minimum allowed value. Will be used by the field's validation logic,
   * and for <i>enabling/disabling
   * the down spinner button</i> (→<code>ext.form.field.SpinnerField.setSpinUpEnabled()</code>).
   * <p>Defaults to Number.NEGATIVE_INFINITY.</p>
   * @default Number.NEGATIVE_INFINITY
   * @see ext.form.field.SpinnerField#setSpinUpEnabled()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-minValue Original Ext JS documentation of 'minValue'
   * @see #setMinValue()
   */
  public native function get minValue():Number;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * @private
   */
  public native function set minValue(value:Number):void;

  [ExtConfig]
  /**
   * Error text to display if the value is not a valid number. For example, this can happen
   * if a valid character like '.' or '-' is left in the field with no number.
   * @default '{0} is not a valid number'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-nanText Original Ext JS documentation of 'nanText'
   */
  public native function get nanText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set nanText(value:String):void;

  [ExtConfig]
  /**
   * Error text to display if the value is negative and →<code>minValue</code> is set to 0.
   * This is used instead of the →<code>minText</code> in that circumstance only.
   * @default 'The value cannot be negative'
   * @see #minValue
   * @see #minText
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-negativeText Original Ext JS documentation of 'negativeText'
   */
  public native function get negativeText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set negativeText(value:String):void;

  [ExtConfig]
  /**
   * Specifies a numeric interval by which the field's value will be incremented or decremented
   * when the user invokes the spinner.
   * @default 1
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-step Original Ext JS documentation of 'step'
   */
  public native function get step():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set step(value:Number):void;

  [ExtConfig]
  /**
   * False to ensure that the →<code>getSubmitValue()</code> method strips
   * always uses <code>.</code> as the separator, regardless of the →<code>decimalSeparator</code>
   * configuration.
   * @default true
   * @see #getSubmitValue()
   * @see #decimalSeparator
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#cfg-submitLocaleSeparator Original Ext JS documentation of 'submitLocaleSeparator'
   */
  public native function get submitLocaleSeparator():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set submitLocaleSeparator(value:Boolean):void;

  [ArrayElementType("String")]
  /**
   * Runs all of Number's validations and returns an array of any errors. Note that this first
   * runs Text's validations, so the returned array is an amalgamation of all field errors.
   * The additional validations run test that the value is a number, and that it is within
   * the configured min and max values.
   * @param value The value to get errors for (defaults to the current field value)
   * @return All validation errors for this field
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#method-getErrors Original Ext JS documentation of 'getErrors'
   */
  override public native function getErrors(value:*):Array;

  /**
   * Replaces any existing →<code>maxValue</code> with the new value.
   * @param value The maximum value
   * @see #maxValue
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#method-setMaxValue Original Ext JS documentation of 'setMaxValue'
   */
  public native function setMaxValue(value:Number):void;

  /**
   * Replaces any existing →<code>minValue</code> with the new value.
   * @param value The minimum value
   * @see #minValue
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Number.html#method-setMinValue Original Ext JS documentation of 'setMinValue'
   */
  public native function setMinValue(value:Number):void;
}
}