package ext.form.field {
import ext.Base;

[Native]
/**
 * This is a singleton object which contains a set of commonly used field validation functions
 * and provides a mechanism for creating reusable custom field validations.
 * The following field validation functions are provided out of the box:
 * <ul>
 * <li>→<code>alpha()</code></li>
 * <li>→<code>alphanum()</code></li>
 * <li>→<code>email()</code></li>
 * <li>→<code>url()</code></li>
 * </ul>
 * <p>VTypes can be applied to a <i>Text Field</i> (→<code>ext.form.field.TextField</code>) using the
 * <code>→ext.form.field.TextField.vtype</code> configuration:</p>
 * <pre>
 * Ext.create('Ext.form.field.Text', {
 *     fieldLabel: 'Email Address',
 *     name: 'email',
 *     vtype: 'email' // applies email validation rules to this field
 * });
 * </pre>
 * <p>To create custom VTypes:</p>
 * <pre>
 * // custom Vtype for vtype:'time'
 * Ext.define('Override.form.field.VTypes', {
 *     override: 'Ext.form.field.VTypes',
 *
 *     // vtype validation function
 *     time: function(value) {
 *         return this.timeRe.test(value);
 *     },
 *     // RegExp for the value to be tested against within the validation function
 *     timeRe: /^([1-9]|1[0-9]):([0-5][0-9])(\s[a|p]m)$/i,
 *     // vtype Text property: The error text to display when the validation function
 *     // returns false
 *     timeText: 'Not a valid time.  Must be in the format "12:34 PM".',
 *     // vtype Mask property: The keystroke filter mask
 *     timeMask: /[\d\s:amp]/i
 * });
 * </pre>
 * <p>In the above example the <code>time</code> function is the validator that will run when field validation
 * occurs, <code>timeText</code> is the error message, and <code>timeMask</code> limits what characters can be typed
 * into the field. Note that the <code>Text</code> and <code>Mask</code> functions must begin with the same name
 * as the validator function.</p>
 * <p>Using a custom validator is the same as using one of the build-in validators - just use the name
 * of the validator function as the <code>→ext.form.field.TextField.vtype</code> configuration on a
 * <i>Text Field</i> (→<code>ext.form.field.TextField</code>):</p>
 * <pre>
 * Ext.create('Ext.form.field.Text', {
 *     fieldLabel: 'Departure Time',
 *     name: 'departureTime',
 *     vtype: 'time' // applies custom time validation rules to this field
 * });
 * </pre>
 * <p>Another example of a custom validator:</p>
 * <pre>
 * // custom Vtype for vtype:'IPAddress'
 * Ext.define('Override.form.field.VTypes', {
 *     override: 'Ext.form.field.VTypes',
 *
 *     IPAddress:  function(value) {
 *         return this.IPAddressRe.test(value);
 *     },
 *     IPAddressRe: /^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/,
 *     IPAddressText: 'Must be a numeric IP address',
 *     IPAddressMask: /[\d\.]/i
 * });
 * </pre>
 * <p>It's important to note that using →<code>ext.Ext.define()</code> with the
 * →<code>ext.ExtClass.override_</code> property means that the custom validator function as well as
 * <code>Text</code> and <code>Mask</code> fields are added as properties of the <code>→ext.form.field.VTypes</code> singleton.</p>
 * <p>Type of singleton VTypes.</p>
 * @see SVTypes#alpha()
 * @see SVTypes#alphanum()
 * @see SVTypes#email()
 * @see SVTypes#url()
 * @see ext.form.field.TextField
 * @see ext.form.field.TextField#vtype
 * @see ext.SExt#define() ext.Ext.define()
 * @see ext.ExtClass#override_
 * @see ext.form.field.#VTypes ext.form.field.VTypes
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.VTypes.html Original Ext JS documentation of 'Ext.form.field.VTypes'
 */
public class SVTypes extends Base {
  /**
   * The keystroke filter mask to be applied on alpha input. Defaults to: /[a-z_]/i
   * @default /[a-z_]/i
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.VTypes.html#property-alphaMask Original Ext JS documentation of 'alphaMask'
   */
  public native function get alphaMask():RegExp;

  /**
   * @private
   */
  public native function set alphaMask(value:RegExp):void;

  /**
   * The error text to display when the alpha validation function returns false.
   * Defaults to: 'This field should only contain letters and _'
   * @default 'This field should only contain letters and _'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.VTypes.html#property-alphaText Original Ext JS documentation of 'alphaText'
   */
  public native function get alphaText():String;

  /**
   * @private
   */
  public native function set alphaText(value:String):void;

  /**
   * The keystroke filter mask to be applied on alphanumeric input. Defaults to: /[a-z0-9_]/i
   * @default /[a-z0-9_]/i
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.VTypes.html#property-alphanumMask Original Ext JS documentation of 'alphanumMask'
   */
  public native function get alphanumMask():RegExp;

  /**
   * @private
   */
  public native function set alphanumMask(value:RegExp):void;

  /**
   * The error text to display when the alphanumeric validation function returns false.
   * Defaults to: 'This field should only contain letters, numbers and _'
   * @default 'This field should only contain letters, numbers and _'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.VTypes.html#property-alphanumText Original Ext JS documentation of 'alphanumText'
   */
  public native function get alphanumText():String;

  /**
   * @private
   */
  public native function set alphanumText(value:String):void;

  /**
   * The keystroke filter mask to be applied on email input. See the →<code>email()</code> method
   * for information about more complex email validation. Defaults to: /[a-z0-9_.-&#64;]/i
   * @default /[\w.\-&#64;'"!#$%&'&#42;+/=?^_`{|}~]/i
   * @see SVTypes#email()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.VTypes.html#property-emailMask Original Ext JS documentation of 'emailMask'
   */
  public native function get emailMask():RegExp;

  /**
   * @private
   */
  public native function set emailMask(value:RegExp):void;

  /**
   * The error text to display when the email validation function returns false.
   * Defaults to: 'This field should be an e-mail address in the format "user&#64;example.com"'
   * @default 'This field should be an e-mail address in the format "user&#64;example.com"'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.VTypes.html#property-emailText Original Ext JS documentation of 'emailText'
   */
  public native function get emailText():String;

  /**
   * @private
   */
  public native function set emailText(value:String):void;

  /**
   * The error text to display when the url validation function returns false.
   * Defaults to: 'This field should be a URL in the format "http:/'+'/www.example.com"'
   * @default 'This field should be a URL in the format "http:/' + '/www.example.com"'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.VTypes.html#property-urlText Original Ext JS documentation of 'urlText'
   */
  public native function get urlText():String;

  /**
   * @private
   */
  public native function set urlText(value:String):void;

  /**
   * The function used to validate alpha values
   * @param value The value
   * @return true if the RegExp test passed, and false if not.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.VTypes.html#method-alpha Original Ext JS documentation of 'alpha'
   */
  public native function alpha(value:String):Boolean;

  /**
   * The function used to validate alphanumeric values
   * @param value The value
   * @return true if the RegExp test passed, and false if not.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.VTypes.html#method-alphanum Original Ext JS documentation of 'alphanum'
   */
  public native function alphanum(value:String):Boolean;

  /**
   * The function used to validate email addresses. Note that complete validation
   * per the email RFC specifications is very complex and beyond the scope of this class,
   * although this function can be overridden if a more comprehensive validation scheme
   * is desired. See the validation section of the <a href="http://en.wikipedia.org/wiki/E-mail_address">Wikipedia article on email addresses</a>
   * for additional information. This implementation is intended to validate the following
   * types of emails:
   * <ul>
   * <li><code>barney&#64;example.de</code></li>
   * <li><code>barney.rubble&#64;example.com</code></li>
   * <li><code>barney-rubble&#64;example.coop</code></li>
   * <li><code>barney+rubble&#64;example.com</code></li>
   * <li><code>barney'rubble&#64;example.com</code></li>
   * <li><code>b.arne.y_r.ubbl.e&#64;example.com</code></li>
   * <li><code>barney4rubble&#64;example.com</code></li>
   * <li><code>barney4rubble!&#64;example.com</code></li>
   * <li><code>_barney+rubble&#64;example.com</code></li>
   * <li><code>"barney+rubble"&#64;example.com</code></li>
   * </ul>
   * @param value The email address
   * @return true if the RegExp test passed, and false if not.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.VTypes.html#method-email Original Ext JS documentation of 'email'
   */
  public native function email(value:String):Boolean;

  /**
   * The function used to validate URLs
   * @param value The URL
   * @return true if the RegExp test passed, and false if not.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.VTypes.html#method-url Original Ext JS documentation of 'url'
   */
  public native function url(value:String):Boolean;
}
}