package ext.form.field {
import ext.dom.CompositeElement;
import ext.dom.Element;
import ext.form.trigger.Trigger;

[Native("Ext.form.field.Text", require)]
/**
 * @see ext.form.field.events.TextField_widthEvent.AUTO_SIZE
 * @eventType ext.form.field.events.TextField_widthEvent.AUTO_SIZE
 */
[Event(name='onAutoSize', type='ext.form.field.events.TextField_widthEvent')]
/**
 * @see ext.form.field.events.TextField_eEvent.KEY_DOWN
 * @eventType ext.form.field.events.TextField_eEvent.KEY_DOWN
 */
[Event(name='onKeyDown', type='ext.form.field.events.TextField_eEvent')]
/**
 * @see ext.form.field.events.TextField_eEvent.KEYPRESS
 * @eventType ext.form.field.events.TextField_eEvent.KEYPRESS
 */
[Event(name='onKeypress', type='ext.form.field.events.TextField_eEvent')]
/**
 * @see ext.form.field.events.TextField_eEvent.KEY_UP
 * @eventType ext.form.field.events.TextField_eEvent.KEY_UP
 */
[Event(name='onKeyUp', type='ext.form.field.events.TextField_eEvent')]
/**
 * @see ext.form.field.events.TextField_eEvent.PASTE
 * @eventType ext.form.field.events.TextField_eEvent.PASTE
 */
[Event(name='onPaste', type='ext.form.field.events.TextField_eEvent')]
/**
 * A basic text field. Can be used as a direct replacement for traditional text inputs,
 * or as the base class for more sophisticated input controls (like →<code>ext.form.field.TextArea</code>
 * and →<code>ext.form.field.ComboBox</code>). Has support for empty-field placeholder values
 * (see →<code>emptyText</code>).
 * <p><b>Validation</b></p>
 * <p>The Text field has a useful set of validations built in:</p>
 * <ul>
 * <li>→<code>allowBlank</code> for making the field required</li>
 * <li>→<code>minLength</code> for requiring a minimum value length</li>
 * <li>→<code>maxLength</code> for setting a maximum value length (with →<code>enforceMaxLength</code> to add it
 * as the <code>maxlength</code> attribute on the input element)</li>
 * <li>→<code>regex</code> to specify a custom regular expression for validation</li>
 * </ul>
 * <p>In addition, custom validations may be added:</p>
 * <ul>
 * <li>→<code>vtype</code> specifies a virtual type implementation from →<code>ext.form.field.VTypes</code>
 * which can contain custom validation logic</li>
 * <li>→<code>validator</code> allows a custom arbitrary function to be called during validation</li>
 * </ul>
 * <p>The details around how and when each of these validation options get used are described in the
 * documentation for →<code>getErrors()</code>.</p>
 * <p>By default, the field value is checked for validity immediately while the user is typing in the
 * field. This can be controlled with the →<code>validateOnChange</code>, →<code>checkChangeEvents</code>, and
 * →<code>checkChangeBuffer</code> configurations. Also see the details on Form Validation in the
 * →<code>ext.form.FormPanel</code> class documentation.</p>
 * <p><b>Masking and Character Stripping</b></p>
 * <p>Text fields can be configured with custom regular expressions to be applied to entered values
 * before validation: see →<code>maskRe</code> and →<code>stripCharsRe</code> for details.</p>
 * <p><b>Example usage</b></p>
 * <p><b>Example:</b></p>
 * <pre>
 * Ext.create('Ext.form.Panel', {
 *     title: 'Contact Info',
 *     width: 300,
 *     bodyPadding: 10,
 *     renderTo: Ext.getBody(),
 *     items: [{
 *         xtype: 'textfield',
 *         name: 'name',
 *         fieldLabel: 'Name',
 *         allowBlank: false  // requires a non-empty value
 *     }, {
 *         xtype: 'textfield',
 *         name: 'email',
 *         fieldLabel: 'Email Address',
 *         vtype: 'email'  // requires value to be a valid email address format
 *     }]
 * });
 * </pre>
 * <p><b>Custom Subclasses</b></p>
 * <p>This class can be extended to provide additional functionality. The example below demonstrates
 * creating a custom search field that uses the HTML5 search input type.</p>
 * <p><b>Example:</b></p>
 * <pre>
 * // A simple subclass of Base that creates a HTML5 search field. Redirects to the
 * // searchUrl when the Enter key is pressed.
 * Ext.define('Ext.form.SearchField', {
 *     extend: 'Ext.form.field.Text',
 *     alias: 'widget.searchfield',
 *
 *     inputType: 'search',
 *
 *     // Config defining the search URL
 *     searchUrl: 'http://www.google.com/search?q={0}',
 *
 *     // Add specialkey listener
 *     initComponent: function() {
 *         this.callParent();
 *         this.on('specialkey', this.checkEnterKey, this);
 *     },
 *
 *     // Handle enter key presses, execute the search if the field has a value
 *     checkEnterKey: function(field, e) {
 *         var value = this.getValue();
 *         if (e.getKey() === e.ENTER &amp;&amp; !Ext.isEmpty(value)) {
 *             location.href = Ext.String.format(this.searchUrl, value);
 *         }
 *     }
 * });
 *
 * Ext.create('Ext.form.Panel', {
 *     title: 'Base Example',
 *     bodyPadding: 5,
 *     width: 250,
 *
 *     // Fields will be arranged vertically, stretched to full width
 *     layout: 'anchor',
 *     defaults: {
 *         anchor: '100%'
 *     },
 *     items: [{
 *         xtype: 'searchfield',
 *         fieldLabel: 'Search',
 *         name: 'query'
 *     }],
 *     renderTo: Ext.getBody()
 * });
 * </pre>
 * @see ext.form.field.TextArea
 * @see ext.form.field.ComboBox
 * @see #emptyText
 * @see #allowBlank
 * @see #minLength
 * @see #maxLength
 * @see #enforceMaxLength
 * @see #regex
 * @see #vtype
 * @see ext.form.field.#VTypes ext.form.field.VTypes
 * @see #validator
 * @see #getErrors()
 * @see #validateOnChange
 * @see #checkChangeEvents
 * @see #checkChangeBuffer
 * @see ext.form.FormPanel
 * @see #maskRe
 * @see #stripCharsRe
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html Original Ext JS documentation of 'Ext.form.field.Text'
 */
public class TextField extends BaseField {
  /**
   * @param config @inheritDoc
   */
  public function TextField(config:TextField = null) {
    super();
  }

  [ExtConfig]
  /**
   * Specify false to validate that the value's length must be &gt; 0. If <code>true</code>, then a blank value
   * is <b>always</b> taken to be valid regardless of any →<code>vtype</code> validation that may be
   * applied.
   * <p>If →<code>vtype</code> validation must still be applied to blank values, configure
   * →<code>validateBlank</code> as <code>true</code>;</p>
   * @default true
   * @see #vtype
   * @see #validateBlank
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-allowBlank Original Ext JS documentation of 'allowBlank'
   */
  public native function get allowBlank():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set allowBlank(value:Boolean):void;

  [ExtConfig]
  /**
   * Specify false to automatically trim the value before validating
   * the whether the value is blank. Setting this to false automatically
   * sets →<code>allowBlank</code> to false.
   * @default true
   * @see #allowBlank
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-allowOnlyWhitespace Original Ext JS documentation of 'allowOnlyWhitespace'
   */
  public native function get allowOnlyWhitespace():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set allowOnlyWhitespace(value:Boolean):void;

  /**
   * ARIA role for this Component, defaults to no role.
   * With no role, no other ARIA attributes are set.
   * @default 'textbox'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#property-ariaRole Original Ext JS documentation of 'ariaRole'
   */
  override public native function get ariaRole():String;

  /**
   * @private
   */
  public native function set ariaRole(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * Specify as <code>false</code> to always show the <code>→inputMask</code>.
   * @default true
   * @since 6.5.0
   * @see #inputMask
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-autoHideInputMask Original Ext JS documentation of 'autoHideInputMask'
   * @see #getAutoHideInputMask()
   * @see #setAutoHideInputMask()
   */
  public native function get autoHideInputMask():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set autoHideInputMask(value:Boolean):void;

  [ExtConfig]
  /**
   * The error text to display if the <b>→<code>allowBlank</code></b> validation fails.
   * @default 'This field is required'
   * @see #allowBlank
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-blankText Original Ext JS documentation of 'blankText'
   */
  public native function get blankText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set blankText(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * The canonical form of <code>→childEls</code> is an object keyed by child's property name
   * with values that are objects with the following properties.
   * <ul>
   * <li><code>→itemId</code> - The id to combine with the Component's id that is the id of the
   * child element.</li>
   * <li><code>→id</code> - The id of the child element.</li>
   * <li><code>leaf</code> - Set to <code>true</code> to ignore content when scanning for childEls. This
   * should be set on things like the generated content for an <code>→ext.view.DataView</code>.</li>
   * <li><code>select</code>: A selector that will be passed to →<code>ext.dom.Element.select()</code>.</li>
   * <li><code>selectNode</code>: A selector that will be passed to
   * →<code>ext.dom.Element.selectNode()</code>.</li>
   * </ul>
   * <p>For example:</p>
   * <pre>
   *  childEls: {
   *      button: true,
   *      buttonText: 'text',
   *      buttonImage: {
   *          itemId: 'image'
   *      }
   *  }
   * </pre>
   * <p>The above is translated into the following complete form:</p>
   * <pre>
   *  childEls: {
   *      button: {
   *          name: 'button',
   *          itemId: 'button'
   *      },
   *      buttonText: {
   *          name: 'buttonText',
   *          itemId: 'text'
   *      },
   *      buttonImage: {
   *          name: 'buttonImage',
   *          itemId: 'image'
   *      }
   *  }
   * </pre>
   * <p>The above can be provided as an array like so:</p>
   * <pre>
   *  childEls: [
   *      'button',
   *      { name: 'buttonText', itemId: 'text' },
   *      { name: 'buttonImage', itemId: 'image' }
   *  }
   * </pre>
   * <p>For example, a Component which renders a title and body text:</p>
   * <p><b>Example:</b></p>
   * <pre>
   * Ext.create('Ext.Component', {
   *     renderTo: Ext.getBody(),
   *     renderTpl: [
   *         '&lt;h1 id="{id}-title" data-ref="title"&gt;{title}&lt;/h1&gt;',
   *         '&lt;p&gt;{msg}&lt;/p&gt;',
   *     ],
   *     renderData: {
   *         title: "Error",
   *         msg: "Something went wrong"
   *     },
   *     childEls: ["title"],
   *     listeners: {
   *         afterrender: function(cmp){
   *             // After rendering the component will have a title property
   *             cmp.title.setStyle({color: "red"});
   *         }
   *     }
   * });
   * </pre>
   * <p><b>Note:</b> <code>childEl</code>s in the →<code>ext.Component.renderTpl</code>
   * must be referenced in a <b>data-ref</b> attribute. Notice in the above example
   * that the "title" <code>childEl</code> is set in the <code>→renderTpl</code> using
   * <b>data-ref="title"</b>.</p>
   * <p>When using <code>select</code>, the property will be an instance of →<code>ext.dom.CompositeElement</code>.
   * In all other cases, the property will be an →<code>ext.dom.Element</code> or <code>null</code>
   * if not found.</p>
   * <p>Care should be taken when using <code>select</code> or <code>selectNode</code> to find child elements.
   * The following issues should be considered:</p>
   * <ul>
   * <li>Performance: using selectors can be 10x slower than id lookup.</li>
   * <li>Over-selecting: selectors are applied after the DOM elements for all children
   * have been rendered, so selectors can match elements from child components
   * (including nested versions of the same component) accidentally.</li>
   * </ul>
   * <p>This above issues are most important when using <code>select</code> since it returns multiple
   * elements.</p>
   * @default [
   *     /&#42;&#42;
   *          &#42; @property {Ext.dom.Element} triggerWrap
   *          &#42; A reference to the element which encapsulates the input field and all
   *          &#42; trigger button(s). Only set after the field has been rendered.
   *          &#42;/
   *     'triggerWrap',
   *     /&#42;&#42;
   *          &#42; @property {Ext.dom.Element} inputWrap
   *          &#42; A reference to the element that wraps the input element. Only set after the
   *          &#42; field has been rendered.
   *          &#42;/
   *     'inputWrap',
   *     'placeholderLabel'
   * ]
   * @see #childEls
   * @see #itemId
   * @see #id
   * @see ext.view.DataView
   * @see ext.dom.Element#select()
   * @see ext.dom.Element#selectNode()
   * @see ext.Component#renderTpl
   * @see #renderTpl
   * @see ext.dom.CompositeElement
   * @see ext.dom.Element
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-childEls Original Ext JS documentation of 'childEls'
   * @see #getChildEls()
   * @see #setChildEls()
   */
  override public native function get childEls():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  override public native function set childEls(value:*):void;

  [ExtConfig]
  /**
   * The sizing and positioning of a Component's internal Elements is the responsibility
   * of the Component's layout manager which sizes a Component's internal structure in response
   * to the Component being sized.
   * <p>Generally, developers will not use this configuration as all provided Components which need
   * their internal elements sizing (Such as <i>input fields</i> (→<code>ext.form.field.BaseField</code>)) come with
   * their own componentLayout managers.</p>
   * <p>The <i>default layout manager</i> (→<code>ext.layout.container.AutoLayout</code>) will be used on instances of the
   * base Ext.Component class which simply sizes the Component's encapsulating element to the
   * height and width specified in the →<code>setSize()</code> method.</p>
   * @default 'textfield'
   * @see ext.form.field.BaseField
   * @see ext.layout.container.AutoLayout
   * @see #setSize()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-componentLayout Original Ext JS documentation of 'componentLayout'
   */
  override public native function get componentLayout():*;

  [ExtConfig]
  /**
   * @private
   */
  override public native function set componentLayout(value:*):void;

  [ExtConfig]
  /**
   * Specify true to disable input keystroke filtering. This will ignore the
   * maskRe field.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-disableKeyFilter Original Ext JS documentation of 'disableKeyFilter'
   */
  public native function get disableKeyFilter():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set disableKeyFilter(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * false to prevent the user from typing text directly into the field; the field can
   * only have its value set programmatically or via an action invoked by a trigger.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-editable Original Ext JS documentation of 'editable'
   * @see #getEditable()
   * @see #setEditable()
   */
  public native function get editable():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set editable(value:Boolean):void;

  [ExtConfig]
  /**
   * The CSS class to apply to an empty field to style the <b>→<code>emptyText</code></b>.
   * This class is automatically added and removed as needed depending on the current field value.
   * @default Ext.baseCSSPrefix + 'form-empty-field'
   * @see #emptyText
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-emptyCls Original Ext JS documentation of 'emptyCls'
   */
  public native function get emptyCls():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set emptyCls(value:String):void;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * The default text to place into an empty field.
   * <p>Note that normally this value will be submitted to the server if this field is enabled;
   * to prevent this you can set the
   * →<code>ext.form.action.Action.submitEmptyText</code> option of
   * →<code>ext.form.BasicForm.submit()</code> to false.</p>
   * <p>Also note that if you use →<code>inputType</code>:'file', →<code>emptyText</code>
   * is not supported and should be avoided.</p>
   * <p>Note that for browsers that support it, setting this property will use the HTML 5 placeholder
   * attribute, and for older browsers that don't support the HTML 5 placeholder attribute
   * the value will be placed directly into the input element itself as the raw value.
   * This means that older browsers will obfuscate the →<code>emptyText</code> value for
   * password input fields.</p>
   * @default ''
   * @see ext.form.action.Action#submitEmptyText
   * @see ext.form.BasicForm#submit()
   * @see #inputType
   * @see #emptyText
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-emptyText Original Ext JS documentation of 'emptyText'
   * @see #getEmptyText()
   * @see #setEmptyText()
   */
  public native function get emptyText():String;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * @private
   */
  public native function set emptyText(value:String):void;

  [ExtConfig]
  /**
   * true to enable the proxying of key events for the HTML input field
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-enableKeyEvents Original Ext JS documentation of 'enableKeyEvents'
   */
  public native function get enableKeyEvents():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set enableKeyEvents(value:Boolean):void;

  [ExtConfig]
  /**
   * True to set the maxLength property on the underlying input field. Defaults to false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-enforceMaxLength Original Ext JS documentation of 'enforceMaxLength'
   */
  public native function get enforceMaxLength():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set enforceMaxLength(value:Boolean):void;

  [ExtConfig]
  /**
   * An extra CSS class to be applied to the body content element in addition to
   * →<code>baseBodyCls</code>.
   * @default Ext.baseCSSPrefix + 'form-text-field-body'
   * @see #baseBodyCls
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-fieldBodyCls Original Ext JS documentation of 'fieldBodyCls'
   */
  override public native function get fieldBodyCls():String;

  [ExtConfig]
  /**
   * @private
   */
  override public native function set fieldBodyCls(value:String):void;

  [ExtConfig]
  /**
   * true if this field should automatically grow and shrink to its content
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-grow Original Ext JS documentation of 'grow'
   */
  public native function get grow():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set grow(value:Boolean):void;

  [ExtConfig]
  /**
   * The maximum width to allow when <code>→grow = true</code>
   * @default 800
   * @see #grow
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-growMax Original Ext JS documentation of 'growMax'
   */
  public native function get growMax():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set growMax(value:Number):void;

  [ExtConfig]
  /**
   * The minimum width to allow when <code>→grow = true</code>
   * @default 30
   * @see #grow
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-growMin Original Ext JS documentation of 'growMin'
   */
  public native function get growMin():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set growMin(value:Number):void;

  [ExtConfig]
  [Bindable]
  /**
   * <code>true</code> to hide all triggers
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-hideTrigger Original Ext JS documentation of 'hideTrigger'
   * @see #getHideTrigger()
   * @see #setHideTrigger()
   */
  public native function get hideTrigger():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set hideTrigger(value:Boolean):void;

  [Deprecated(replacement="inputWrap", since="5.0")]
  /**
   * A reference to the element that wraps the input element. Only set after the
   * field has been rendered.
   * @default me.inputWrap
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#property-inputCell Original Ext JS documentation of 'inputCell'
   */
  public native function get inputCell():Element;

  /**
   * @private
   */
  public native function set inputCell(value:Element):void;

  [ExtConfig]
  [Bindable]
  /**
   * <b>Important:</b> To use this config you must require <code>→ext.field.InputMaskField</code> or
   * use a complete framework build. The logic to implement an <code>→inputMask</code> is not
   * automatically included in a build.
   * @default null
   * @since 6.5.0
   * @see ext.field.InputMaskField
   * @see #inputMask
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-inputMask Original Ext JS documentation of 'inputMask'
   * @see #getInputMask()
   * @see #setInputMask()
   */
  public native function get inputMask():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set inputMask(value:*):void;

  /**
   * A reference to the element that wraps the input element. Only set after the
   * field has been rendered.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#property-inputWrap Original Ext JS documentation of 'inputWrap'
   */
  public native function get inputWrap():Element;

  /**
   * @private
   */
  public native function set inputWrap(value:Element):void;

  [ExtConfig]
  /**
   * The CSS class that is added to the element wrapping the input element
   * @default Ext.baseCSSPrefix + 'form-text-wrap'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-inputWrapCls Original Ext JS documentation of 'inputWrapCls'
   */
  public native function get inputWrapCls():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set inputWrapCls(value:String):void;

  [ExtConfig]
  /**
   * An input mask regular expression that will be used to filter keystrokes
   * (character being typed) that do not match.
   * Note: It does not filter characters already in the input.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-maskRe Original Ext JS documentation of 'maskRe'
   */
  public native function get maskRe():RegExp;

  [ExtConfig]
  /**
   * @private
   */
  public native function set maskRe(value:RegExp):void;

  [ExtConfig]
  /**
   * Maximum input field length allowed by validation. This behavior is intended to
   * provide instant feedback to the user by improving usability to allow pasting and editing
   * or overtyping and back tracking. To restrict the maximum number of characters that can be
   * entered into the field use the
   * <b>→<code>ext.form.field.TextField.enforceMaxLength</code></b> option.
   * <p>Defaults to Number.MAX_VALUE.</p>
   * @default Number.MAX_VALUE
   * @see ext.form.field.TextField#enforceMaxLength
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-maxLength Original Ext JS documentation of 'maxLength'
   */
  public native function get maxLength():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set maxLength(value:Number):void;

  [ExtConfig]
  /**
   * Error text to display if the <b><i>maximum length</i> (→<code>maxLength</code>)</b> validation fails.
   * @default 'The maximum length for this field is {0}'
   * @see #maxLength
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-maxLengthText Original Ext JS documentation of 'maxLengthText'
   */
  public native function get maxLengthText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set maxLengthText(value:String):void;

  [ExtConfig]
  /**
   * Minimum input field length required
   * @default 0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-minLength Original Ext JS documentation of 'minLength'
   */
  public native function get minLength():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set minLength(value:Number):void;

  [ExtConfig]
  /**
   * Error text to display if the <b><i>minimum length</i> (→<code>minLength</code>)</b> validation fails.
   * @default 'The minimum length for this field is {0}'
   * @see #minLength
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-minLengthText Original Ext JS documentation of 'minLengthText'
   */
  public native function get minLengthText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set minLengthText(value:String):void;

  [ExtConfig]
  /**
   * <code>true</code> to prevent the user from changing the field, and hide all triggers.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-readOnly Original Ext JS documentation of 'readOnly'
   * @see #setReadOnly()
   */
  override public native function get readOnly():Boolean;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * @private
   */
  override public native function set readOnly(value:Boolean):void;

  [ExtConfig]
  /**
   * A JavaScript RegExp object to be tested against the field value during validation.
   * If the test fails, the field will be marked invalid using
   * either <b>→<code>regexText</code></b> or <b>→<code>invalidText</code></b>.
   * @see #regexText
   * @see #invalidText
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-regex Original Ext JS documentation of 'regex'
   */
  public native function get regex():RegExp;

  [ExtConfig]
  /**
   * @private
   */
  public native function set regex(value:RegExp):void;

  [ExtConfig]
  /**
   * The error text to display if <b>→<code>regex</code></b> is used and the test fails during validation
   * @default ''
   * @see #regex
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-regexText Original Ext JS documentation of 'regexText'
   */
  public native function get regexText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set regexText(value:String):void;

  [ExtConfig]
  /**
   * <code>true</code> to attach a <i>click repeater</i> (→<code>ext.util.ClickRepeater</code>) to the trigger(s).
   * Click repeating behavior can also be configured on the individual <i>trigger instances</i> (→<code>triggers</code>) using the trigger's →<code>ext.form.trigger.Trigger.repeatClick</code> config.
   * @default false
   * @see ext.util.ClickRepeater
   * @see #triggers
   * @see ext.form.trigger.Trigger#repeatClick
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-repeatTriggerClick Original Ext JS documentation of 'repeatTriggerClick'
   */
  public native function get repeatTriggerClick():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set repeatTriggerClick(value:Boolean):void;

  [ExtConfig]
  /**
   * The CSS class to apply to a required field, i.e. a field where <b>→<code>allowBlank</code></b>
   * is false.
   * @default Ext.baseCSSPrefix + 'form-required-field'
   * @see #allowBlank
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-requiredCls Original Ext JS documentation of 'requiredCls'
   */
  public native function get requiredCls():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set requiredCls(value:String):void;

  [ExtConfig]
  /**
   * <code>true</code> to automatically select any existing field text when the field receives input
   * focus. Only applies when →<code>editable</code> = true
   * @default false
   * @see #editable
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-selectOnFocus Original Ext JS documentation of 'selectOnFocus'
   */
  public native function get selectOnFocus():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set selectOnFocus(value:Boolean):void;

  [Deprecated(replacement="width", since="6.5.0")]
  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * An initial value for the 'size' attribute on the text input element. This is only
   * used if the field has no configured →<code>width</code> and is not given a width by its
   * container's layout. Defaults to 20.
   * @see #width
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-size Original Ext JS documentation of 'size'
   * @see #getSize()
   */
  public native function get size():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set size(value:Number):void;

  [ArrayElementType("String")]
  [ExtConfig]
  /**
   * An array of events that, when fired, should trigger this object to
   * save its state. <code>→stateEvents</code> defaults to the <code>→stateEvents</code> associated with the
   * component you are using. Any events you statically set will be appended to that list.
   * <code>→stateEvents</code> may be any type of event supported by this object, including
   * browser or custom events (e.g., <code>['click', 'customerchange']</code>).
   * <p>See <code>→stateful</code> for an explanation of saving and
   * restoring object state.
   * By default the following stateEvents are added:</p>
   * <ul>
   * <li>→<code>event:onResize</code> - <i>(added by Ext.Component)</i></li>
   * <li>→<code>event:onChange</code></li>
   * </ul>
   * @see #stateEvents
   * @see #stateful
   * @see #event:onResize
   * @see #event:onChange
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-stateEvents Original Ext JS documentation of 'stateEvents'
   */
  override public native function get stateEvents():Array;

  [ExtConfig]
  /**
   * @private
   */
  override public native function set stateEvents(value:Array):void;

  [ExtConfig]
  /**
   * A JavaScript RegExp object used to strip unwanted content from the value
   * during input. If <code>→stripCharsRe</code> is specified,
   * every <i>character sequence</i> matching <code>→stripCharsRe</code> will be removed.
   * @see #stripCharsRe
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-stripCharsRe Original Ext JS documentation of 'stripCharsRe'
   */
  public native function get stripCharsRe():RegExp;

  [ExtConfig]
  /**
   * @private
   */
  public native function set stripCharsRe(value:RegExp):void;

  [Deprecated(message="A composite of all the trigger button elements. Only set after the field has\nbeen rendered.\n", since="5.0")]
  /**
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#property-triggerEl Original Ext JS documentation of 'triggerEl'
   */
  public native function get triggerEl():CompositeElement;

  /**
   * @private
   */
  public native function set triggerEl(value:CompositeElement):void;

  /**
   * A reference to the element which encapsulates the input field and all
   * trigger button(s). Only set after the field has been rendered.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#property-triggerWrap Original Ext JS documentation of 'triggerWrap'
   */
  public native function get triggerWrap():Element;

  /**
   * @private
   */
  public native function set triggerWrap(value:Element):void;

  [ExtConfig]
  /**
   * The CSS class that is added to the div wrapping the input element and trigger button(s).
   * @default Ext.baseCSSPrefix + 'form-trigger-wrap'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-triggerWrapCls Original Ext JS documentation of 'triggerWrapCls'
   */
  public native function get triggerWrapCls():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set triggerWrapCls(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * <i>Triggers</i> (→<code>ext.form.trigger.Trigger</code>) to use in this field. The keys in
   * this object are unique identifiers for the triggers. The values in this object
   * are →<code>ext.form.trigger.Trigger</code> configuration objects.
   * <pre>
   * Ext.create('Ext.form.field.Text', {
   *     renderTo: document.body,
   *     fieldLabel: 'My Custom Field',
   *     triggers: {
   *         foo: {
   *             cls: 'my-foo-trigger',
   *             handler: function() {
   *                 console.log('foo trigger clicked');
   *             }
   *         },
   *         bar: {
   *             cls: 'my-bar-trigger',
   *             handler: function() {
   *                 console.log('bar trigger clicked');
   *             }
   *         }
   *     }
   * });
   * </pre>
   * <p>The weight value may be a negative value in order to position custom triggers
   * ahead of default triggers like that of ComboBox.</p>
   * <pre>
   * Ext.create('Ext.form.field.ComboBox', {
   *     renderTo: Ext.getBody(),
   *     fieldLabel: 'My Custom Field',
   *     triggers: {
   *         foo: {
   *             cls: 'my-foo-trigger',
   *             weight: -2, // negative to place before default triggers
   *             handler: function() {
   *                 console.log('foo trigger clicked');
   *             }
   *         },
   *         bar: {
   *             cls: 'my-bar-trigger',
   *             weight: -1,
   *             handler: function() {
   *                 console.log('bar trigger clicked');
   *             }
   *         }
   *     }
   * });
   * </pre>
   * @default undefined
   * @see ext.form.trigger.Trigger
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-triggers Original Ext JS documentation of 'triggers'
   * @see #getTriggers()
   * @see #setTriggers()
   */
  public native function get triggers():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set triggers(value:Object):void;

  [ExtConfig]
  /**
   * Specify as <code>true</code> to modify the behaviour of →<code>allowBlank</code> so that blank values
   * are not passed as valid, but are subject to any configure →<code>vtype</code> validation.
   * @default false
   * @see #allowBlank
   * @see #vtype
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-validateBlank Original Ext JS documentation of 'validateBlank'
   */
  public native function get validateBlank():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set validateBlank(value:Boolean):void;

  [ExtConfig]
  /**
   * A custom validation function to be called during field validation (→<code>getErrors()</code>).
   * If specified, this function will be called first, allowing the developer to override
   * the default validation process.
   * <pre>
   * Ext.create('Ext.form.field.Text', {
   *     renderTo: document.body,
   *     name: 'phone',
   *     fieldLabel: 'Phone Number',
   *     validator: function(val) {
   *         // remove non-numeric characters
   *         var tn = val.replace(/[^0-9]/g,''),
   *             errMsg = "Must be a 10 digit telephone number";
   *         // if the numeric value is not 10 digits return an error message
   *         return (tn.length === 10) ? true : errMsg;
   *     }
   * });
   * </pre>
   * <ul>
   * <li><code>value:Object</code> —
   * The current field value
   * </li>
   * <li><code>null</code> —
   * response
   * <ul>
   * <li>True if the value is valid</li>
   * <li>An error message if the value is invalid</li>
   * </ul>
   * </li>
   * </ul>
   * @see #getErrors()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-validator Original Ext JS documentation of 'validator'
   */
  public native function get validator():Function;

  [ExtConfig]
  /**
   * @private
   */
  public native function set validator(value:Function):void;

  [ExtConfig]
  /**
   * A validation type name as defined in →<code>ext.form.field.VTypes</code>
   * @see ext.form.field.#VTypes ext.form.field.VTypes
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-vtype Original Ext JS documentation of 'vtype'
   */
  public native function get vtype():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set vtype(value:String):void;

  [ExtConfig]
  /**
   * A custom error message to display in place of the default message provided for the
   * <b><code>→vtype</code></b> currently set for this field.
   * <b>Note</b>: only applies if <b><code>→vtype</code></b> is set, else ignored.
   * @see #vtype
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#cfg-vtypeText Original Ext JS documentation of 'vtypeText'
   */
  public native function get vtypeText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set vtypeText(value:String):void;

  /**
   * Automatically grows the field to accommodate the width of the text up to the maximum
   * field width allowed. This only takes effect if →<code>grow</code> = true, and fires the
   * →<code>event:onAutoSize</code> event if the width changes.
   * @see #grow
   * @see #event:onAutoSize
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#method-autoSize Original Ext JS documentation of 'autoSize'
   */
  public native function autoSize():void;

  /**
   * Returns the value of <code>autoHideInputMask</code>.
   * @see #autoHideInputMask
   */
  public native function getAutoHideInputMask():Boolean;

  /**
   * Returns the value of <code>childEls</code>.
   * @see #childEls
   */
  override public native function getChildEls():*;

  /**
   * Returns the value of <code>editable</code>.
   * @see #editable
   */
  public native function getEditable():Boolean;

  /**
   * Returns the value of this field's →<code>emptyText</code>
   * @return The value of this field's emptyText
   * @see #emptyText
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#method-getEmptyText Original Ext JS documentation of 'getEmptyText'
   */
  public native function getEmptyText():String;

  [ArrayElementType("String")]
  /**
   * Validates a value according to the field's validation rules and returns an array of errors
   * for any failing validations. Validation rules are processed in the following order:
   * <ol>
   * <li>
   * <p><b>Field specific validator</b></p>
   * <p>A validator offers a way to customize and reuse a validation specification.
   * If a field is configured with a <code>→validator</code>
   * function, it will be passed the current field value. The <code>→validator</code>
   * function is expected to return either:</p>
   * <ul>
   * <li>Boolean <code>true</code> if the value is valid (validation continues).</li>
   * <li>a String to represent the invalid message if invalid (validation halts).</li>
   * </ul>
   * </li>
   * <li>
   * <p><b>Basic Validation</b></p>
   * <p>If the <code>→validator</code> has not halted validation,
   * basic validation proceeds as follows:</p>
   * <ul>
   * <li>
   * <p><code>→allowBlank</code> : (Invalid message = <code>→blankText</code>)</p>
   * <p>Depending on the configuration of <code>→allowBlank</code>, a
   * blank field will cause validation to halt at this step and return
   * Boolean true or false accordingly.</p>
   * </li>
   * <li>
   * <p><code>→minLength</code> : (Invalid message = <code>→minLengthText</code>)</p>
   * <p>If the passed value does not satisfy the <code>→minLength</code>
   * specified, validation halts.</p>
   * </li>
   * <li>
   * <p><code>→maxLength</code> : (Invalid message = <code>→maxLengthText</code>)</p>
   * <p>If the passed value does not satisfy the <code>→maxLength</code>
   * specified, validation halts.</p>
   * </li>
   * </ul>
   * </li>
   * <li>
   * <p><b>Preconfigured Validation Types (VTypes)</b></p>
   * <p>If none of the prior validation steps halts validation, a field
   * configured with a <code>→vtype</code> will utilize the
   * corresponding →<code>ext.form.field.VTypes</code> validation function.
   * If invalid, either the field's <code>→vtypeText</code> or
   * the VTypes vtype Text property will be used for the invalid message.
   * Keystrokes on the field will be filtered according to the VTypes
   * vtype Mask property.</p>
   * </li>
   * <li>
   * <p><b>Field specific regex test</b></p>
   * <p>If none of the prior validation steps halts validation, a field's
   * configured <code>→regex</code> test will be processed.
   * The invalid message for this test is configured with <code>→regexText</code></p>
   * </li>
   * </ol>
   * @param value The value to validate. The processed raw value will be used
   * if nothing is passed.
   * @return Array of any validation errors
   * @see #validator
   * @see #allowBlank
   * @see #blankText
   * @see #minLength
   * @see #minLengthText
   * @see #maxLength
   * @see #maxLengthText
   * @see #vtype
   * @see ext.form.field.#VTypes ext.form.field.VTypes
   * @see #vtypeText
   * @see #regex
   * @see #regexText
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#method-getErrors Original Ext JS documentation of 'getErrors'
   */
  override public native function getErrors(value:*):Array;

  /**
   * Returns the value of <code>hideTrigger</code>.
   * @see #hideTrigger
   */
  public native function getHideTrigger():Boolean;

  /**
   * Returns the value of <code>inputMask</code>.
   * @see #inputMask
   */
  public native function getInputMask():*;

  /**
   * Returns the selection range of an input element as an array of three values...
   * <pre>
   *  [ start, end, direction ]
   * </pre>
   * <p>These have the same meaning as the parameters to <code>→selectText()</code>.</p>
   * @since 6.5.0
   * @see #selectText()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#method-getTextSelection Original Ext JS documentation of 'getTextSelection'
   */
  public native function getTextSelection():Array;

  /**
   * Returns the trigger with the given id
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#method-getTrigger Original Ext JS documentation of 'getTrigger'
   */
  public native function getTrigger(id:String):Trigger;

  [Deprecated(message="This method was removed.\n", since="5.0")]
  /**
   * Get the total width of the trigger button area.
   * @return The total trigger width
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#method-getTriggerWidth Original Ext JS documentation of 'getTriggerWidth'
   */
  public native function getTriggerWidth():Number;

  /**
   * Returns the value of <code>triggers</code>.
   * @see #triggers
   */
  public native function getTriggers():Object;

  /**
   * Performs any necessary manipulation of a raw String value to prepare it for conversion and/or
   * <i>validation</i> (→<code>validate()</code>). For text fields this applies the configured
   * →<code>stripCharsRe</code> to the raw value.
   * @param value The unprocessed string value
   * @return The processed string value
   * @see #validate()
   * @see #stripCharsRe
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#method-processRawValue Original Ext JS documentation of 'processRawValue'
   */
  override public native function processRawValue(value:Object):Object;

  /**
   * Select the specified contents of the input field (all by default).
   * @param start
   * @default 0
   * @param end
   * @param direction Pass "f" for forward,
   * "b" for backwards.
   * @default "f"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#method-selectText Original Ext JS documentation of 'selectText'
   */
  public native function selectText(start:Number = 0, end:Number = NaN, direction:String = "f"):void;

  /**
   * Sets the value of <code>autoHideInputMask</code>.
   * @param autoHideInputMask The new value.
   * @see #autoHideInputMask
   */
  public native function setAutoHideInputMask(autoHideInputMask:Boolean):void;

  /**
   * Sets the value of <code>childEls</code>.
   * @param childEls The new value.
   * @see #childEls
   */
  override public native function setChildEls(childEls:*):void;

  /**
   * Sets the value of <code>editable</code>.
   * @param editable The new value.
   * @see #editable
   */
  public native function setEditable(editable:Boolean):void;

  /**
   * Sets the default text to place into an empty field
   * @param value The →<code>emptyText</code> value for this field
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#method-setEmptyText Original Ext JS documentation of 'setEmptyText'
   * @see #emptyText
   */
  public native function setEmptyText(value:String):TextField;

  /**
   * Sets the value of <code>hideTrigger</code>.
   * @param hideTrigger The new value.
   * @see #hideTrigger
   */
  public native function setHideTrigger(hideTrigger:Boolean):void;

  /**
   * Sets the value of <code>inputMask</code>.
   * @param inputMask The new value.
   * @see #inputMask
   */
  public native function setInputMask(inputMask:*):void;

  /**
   * Sets the read-only state of this field.
   * @param readOnly True to prevent the user changing the field and explicitly
   * hide the trigger(s). Setting this to true will supersede settings editable and
   * hideTrigger. Setting this to false will defer back to →<code>editable</code> and
   * →<code>hideTrigger</code>.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.form.field.Text.html#method-setReadOnly Original Ext JS documentation of 'setReadOnly'
   * @see #editable
   * @see #hideTrigger
   */
  override public native function setReadOnly(readOnly:Boolean):void;

  /**
   * Sets the value of <code>triggers</code>.
   * @param triggers The new value.
   * @see #triggers
   */
  public native function setTriggers(triggers:Object):void;
}
}