package ext.grid.column {
import ext.dom.Element;
import ext.form.field.IField;
import ext.grid.header.HeaderContainer;

[Native("Ext.grid.column.Column", require)]
/**
 * This class specifies the definition for a column inside a →<code>ext.grid.GridPanel</code>. It encompasses
 * both the grid header configuration as well as displaying data within the grid itself. If the
 * →<code>columns</code> configuration is specified, this column will become a column group and can
 * contain other columns inside. In general, this class will not be created directly, rather
 * an array of column configurations will be passed to the grid:
 * <p><b>Example:</b></p>
 * <pre>
 * Ext.create('Ext.data.Store', {
 *     storeId:'employeeStore',
 *     fields:['firstname', 'lastname', 'seniority', 'dep', 'hired'],
 *     data:[
 *         {firstname:"Michael", lastname:"Scott", seniority:7, dep:"Management", hired:"01/10/2004"},
 *         {firstname:"Dwight", lastname:"Schrute", seniority:2, dep:"Sales", hired:"04/01/2004"},
 *         {firstname:"Jim", lastname:"Halpert", seniority:3, dep:"Sales", hired:"02/22/2006"},
 *         {firstname:"Kevin", lastname:"Malone", seniority:4, dep:"Accounting", hired:"06/10/2007"},
 *         {firstname:"Angela", lastname:"Martin", seniority:5, dep:"Accounting", hired:"10/21/2008"}
 *     ]
 * });
 *
 * Ext.create('Ext.grid.Panel', {
 *     title: 'Column Demo',
 *     store: Ext.data.StoreManager.lookup('employeeStore'),
 *     columns: [
 *         {text: 'First Name',  dataIndex:'firstname'},
 *         {text: 'Last Name',  dataIndex:'lastname'},
 *         {text: 'Hired Month',  dataIndex:'hired', xtype:'datecolumn', format:'M'},
 *         {text: 'Department (Yrs)', xtype:'templatecolumn', tpl:'{dep} ({seniority})'}
 *     ],
 *     width: 400,
 *     forceFit: true,
 *     renderTo: Ext.getBody()
 * });
 * </pre>
 * <p><b>Convenience Subclasses</b></p>
 * <p>There are several column subclasses that provide default rendering for various data types</p>
 * <ul>
 * <li>→<code>ext.grid.column.ActionColumn</code>: Renders icons that can respond to click events inline</li>
 * <li>→<code>ext.grid.column.BooleanColumn</code>: Renders for boolean values</li>
 * <li>→<code>ext.grid.column.DateColumn</code>: Renders for date values</li>
 * <li>→<code>ext.grid.column.NumberColumn</code>: Renders for numeric values</li>
 * <li>→<code>ext.grid.column.TemplateColumn</code>: Renders a value using an →<code>ext.XTemplate</code> using the record data</li>
 * </ul>
 * <p><b>Setting Widths</b></p>
 * <p>The columns are laid out by a →<code>ext.layout.container.HBoxLayout</code> layout, so a column can either
 * be given an explicit width value or a →<code>flex</code> configuration. If no width is specified the grid will
 * automatically the size the column to 100px.</p>
 * <p>Group columns (columns with <i>child columns</i> (→<code>columns</code>)) may be sized using →<code>flex</code>,
 * in which case they will apply <code>forceFit</code> to their child columns so as not to leave blank space.</p>
 * <p>If a group column is not flexed, its width is calculated by measuring the width of the
 * child columns, so a width option should not be specified in that case.</p>
 * <p><b>Header Options</b></p>
 * <ul>
 * <li>→<code>text</code>: Sets the header text for the column</li>
 * <li>→<code>sortable</code>: Specifies whether the column can be sorted by clicking the header or using the column menu</li>
 * <li>→<code>hideable</code>: Specifies whether the column can be hidden using the column menu</li>
 * <li>→<code>menuDisabled</code>: Disables the column header menu</li>
 * <li>→<code>draggable</code>: Specifies whether the column header can be reordered by dragging</li>
 * <li>→<code>groupable</code>: Specifies whether the grid can be grouped by the column dataIndex. See also →<code>ext.grid.feature.Grouping</code></li>
 * </ul>
 * <p><b>Data Options</b></p>
 * <ul>
 * <li>→<code>dataIndex</code>: The dataIndex is the field in the underlying →<code>ext.data.Store</code> to use as the value for the column.</li>
 * <li>→<code>ext.grid.column.Column.renderer</code>: Allows the underlying store
 * value to be transformed before being displayed in the grid</li>
 * </ul>
 * <p><b><i>State saving</i></b></p>
 * <p>When the owning <i>Grid</i> (→<code>ext.grid.GridPanel</code>) is configured
 * →<code>ext.grid.GridPanel.stateful</code>, it will save its column state (order and width)
 * encapsulated within the default Panel state of changed width and height and
 * collapsed/expanded state.</p>
 * <p>On a <code>→stateful</code> grid, not only should the Grid have a
 * →<code>ext.grid.GridPanel.stateId</code>, each column of the grid should also be configured
 * with a →<code>stateId</code> which identifies that column locally within the grid.</p>
 * <p>Omitting the <code>→stateId</code> config from the columns results in columns with generated
 * internal ID's. The generated ID's are subject to change on each page load
 * making it impossible for the state manager to restore the previous state of the
 * columns.</p>
 * @see ext.grid.GridPanel
 * @see #columns
 * @see ext.grid.column.ActionColumn
 * @see ext.grid.column.BooleanColumn
 * @see ext.grid.column.DateColumn
 * @see ext.grid.column.NumberColumn
 * @see ext.grid.column.TemplateColumn
 * @see ext.XTemplate
 * @see ext.layout.container.HBoxLayout
 * @see #flex
 * @see #text
 * @see #sortable
 * @see #hideable
 * @see #menuDisabled
 * @see #draggable
 * @see #groupable
 * @see ext.grid.feature.Grouping
 * @see #dataIndex
 * @see ext.data.Store
 * @see ext.grid.column.Column#renderer
 * @see ext.grid.GridPanel#stateful
 * @see #stateful
 * @see ext.grid.GridPanel#stateId
 * @see #stateId
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html Original Ext JS documentation of 'Ext.grid.column.Column'
 */
public class Column extends HeaderContainer {
  /**
   * @param config @inheritDoc
   */
  public function Column(config:Column = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * Sets the alignment of the header and rendered columns.
   * Possible values are: <code>'start'</code>, <code>'center'</code>, and <code>'end'</code>.
   * <p>Since 6.2.0, <code>'left'</code> and <code>'right'</code> will still work, but retain their meaning
   * even when the application is in RTL mode.</p>
   * <p><code>'start'</code> and <code>'end'</code> always conform to the locale's text direction.</p>
   * @default 'start'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-align Original Ext JS documentation of 'align'
   * @see #getAlign()
   * @see #setAlign()
   */
  public native function get align():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set align(value:String):void;

  [ExtConfig]
  /**
   * Configure as <code>false</code> to remove all cells in this column from navigation.
   * <p>This is currently used by the PivotGrid package to create columns which have
   * no semantic role, but are purely for visual indentation purposes.</p>
   * @default true
   * @since 6.2.0.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-cellFocusable Original Ext JS documentation of 'cellFocusable'
   */
  public native function get cellFocusable():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set cellFocusable(value:Boolean):void;

  [ExtConfig]
  /**
   * True to allow whitespace in this column's cells to wrap, and cause taller column height where
   * necessary.
   * <p>This implicitly sets the →<code>variableRowHeight</code> config to <code>true</code></p>
   * @default false
   * @see #variableRowHeight
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-cellWrap Original Ext JS documentation of 'cellWrap'
   */
  public native function get cellWrap():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set cellWrap(value:Boolean):void;

  [ExtConfig]
  /**
   * An optional array of sub-column definitions. This column becomes a group, and houses
   * the columns defined in the <code>→columns</code> config.
   * <p>Group columns may not be sortable. But they may be hideable and moveable. And you may move
   * headers into and out of a group. Note that if all sub columns are dragged out of a group,
   * the group is destroyed.</p>
   * @see #columns
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-columns Original Ext JS documentation of 'columns'
   */
  public native function get columns():Array;

  [ExtConfig]
  /**
   * @private
   */
  public native function set columns(value:Array):void;

  [ExtConfig]
  /**
   * The name of the field in the grid's →<code>ext.data.Store</code>'s →<code>ext.data.Model</code>
   * definition from which to draw the column's value. <b>Required.</b>
   * @see ext.data.Store
   * @see ext.data.Model
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-dataIndex Original Ext JS documentation of 'dataIndex'
   */
  public native function get dataIndex():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set dataIndex(value:String):void;

  [ExtConfig]
  /**
   * This text will be announced by Assistive Technologies such as screen readers when
   * a cell with changed ("dirty") value is focused.
   * @default "Cell value has been edited"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-dirtyText Original Ext JS documentation of 'dirtyText'
   */
  public native function get dirtyText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set dirtyText(value:String):void;

  [ExtConfig]
  /**
   * False to disable drag-drop reordering of this column.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-draggable Original Ext JS documentation of 'draggable'
   */
  override public native function get draggable():*;

  [ExtConfig]
  /**
   * @private
   */
  override public native function set draggable(value:*):void;

  [ExtConfig]
  /**
   * A renderer to be used in conjunction with
   * →<code>ext.grid.plugin.RowEditingPlugin</code>. This renderer is used to display a
   * custom value for non-editable fields.
   * <p><b>Note:</b> The editRenderer is called when the roweditor is initially shown.
   * Changes to the record during editing will not call editRenderer.</p>
   * <pre>
   * var store = Ext.create('Ext.data.Store', {
   *     fields: ['name', 'email'],
   *     data: [{
   *         "name": "Finn",
   *         "email": "finn&#64;adventuretime.com"
   *     }, {
   *         "name": "Jake",
   *         "email": "jake&#64;adventuretime.com"
   *     }]
   * });
   *
   * Ext.create('Ext.grid.Panel', {
   *     title: 'Land Of Ooo',
   *     store: store,
   *     columns: [{
   *         text: 'Name',
   *         dataIndex: 'name',
   *         editRenderer: function(value){
   *             return '&lt;span style="color:gray;"&gt;' + value + '&lt;/span&gt;';
   *           }
   *     }, {
   *         text: 'Email',
   *         dataIndex: 'email',
   *         flex: 1,
   *         editor: {
   *             xtype: 'textfield',
   *             allowBlank: false
   *         }
   *     }],
   *     plugins: {
   *         rowediting: {
   *             clicksToEdit: 1
   *         }
   *     },
   *     height: 200,
   *     width: 400,
   *     renderTo: document.body
   * });
   * </pre>
   * <ul>
   * <li><code>value:Object</code> —
   * The data value for the current cell
   * <pre>
   * editRenderer: function(value){
   *     // evaluates `value` to append either `person' or `people`
   *     return Ext.util.Format.plural(value, 'person', 'people');
   * }
   * </pre>
   * </li>
   * <li><code>metaData:Object</code> —
   * <b>Note:</b> The metadata param is passed to the
   * editRenderer, but is not used.
   * </li>
   * <li><code>record:ext.data.Model</code> —
   * The record for the current row
   * <pre>
   * editRenderer: function (value, metaData, record) {
   *     // evaluate the record's `updated` field and if truthy return the value
   *     // from the `newVal` field, else return value
   *     var updated = record.get('updated');
   *     return updated ? record.get('newVal') : value;
   * }
   * </pre>
   * </li>
   * <li><code>rowIndex:Number</code> —
   * The index of the current row
   * <pre>
   * editRenderer: function (value, metaData, record, rowIndex) {
   *     // style the value differently for even / odd values
   *     var odd = (rowIndex % 2 === 0),
   *         color = (odd ? 'gray' : 'red');
   *     return '&lt;span style="color:' + color + ';"&gt;' + value + '&lt;/span&gt;';
   * }
   * </pre>
   * </li>
   * <li><code>colIndex:Number</code> —
   * The index of the current column
   * </li>
   * <li><code>store:ext.data.Store</code> —
   * The data store
   * <pre>
   * editRenderer: function (value, metaData, record, rowIndex, colIndex, store) {
   *     // style the cell differently depending on how the value relates to the
   *     // average of all values
   *     var average = store.average('grades'),
   *         status = (value &lt; average) ? 'needsImprovement' : 'satisfactory';
   *     return '&lt;span class="' + status + '"&gt;' + value + '&lt;/span&gt;';
   * }
   * </pre>
   * </li>
   * <li><code>view:ext.view.DataView</code> —
   * The data view
   * <pre>
   * editRenderer: function (value, metaData, record, rowIndex, colIndex, store, view) {
   *     // style the value using the dataIndex of the column
   *     var headerCt = this.getHeaderContainer(),
   *         column = headerCt.getHeaderAtIndex(colIndex);
   *
   *     return '&lt;span class="app-' + column.dataIndex + '"&gt;' + value + '&lt;/span&gt;';
   * }
   * </pre>
   * </li>
   * <li><code>null:String</code> —
   * The HTML string to be rendered.
   * </li>
   * </ul>
   * @see ext.grid.plugin.RowEditingPlugin
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-editRenderer Original Ext JS documentation of 'editRenderer'
   */
  public native function get editRenderer():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set editRenderer(value:*):void;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * An optional xtype or config object for a →<code>ext.form.field.Field</code> to use
   * for editing.
   * Only applicable if the grid is using an →<code>ext.grid.plugin.GridEditingPlugin</code> plugin.
   * <p><b>Note:</b> The →<code>ext.form.field.HtmlEditor</code> field is not a
   * supported editor field type.</p>
   * @see ext.form.field.Field
   * @see ext.grid.plugin.GridEditingPlugin
   * @see ext.form.field.HtmlEditor
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-editor Original Ext JS documentation of 'editor'
   * @see #getEditor()
   * @see #setEditor()
   */
  public native function get editor():IField;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * @private
   */
  public native function set editor(value:IField):void;

  [ExtConfig]
  /**
   * The text to display in empty cells (cells with a value of <code>undefined</code>, <code>null</code>, or <code>''</code>).
   * <p>Defaults to <code>&amp;#160;</code> aka <code>&amp;nbsp;</code>.</p>
   * @default undefined
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-emptyCellText Original Ext JS documentation of 'emptyCellText'
   */
  public native function get emptyCellText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set emptyCellText(value:String):void;

  [ExtConfig]
  /**
   * True to enable text selection inside grid cells in this column.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-enableTextSelection Original Ext JS documentation of 'enableTextSelection'
   */
  public native function get enableTextSelection():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set enableTextSelection(value:Boolean):void;

  [ExtConfig]
  /**
   * During data export via the →<code>null</code> plugin the data for
   * this column could be formatted in multiple ways...
   * <ul>
   * <li>using the <code>exportStyle.format</code></li>
   * <li>using the <code>→formatter</code> if no <code>→exportStyle</code> is defined</li>
   * <li>using the <code>→exportRenderer</code></li>
   * </ul>
   * <p>If you want to use the <code>→renderer</code> defined on this column then set <code>→exportRenderer</code>
   * to <code>true</code>. Beware that this should only happen if the <code>→renderer</code> deals only with
   * data on the record or value and it does NOT style the cell or returns an html
   * string.</p>
   * <pre>
   *  {
   *      xtype: 'numbercolumn',
   *      dataIndex: 'price',
   *      text: 'Price',
   *      renderer: function (value, metaData, record, rowIndex, colIndex, store, view) {
   *          return Ext.util.Format.currency(value);
   *      },
   *      exportRenderer: true
   *  }
   * </pre>
   * <p>If you don't want to use the <code>→renderer</code> during export but you still want to format
   * the value in a special way then you can provide a function to <code>→exportRenderer</code> or
   * a string (which is a function name on the ViewController).
   * The provided function has the same signature as the renderer.</p>
   * <pre>
   *  {
   *      xtype: 'numbercolumn',
   *      dataIndex: 'price',
   *      text: 'Price',
   *      exportRenderer: function (value, metaData, record, rowIndex, colIndex, store, view){
   *          return Ext.util.Format.currency(value);
   *      }
   *  }
   *
   *
   *  {
   *      xtype: 'numbercolumn',
   *      dataIndex: 'price',
   *      text: 'Price',
   *      exportRenderer: 'exportAsCurrency' // this is a function on the ViewController
   *  }
   * </pre>
   * <p>If <code>exportStyle.format</code>, <code>→formatter</code> and <code>→exportRenderer</code> are all defined on the
   * column then the <code>→exportStyle</code> wins and will be used to format the data for this
   * column.</p>
   * @default false
   * @see null
   * @see #formatter
   * @see #exportStyle
   * @see #exportRenderer
   * @see #renderer
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-exportRenderer Original Ext JS documentation of 'exportRenderer'
   */
  public native function get exportRenderer():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set exportRenderer(value:*):void;

  [ExtConfig]
  /**
   * A style definition that is used during data export via the →<code>null</code>.
   * This style will be applied to the columns generated in the exported file.
   * <p>You could define it as a single object that will be used by all exporters:</p>
   * <pre>
   *  {
   *      xtype: 'numbercolumn',
   *      dataIndex: 'price',
   *      text: 'Price',
   *      exportStyle: {
   *          format: 'Currency',
   *          alignment: {
   *              horizontal: 'Right'
   *          },
   *          font: {
   *              italic: true
   *          }
   *      }
   *  }
   * </pre>
   * <p>You could also define it as an array of objects, each object having a <code>type</code>
   * that specifies by which exporter will be used:</p>
   * <pre>
   *  {
   *      xtype: 'numbercolumn',
   *      dataIndex: 'price',
   *      text: 'Price',
   *      exportStyle: [{
   *          type: 'html', // used by the `→<code>html</code>` exporter
   *          format: 'Currency',
   *          alignment: {
   *              horizontal: 'Right'
   *          },
   *          font: {
   *              italic: true
   *          }
   *      },{
   *          type: 'csv', // used by the `csv` exporter
   *          format: 'General'
   *      }]
   *  }
   * </pre>
   * <p>Or you can define it as an array of objects that has:</p>
   * <ul>
   * <li>
   * <p>one object with no <code>type</code> key that is considered the style to use by all exporters</p>
   * </li>
   * <li>
   * <p>objects with the <code>type</code> key defined that are exceptions of the above rule</p>
   * <p>{
   * xtype: 'numbercolumn',
   * dataIndex: 'price',
   * text: 'Price',
   * exportStyle: [{
   * // no type defined means this is the default
   * format: 'Currency',
   * alignment: {
   * horizontal: 'Right'
   * },
   * font: {
   * italic: true
   * }
   * },{
   * type: 'csv', // only the CSV exporter has a special style
   * format: 'General'
   * }]
   * }</p>
   * </li>
   * </ul>
   * @default null
   * @see null
   * @see #html
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-exportStyle Original Ext JS documentation of 'exportStyle'
   */
  public native function get exportStyle():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set exportStyle(value:*):void;

  [ExtConfig]
  /**
   * This config is similar to →<code>exportRenderer</code> but is applied to summary
   * records.
   * @default false
   * @see #exportRenderer
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-exportSummaryRenderer Original Ext JS documentation of 'exportSummaryRenderer'
   */
  public native function get exportSummaryRenderer():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set exportSummaryRenderer(value:*):void;

  [Deprecated(replacement="editor", since="4.0.5")]
  [ExtConfig]
  /**
   * Alias for →<code>editor</code>.
   * @see #editor
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-field Original Ext JS documentation of 'field'
   */
  public native function get field():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set field(value:*):void;

  [Deprecated(replacement="resizable", since="4.0")]
  [ExtConfig]
  /**
   * True to prevent the column from being resizable.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-fixed Original Ext JS documentation of 'fixed'
   */
  override public native function get fixed():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  override public native function set fixed(value:Boolean):void;

  [ExtConfig]
  /**
   * This config accepts a format specification as would be used in a <code>→ext.Template</code>
   * formatted token. For example <code>'round(2)'</code> to round numbers to 2 decimal places
   * or <code>'date("Y-m-d")'</code> to format a Date.
   * <p>In previous releases the <code>→renderer</code> config had limited abilities to use one
   * of the <code>→ext.util.Format</code> methods but <code>→formatter</code> now replaces that usage and
   * can also handle formatting parameters.</p>
   * <p>When the value begins with <code>"this."</code> (for example, <code>"this.foo(2)"</code>), the
   * implied scope on which "foo" is found is the <code>→scope</code> config for the column.</p>
   * <p>If the <code>→scope</code> is not given, or implied using a prefix of <code>"this"</code>, then either the
   * <i>ViewController</i> (→<code>getController()</code>) or the closest ancestor component
   * configured as →<code>defaultListenerScope</code> is assumed to be the object
   * with the method.</p>
   * @since 5.0.0
   * @see ext.Template
   * @see #renderer
   * @see ext.util.#Format ext.util.Format
   * @see #formatter
   * @see #scope
   * @see #getController()
   * @see #defaultListenerScope
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-formatter Original Ext JS documentation of 'formatter'
   */
  public native function get formatter():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set formatter(value:String):void;

  [ExtConfig]
  /**
   * If the grid uses a →<code>ext.grid.feature.Grouping</code>, this option may be used to disable
   * the header menu item to group by the column selected. By default, the header menu group
   * option is enabled. Set to false to disable (but still show) the group option
   * in the header menu for the column.
   * @see ext.grid.feature.Grouping
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-groupable Original Ext JS documentation of 'groupable'
   */
  public native function get groupable():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set groupable(value:Boolean):void;

  [Deprecated(replacement="text", since="4.0")]
  [ExtConfig]
  /**
   * The header text.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-header Original Ext JS documentation of 'header'
   */
  public native function get header():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set header(value:String):void;

  [ExtConfig]
  /**
   * False to prevent the user from hiding this column.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-hideable Original Ext JS documentation of 'hideable'
   */
  public native function get hideable():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set hideable(value:Boolean):void;

  [ExtConfig]
  /**
   * This flag indicates that this column will be ignored when grid data is exported.
   * <p>When grid data is exported you may want to export only some columns that are important
   * and not everything. Widget, check and action columns are not relevant when data is
   * exported. You can set this flag on any column that you want to be ignored during export.</p>
   * <p>This is used by <i>clipboard plugin</i> (→<code>ext.grid.plugin.GridClipboardPlugin</code>) and
   * <i>exporter plugin</i> (→<code>null</code>).</p>
   * @default false
   * @see ext.grid.plugin.GridClipboardPlugin
   * @see null
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-ignoreExport Original Ext JS documentation of 'ignoreExport'
   */
  public native function get ignoreExport():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set ignoreExport(value:Boolean):void;

  /**
   * Set in this class to identify, at runtime, instances which are not instances of the
   * HeaderContainer base class, but are in fact simple column headers.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#property-isColumn Original Ext JS documentation of 'isColumn'
   */
  public native function get isColumn():Boolean;

  [Deprecated(message="see isColumn\nSet in this class to identify, at runtime, instances which are not instances of the\nHeaderContainer base class, but are in fact, the subclass: Header.\n", since="6.5.0")]
  /**
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#property-isHeader Original Ext JS documentation of 'isHeader'
   */
  public native function get isHeader():Boolean;

  /**
   * @private
   */
  public native function set isHeader(value:Boolean):void;

  [ExtConfig]
  /**
   * If the grid is configured with →<code>ext.panel.TablePanel.enableLocking</code>, or has
   * columns which are configured with a →<code>locked</code> value, this option may be used to disable
   * user-driven locking or unlocking of this column. This column will remain in the side
   * into which its own →<code>locked</code> configuration placed it.
   * @see ext.panel.TablePanel#enableLocking
   * @see #locked
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-lockable Original Ext JS documentation of 'lockable'
   */
  public native function get lockable():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set lockable(value:Boolean):void;

  [ExtConfig]
  /**
   * True to lock this column in place. Implicitly enables locking on the grid.
   * See also →<code>ext.grid.GridPanel.enableLocking</code>.
   * @default false
   * @see ext.grid.GridPanel#enableLocking
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-locked Original Ext JS documentation of 'locked'
   */
  public native function get locked():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set locked(value:Boolean):void;

  [ExtConfig]
  /**
   * True to disable the column header menu containing sort/hide options.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-menuDisabled Original Ext JS documentation of 'menuDisabled'
   */
  public native function get menuDisabled():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set menuDisabled(value:Boolean):void;

  [ExtConfig]
  /**
   * The text to render in the column visibility selection menu for this column. If not
   * specified, will default to the text value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-menuText Original Ext JS documentation of 'menuText'
   */
  public native function get menuText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set menuText(value:String):void;

  [ExtConfig]
  /**
   * This flag indicates that the renderer produces HTML.
   * <p>If this column is going to be updated rapidly, and the
   * →<code>ext.grid.column.Column.renderer</code> or →<code>updater</code> only produces
   * text, then to avoid the expense of HTML parsing and element production during the
   * update, this property may be configured as <code>false</code>.</p>
   * @default true
   * @see ext.grid.column.Column#renderer
   * @see #updater
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-producesHTML Original Ext JS documentation of 'producesHTML'
   */
  public native function get producesHTML():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set producesHTML(value:Boolean):void;

  [ExtConfig]
  /**
   * A renderer is an 'interceptor' method which can be used to transform data (value,
   * appearance, etc.) before it is rendered. Example:
   * <p><b>NOTE:</b> In previous releases, a string was treated as a method on
   * <code>→ext.util.Format</code> but that is now handled by the →<code>formatter</code> config.</p>
   * @default false
   * <ul>
   * <li><code>value:Object</code> —
   * The data value for the current cell
   * <pre>
   * renderer: function(value){
   *     // evaluates `value` to append either `person' or `people`
   *     return Ext.util.Format.plural(value, 'person', 'people');
   * }
   * </pre>
   * </li>
   * <li><code>metaData:Object</code> —
   * A collection of metadata about the current cell; can be
   * used or modified by the renderer. Recognized properties are: <code>→tdCls</code>, <code>tdAttr</code>,
   * and <code>tdStyle</code>.
   * <p>To add style attributes to the <code>&amp;lt;td&gt;</code> element, you must use the <code>tdStyle</code>
   * property. Using a style attribute in the <code>tdAttr</code> property will override the
   * styles the column sets, such as the width which will break the rendering.</p>
   * <p>You can see an example of using the metaData parameter below.</p>
   * <pre>
   *  Ext.create('Ext.data.Store', {
   *       storeId: 'simpsonsStore',
   *       fields: ['class', 'attr', 'style'],
   *       data: {
   *           'class': 'red-bg',
   *           'attr': 'lightyellow',
   *           'style': 'red'
   *       }
   *  });
   *
   *  Ext.create('Ext.grid.Panel', {
   *       title: 'Simpsons',
   *       store: Ext.data.StoreManager.lookup('simpsonsStore'),
   *       columns: [{
   *           text: 'Name',
   *           dataIndex: 'class',
   *           renderer: function (value, metaData) {
   *               metaData.tdCls = value;
   *               return value;
   *           }
   *       }, {
   *           text: 'Email',
   *           dataIndex: 'attr',
   *           flex: 1,
   *           renderer: function (value, metaData) {
   *               metaData.tdAttr = 'bgcolor="' + value + '"';
   *               return value;
   *           }
   *       }, {
   *           text: 'Phone',
   *           dataIndex: 'style',
   *           renderer: function (value, metaData) {
   *               metaData.tdStyle = 'color:' + value;
   *               return value;
   *           }
   *       }],
   *       height: 200,
   *       width: 400,
   *       renderTo: Ext.getBody()
   *   });
   * </pre>
   * </li>
   * <li><code>record:ext.data.Model</code> —
   * The record for the current row
   * <pre>
   * renderer: function (value, metaData, record) {
   *     // evaluate the record's `updated` field and if truthy return the value
   *     // from the `newVal` field, else return value
   *     var updated = record.get('updated');
   *     return updated ? record.get('newVal') : value;
   * }
   * </pre>
   * </li>
   * <li><code>rowIndex:Number</code> —
   * The index of the current row
   * <pre>
   * renderer: function (value, metaData, record, rowIndex) {
   *     // style the cell differently for even / odd values
   *     var odd = (rowIndex % 2 === 0);
   *     metaData.tdStyle = 'color:' + (odd ? 'gray' : 'red');
   * }
   * </pre>
   * </li>
   * <li><code>colIndex:Number</code> —
   * The index of the current column
   * <pre>
   * var myRenderer = function(value, metaData, record, rowIndex, colIndex) {
   *     if (colIndex === 0) {
   *         metaData.tdAttr = 'data-qtip=' + value;
   *     }
   *     // additional logic to apply to values in all columns
   *     return value;
   * }
   *
   * // using the same renderer on all columns you can process the value for
   * // each column with the same logic and only set a tooltip on the first column
   * renderer: myRenderer
   * </pre>
   * <p><i>See also →<code>ext.tip.QuickTipManager</code></i></p>
   * </li>
   * <li><code>store:ext.data.Store</code> —
   * The data store
   * <pre>
   * renderer: function (value, metaData, record, rowIndex, colIndex, store) {
   *     // style the cell differently depending on how the value relates to the
   *     // average of all values
   *     var average = store.average('grades');
   *     metaData.tdCls = (value &lt; average) ? 'needsImprovement' : 'satisfactory';
   *     return value;
   * }
   * </pre>
   * </li>
   * <li><code>view:ext.view.DataView</code> —
   * The data view
   * <pre>
   * renderer: function (value, metaData, record, rowIndex, colIndex, store, view) {
   *     // style the cell using the dataIndex of the column
   *     var headerCt = this.getHeaderContainer(),
   *         column = headerCt.getHeaderAtIndex(colIndex);
   *
   *     metaData.tdCls = 'app-' + column.dataIndex;
   *     return value;
   * }
   * </pre>
   * </li>
   * <li><code>null:String</code> —
   * The HTML string to be rendered.
   * </li>
   * </ul>
   * @see ext.util.#Format ext.util.Format
   * @see #formatter
   * @see #tdCls
   * @see ext.tip.#QuickTipManager ext.tip.QuickTipManager
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-renderer Original Ext JS documentation of 'renderer'
   */
  public native function get renderer():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set renderer(value:*):void;

  [ExtConfig]
  /**
   * False to prevent the column from being resizable.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-resizable Original Ext JS documentation of 'resizable'
   */
  override public native function get resizable():*;

  [ExtConfig]
  /**
   * @private
   */
  override public native function set resizable(value:*):void;

  [ExtConfig]
  /**
   * The scope to use when calling the
   * →<code>ext.grid.column.Column.renderer</code> function.
   * @see ext.grid.column.Column#renderer
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-scope Original Ext JS documentation of 'scope'
   */
  public native function get scope():Object;

  [ExtConfig]
  /**
   * @private
   */
  public native function set scope(value:Object):void;

  [ExtConfig]
  /**
   * False to disable sorting of this column. Whether local/remote sorting is used is specified in
   * <code>→ext.data.Store.remoteSort</code>.
   * @default true
   * @see ext.data.Store#remoteSort
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-sortable Original Ext JS documentation of 'sortable'
   */
  override public native function get sortable():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  override public native function set sortable(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * A Sorter, or sorter config object to apply when the standard user interface
   * sort gesture is invoked. This is usually clicking this column header, but
   * there are also menu options to sort ascending or descending.
   * <p>Note that a sorter may also be specified as a function which accepts two
   * records to compare.</p>
   * <p>In 6.2.0, a <code>controller (→ext.app.ViewController)</code> method can be used
   * like so:</p>
   * <pre>
   *  sorter: {
   *      sorterFn: 'sorterMethodName'
   *  }
   * </pre>
   * @default null
   * @since 6.0.1
   * @see ext.app.ViewController
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-sorter Original Ext JS documentation of 'sorter'
   * @see #getSorter()
   * @see #setSorter()
   */
  public native function get sorter():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set sorter(value:*):void;

  [ExtConfig]
  /**
   * An identifier which identifies this column uniquely within the owning grid's
   * <i>state</i> (→<code>stateful</code>).
   * <p>This does not have to be <i>globally</i> unique. A column's state is not saved standalone.
   * It is encapsulated within the owning grid's state.</p>
   * @see #stateful
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-stateId Original Ext JS documentation of 'stateId'
   */
  override public native function get stateId():String;

  [ExtConfig]
  /**
   * @private
   */
  override public native function set stateId(value:String):void;

  [ExtConfig]
  /**
   * A renderer to be used in conjunction with the →<code>ext.grid.feature.Summary</code> or
   * →<code>ext.grid.feature.GroupingSummary</code> features. This renderer is used to
   * display a summary value for this column.
   * @see ext.grid.feature.Summary
   * @see ext.grid.feature.GroupingSummary
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-summaryRenderer Original Ext JS documentation of 'summaryRenderer'
   */
  public native function get summaryRenderer():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set summaryRenderer(value:*):void;

  [ExtConfig]
  /**
   * A CSS class names to apply to the table cells for this column.
   * @default ''
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-tdCls Original Ext JS documentation of 'tdCls'
   */
  public native function get tdCls():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set tdCls(value:String):void;

  [ExtConfig]
  /**
   * The header text to be used as innerHTML (html tags are accepted) to display in the Grid.
   * @default '\xa0'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-text Original Ext JS documentation of 'text'
   * @see #setText()
   */
  public native function get text():String;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * @private
   */
  public native function set text(value:String):void;

  /**
   * Element that contains the text in column header.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#property-textEl Original Ext JS documentation of 'textEl'
   */
  public native function get textEl():Element;

  /**
   * @private
   */
  public native function set textEl(value:Element):void;

  [ExtConfig]
  /**
   * A tooltip to display for this column header
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-tooltip Original Ext JS documentation of 'tooltip'
   */
  public native function get tooltip():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set tooltip(value:String):void;

  [ExtConfig]
  /**
   * The type of →<code>tooltip</code> to use. Either 'qtip' for QuickTips or 'title' for title
   * attribute.
   * @default "qtip"
   * @see #tooltip
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-tooltipType Original Ext JS documentation of 'tooltipType'
   */
  public native function get tooltipType():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set tooltipType(value:String):void;

  /**
   * Element that acts as button for column header dropdown menu.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#property-triggerEl Original Ext JS documentation of 'triggerEl'
   */
  public native function get triggerEl():Element;

  /**
   * @private
   */
  public native function set triggerEl(value:Element):void;

  [ExtConfig]
  /**
   * An updater is a method which is used when records are updated, and an <i>existing</i> grid row
   * needs updating. The method is passed the cell element and may manipulate it in any way.
   * <p><b>Note</b>: The updater is required to insert the →<code>emptyCellText</code> if there
   * is no value in the cell.</p>
   * <pre>
   * Ext.create('Ext.grid.Panel', {
   *     title: 'Grades',
   *     store: {
   *         fields: ['originalScore', 'newScore'],
   *         data: [{
   *             originalScore: 70,
   *             newScore: 70
   *         }]
   *     },
   *     columns: [{
   *         text: 'Score',
   *         dataIndex: 'newScore',
   *         editor: 'numberfield',
   *         flex: 1,
   *         updater: function (cell, value, record, view) {
   *             var inner = Ext.get(cell).first(),
   *                 originalScore = record.get('originalScore'),
   *                 color = (value === originalScore)
   *                     ? 'black' : (value &gt; originalScore) ? 'green' : 'red';
   *
   *             // set the color based on the current value
   *             // relative to the originalScore value
   *             // &#42; same   = black
   *             // &#42; higher = green
   *             // &#42; less   = red
   *             inner.applyStyles({
   *                 color: color
   *             });
   *             // pass the value to the cell's inner el
   *             inner.setHtml(value);
   *         }
   *     }],
   *     height: 200,
   *     width: 400,
   *     renderTo: document.body,
   *     tbar: [{
   *         xtype: 'numberfield',
   *         fieldLabel: 'New Score',
   *         value: 70,
   *         listeners: {
   *             change: function (field, newValue) {
   *                 this.up('grid').getStore().first().set('newScore', newValue);
   *             }
   *         }
   *     }]
   * });
   * </pre>
   * <p>If a string is passed it is assumed to be the name of a method defined by the
   * <i>ViewController</i> (→<code>getController()</code>) or an ancestor component configured as
   * →<code>defaultListenerScope</code>.</p>
   * <ul>
   * <li><code>cell:js.HTMLElement</code> (optional) —
   * The HTML cell element to update.
   * </li>
   * <li><code>value:Object</code> (optional) —
   * The data value for the current cell
   * </li>
   * <li><code>record:ext.data.Model</code> (optional) —
   * The record for the current row
   * </li>
   * <li><code>view:ext.view.DataView</code> (optional) —
   * The current view
   * <p><b>Note</b>: The updater is required to insert the →<code>emptyCellText</code> if there is no value
   * in the cell.</p>
   * </li>
   * </ul>
   * @see #emptyCellText
   * @see #getController()
   * @see #defaultListenerScope
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-updater Original Ext JS documentation of 'updater'
   */
  public native function get updater():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set updater(value:*):void;

  [ExtConfig]
  /**
   * True to indicate that data in this column may take on an unpredictable height, possibly
   * differing from row to row.
   * <p>If this is set, then View refreshes, and removal and addition of new rows will result
   * in an ExtJS layout of the grid in order to adjust for possible addition/removal of scrollbars
   * in the case of data changing height.</p>
   * <p>This config also tells the View's buffered renderer that row heights are unpredictable,
   * and must be remeasured as the view is refreshed.</p>
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#cfg-variableRowHeight Original Ext JS documentation of 'variableRowHeight'
   */
  public native function get variableRowHeight():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set variableRowHeight(value:Boolean):void;

  /**
   * Sizes this Column to fit the max content width.
   * <i>Note that group columns shrink-wrap around the size of leaf columns. Auto sizing
   * a group column auto-sizes descendant leaf columns.</i>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#method-autoSize Original Ext JS documentation of 'autoSize'
   */
  public native function autoSize():void;

  /**
   * When defined this will take precedence over the
   * →<code>ext.grid.column.Column.renderer</code> config.
   * This is meant to be defined in subclasses that wish to supply their own renderer.
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @see ext.grid.column.Column#renderer
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#method-defaultRenderer Original Ext JS documentation of 'defaultRenderer'
   */
  protected native function defaultRenderer():void;

  /**
   * Returns the value of <code>align</code>.
   * @see #align
   */
  public native function getAlign():String;

  /**
   * Retrieves the editing field for editing associated with this header. If the
   * field has not been instantiated it will be created.
   * <p><b>Note:</b> This method will only have an implementation if an Editing plugin has
   * been enabled on the grid (<i>cellediting</i> (→<code>ext.grid.plugin.CellEditingPlugin</code>) /
   * <i>rowediting</i> (→<code>ext.grid.plugin.RowEditingPlugin</code>)).</p>
   * @param record The →<code>ext.data.Model</code> instance being edited.
   * @param defaultField An xtype or config object for a
   * →<code>ext.form.field.Field</code> to be created as the default editor if it does
   * not already exist
   * @return The editor field associated with
   * this column. Returns false if there is no field associated with the
   * →<code>ext.grid.column.Column</code>.
   * @see ext.grid.plugin.CellEditingPlugin
   * @see ext.grid.plugin.RowEditingPlugin
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#method-getEditor Original Ext JS documentation of 'getEditor'
   * @see ext.data.Model
   * @see ext.form.field.Field
   * @see ext.grid.column.Column
   */
  public native function getEditor(record:Object = null, defaultField:* = undefined):*;

  /**
   * Returns the index of this column only if this column is a base level Column. If it
   * is a group column, it returns <code>false</code>.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#method-getIndex Original Ext JS documentation of 'getIndex'
   */
  public native function getIndex():Number;

  /**
   * Returns the parameter to sort upon when sorting this header.
   * By default this returns the dataIndex and will not need to be overridden in most cases.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#method-getSortParam Original Ext JS documentation of 'getSortParam'
   */
  public native function getSortParam():String;

  /**
   * Returns the value of <code>sorter</code>.
   * @see #sorter
   */
  public native function getSorter():*;

  /**
   * Returns the index of this column in the list of <i>visible</i> columns only if this column
   * is a base level Column. If it is a group column, it returns <code>false</code>.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#method-getVisibleIndex Original Ext JS documentation of 'getVisibleIndex'
   */
  public native function getVisibleIndex():Number;

  /**
   * Determines whether the UI should be allowed to offer an option to hide this column.
   * <p>A column may <i>not</i> be hidden if to do so would leave the grid with no visible columns.</p>
   * <p>This is used to determine the enabled/disabled state of header hide menu items.</p>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#method-isHideable Original Ext JS documentation of 'isHideable'
   */
  public native function isHideable():void;

  /**
   * Determines whether the UI should be allowed to offer an option to lock or unlock this column.
   * Note that this includes dragging a column into the opposite side of a
   * <i>lockable</i> (→<code>ext.panel.TablePanel.enableLocking</code>) grid.
   * <p>A column may <i>not</i> be moved from one side to the other of a
   * <i>lockable</i> (→<code>ext.panel.TablePanel.enableLocking</code>) grid if to do so would leave one side
   * with no visible columns.</p>
   * <p>This is used to determine the enabled/disabled state of the lock/unlock
   * menu item used in <i>lockable</i> (→<code>ext.panel.TablePanel.enableLocking</code>) grids, and to determine
   * droppabilty when dragging a header.</p>
   * @see ext.panel.TablePanel#enableLocking
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#method-isLockable Original Ext JS documentation of 'isLockable'
   */
  public native function isLockable():void;

  /**
   * Determines whether this column is in the locked side of a grid.
   * It may be a descendant node of a locked column and as such will <i>not</i> have the
   * →<code>locked</code> flag set.
   * @see #locked
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#method-isLocked Original Ext JS documentation of 'isLocked'
   */
  public native function isLocked():void;

  /**
   * Sets the value of <code>align</code>.
   * @param align The new value.
   * @see #align
   */
  public native function setAlign(align:String):void;

  /**
   * Sets the form field to be used for editing.
   * <p><b>Note:</b> This method will only have an implementation if an Editing plugin has
   * been enabled on the grid (<i>cellediting</i> (→<code>ext.grid.plugin.CellEditingPlugin</code>) /
   * <i>rowediting</i> (→<code>ext.grid.plugin.RowEditingPlugin</code>)).</p>
   * @param field An object representing a field to be created.
   * If no xtype is specified a 'textfield' is assumed.
   * @see ext.grid.plugin.CellEditingPlugin
   * @see ext.grid.plugin.RowEditingPlugin
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#method-setEditor Original Ext JS documentation of 'setEditor'
   */
  public native function setEditor(field:IField):void;

  /**
   * Sets the value of <code>sorter</code>.
   * @param sorter The new value.
   * @see #sorter
   */
  public native function setSorter(sorter:*):void;

  /**
   * Sets the header text for this Column.
   * @param text The header to display on this Column.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Column.html#method-setText Original Ext JS documentation of 'setText'
   */
  public native function setText(text:String):void;
}
}